"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Ingestion = void 0;
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_lambda_event_sources_1 = require("@aws-cdk/aws-lambda-event-sources");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const aws_s3_deployment_1 = require("@aws-cdk/aws-s3-deployment");
const aws_sqs_1 = require("@aws-cdk/aws-sqs");
const aws_stepfunctions_1 = require("@aws-cdk/aws-stepfunctions");
const aws_stepfunctions_tasks_1 = require("@aws-cdk/aws-stepfunctions-tasks");
const core_1 = require("@aws-cdk/core");
const config_file_1 = require("../../config-file");
const deep_link_1 = require("../../deep-link");
const runbook_url_1 = require("../../runbook-url");
const storage_1 = require("../../s3/storage");
const _lambda_architecture_1 = require("../_lambda-architecture");
const constants_1 = require("../shared/constants");
const constants_2 = require("./constants");
const ingestion_1 = require("./ingestion");
const re_ingest_1 = require("./re-ingest");
/**
 * The ingestion function receives messages from discovery integrations and
 * processes their payloads into the provided S3 Bucket.
 *
 * This function is also an `IGrantable`, so that it can be granted permissions
 * to read from the source S3 buckets.
 */
class Ingestion extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b, _c, _d;
        super(scope, id);
        this.queueRetentionPeriod = core_1.Duration.days(14);
        this.deadLetterQueue = new aws_sqs_1.Queue(this, 'DLQ', {
            encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
            retentionPeriod: this.queueRetentionPeriod,
            visibilityTimeout: core_1.Duration.minutes(15),
        });
        this.queue = new aws_sqs_1.Queue(this, 'Queue', {
            deadLetterQueue: {
                maxReceiveCount: 5,
                queue: this.deadLetterQueue,
            },
            encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
            retentionPeriod: this.queueRetentionPeriod,
            visibilityTimeout: core_1.Duration.minutes(15),
        });
        const configFilename = 'config.json';
        const config = new config_file_1.ConfigFile(configFilename, JSON.stringify({
            packageLinks: (_a = props.packageLinks) !== null && _a !== void 0 ? _a : [],
            packageTags: (_b = props.packageTags) !== null && _b !== void 0 ? _b : [],
        }));
        const storageFactory = storage_1.S3StorageFactory.getOrCreate(this);
        const configBucket = storageFactory.newBucket(this, 'ConfigBucket', {
            blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
            enforceSSL: true,
            versioned: true,
        });
        new aws_s3_deployment_1.BucketDeployment(this, 'DeployIngestionConfiguration', {
            sources: [aws_s3_deployment_1.Source.asset(config.dir)],
            destinationBucket: configBucket,
        });
        const environment = {
            AWS_EMF_ENVIRONMENT: 'Local',
            BUCKET_NAME: props.bucket.bucketName,
            CONFIG_BUCKET_NAME: configBucket.bucketName,
            CONFIG_FILE_KEY: configFilename,
            STATE_MACHINE_ARN: props.orchestration.stateMachine.stateMachineArn,
        };
        if (props.codeArtifact) {
            environment.CODE_ARTIFACT_REPOSITORY_ENDPOINT = props.codeArtifact.publishingRepositoryNpmEndpoint;
            environment.CODE_ARTIFACT_DOMAIN_NAME = props.codeArtifact.repositoryDomainName;
            environment.CODE_ARTIFACT_DOMAIN_OWNER = props.codeArtifact.repositoryDomainOwner;
        }
        const handler = new ingestion_1.Ingestion(this, 'Default', {
            description: '[ConstructHub/Ingestion] Ingests new package versions into the Construct Hub',
            environment,
            logRetention: (_c = props.logRetention) !== null && _c !== void 0 ? _c : aws_logs_1.RetentionDays.TEN_YEARS,
            memorySize: 10240,
            timeout: core_1.Duration.minutes(15),
            tracing: aws_lambda_1.Tracing.ACTIVE,
        });
        this.function = handler;
        configBucket.grantRead(handler);
        props.bucket.grantWrite(this.function);
        (_d = props.codeArtifact) === null || _d === void 0 ? void 0 : _d.grantPublishToRepository(handler);
        props.orchestration.stateMachine.grantStartExecution(this.function);
        this.function.addEventSource(new aws_lambda_event_sources_1.SqsEventSource(this.queue, { batchSize: 1 }));
        // This event source is disabled, and can be used to re-process dead-letter-queue messages
        this.function.addEventSource(new aws_lambda_event_sources_1.SqsEventSource(this.deadLetterQueue, { batchSize: 1, enabled: false }));
        // Reprocess workflow
        const reprocessQueue = new aws_sqs_1.Queue(this, 'ReprocessQueue', {
            deadLetterQueue: {
                maxReceiveCount: 5,
                queue: this.deadLetterQueue,
            },
            encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
            retentionPeriod: this.queueRetentionPeriod,
            // Visibility timeout of 15 minutes matches the Lambda maximum execution time.
            visibilityTimeout: core_1.Duration.minutes(15),
        });
        props.bucket.grantRead(this.function, `${constants_1.STORAGE_KEY_PREFIX}*${constants_1.PACKAGE_KEY_SUFFIX}`);
        this.function.addEventSource(new aws_lambda_event_sources_1.SqsEventSource(reprocessQueue, { batchSize: 1 }));
        new ReprocessIngestionWorkflow(this, 'ReprocessWorkflow', { bucket: props.bucket, queue: reprocessQueue });
        this.grantPrincipal = this.function.grantPrincipal;
        props.monitoring.addHighSeverityAlarm('Ingestion Dead-Letter Queue not empty', new aws_cloudwatch_1.MathExpression({
            expression: 'm1 + m2',
            usingMetrics: {
                m1: this.deadLetterQueue.metricApproximateNumberOfMessagesVisible({ period: core_1.Duration.minutes(1) }),
                m2: this.deadLetterQueue.metricApproximateNumberOfMessagesNotVisible({ period: core_1.Duration.minutes(1) }),
            },
        }).createAlarm(this, 'DLQAlarm', {
            alarmName: `${this.node.path}/DLQNotEmpty`,
            alarmDescription: [
                'The dead-letter queue for the Ingestion function is not empty!',
                '',
                `RunBook: ${runbook_url_1.RUNBOOK_URL}`,
                '',
                `Direct link to the queue: ${deep_link_1.sqsQueueUrl(this.deadLetterQueue)}`,
                `Direct link to the function: ${deep_link_1.lambdaFunctionUrl(this.function)}`,
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
            // SQS does not emit metrics if the queue has been empty for a while, which is GOOD.
            treatMissingData: aws_cloudwatch_1.TreatMissingData.NOT_BREACHING,
        }));
        props.monitoring.addHighSeverityAlarm('Ingestion failures', this.function.metricErrors().createAlarm(this, 'FailureAlarm', {
            alarmName: `${this.node.path}/Failure`,
            alarmDescription: [
                'The Ingestion function is failing!',
                '',
                `RunBook: ${runbook_url_1.RUNBOOK_URL}`,
                '',
                `Direct link to the function: ${deep_link_1.lambdaFunctionUrl(this.function)}`,
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 2,
            threshold: 1,
            // Lambda only emits metrics when the function is invoked. No invokation => no errors.
            treatMissingData: aws_cloudwatch_1.TreatMissingData.NOT_BREACHING,
        }));
    }
    metricFoundLicenseFile(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "FoundLicenseFile" /* FOUND_LICENSE_FILE */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    metricIneligibleLicense(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "IneligibleLicense" /* INELIGIBLE_LICENSE */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    metricInvalidAssembly(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "InvalidAssembly" /* INVALID_ASSEMBLY */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    metricInvalidTarball(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "InvalidTarball" /* INVALID_TARBALL */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    /**
     * This metrics is the total count of packages that were rejected due to
     * mismatched identity (name, version, license) between the `package.json`
     * file and te `.jsii` attribute.
     */
    metricMismatchedIdentityRejections(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "MismatchedIdentityRejections" /* MISMATCHED_IDENTITY_REJECTIONS */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
}
exports.Ingestion = Ingestion;
/**
 * A StepFunctions State Machine to reprocess every currently indexed package
 * through the ingestion function. This should not be frequently required, but
 * can come in handy at times.
 *
 * For more information, refer to the runbook at
 * https://github.com/cdklabs/construct-hub/blob/main/docs/operator-runbook.md
 */
class ReprocessIngestionWorkflow extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const lambdaFunction = new re_ingest_1.ReIngest(this, 'Function', {
            architecture: _lambda_architecture_1.gravitonLambdaIfAvailable(this),
            description: '[ConstructHub/Ingestion/ReIngest] The function used to reprocess packages through ingestion',
            environment: { BUCKET_NAME: props.bucket.bucketName, QUEUE_URL: props.queue.queueUrl },
            memorySize: 10240,
            tracing: aws_lambda_1.Tracing.ACTIVE,
            timeout: core_1.Duration.minutes(3),
        });
        props.queue.grantSendMessages(lambdaFunction);
        props.bucket.grantRead(lambdaFunction, `${constants_1.STORAGE_KEY_PREFIX}*${constants_1.METADATA_KEY_SUFFIX}`);
        props.bucket.grantRead(lambdaFunction, `${constants_1.STORAGE_KEY_PREFIX}*${constants_1.PACKAGE_KEY_SUFFIX}`);
        // Need to physical-name the state machine so it can self-invoke.
        const stateMachineName = stateMachineNameFrom(this.node.path);
        const listBucket = new aws_stepfunctions_1.Choice(this, 'Has a ContinuationToken?')
            .when(aws_stepfunctions_1.Condition.isPresent('$.ContinuationToken'), new aws_stepfunctions_tasks_1.CallAwsService(this, 'S3.ListObjectsV2(NextPage)', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                ContinuationToken: aws_stepfunctions_1.JsonPath.stringAt('$.ContinuationToken'),
                Prefix: constants_1.STORAGE_KEY_PREFIX,
            },
            resultPath: '$.response',
        }).addRetry({ errors: ['S3.SdkClientException'] }))
            .otherwise(new aws_stepfunctions_tasks_1.CallAwsService(this, 'S3.ListObjectsV2(FirstPage)', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                Prefix: constants_1.STORAGE_KEY_PREFIX,
            },
            resultPath: '$.response',
        }).addRetry({ errors: ['S3.SdkClientException'] })).afterwards();
        const process = new aws_stepfunctions_1.Map(this, 'Process Result', {
            itemsPath: '$.response.Contents',
            resultPath: aws_stepfunctions_1.JsonPath.DISCARD,
        }).iterator(new aws_stepfunctions_1.Choice(this, 'Is metadata object?')
            .when(aws_stepfunctions_1.Condition.stringMatches('$.Key', `*${constants_1.METADATA_KEY_SUFFIX}`), new aws_stepfunctions_tasks_1.LambdaInvoke(this, 'Send for reprocessing', { lambdaFunction })
            // Ample retries here... We should never fail because of throttling....
            .addRetry({ errors: ['Lambda.TooManyRequestsException'], backoffRate: 1.1, interval: core_1.Duration.minutes(1), maxAttempts: 30 }))
            .otherwise(new aws_stepfunctions_1.Succeed(this, 'Nothing to do')));
        listBucket.next(new aws_stepfunctions_1.Choice(this, 'Is there more?')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), new aws_stepfunctions_tasks_1.StepFunctionsStartExecution(this, 'Continue as new', {
            associateWithParent: true,
            stateMachine: aws_stepfunctions_1.StateMachine.fromStateMachineArn(this, 'ThisStateMachine', core_1.Stack.of(this).formatArn({
                arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME,
                service: 'states',
                resource: 'stateMachine',
                resourceName: stateMachineName,
            })),
            input: aws_stepfunctions_1.TaskInput.fromObject({ ContinuationToken: aws_stepfunctions_1.JsonPath.stringAt('$.response.NextContinuationToken') }),
            integrationPattern: aws_stepfunctions_1.IntegrationPattern.REQUEST_RESPONSE,
            resultPath: aws_stepfunctions_1.JsonPath.DISCARD,
        }).addRetry({ errors: ['StepFunctions.ExecutionLimitExceeded'] })).afterwards({ includeOtherwise: true })
            .next(process));
        const stateMachine = new aws_stepfunctions_1.StateMachine(this, 'StateMachine', {
            definition: listBucket,
            stateMachineName,
            timeout: core_1.Duration.hours(1),
        });
        props.bucket.grantRead(stateMachine);
        props.queue.grantSendMessages(stateMachine);
    }
}
/**
 * This turns a node path into a valid state machine name, to try and improve
 * the StepFunction's AWS console experience while minimizing the risk for
 * collisons.
 */
function stateMachineNameFrom(nodePath) {
    // Poor man's replace all...
    return nodePath.split(/[^a-z0-9+!@.()=_'-]+/i).join('.');
}
//# sourceMappingURL=data:application/json;base64,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