"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BackendDashboard = void 0;
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const core_1 = require("@aws-cdk/core");
const package_versions_table_widget_1 = require("./backend/inventory/package-versions-table-widget");
const constants_1 = require("./backend/shared/constants");
const language_1 = require("./backend/shared/language");
const deep_link_1 = require("./deep-link");
const metric_utils_1 = require("./metric-utils");
class BackendDashboard extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const reports = [[
                new package_versions_table_widget_1.PackageVersionsTableWidget(this, 'UninstallablePackages', {
                    title: 'Package Versions Report | Uninstallable',
                    description: [
                        "These packages could not be installed. Note that currently they will also appear in the 'missing' documentation reports.",
                        '',
                        "The specific error can be found in the package directory inside a file named 'uninstallable'",
                    ].join('\n'),
                    bucket: props.packageData,
                    key: constants_1.UNINSTALLABLE_PACKAGES_REPORT,
                    height: 6,
                    width: 24,
                }),
            ]];
        for (const language of language_1.DocumentationLanguage.ALL) {
            for (const report of this.perLanguageReports(language, props.packageData)) {
                // put every report in a new line
                reports.push([report]);
            }
        }
        new aws_cloudwatch_1.Dashboard(this, 'Reports', {
            dashboardName: `${props.dashboardName}-reports`,
            periodOverride: aws_cloudwatch_1.PeriodOverride.AUTO,
            start: '-P1W',
            widgets: reports,
        });
        new aws_cloudwatch_1.Dashboard(this, 'Graphs', {
            dashboardName: `${props.dashboardName}-graphs`,
            periodOverride: aws_cloudwatch_1.PeriodOverride.AUTO,
            start: '-P1W',
            widgets: [
                [
                    new aws_cloudwatch_1.TextWidget({
                        height: 2,
                        width: 24,
                        markdown: [
                            '# Catalog Overview',
                            '',
                            `[button:primary:Package Data](${deep_link_1.s3ObjectUrl(props.packageData)})`,
                            `[button:Catalog Builder](${deep_link_1.lambdaFunctionUrl(props.orchestration.catalogBuilder.function)})`,
                            `[button:Inventory Canary](${deep_link_1.lambdaFunctionUrl(props.inventory.function)})`,
                            `[button:Search Canary Log Group](${deep_link_1.lambdaSearchLogGroupUrl(props.inventory.function)})`,
                        ].join('\n'),
                    }),
                ],
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Catalog Size',
                        left: [
                            props.inventory.metricSubmoduleCount({ label: 'Submodules' }),
                            props.inventory.metricPackageVersionCount({ label: 'Package Versions' }),
                            props.inventory.metricPackageMajorCount({ label: 'Package Majors' }),
                            props.inventory.metricPackageCount({ label: 'Packages' }),
                        ],
                        leftYAxis: { min: 0 },
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Catalog Issues',
                        left: [
                            props.inventory.metricUnknownObjectCount({ label: 'Unknown' }),
                            props.inventory.metricMissingAssemblyCount({ label: 'Missing Assembly' }),
                            props.inventory.metricMissingPackageMetadataCount({ label: 'Missing Metadata' }),
                            props.inventory.metricMissingPackageTarballCount({ label: 'Missing Tarball' }),
                            props.inventory.metricUninstallablePackageCount({ label: 'Uninstallable Package' }),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            props.orchestration.catalogBuilder.metricMissingConstructFrameworkCount({ label: 'No Construct Framework' }),
                            props.orchestration.catalogBuilder.metricMissingConstructFrameworkVersionCount({ label: 'No Construct Framework Version' }),
                        ],
                        rightYAxis: { min: 0 },
                    }),
                ],
                ...this.catalogOverviewLanguageSections(props),
                ...renderPackageSourcesWidgets(props.packageSources),
                [
                    new aws_cloudwatch_1.TextWidget({
                        height: 2,
                        width: 24,
                        markdown: [
                            '# Ingestion Function',
                            '',
                            `[button:Ingestion Function](${deep_link_1.lambdaFunctionUrl(props.ingestion.function)})`,
                            `[button:primary:Search Log Group](${deep_link_1.lambdaSearchLogGroupUrl(props.ingestion.function)})`,
                            `[button:DLQ](${deep_link_1.sqsQueueUrl(props.ingestion.deadLetterQueue)})`,
                        ].join('\n'),
                    }),
                ],
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Function Health',
                        left: [
                            metric_utils_1.fillMetric(props.ingestion.function.metricInvocations({ label: 'Invocations' })),
                            metric_utils_1.fillMetric(props.ingestion.function.metricErrors({ label: 'Errors' })),
                        ],
                        leftYAxis: { min: 0 },
                        period: core_1.Duration.minutes(1),
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Input Queue',
                        left: [
                            props.ingestion.queue.metricApproximateNumberOfMessagesVisible({ label: 'Visible Messages', period: core_1.Duration.minutes(1) }),
                            props.ingestion.queue.metricApproximateNumberOfMessagesNotVisible({ label: 'Hidden Messages', period: core_1.Duration.minutes(1) }),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            props.ingestion.queue.metricApproximateAgeOfOldestMessage({ label: 'Oldest Message Age', period: core_1.Duration.minutes(1) }),
                        ],
                        rightAnnotations: [{
                                color: '#ffa500',
                                label: '10 Minutes',
                                value: core_1.Duration.minutes(10).toSeconds(),
                            }],
                        rightYAxis: { min: 0 },
                        period: core_1.Duration.minutes(1),
                    }),
                ],
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Input Quality',
                        left: [
                            metric_utils_1.fillMetric(props.ingestion.metricInvalidAssembly({ label: 'Invalid Assemblies' })),
                            metric_utils_1.fillMetric(props.ingestion.metricInvalidTarball({ label: 'Invalid Tarball' })),
                            metric_utils_1.fillMetric(props.ingestion.metricIneligibleLicense({ label: 'Ineligible License' })),
                            metric_utils_1.fillMetric(props.ingestion.metricMismatchedIdentityRejections({ label: 'Mismatched Identity' })),
                            metric_utils_1.fillMetric(props.ingestion.metricFoundLicenseFile({ label: 'Found License file' })),
                        ],
                        leftYAxis: { label: 'Count', min: 0, showUnits: false },
                        stacked: true,
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Dead Letters',
                        left: [
                            props.ingestion.deadLetterQueue.metricApproximateNumberOfMessagesVisible({ label: 'Visible Messages' }),
                            props.ingestion.deadLetterQueue.metricApproximateNumberOfMessagesNotVisible({ label: 'Invisible Messages' }),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            props.ingestion.deadLetterQueue.metricApproximateAgeOfOldestMessage({ label: 'Oldest Message Age' }),
                        ],
                        rightAnnotations: [{
                                color: '#ff7f0e',
                                label: '10 days',
                                value: core_1.Duration.days(10).toSeconds(),
                            }, {
                                color: '#ff0000',
                                label: '14 days (DLQ Retention)',
                                value: core_1.Duration.days(14).toSeconds(),
                            }],
                        rightYAxis: { min: 0 },
                        period: core_1.Duration.minutes(1),
                    }),
                ],
                [
                    new aws_cloudwatch_1.TextWidget({
                        height: 2,
                        width: 24,
                        markdown: [
                            '# Orchestration',
                            '',
                            `[button:primary:State Machine](${deep_link_1.stateMachineUrl(props.orchestration.stateMachine)})`,
                            `[button:DLQ](${deep_link_1.sqsQueueUrl(props.orchestration.deadLetterQueue)})`,
                            `[button:Redrive DLQ](${deep_link_1.lambdaFunctionUrl(props.orchestration.redriveFunction)})`,
                            `[button:Regenerate All Documentation](${deep_link_1.stateMachineUrl(props.orchestration.regenerateAllDocumentation)})`,
                        ].join('\n'),
                    }),
                ],
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'State Machine Executions',
                        left: [
                            metric_utils_1.fillMetric(props.orchestration.stateMachine.metricStarted({ label: 'Started' })),
                            metric_utils_1.fillMetric(props.orchestration.stateMachine.metricSucceeded({ label: 'Succeeded' })),
                            metric_utils_1.fillMetric(props.orchestration.stateMachine.metricAborted({ label: 'Aborted' })),
                            metric_utils_1.fillMetric(props.orchestration.stateMachine.metricFailed({ label: 'Failed' })),
                            metric_utils_1.fillMetric(props.orchestration.stateMachine.metricThrottled({ label: 'Throttled' })),
                            metric_utils_1.fillMetric(props.orchestration.stateMachine.metricTimedOut({ label: 'Timed Out' })),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            props.orchestration.stateMachine.metricTime({ label: 'Duration' }),
                        ],
                        rightYAxis: { min: 0 },
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Dead Letter Queue',
                        left: [
                            props.orchestration.deadLetterQueue.metricApproximateNumberOfMessagesVisible({ label: 'Visible Messages' }),
                            props.orchestration.deadLetterQueue.metricApproximateNumberOfMessagesNotVisible({ label: 'Invisible Messages' }),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            props.orchestration.deadLetterQueue.metricApproximateAgeOfOldestMessage({ label: 'Oldest Message Age' }),
                        ],
                        rightAnnotations: [{
                                color: '#ff7f0e',
                                label: '10 days',
                                value: core_1.Duration.days(10).toSeconds(),
                            }, {
                                color: '#ff0000',
                                label: '14 days (DLQ Retention)',
                                value: core_1.Duration.days(14).toSeconds(),
                            }],
                        rightYAxis: { min: 0 },
                        period: core_1.Duration.minutes(1),
                    }),
                ],
                // deny list
                // ----------------------------------------------
                [
                    new aws_cloudwatch_1.TextWidget({
                        height: 2,
                        width: 24,
                        markdown: [
                            '# Deny List',
                            '',
                            `[button:primary:Deny List Object](${deep_link_1.s3ObjectUrl(props.denyList.bucket, props.denyList.objectKey)})`,
                            `[button:Prune Function](${deep_link_1.lambdaFunctionUrl(props.denyList.prune.pruneHandler)})`,
                            `[button:Prune Logs](${deep_link_1.lambdaSearchLogGroupUrl(props.denyList.prune.pruneHandler)})`,
                            `[button:Delete Queue](${deep_link_1.sqsQueueUrl(props.denyList.prune.queue)})`,
                            `[button:Delete Logs](${deep_link_1.lambdaSearchLogGroupUrl(props.denyList.prune.deleteHandler)})`,
                        ].join('\n'),
                    }),
                ],
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Deny List',
                        left: [
                            metric_utils_1.fillMetric(props.denyList.metricDenyListRules({ label: 'Rules' }), 'REPEAT'),
                            props.denyList.prune.queue.metricNumberOfMessagesDeleted({ label: 'Deleted Files' }),
                        ],
                        leftYAxis: { min: 0 },
                        period: core_1.Duration.minutes(5),
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Prune Function Health',
                        left: [
                            metric_utils_1.fillMetric(props.denyList.prune.pruneHandler.metricInvocations({ label: 'Invocations' })),
                            metric_utils_1.fillMetric(props.denyList.prune.pruneHandler.metricErrors({ label: 'Errors' })),
                        ],
                        leftYAxis: { min: 0 },
                        period: core_1.Duration.minutes(5),
                    }),
                ],
                ...(props.packageStats ? renderPackageStatsWidgets(props.packageStats) : []),
                ...renderVersionTrackerWidgets(props.versionTracker),
            ],
        });
    }
    perLanguageReports(language, packageData) {
        return [
            new package_versions_table_widget_1.PackageVersionsTableWidget(this, `MissingDocs-${language.name}`, {
                title: `Package Versions Report | Missing Documentation | _${language.name}_`,
                description: [
                    `These packages are missing ${language.name} documentation.`,
                    'Apart from the uninstallable packages, this report should stay empty',
                    '',
                    'To investigate inspect the orchestration DLQ.',
                ].join('\n'),
                bucket: packageData,
                key: constants_1.missingDocumentationReport(language),
                height: 6,
                width: 24,
            }),
            new package_versions_table_widget_1.PackageVersionsTableWidget(this, `CorruptAssemblyDocs-${language.name}`, {
                title: `Package Versions Report | Corrupt Assembly | _${language.name}_`,
                description: [
                    `These packages are missing ${language.name} documentation because of a corrupted assembly.`,
                    '',
                    "The specific error can be found in the package directory inside files suffixed with '.corruptassembly'",
                ].join('\n'),
                bucket: packageData,
                key: constants_1.corruptAssemblyReport(language),
                height: 6,
                width: 24,
            }),
        ];
    }
    *catalogOverviewLanguageSections({ inventory, orchestration }) {
        yield [
            new aws_cloudwatch_1.TextWidget({
                height: 2,
                width: 24,
                markdown: [
                    '# Documentation Generation',
                    '',
                    `[button:primary:Transliterator Logs](${deep_link_1.logGroupUrl(orchestration.transliterator.logGroup)})`,
                    `[button:Transliterator ECS Cluster](${deep_link_1.ecsClusterUrl(orchestration.ecsCluster)})`,
                ].join('\n'),
            }),
        ];
        const mFargateUsage = new aws_cloudwatch_1.Metric({
            dimensionsMap: {
                Class: 'None',
                Resource: 'OnDemand',
                Service: 'Fargate',
                Type: 'Resource',
            },
            metricName: 'ResourceCount',
            namespace: 'AWS/Usage',
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
        });
        yield [
            new aws_cloudwatch_1.GraphWidget({
                height: 6,
                width: 12,
                title: 'Fargate Resources',
                left: [
                    mFargateUsage.with({ label: 'Fargate Usage (On-Demand)' }),
                    new aws_cloudwatch_1.MathExpression({
                        expression: 'SERVICE_QUOTA(mFargateUsage)',
                        label: 'Fargate Quota (On-Demand)',
                        usingMetrics: { mFargateUsage },
                    }),
                ],
                leftYAxis: { min: 0 },
                right: [
                    orchestration.metricEcsCpuUtilization({ label: 'CPU Utilization' }),
                    orchestration.metricEcsMemoryUtilization({ label: 'Memory Utilization' }),
                ],
                rightYAxis: { label: 'Percent', min: 0, max: 100, showUnits: false },
            }),
            new aws_cloudwatch_1.GraphWidget({
                height: 6,
                width: 12,
                title: 'ECS Resources',
                left: [
                    metric_utils_1.fillMetric(orchestration.metricEcsNetworkRxBytes({ label: 'Received Bytes' })),
                    metric_utils_1.fillMetric(orchestration.metricEcsNetworkTxBytes({ label: 'Transmitted Bytes' })),
                ],
                leftYAxis: { min: 0 },
                right: [
                    metric_utils_1.fillMetric(orchestration.metricEcsTaskCount({ label: 'Task Count' })),
                ],
                rightYAxis: { min: 0 },
            }),
        ];
        for (const language of language_1.DocumentationLanguage.ALL) {
            yield [
                new aws_cloudwatch_1.TextWidget({
                    height: 1,
                    width: 24,
                    markdown: `## Language: ${language.toString()}`,
                }),
            ];
            yield [
                new aws_cloudwatch_1.GraphWidget({
                    height: 6,
                    width: 12,
                    title: 'Package Versions',
                    left: [
                        inventory.metricSupportedPackageVersionCount(language, { label: 'Available', color: '#2ca02c' }),
                        inventory.metricCorruptAssemblyPackageVersionCount(language, { label: 'Corrupt Assembly', color: '#3542D7' }),
                        inventory.metricUnsupportedPackageVersionCount(language, { label: 'Unsupported', color: '#9467bd' }),
                        inventory.metricMissingPackageVersionCount(language, { label: 'Missing', color: '#d62728' }),
                    ],
                    leftYAxis: { showUnits: false },
                    view: aws_cloudwatch_1.GraphWidgetView.PIE,
                }),
                new aws_cloudwatch_1.GraphWidget({
                    height: 6,
                    width: 12,
                    title: 'Package Version Submodules',
                    left: [
                        inventory.metricSupportedSubmoduleCount(language, { label: 'Available', color: '#2ca02c' }),
                        inventory.metricCorruptAssemblySubmoduleCount(language, { label: 'Corrupt Assembly', color: '#3542D7' }),
                        inventory.metricUnsupportedSubmoduleCount(language, { label: 'Unsupported', color: '#9467bd' }),
                        inventory.metricMissingSubmoduleCount(language, { label: 'Missing', color: '#d62728' }),
                    ],
                    leftYAxis: { showUnits: false },
                    view: aws_cloudwatch_1.GraphWidgetView.PIE,
                }),
            ];
        }
    }
}
exports.BackendDashboard = BackendDashboard;
function* renderPackageSourcesWidgets(packageSources) {
    var _a;
    for (const packageSource of packageSources) {
        yield [
            new aws_cloudwatch_1.TextWidget({
                height: 2,
                width: 24,
                markdown: [
                    `# ${packageSource.name}`,
                    '',
                    ...((_a = packageSource.links) !== null && _a !== void 0 ? _a : [])
                        .map(({ name, primary, url }) => `[${primary ? 'button:primary' : 'button'}:${name}](${url})`),
                ].join('\n'),
            }),
        ];
        yield* packageSource.dashboardWidgets;
    }
    return;
}
function renderPackageStatsWidgets(packageStats) {
    return [
        [
            new aws_cloudwatch_1.TextWidget({
                height: 2,
                width: 24,
                markdown: [
                    '# Package Stats',
                    '',
                    `[button:primary:Package Stats Object](${deep_link_1.s3ObjectUrl(packageStats.bucket, packageStats.statsKey)})`,
                    `[button:Package Stats Function](${deep_link_1.lambdaFunctionUrl(packageStats.handler)})`,
                    `[button:Package Stats Logs](${deep_link_1.lambdaSearchLogGroupUrl(packageStats.handler)})`,
                ].join('\n'),
            }),
        ],
        [
            new aws_cloudwatch_1.GraphWidget({
                height: 6,
                width: 12,
                title: 'Number of Package Stats Recorded',
                left: [
                    metric_utils_1.fillMetric(packageStats.metricPackagesCount({ label: 'Packages with stats' }), 'REPEAT'),
                ],
                leftYAxis: { min: 0 },
            }),
            new aws_cloudwatch_1.GraphWidget({
                height: 6,
                width: 12,
                title: 'Invocation Duration',
                left: [
                    packageStats.handler.metricDuration({ label: 'Duration' }),
                ],
                leftYAxis: { min: 0 },
                rightAnnotations: [{
                        color: '#ffa500',
                        label: '15 minutes (Lambda timeout)',
                        value: core_1.Duration.minutes(15).toSeconds(),
                    }],
            }),
        ],
    ];
}
function renderVersionTrackerWidgets(versionTracker) {
    return [
        [
            new aws_cloudwatch_1.TextWidget({
                height: 2,
                width: 24,
                markdown: [
                    '# Version Tracker',
                    '',
                    `[button:primary:Versions Object](${deep_link_1.s3ObjectUrl(versionTracker.bucket, constants_1.VERSION_TRACKER_KEY)})`,
                    `[button:Version Tracker Function](${deep_link_1.lambdaFunctionUrl(versionTracker.handler)})`,
                    `[button:Version Tracker Logs](${deep_link_1.lambdaSearchLogGroupUrl(versionTracker.handler)})`,
                ].join('\n'),
            }),
        ],
        [
            new aws_cloudwatch_1.GraphWidget({
                height: 6,
                width: 12,
                title: 'Number of Package Versions Recorded',
                left: [
                    metric_utils_1.fillMetric(versionTracker.metricTrackedVersionsCount({ label: 'Package versions recorded' }), 'REPEAT'),
                ],
                leftYAxis: { min: 0 },
            }),
            new aws_cloudwatch_1.GraphWidget({
                height: 6,
                width: 12,
                title: 'Invocation Duration',
                left: [
                    versionTracker.handler.metricDuration({ label: 'Duration' }),
                ],
                leftYAxis: { min: 0 },
                rightAnnotations: [{
                        color: '#ffa500',
                        label: '1 minutes (Lambda timeout)',
                        value: core_1.Duration.minutes(1).toSeconds(),
                    }],
            }),
        ],
    ];
}
//# sourceMappingURL=data:application/json;base64,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