# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['pytest_bdd']

package_data = \
{'': ['*'], 'pytest_bdd': ['templates/*']}

install_requires = \
['Mako', 'parse', 'parse-type', 'pytest>=6.2.0', 'typing-extensions']

entry_points = \
{'console_scripts': ['pytest-bdd = pytest_bdd.scripts:main'],
 'pytest11': ['pytest-bdd = pytest_bdd.plugin']}

setup_kwargs = {
    'name': 'pytest-bdd',
    'version': '6.1.1',
    'description': 'BDD for pytest',
    'long_description': 'BDD library for the pytest runner\n=================================\n\n.. image:: http://img.shields.io/pypi/v/pytest-bdd.svg\n   :target: https://pypi.python.org/pypi/pytest-bdd\n.. image:: https://codecov.io/gh/pytest-dev/pytest-bdd/branch/master/graph/badge.svg\n  :target: https://codecov.io/gh/pytest-dev/pytest-bdd\n.. image:: https://travis-ci.org/pytest-dev/pytest-bdd.svg?branch=master\n    :target: https://travis-ci.org/pytest-dev/pytest-bdd\n.. image:: https://readthedocs.org/projects/pytest-bdd/badge/?version=stable\n    :target: https://readthedocs.org/projects/pytest-bdd/\n    :alt: Documentation Status\n\npytest-bdd implements a subset of the Gherkin language to enable automating project\nrequirements testing and to facilitate behavioral driven development.\n\nUnlike many other BDD tools, it does not require a separate runner and benefits from\nthe power and flexibility of pytest. It enables unifying unit and functional\ntests, reduces the burden of continuous integration server configuration and allows the reuse of\ntest setups.\n\nPytest fixtures written for unit tests can be reused for setup and actions\nmentioned in feature steps with dependency injection. This allows a true BDD\njust-enough specification of the requirements without maintaining any context object\ncontaining the side effects of Gherkin imperative declarations.\n\n.. _behave: https://pypi.python.org/pypi/behave\n.. _pytest-splinter: https://github.com/pytest-dev/pytest-splinter\n\nInstall pytest-bdd\n------------------\n\n::\n\n    pip install pytest-bdd\n\n\nExample\n-------\n\nAn example test for a blog hosting software could look like this.\nNote that pytest-splinter_ is used to get the browser fixture.\n\n.. code-block:: gherkin\n\n    # content of publish_article.feature\n\n    Feature: Blog\n        A site where you can publish your articles.\n\n        Scenario: Publishing the article\n            Given I\'m an author user\n            And I have an article\n\n            When I go to the article page\n            And I press the publish button\n\n            Then I should not see the error message\n            And the article should be published  # Note: will query the database\n\nNote that only one feature is allowed per feature file.\n\n.. code-block:: python\n\n    # content of test_publish_article.py\n\n    from pytest_bdd import scenario, given, when, then\n\n    @scenario(\'publish_article.feature\', \'Publishing the article\')\n    def test_publish():\n        pass\n\n\n    @given("I\'m an author user")\n    def author_user(auth, author):\n        auth[\'user\'] = author.user\n\n\n    @given("I have an article", target_fixture="article")\n    def article(author):\n        return create_test_article(author=author)\n\n\n    @when("I go to the article page")\n    def go_to_article(article, browser):\n        browser.visit(urljoin(browser.url, \'/manage/articles/{0}/\'.format(article.id)))\n\n\n    @when("I press the publish button")\n    def publish_article(browser):\n        browser.find_by_css(\'button[name=publish]\').first.click()\n\n\n    @then("I should not see the error message")\n    def no_error_message(browser):\n        with pytest.raises(ElementDoesNotExist):\n            browser.find_by_css(\'.message.error\').first\n\n\n    @then("the article should be published")\n    def article_is_published(article):\n        article.refresh()  # Refresh the object in the SQLAlchemy session\n        assert article.is_published\n\n\nScenario decorator\n------------------\n\nFunctions decorated with the `scenario` decorator behave like a normal test function,\nand they will be executed after all scenario steps.\n\n\n.. code-block:: python\n\n    from pytest_bdd import scenario, given, when, then\n\n    @scenario(\'publish_article.feature\', \'Publishing the article\')\n    def test_publish(browser):\n        assert article.title in browser.html\n\n\n.. NOTE:: It is however encouraged to try as much as possible to have your logic only inside the Given, When, Then steps.\n\n\nStep aliases\n------------\n\nSometimes, one has to declare the same fixtures or steps with\ndifferent names for better readability. In order to use the same step\nfunction with multiple step names simply decorate it multiple times:\n\n.. code-block:: python\n\n    @given("I have an article")\n    @given("there\'s an article")\n    def article(author, target_fixture="article"):\n        return create_test_article(author=author)\n\nNote that the given step aliases are independent and will be executed\nwhen mentioned.\n\nFor example if you associate your resource to some owner or not. Admin\nuser can’t be an author of the article, but articles should have a\ndefault author.\n\n.. code-block:: gherkin\n\n    Feature: Resource owner\n        Scenario: I\'m the author\n            Given I\'m an author\n            And I have an article\n\n\n        Scenario: I\'m the admin\n            Given I\'m the admin\n            And there\'s an article\n\n\nStep arguments\n--------------\n\nOften it\'s possible to reuse steps giving them a parameter(s).\nThis allows to have single implementation and multiple use, so less code.\nAlso opens the possibility to use same step twice in single scenario and with different arguments!\nAnd even more, there are several types of step parameter parsers at your disposal\n(idea taken from behave_ implementation):\n\n.. _pypi_parse: http://pypi.python.org/pypi/parse\n.. _pypi_parse_type: http://pypi.python.org/pypi/parse_type\n\n**string** (the default)\n    This is the default and can be considered as a `null` or `exact` parser. It parses no parameters\n    and matches the step name by equality of strings.\n**parse** (based on: pypi_parse_)\n    Provides a simple parser that replaces regular expressions for\n    step parameters with a readable syntax like ``{param:Type}``.\n    The syntax is inspired by the Python builtin ``string.format()``\n    function.\n    Step parameters must use the named fields syntax of pypi_parse_\n    in step definitions. The named fields are extracted,\n    optionally type converted and then used as step function arguments.\n    Supports type conversions by using type converters passed via `extra_types`\n**cfparse** (extends: pypi_parse_, based on: pypi_parse_type_)\n    Provides an extended parser with "Cardinality Field" (CF) support.\n    Automatically creates missing type converters for related cardinality\n    as long as a type converter for cardinality=1 is provided.\n    Supports parse expressions like:\n    * ``{values:Type+}`` (cardinality=1..N, many)\n    * ``{values:Type*}`` (cardinality=0..N, many0)\n    * ``{value:Type?}``  (cardinality=0..1, optional)\n    Supports type conversions (as above).\n**re**\n    This uses full regular expressions to parse the clause text. You will\n    need to use named groups "(?P<name>...)" to define the variables pulled\n    from the text and passed to your ``step()`` function.\n    Type conversion can only be done via `converters` step decorator argument (see example below).\n\nThe default parser is `string`, so just plain one-to-one match to the keyword definition.\nParsers except `string`, as well as their optional arguments are specified like:\n\nfor `cfparse` parser\n\n.. code-block:: python\n\n    from pytest_bdd import parsers\n\n    @given(\n        parsers.cfparse("there are {start:Number} cucumbers", extra_types={"Number": int}),\n        target_fixture="cucumbers",\n    )\n    def given_cucumbers(start):\n        return {"start": start, "eat": 0}\n\nfor `re` parser\n\n.. code-block:: python\n\n    from pytest_bdd import parsers\n\n    @given(\n        parsers.re(r"there are (?P<start>\\d+) cucumbers"),\n        converters={"start": int},\n        target_fixture="cucumbers",\n    )\n    def given_cucumbers(start):\n        return {"start": start, "eat": 0}\n\n\nExample:\n\n.. code-block:: gherkin\n\n    Feature: Step arguments\n        Scenario: Arguments for given, when, then\n            Given there are 5 cucumbers\n\n            When I eat 3 cucumbers\n            And I eat 2 cucumbers\n\n            Then I should have 0 cucumbers\n\n\nThe code will look like:\n\n.. code-block:: python\n\n    from pytest_bdd import scenarios, given, when, then, parsers\n\n\n    scenarios("arguments.feature")\n\n\n    @given(parsers.parse("there are {start:d} cucumbers"), target_fixture="cucumbers")\n    def given_cucumbers(start):\n        return {"start": start, "eat": 0}\n\n\n    @when(parsers.parse("I eat {eat:d} cucumbers"))\n    def eat_cucumbers(cucumbers, eat):\n        cucumbers["eat"] += eat\n\n\n    @then(parsers.parse("I should have {left:d} cucumbers"))\n    def should_have_left_cucumbers(cucumbers, left):\n        assert cucumbers["start"] - cucumbers["eat"] == left\n\nExample code also shows possibility to pass argument converters which may be useful if you need to postprocess step\narguments after the parser.\n\nYou can implement your own step parser. It\'s interface is quite simple. The code can looks like:\n\n.. code-block:: python\n\n    import re\n    from pytest_bdd import given, parsers\n\n\n    class MyParser(parsers.StepParser):\n        """Custom parser."""\n\n        def __init__(self, name, **kwargs):\n            """Compile regex."""\n            super().__init__(name)\n            self.regex = re.compile(re.sub("%(.+)%", "(?P<\\1>.+)", self.name), **kwargs)\n\n        def parse_arguments(self, name):\n            """Get step arguments.\n\n            :return: `dict` of step arguments\n            """\n            return self.regex.match(name).groupdict()\n\n        def is_matching(self, name):\n            """Match given name with the step name."""\n            return bool(self.regex.match(name))\n\n\n    @given(parsers.parse("there are %start% cucumbers"), target_fixture="cucumbers")\n    def given_cucumbers(start):\n        return {"start": start, "eat": 0}\n\n\nOverride fixtures via given steps\n---------------------------------\n\nDependency injection is not a panacea if you have complex structure of your test setup data. Sometimes there\'s a need\nsuch a given step which would imperatively change the fixture only for certain test (scenario), while for other tests\nit will stay untouched. To allow this, special parameter `target_fixture` exists in the `given` decorator:\n\n.. code-block:: python\n\n    from pytest_bdd import given\n\n    @pytest.fixture\n    def foo():\n        return "foo"\n\n\n    @given("I have injecting given", target_fixture="foo")\n    def injecting_given():\n        return "injected foo"\n\n\n    @then(\'foo should be "injected foo"\')\n    def foo_is_foo(foo):\n        assert foo == \'injected foo\'\n\n\n.. code-block:: gherkin\n\n    Feature: Target fixture\n        Scenario: Test given fixture injection\n            Given I have injecting given\n            Then foo should be "injected foo"\n\n\nIn this example existing fixture `foo` will be overridden by given step `I have injecting given` only for scenario it\'s\nused in.\n\nSometimes it is also useful to let `when` and `then` steps to provide a fixture as well.\nA common use case is when we have to assert the outcome of an HTTP request:\n\n.. code-block:: python\n\n    # content of test_blog.py\n\n    from pytest_bdd import scenarios, given, when, then\n\n    from my_app.models import Article\n\n    scenarios("blog.feature")\n\n\n    @given("there is an article", target_fixture="article")\n    def there_is_an_article():\n        return Article()\n\n\n    @when("I request the deletion of the article", target_fixture="request_result")\n    def there_should_be_a_new_article(article, http_client):\n        return http_client.delete(f"/articles/{article.uid}")\n\n\n    @then("the request should be successful")\n    def article_is_published(request_result):\n        assert request_result.status_code == 200\n\n\n.. code-block:: gherkin\n\n    # content of blog.feature\n\n    Feature: Blog\n        Scenario: Deleting the article\n            Given there is an article\n\n            When I request the deletion of the article\n\n            Then the request should be successful\n\n\nMultiline steps\n---------------\n\nAs Gherkin, pytest-bdd supports multiline steps\n(a.k.a. `Doc Strings <https://cucumber.io/docs/gherkin/reference/#doc-strings>`_).\nBut in much cleaner and powerful way:\n\n.. code-block:: gherkin\n\n    Feature: Multiline steps\n        Scenario: Multiline step using sub indentation\n            Given I have a step with:\n                Some\n                Extra\n                Lines\n            Then the text should be parsed with correct indentation\n\nStep is considered as multiline one, if the **next** line(s) after it\'s first line, is indented relatively\nto the first line. The step name is then simply extended by adding further lines with newlines.\nIn the example above, the Given step name will be:\n\n.. code-block:: python\n\n    \'I have a step with:\\nSome\\nExtra\\nLines\'\n\nYou can of course register step using full name (including the newlines), but it seems more practical to use\nstep arguments and capture lines after first line (or some subset of them) into the argument:\n\n.. code-block:: python\n\n    from pytest_bdd import given, then, scenario, parsers\n\n\n    scenarios("multiline.feature")\n\n\n    @given(parsers.parse("I have a step with:\\n{content}"), target_fixture="text")\n    def given_text(content):\n        return content\n\n\n    @then("the text should be parsed with correct indentation")\n    def text_should_be_correct(text):\n        assert text == "Some\\nExtra\\nLines"\n\n\nScenarios shortcut\n------------------\n\nIf you have relatively large set of feature files, it\'s boring to manually bind scenarios to the tests using the scenario decorator. Of course with the manual approach you get all the power to be able to additionally parametrize the test, give the test function a nice name, document it, etc, but in the majority of the cases you don\'t need that.\nInstead, you want to bind all the scenarios found in the ``features`` folder(s) recursively automatically, by using the ``scenarios`` helper.\n\n.. code-block:: python\n\n    from pytest_bdd import scenarios\n\n    # assume \'features\' subfolder is in this file\'s directory\n    scenarios(\'features\')\n\nThat\'s all you need to do to bind all scenarios found in the ``features`` folder!\nNote that you can pass multiple paths, and those paths can be either feature files or feature folders.\n\n\n.. code-block:: python\n\n    from pytest_bdd import scenarios\n\n    # pass multiple paths/files\n    scenarios(\'features\', \'other_features/some.feature\', \'some_other_features\')\n\nBut what if you need to manually bind certain scenario, leaving others to be automatically bound?\nJust write your scenario in a "normal" way, but ensure you do it **before** the call of ``scenarios`` helper.\n\n\n.. code-block:: python\n\n    from pytest_bdd import scenario, scenarios\n\n    @scenario(\'features/some.feature\', \'Test something\')\n    def test_something():\n        pass\n\n    # assume \'features\' subfolder is in this file\'s directory\n    scenarios(\'features\')\n\nIn the example above, the ``test_something`` scenario binding will be kept manual, other scenarios found in the ``features`` folder will be bound automatically.\n\n\nScenario outlines\n-----------------\n\nScenarios can be parametrized to cover few cases. These are called `Scenario Outlines <https://cucumber.io/docs/gherkin/reference/#scenario-outline>`_ in Gherkin, and the variable templates are written using angular brackets (e.g. ``<var_name>``).\n\nExample:\n\n.. code-block:: gherkin\n\n    # content of scenario_outlines.feature\n\n    Feature: Scenario outlines\n        Scenario Outline: Outlined given, when, then\n            Given there are <start> cucumbers\n            When I eat <eat> cucumbers\n            Then I should have <left> cucumbers\n\n            Examples:\n            | start | eat | left |\n            |  12   |  5  |  7   |\n\n.. code-block:: python\n\n    from pytest_bdd import scenarios, given, when, then, parsers\n\n\n    scenarios("scenario_outlines.feature")\n\n\n    @given(parsers.parse("there are {start:d} cucumbers"), target_fixture="cucumbers")\n    def given_cucumbers(start):\n        return {"start": start, "eat": 0}\n\n\n    @when(parsers.parse("I eat {eat:d} cucumbers"))\n    def eat_cucumbers(cucumbers, eat):\n        cucumbers["eat"] += eat\n\n\n    @then(parsers.parse("I should have {left:d} cucumbers"))\n    def should_have_left_cucumbers(cucumbers, left):\n        assert cucumbers["start"] - cucumbers["eat"] == left\n\n\nOrganizing your scenarios\n-------------------------\n\nThe more features and scenarios you have, the more important becomes the question about their organization.\nThe things you can do (and that is also a recommended way):\n\n* organize your feature files in the folders by semantic groups:\n\n::\n\n    features\n    │\n    ├──frontend\n    │  │\n    │  └──auth\n    │     │\n    │     └──login.feature\n    └──backend\n       │\n       └──auth\n          │\n          └──login.feature\n\nThis looks fine, but how do you run tests only for certain feature?\nAs pytest-bdd uses pytest, and bdd scenarios are actually normal tests. But test files\nare separate from the feature files, the mapping is up to developers, so the test files structure can look\ncompletely different:\n\n::\n\n    tests\n    │\n    └──functional\n       │\n       └──test_auth.py\n          │\n          └ """Authentication tests."""\n            from pytest_bdd import scenario\n\n            @scenario(\'frontend/auth/login.feature\')\n            def test_logging_in_frontend():\n                pass\n\n            @scenario(\'backend/auth/login.feature\')\n            def test_logging_in_backend():\n                pass\n\n\nFor picking up tests to run we can use\n`tests selection <https://pytest.org/en/7.1.x/how-to/usage.html#specifying-which-tests-to-run>`_ technique. The problem is that\nyou have to know how your tests are organized, knowing only the feature files organization is not enough.\nCucumber uses `tags <https://cucumber.io/docs/cucumber/api/#tags>`_ as a way of categorizing your features\nand scenarios, which pytest-bdd supports. For example, we could have:\n\n.. code-block:: gherkin\n\n    @login @backend\n    Feature: Login\n\n      @successful\n      Scenario: Successful login\n\n\npytest-bdd uses `pytest markers <http://pytest.org/latest/mark.html>`_ as a `storage` of the tags for the given\nscenario test, so we can use standard test selection:\n\n.. code-block:: bash\n\n    pytest -m "backend and login and successful"\n\nThe feature and scenario markers are not different from standard pytest markers, and the ``@`` symbol is stripped out automatically to allow test selector expressions. If you want to have bdd-related tags to be distinguishable from the other test markers, use prefix like ``bdd``.\nNote that if you use pytest ``--strict`` option, all bdd tags mentioned in the feature files should be also in the ``markers`` setting of the ``pytest.ini`` config. Also for tags please use names which are python-compatible variable names, eg starts with a non-number, underscore alphanumeric, etc. That way you can safely use tags for tests filtering.\n\nYou can customize how tags are converted to pytest marks by implementing the\n``pytest_bdd_apply_tag`` hook and returning ``True`` from it:\n\n.. code-block:: python\n\n   def pytest_bdd_apply_tag(tag, function):\n       if tag == \'todo\':\n           marker = pytest.mark.skip(reason="Not implemented yet")\n           marker(function)\n           return True\n       else:\n           # Fall back to the default behavior of pytest-bdd\n           return None\n\nTest setup\n----------\n\nTest setup is implemented within the Given section. Even though these steps\nare executed imperatively to apply possible side-effects, pytest-bdd is trying\nto benefit of the PyTest fixtures which is based on the dependency injection\nand makes the setup more declarative style.\n\n.. code-block:: python\n\n    @given("I have a beautiful article", target_fixture="article")\n    def article():\n        return Article(is_beautiful=True)\n\nThe target PyTest fixture "article" gets the return value and any other step can depend on it.\n\n.. code-block:: gherkin\n\n    Feature: The power of PyTest\n        Scenario: Symbolic name across steps\n            Given I have a beautiful article\n            When I publish this article\n\nWhen step is referring the article to publish it.\n\n.. code-block:: python\n\n    @when("I publish this article")\n    def publish_article(article):\n        article.publish()\n\n\nMany other BDD toolkits operate a global context and put the side effects there.\nThis makes it very difficult to implement the steps, because the dependencies\nappear only as the side-effects in the run-time and not declared in the code.\nThe publish article step has to trust that the article is already in the context,\nhas to know the name of the attribute it is stored there, the type etc.\n\nIn pytest-bdd you just declare an argument of the step function that it depends on\nand the PyTest will make sure to provide it.\n\nStill side effects can be applied in the imperative style by design of the BDD.\n\n.. code-block:: gherkin\n\n    Feature: News website\n        Scenario: Publishing an article\n            Given I have a beautiful article\n            And my article is published\n\nFunctional tests can reuse your fixture libraries created for the unit-tests and upgrade\nthem by applying the side effects.\n\n.. code-block:: python\n\n    @pytest.fixture\n    def article():\n        return Article(is_beautiful=True)\n\n\n    @given("I have a beautiful article")\n    def i_have_a_beautiful_article(article):\n        pass\n\n\n    @given("my article is published")\n    def published_article(article):\n        article.publish()\n        return article\n\n\nThis way side-effects were applied to our article and PyTest makes sure that all\nsteps that require the "article" fixture will receive the same object. The value\nof the "published_article" and the "article" fixtures is the same object.\n\nFixtures are evaluated only once within the PyTest scope and their values are cached.\n\n\nBackgrounds\n-----------\n\nIt\'s often the case that to cover certain feature, you\'ll need multiple scenarios. And it\'s logical that the\nsetup for those scenarios will have some common parts (if not equal). For this, there are `backgrounds`.\npytest-bdd implements `Gherkin backgrounds <https://cucumber.io/docs/gherkin/reference/#background>`_ for\nfeatures.\n\n.. code-block:: gherkin\n\n    Feature: Multiple site support\n\n      Background:\n        Given a global administrator named "Greg"\n        And a blog named "Greg\'s anti-tax rants"\n        And a customer named "Wilson"\n        And a blog named "Expensive Therapy" owned by "Wilson"\n\n      Scenario: Wilson posts to his own blog\n        Given I am logged in as Wilson\n        When I try to post to "Expensive Therapy"\n        Then I should see "Your article was published."\n\n      Scenario: Greg posts to a client\'s blog\n        Given I am logged in as Greg\n        When I try to post to "Expensive Therapy"\n        Then I should see "Your article was published."\n\nIn this example, all steps from the background will be executed before all the scenario\'s own given\nsteps, adding possibility to prepare some common setup for multiple scenarios in a single feature.\nAbout background best practices, please read Gherkin\'s\n`Tips for using Background <https://cucumber.io/docs/gherkin/reference/#tips-for-using-background>`_.\n\n.. NOTE:: There is only step "Given" should be used in "Background" section,\n          steps "When" and "Then" are prohibited, because their purpose are\n          related to actions and consuming outcomes, that is conflict with\n          "Background" aim - prepare system for tests or "put the system\n          in a known state" as "Given" does it.\n          The statement above is applied for strict Gherkin mode, which is\n          enabled by default.\n\n\nReusing fixtures\n----------------\n\nSometimes scenarios define new names for the existing fixture that can be\ninherited (reused). For example, if we have pytest fixture:\n\n\n.. code-block:: python\n\n    @pytest.fixture\n    def article():\n       """Test article."""\n       return Article()\n\n\nThen this fixture can be reused with other names using given():\n\n\n.. code-block:: python\n\n    @given(\'I have beautiful article\')\n    def i_have_an_article(article):\n       """I have an article."""\n\n\nReusing steps\n-------------\n\nIt is possible to define some common steps in the parent conftest.py and\nsimply expect them in the child test file.\n\n.. code-block:: gherkin\n\n    # content of common_steps.feature\n\n    Scenario: All steps are declared in the conftest\n        Given I have a bar\n        Then bar should have value "bar"\n\n.. code-block:: python\n\n    # content of conftest.py\n\n    from pytest_bdd import given, then\n\n\n    @given("I have a bar", target_fixture="bar")\n    def bar():\n        return "bar"\n\n\n    @then(\'bar should have value "bar"\')\n    def bar_is_bar(bar):\n        assert bar == "bar"\n\n.. code-block:: python\n\n    # content of test_common.py\n\n    @scenario("common_steps.feature", "All steps are declared in the conftest")\n    def test_conftest():\n        pass\n\nThere are no definitions of the steps in the test file. They were\ncollected from the parent conftest.py.\n\n\nDefault steps\n-------------\n\nHere is the list of steps that are implemented inside of the pytest-bdd:\n\ngiven\n    * trace - enters the `pdb` debugger via `pytest.set_trace()`\nwhen\n    * trace - enters the `pdb` debugger via `pytest.set_trace()`\nthen\n    * trace - enters the `pdb` debugger via `pytest.set_trace()`\n\n\nFeature file paths\n------------------\n\nBy default, pytest-bdd will use current module\'s path as base path for finding feature files, but this behaviour can be changed in the pytest configuration file (i.e. `pytest.ini`, `tox.ini` or `setup.cfg`) by declaring the new base path in the `bdd_features_base_dir` key. The path is interpreted as relative to the `pytest root directory <https://docs.pytest.org/en/latest/reference/customize.html#rootdir>`__.\nYou can also override features base path on a per-scenario basis, in order to override the path for specific tests.\n\npytest.ini:\n\n.. code-block:: ini\n\n    [pytest]\n    bdd_features_base_dir = features/\n\ntests/test_publish_article.py:\n\n.. code-block:: python\n\n    from pytest_bdd import scenario\n\n\n    @scenario("foo.feature", "Foo feature in features/foo.feature")\n    def test_foo():\n        pass\n\n\n    @scenario(\n        "foo.feature",\n        "Foo feature in tests/local-features/foo.feature",\n        features_base_dir="./local-features/",\n    )\n    def test_foo_local():\n        pass\n\n\nThe `features_base_dir` parameter can also be passed to the `@scenario` decorator.\n\n\nAvoid retyping the feature file name\n------------------------------------\n\nIf you want to avoid retyping the feature file name when defining your scenarios in a test file, use ``functools.partial``.\nThis will make your life much easier when defining multiple scenarios in a test file. For example:\n\n.. code-block:: python\n\n    # content of test_publish_article.py\n\n    from functools import partial\n\n    import pytest_bdd\n\n\n    scenario = partial(pytest_bdd.scenario, "/path/to/publish_article.feature")\n\n\n    @scenario("Publishing the article")\n    def test_publish():\n        pass\n\n\n    @scenario("Publishing the article as unprivileged user")\n    def test_publish_unprivileged():\n        pass\n\n\nYou can learn more about `functools.partial <https://docs.python.org/3/library/functools.html#functools.partial>`_\nin the Python docs.\n\n\nProgrammatic step generation\n----------------------------\nSometimes you have step definitions that would be much easier to automate rather than writing manually over and over again.\nThis is common, for example, when using libraries like `pytest-factoryboy <https://pytest-factoryboy.readthedocs.io/>`_ that automatically creates fixtures.\nWriting step definitions for every model can become a tedious task.\n\nFor this reason, pytest-bdd provides a way to generate step definitions automatically.\n\nThe trick is to pass the ``stacklevel`` parameter to the ``given``, ``when``, ``then``, ``step`` decorators. This will instruct them to inject the step fixtures in the appropriate module, rather than just injecting them in the caller frame.\n\nLet\'s look at a concrete example; let\'s say you have a class ``Wallet`` that has some amount for each currency:\n\n.. code-block:: python\n\n    # contents of wallet.py\n\n    import dataclass\n\n    @dataclass\n    class Wallet:\n        verified: bool\n\n        amount_eur: int\n        amount_usd: int\n        amount_gbp: int\n        amount_jpy: int\n\n\nYou can use pytest-factoryboy to automatically create model fixtures for this class:\n\n.. code-block:: python\n\n    # contents of wallet_factory.py\n\n    from wallet import Wallet\n\n    import factory\n    from pytest_factoryboy import register\n\n    class WalletFactory(factory.Factory):\n        class Meta:\n            model = Wallet\n\n        amount_eur = 0\n        amount_usd = 0\n        amount_gbp = 0\n        amount_jpy = 0\n\n    register(Wallet)  # creates the "wallet" fixture\n    register(Wallet, "second_wallet")  # creates the "second_wallet" fixture\n\n\nNow we can define a function ``generate_wallet_steps(...)`` that creates the steps for any wallet fixture (in our case, it will be ``wallet`` and ``second_wallet``):\n\n.. code-block:: python\n\n    # contents of wallet_steps.py\n\n    import re\n    from dataclasses import fields\n\n    import factory\n    import pytest\n    from pytest_bdd import given, when, then, scenarios, parsers\n\n\n    def generate_wallet_steps(model_name="wallet", stacklevel=1):\n        stacklevel += 1\n\n        human_name = model_name.replace("_", " ")  # "second_wallet" -> "second wallet"\n\n        @given(f"I have a {human_name}", target_fixture=model_name, stacklevel=stacklevel)\n        def _(request):\n            return request.getfixturevalue(model_name)\n\n        # Generate steps for currency fields:\n        for field in fields(Wallet):\n            match = re.fullmatch(r"amount_(?P<currency>[a-z]{3})", field.name)\n            if not match:\n                continue\n            currency = match["currency"]\n\n            @given(\n                parsers.parse(f"I have {{value:d}} {currency.upper()} in my {human_name}"),\n                target_fixture=f"{model_name}__amount_{currency}",\n                stacklevel=stacklevel,\n            )\n            def _(value: int) -> int:\n                return value\n\n            @then(\n                parsers.parse(f"I should have {{value:d}} {currency.upper()} in my {human_name}"),\n                stacklevel=stacklevel,\n            )\n            def _(value: int, _currency=currency, _model_name=model_name) -> None:\n                wallet = request.getfixturevalue(_model_name)\n                assert getattr(wallet, f"amount_{_currency}") == value\n\n    # Inject the steps into the current module\n    generate_wallet_steps("wallet")\n    generate_wallet_steps("second_wallet")\n\n\nThis last file, ``wallet_steps.py``, now contains all the step definitions for our "wallet" and "second_wallet" fixtures.\n\nWe can now define a scenario like this:\n\n.. code-block:: gherkin\n\n    # contents of wallet.feature\n    Feature: A feature\n\n        Scenario: Wallet EUR amount stays constant\n            Given I have 10 EUR in my wallet\n            And I have a wallet\n            Then I should have 10 EUR in my wallet\n\n        Scenario: Second wallet JPY amount stays constant\n            Given I have 100 JPY in my second wallet\n            And I have a second wallet\n            Then I should have 100 JPY in my second wallet\n\n\nand finally a test file that puts it all together and run the scenarios:\n\n.. code-block:: python\n\n    # contents of test_wallet.py\n\n    from pytest_factoryboy import scenarios\n\n    from wallet_factory import *  # import the registered fixtures "wallet" and "second_wallet"\n    from wallet_steps import *  # import all the step definitions into this test file\n\n    scenarios("wallet.feature")\n\n\nHooks\n-----\n\npytest-bdd exposes several `pytest hooks <https://docs.pytest.org/en/7.1.x/reference/reference.html#hooks>`_\nwhich might be helpful building useful reporting, visualization, etc. on top of it:\n\n* pytest_bdd_before_scenario(request, feature, scenario) - Called before scenario is executed\n\n* pytest_bdd_after_scenario(request, feature, scenario) - Called after scenario is executed\n  (even if one of steps has failed)\n\n* pytest_bdd_before_step(request, feature, scenario, step, step_func) - Called before step function\n  is executed and it\'s arguments evaluated\n\n* pytest_bdd_before_step_call(request, feature, scenario, step, step_func, step_func_args) - Called before step\n  function is executed with evaluated arguments\n\n* pytest_bdd_after_step(request, feature, scenario, step, step_func, step_func_args) - Called after step function\n  is successfully executed\n\n* pytest_bdd_step_error(request, feature, scenario, step, step_func, step_func_args, exception) - Called when step\n  function failed to execute\n\n* pytest_bdd_step_func_lookup_error(request, feature, scenario, step, exception) - Called when step lookup failed\n\n\nBrowser testing\n---------------\n\nTools recommended to use for browser testing:\n\n* pytest-splinter_ - pytest `splinter <https://splinter.readthedocs.io/>`_ integration for the real browser testing\n\n\nReporting\n---------\n\nIt\'s important to have nice reporting out of your bdd tests. Cucumber introduced some kind of standard for\n`json format <https://www.relishapp.com/cucumber/cucumber/docs/json-output-formatter>`_\nwhich can be used for, for example, by `this <https://plugins.jenkins.io/cucumber-testresult-plugin/>`_ Jenkins\nplugin.\n\nTo have an output in json format:\n\n::\n\n    pytest --cucumberjson=<path to json report>\n\nThis will output an expanded (meaning scenario outlines will be expanded to several scenarios) cucumber format.\n\nTo enable gherkin-formatted output on terminal, use\n\n::\n\n    pytest --gherkin-terminal-reporter\n\n\nTest code generation helpers\n----------------------------\n\nFor newcomers it\'s sometimes hard to write all needed test code without being frustrated.\nTo simplify their life, simple code generator was implemented. It allows to create fully functional\nbut of course empty tests and step definitions for given a feature file.\nIt\'s done as a separate console script provided by pytest-bdd package:\n\n::\n\n    pytest-bdd generate <feature file name> .. <feature file nameN>\n\nIt will print the generated code to the standard output so you can easily redirect it to the file:\n\n::\n\n    pytest-bdd generate features/some.feature > tests/functional/test_some.py\n\n\nAdvanced code generation\n------------------------\n\nFor more experienced users, there\'s smart code generation/suggestion feature. It will only generate the\ntest code which is not yet there, checking existing tests and step definitions the same way it\'s done during the\ntest execution. The code suggestion tool is called via passing additional pytest arguments:\n\n::\n\n    pytest --generate-missing --feature features tests/functional\n\nThe output will be like:\n\n::\n\n    ============================= test session starts ==============================\n    platform linux2 -- Python 2.7.6 -- py-1.4.24 -- pytest-2.6.2\n    plugins: xdist, pep8, cov, cache, bdd, bdd, bdd\n    collected 2 items\n\n    Scenario is not bound to any test: "Code is generated for scenarios which are not bound to any tests" in feature "Missing code generation" in /tmp/pytest-552/testdir/test_generate_missing0/tests/generation.feature\n    --------------------------------------------------------------------------------\n\n    Step is not defined: "I have a custom bar" in scenario: "Code is generated for scenario steps which are not yet defined(implemented)" in feature "Missing code generation" in /tmp/pytest-552/testdir/test_generate_missing0/tests/generation.feature\n    --------------------------------------------------------------------------------\n    Please place the code above to the test file(s):\n\n    @scenario(\'tests/generation.feature\', \'Code is generated for scenarios which are not bound to any tests\')\n    def test_Code_is_generated_for_scenarios_which_are_not_bound_to_any_tests():\n        """Code is generated for scenarios which are not bound to any tests."""\n\n\n    @given("I have a custom bar")\n    def I_have_a_custom_bar():\n        """I have a custom bar."""\n\nAs as side effect, the tool will validate the files for format errors, also some of the logic bugs, for example the\nordering of the types of the steps.\n\n\n.. _Migration from 5.x.x:\n\nMigration of your tests from versions 5.x.x\n-------------------------------------------\n\nThe primary focus of the pytest-bdd is the compatibility with the latest gherkin developments\ne.g. multiple scenario outline example tables with tags support etc.\n\nIn order to provide the best compatibility it is best to support the features described in the official\ngherkin reference. This means deprecation of some non-standard features that were implemented in pytest-bdd.\n\n\nRemoval of the feature examples\n^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^\nThe example tables on the feature level are no longer supported. If you had examples on the feature level, you should copy them to each individual scenario.\n\n\nRemoval of the vertical examples\n^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^\nVertical example tables are no longer supported since the official gherkin doesn\'t support them.\nThe example tables should have horizontal orientation.\n\n\nStep arguments are no longer fixtures\n^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^\nStep parsed arguments conflicted with the fixtures. Now they no longer define fixture.\nIf the fixture has to be defined by the step the target_fixture param should be used.\n\n\nVariable templates in steps are only parsed for Scenario Outlines\n^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^\nIn previous versions of pytest, steps containing ``<variable>`` would be parsed both by ``Scenario`` and ``Scenario Outline``.\nNow they are only parsed within a ``Scenario Outline``.\n\n\n.. _Migration from 4.x.x:\n\nMigration of your tests from versions 4.x.x\n-------------------------------------------\n\nReplace usage of <parameter> inside step definitions with parsed {parameter}\n^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^\nTemplated steps (e.g. ``@given("there are <start> cucumbers")``) should now the use step argument parsers in order to match the scenario outlines and get the values from the example tables. The values from the example tables are no longer passed as fixtures, although if you define your step to use a parser, the parameters will be still provided as fixtures.\n\n.. code-block:: python\n\n    # Old step definition:\n    @given("there are <start> cucumbers")\n    def given_cucumbers(start):\n        pass\n\n\n    # New step definition:\n    @given(parsers.parse("there are {start} cucumbers"))\n    def given_cucumbers(start):\n        pass\n\n\nScenario `example_converters` are removed in favor of the converters provided on the step level:\n\n.. code-block:: python\n\n    # Old code:\n    @given("there are <start> cucumbers")\n    def given_cucumbers(start):\n        return {"start": start}\n\n    @scenario("outline.feature", "Outlined", example_converters={"start": float})\n    def test_outline():\n        pass\n\n\n    # New code:\n    @given(parsers.parse("there are {start} cucumbers"), converters={"start": float})\n    def given_cucumbers(start):\n        return {"start": start}\n\n    @scenario("outline.feature", "Outlined")\n    def test_outline():\n        pass\n\n\nRefuse combining scenario outline and pytest parametrization\n^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^\nThe significant downside of combining scenario outline and pytest parametrization approach was inability to see the\ntest table from the feature file.\n\n\n.. _Migration from 3.x.x:\n\nMigration of your tests from versions 3.x.x\n-------------------------------------------\n\n\nGiven steps are no longer fixtures. In case it is needed to make given step setup a fixture\nthe target_fixture parameter should be used.\n\n\n.. code-block:: python\n\n    @given("there\'s an article", target_fixture="article")\n    def there_is_an_article():\n        return Article()\n\n\nGiven steps no longer have fixture parameter. In fact the step may depend on multiple fixtures.\nJust normal step declaration with the dependency injection should be used.\n\n.. code-block:: python\n\n    @given("there\'s an article")\n    def there_is_an_article(article):\n        pass\n\n\nStrict gherkin option is removed, so the ``strict_gherkin`` parameter can be removed from the scenario decorators\nas well as ``bdd_strict_gherkin`` from the ini files.\n\nStep validation handlers for the hook ``pytest_bdd_step_validation_error`` should be removed.\n\nLicense\n-------\n\nThis software is licensed under the `MIT License <https://opensource.org/licenses/MIT>`_.\n\n© 2013 Oleg Pidsadnyi, Anatoly Bubenkov and others\n',
    'author': 'Oleg Pidsadnyi',
    'author_email': 'oleg.pidsadnyi@gmail.com',
    'maintainer': 'Alessio Bogon',
    'maintainer_email': '778703+youtux@users.noreply.github.com',
    'url': 'https://pytest-bdd.readthedocs.io/',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
