# -*- coding: utf-8 -*-

from typing import (
    Optional,
    Union
)

from . import variables as gv
from .exception import auto_raise, APINetworkError, APIJSONParesError, APIServerResponseError
from .parser import response_json_stripper
from .router import rew_get
from .session import UserSession


def getOfficialVideoPost(
        video_seq: Union[str, int],
        session: UserSession = None,
        silent: bool = False
) -> Optional[dict]:
    """Get detailed official video post data.

    Arguments:
        video_seq (:class:`str`) : Unique seq id of the video post to load data.
        session (:class:`vlivepy.UserSession`, optional) : Session for loading data with permission, defaults to None.
        silent (:class:`bool`, optional) : Return None instead of raising exception, defaults to False.

    Returns:
        :class:`dict`. Parsed json data
    """

    sr = rew_get(**gv.endpoint_official_video_post(video_seq),
                 session=session, wait=0.5, status=[200, 403])

    if sr.success:
        return response_json_stripper(sr.response.json(), silent=silent)
    else:
        auto_raise(APINetworkError, silent)

    return None


def getLivePlayInfo(
        video_seq: Union[str, int],
        session: UserSession = None,
        vpdid2: str = None,
        silent: bool = False
) -> Optional[dict]:
    """Get detailed play info of live.

    Arguments:
        video_seq (:class:`str`) : Unique seq id of the video post to load data.
        session (:class:`vlivepy.UserSession`, optional) : Session for loading data with permission, defaults to None.
        vpdid2 (:class:`str`, optional) : User vpdid2 data, defaults to None.
            It can be automatically generated by :obj:`session`.
        silent (:class:`bool`, optional) : Return None instead of raising exception, defaults to False.

    Returns:
        :class:`dict`. Parsed json data
    """

    # Get vpdid2, if session is valid
    if session is not None and vpdid2 is None:
        vpdid2 = getVpdid2(session, silent=silent)

    # Make request
    sr = rew_get(**gv.endpoint_live_play_info(video_seq, vpdid2),
                 session=session, status=[200, 403])

    if sr.success:
        json_response = sr.response.json()
        if json_response['code'] != 1000:
            auto_raise(
                APIServerResponseError("Video-%s is not live or reserved live. It may be a VOD" % video_seq),
                silent
            )
        else:
            return response_json_stripper(json_response, silent=silent)
    else:
        auto_raise(APINetworkError, silent)

    return None


def getLiveStatus(
        video_seq: Union[str, int],
        silent: bool = False
) -> Optional[dict]:
    """Get simplified status of live.

    Arguments:
        video_seq (:class:`str`) : Unique seq id of the video post to load data.
        silent (:class:`bool`, optional) : Return None instead of raising exception, defaults to False.

    Returns:
        :class:`dict`. Parsed json data
    """

    # Make request
    sr = rew_get(**gv.endpoint_live_status(video_seq),
                 wait=0.2, status=[200])

    if sr.success:
        json_response = sr.response.json()
        if json_response['code'] != 1000:
            auto_raise(
                APIServerResponseError("Video-%s is not live or reserved live. It may be a VOD" % video_seq),
                silent
            )
        else:
            return response_json_stripper(json_response, silent=silent)
    else:
        auto_raise(APINetworkError, silent)

    return None


def getVodId(
        video_seq: Union[str, int],
        silent: bool = False
) -> Optional[str]:
    """Video utility for parsing VOD id from official video post data.
    This internally uses :func:`getOfficialVideoPost` function and parse VOD id from it.

    Arguments:
        video_seq (:class:`str`) : Unique seq id of the video post to load data.
        silent (:class:`bool`, optional) : Return None instead of raising exception, defaults to False.

    Returns:
        :class:`str`. Parsed VOD id
    """

    data = getOfficialVideoPost(video_seq, silent=silent)

    if data is not None:
        if 'officialVideo' in data:
            if 'vodId' in data['officialVideo']:
                return data['officialVideo']['vodId']
            else:
                auto_raise(APIJSONParesError("Given data is live data"), silent=silent)
        else:
            auto_raise(APIJSONParesError("Given data is post data"), silent=silent)

    return None


def getInkeyData(
        video_seq: Union[str, int],
        session: UserSession = None,
        silent: bool = False
) -> Optional[dict]:
    """Get InKey data of current session and video.

    Arguments:
        video_seq (:class:`str`) : Unique seq id of the video post to load data.
        session (:class:`vlivepy.UserSession`, optional) : Session for loading data with permission, defaults to None.
        silent (:class:`bool`, optional) : Return None instead of raising exception, defaults to False.

    Returns:
        :class:`dict`. Parsed json data
    """

    # Make request
    sr = rew_get(**gv.endpoint_vod_inkey(video_seq),
                 wait=0.5, session=session, status=[200])

    if sr.success:
        return response_json_stripper(sr.response.json(), silent=silent)
    else:
        auto_raise(APINetworkError, silent)

    return None


def getVpdid2(
        session: UserSession,
        silent: bool = False
) -> Optional[str]:
    """Video utility for get user's vpdid2 info.
    This internally uses :func:`getInkeyData` function with :code:`video_seq="142851"` param and parse vpdid2 from it.

    Arguments:
        session (:class:`vlivepy.UserSession`) : Session for loading data.
        silent (:class:`bool`, optional) : Return None instead of raising exception, defaults to False.

    Returns:
        :class:`str`. Parsed vpdid2.
    """

    inkey = getInkeyData("142851", session=session, silent=silent)
    if inkey is None:
        return None
    else:
        if 'vpdid2' not in inkey:
            auto_raise(APIJSONParesError("Server didn't return vpdid2"), silent=silent)
        return inkey['vpdid2']


def getVodPlayInfo(
        video_seq: Union[str, int],
        vod_id: str = None,
        session: UserSession = None,
        silent: bool = False
) -> Optional[dict]:
    """Get detailed play info of VOD.

    Arguments:
        video_seq (:class:`str`) : Unique seq id of the video post to load data.
        vod_id (:class:`str`, optional) : Unique id of the VOD to load data, defaults to None.
            It can be automatically generated with func:`getVodId`.
        session (:class:`vlivepy.UserSession`, optional) : Session for loading data with permission, defaults to None.
        silent (:class:`bool`, optional) : Return None instead of raising exception, defaults to False.

    Returns:
        :class:`dict`. Parsed json data
    """

    inkey = getInkeyData(video_seq, session=session, silent=silent)['inkey']
    if vod_id is None:
        vod_id = getVodId(video_seq)

    # make request
    sr = rew_get(**gv.endpoint_vod_play_info(vod_id, inkey),
                 session=session, wait=0.3, status=[200, 403])

    if sr.success:
        return response_json_stripper(sr.response.json(), silent=silent)
    else:
        auto_raise(APINetworkError, silent=silent)


def getOfficialVideoData(
        video_seq: Union[str, int],
        session: UserSession = None,
        silent: bool = False
) -> Optional[dict]:
    """Video utility for parsing VOD id from official video post data.
    This internally uses :func:`getOfficialVideoPost` function and parse VOD id from it.

    Arguments:
        video_seq (:class:`str`) : Unique seq id of the video post to load data.
        session (:class:`vlivepy.UserSession`, optional) : Session for loading data, defaults to None.
        silent (:class:`bool`, optional) : Return None instead of raising exception, defaults to False.

    Returns:
        :class:`dict`. Parsed json data
    """
    ovp = getOfficialVideoPost(video_seq, session=session, silent=silent)
    if ovp:
        return ovp['officialVideo']

    return None
