#############################################################################
#
#   HEADER: saLibrary.py
#   SUMMARY: The saLibrary.py module implements all level and aspect of several 
#   core features of satool framework.
#   The main core features are series of user-interfacing API and API-helper functions
#   roughly can be categorized into following major features:
#   1. LOG MANAGEMENT - implements all aspects of logging feature in satool framework.
#   2. GLOBAL CONFIGURATION SETTING MANAGEMEN - implements the API for and manages configuration variables.
#   3. RUNTIME CONFIGURATION SETTING MANAGEMENT - implements the API for and manages run-time 
#   configuration variables.
#   
#   The sections below gives more details of each core features listed above:
#   
#   1. LOG MANAGEMENT DETAILS.
#   There are currently 3 different levels of logging is implemented.
#   with 1 more future implementation is awaiting.
#
#   - SUMMARY LOG - is the most concise log generated by running the test scripts
#   and normally used to inspect the result of running test scripts by giving bird's eye
#   view of the test script resuls. It is designed to be used for ~90% of the time by test
#   executer in order to take advantage of automation framework.
#
#   LEVEL-1 log - is more bit more verbose than the summary log that it contains
#   additional information and includes whole of summary log. When summary log does
#   not contain enough information regarding the test results, this level might
#   contain clue to the results.
#
#   LEVEL-2 log - is more verbose log and normally used for deep-level debugging
#   purposes.
#
#   (future implementation)
#   LEVEL-3 log - is the most verbose log and likely to gather almost all outputs
#   from test scripts, therefore tends to be large text size along with complete information
#   about how the test has been run. It is completely inadequate to scour over level 3
#   log to inspect the test result. The L-3 log contains more extensive information
#   about level 2 log in that it gathers the output as if all the function, API-s debug
#   switch has been turned off at once. The L-3 level log is used in extreme circumstances
#   to debug and troubleshoot most complicated technical problems.
#
#   The output file contains various level of logs are as follows:
#   generic format:
#   driver.py.<UCSM-IP><BLADE-LOCATION><DATE><TIME><LOG-LEVEL-SPECIFIER>.log
#   The summary log has no LOG-LEVEL-SPECIFIER as shown below:
#   Examples below shows the log file(s) generated by running the blade with ucsm-ip
#   10.193.225.211 locatd in 1/3 of chassis with time stamp: 2017 Sep 22 at 1:40pm.
#
#   which functions are used to output to which LOG-LEVEL?
#   The generic rules is that, the function that outputs to specific N will also
#   output to all levels that are more verbose than N and will not output to a log
#   that are less verbose. i.e. Function A outputs to LOG-1 will also output to LOG-2
#   but not summary log since it is less verbose.
#   
#   With this rule, following are list of functions and their LOG levels. The list might not be 
#   complete at times as more functions are added. 
#
#   1. summary log functions:
#   printWarning
#   printErr 
#   printInfoToFile
#   printSingBarInfo
#   printDoubleBarInfo (not that as per rules above, all the summary log functions will output 
#   to all other L1, L2 logs as they are more verbose.
#
#   2. L-1 level functions:
#   printWarnMinor
#
#   3. L-2 level functions:
#   printDbg
#   printSingleBar
#   printDoubleBar
#   printSeq
#   
#   Following functions are helper class functions to be called by the other print functions mentioned above, therefore
#   it is not recommended to call 'em directly:
#   
#   printToFile
#   printToFileL1
#   printToFileL2
#   printToFileL3
#   
#   When helper class functions mentioned above are not used by certain print functions, then their output is not 
#   logged to any log file. Those functions use python's standard print function and therefore their output can
#   only be captured from stdout terminal:
#
#   printAttn

#   2. GLOBAL CONFIGURATION SETTING MANAGEMENT - implements the API for and manages configuration variables.
#   Global confguration setting files contain, ideally, all configurable variables and parameters used throughout
#   the satool framework and serves as the one-spot-for-all place for all configurable variables. Sometime it might be
#   unrealistic ideally to transfer all configurable variables into one file given the sheer amount of coding involved
#   throughout the framework, however the feature is implemented with a following forward mentality:
#   - Provide user/developer to manage any configurable variable in one consistent place, transitioning specific setting and variable
#   to locally somewhere in a scattered manner to a global management one-spot-for-all place will involve minimal code and effort.
#   - No other mechanism or places or any other specific configurable file is allowed to prevent scattering of configurable variable.
#   
#   Implementation details - Since there many different models and blades are being managed different configuration files are created
#   for each blade model and location. However user rarely, if ever deals with the nuts and bolts of managing the file directly instead
#   series of API functions will let user to deal with only UCSM IP and blade location.
#   
#   The configuration management files consist of following:
#   1. There is single, central, generic configuration file called <PYTHON_ROOT_FOLDER>/api/config.txt - serves as a templace for all other 
#   blade-bound blade-specific configuration file. If certain blade has no specific configuration file created for it, them 
#   ./edit.config.py utility can be used to create new one by duplicating api/config.txt file. Therefore, api/config.txt serves as a
#   seed or template for any new conifugration file for specific blade configuration file and if any new configuration is introduced 
#   it is advised to place it in the api/config.txt so that all blade-specific confinguration files created in the future will inherit it.
#   Another important aspect of generic configuration file is to provide default value for any configuration value when the consumer 
#   of this configuration feature is looking for it in the blade specific configuration file and it is not found. In that case, if its default 
#   fail-safe value is defined in generic configuration file, corresponding API will potentially pick up the value from generic configuration file.
#   Still, there are certain configuration variable that preferably cause the test script to exit with error rather than moving away with default
#   value, therefore that choice is dictated by the CONFIG_LEVEL_<N> separator defined in the api/config.txt file. For more information regarding 
#   this aspect, refer to the header of api/config.txt where it is elaborated.
#
#   2. Blade specific configuration file is created and managed by several API function listed below:
#   getGlobal - will retrieve the particular configuration from blade specific configuration file. If not found, based on the CONFIG_LEVEL_<N>, setting
#   if might result in EXIT_ERR or may fetch the default value from generic configuration file.
#   ./edit.config.py can be used to create from api/config.txt or duplicate another blade specific configuration file. For more information, 
#   use ./edit.config.py --help. 
#   Example usage:
#   - Create new blade-bound blade specific configuration file located in <UCSM IP><LOC>: ./edit.config.py -c <UCSM IP> <LOC>.
#   - Open the blade-bound blade specific configuration file in text editor: ./edit.config.py -o <UCSM IP> <LOC>.
#   - Duplicate the blade-bound specific configuration file bound to <UCSM><LOC> to another configuration file bound to <UCSM1><LOC1>:
#   ./edit.config.py -c <UCSM> <LOC> <UCSM1> <LOC1>
#   If there are blades with different models swapped in the same slot, the different configuration files created for each blade model even though
#   the locations are same. This is to manage the blade configuration files differently based on the blade model. However it can not differentiate between
#   same blade models that are different in other aspect i.e. serial No., revision etc., That is if same blade  is replaced with another blade with same model
#   it can not differentiate. By utilizing this series of API-s and utilities, user has no need to manage and deal with the configuration files directly.
#   The configuration files are all created and managed in the location: <PYTHON_ROOT_DIR>/api/configuration/ with the naming convention:
#   config.<UCSM-IP><LOC><MODEL> i.e: config.10.193.225.121.1.8.UCSB-B200-M4

#   3. RUNTIME CONFIGURATION SETTING MANAGEMENT - implements the API for and manages run-time 

# Import modules section. 

import pexpect
import time
import re
import os
import sys
import string 
import inspect
import telnetlib

#for cid server:
import subprocess
import getpass

from time import gmtime, strftime
#from ucs import *

# Variables and constants definition section. 

expectTermination= ".*#.*"
sam = None
global ip
bmcIp = None
bladeSpName = None
argIp = None
argBladePos = None
fp = None
UNIFIED_LOG = 0
TIMEOUT_PREREQ_DISPLAY = 5
PYTHON_ROOT_DIR = os.environ.get('CURRENT_TREE')
FiListUsFileName = None

# RH1 server login credentials.

RH1_IP = "10.193.191.11"
RH1_USERNAME = None
RH1_PASSWORD = None

# Function API return value definitions. Those are the pre-defined return value definitions use for 
# only function api-s defined in api/*.py files. They are not the test return values for main test
# scripts residing in tests/*/*.py files. 

EXIT_ERR = None     # Most used definition to signal error, most API-s use to return whene there are error condition.
                    # Depending on the nature of the error, it may or may not impact the test result under way.
EXIT_FAIL = 4       # Normal logic error, used to differentiate from EXIT_ERR in some places.
EXIT_NO_ERR = -1    # Non-error condition but current function must exit.
SUCCESS = 1         # Success, usually indicate the current function call resulted in success if no return value is needed.
EXIT_WARN = 3       # Exit condition with success but add'l action may need to be depending on the calling function.
EXIT_PASSWD = 5     # Exit condition where password should be supplied in subsequent calls to cli_with_ret.

# Efi shell/linux command return statuses, used in "cli_with_ret".

EXIT_ERR_CMD_NOT_AVAIL  = 10 # command not available
EXIT_ERR_NO_PROMPT      = 11 # can not return back to efi shell prompt.

# Test script return value (return to test harness file: driver.py) definitions.
# These are pre-defined return values used by main test scripts (tests/*/*.py) 
# and defines the actual TEST results.

RET_PASS = 0        # final result is pass. No review is needed.
RET_PASS_E = 1      # final result is pass but there are some incidents occurred during the test that
                    # should be reviewed in the log. 
RET_FAIL = 2        # final result is fail. 
RET_BLOCK = 3       # final result is block, this is different than fail in that, error encountered
                    # during test does not necessarily mean fail logic, rather causing test unable to continue.
RET_INSPECT = 4     # when test is not intelligent enough to decide and return PASS/FAIL, result however 
                    # still needs the automated execution, might use this. Log review is needed.
RET_EXCEPT = 5      # test has coding or runtime error causing exception.
RET_SKIP = 6

RET_TERM = 50       # unable to restore blade to default state, in which it will affect the next script execution.
retFailReason = "Unknown error. "

# bit definitions used for easier bit manupulation

BIT0 = 1 << 0
BIT1 = 1 << 1
BIT2 = 1 << 2
BIT3 = 1 << 3
BIT4 = 1 << 4
BIT5 = 1 << 5
BIT6 = 1 << 6
BIT7 = 1 << 7
BIT8 = 1 << 8
BIT9 = 1 << 9
BIT10 = 1 << 10
BIT11 = 1 << 11
BIT12 = 1 << 12
BIT13 = 1 << 13
BIT14 = 1 << 14
BIT15 = 1 << 15
BIT16 = 1 << 16
BIT17 = 1 << 17
BIT18 = 1 << 18
BIT19 = 1 << 19
BIT20 = 1 << 20
BIT21 = 1 << 21
BIT22 = 1 << 22
BIT23 = 1 << 23
BIT24 = 1 << 24
BIT25 = 1 << 25
BIT26 = 1 << 26
BIT27 = 1 << 27
BIT28 = 1 << 28
BIT29 = 1 << 29
BIT30 = 1 << 30
BIT31 = 1 << 31
BIT32 = 1 << 32
BIT33 = 1 << 33
BIT34 = 1 << 34
BIT35 = 1 << 35
BIT36 = 1 << 36
BIT37 = 1 << 37
BIT38 = 1 << 38
BIT39 = 1 << 39
BIT40 = 1 << 40
BIT41 = 1 << 41
BIT42 = 1 << 42
BIT43 = 1 << 43
BIT44 = 1 << 44
BIT45 = 1 << 45
BIT46 = 1 << 46
BIT47 = 1 << 47
BIT48 = 1 << 48
BIT49 = 1 << 49
BIT50 = 1 << 50
BIT51 = 1 << 51
BIT52 = 1 << 52
BIT53 = 1 << 53
BIT54 = 1 << 54
BIT55 = 1 << 55
BIT56 = 1 << 56
BIT57 = 1 << 57
BIT58 = 1 << 58
BIT59 = 1 << 59
BIT60 = 1 << 60
BIT61 = 1 << 61
BIT62 = 1 << 62
BIT63 = 1 << 63

#   Following return codes are used in conjunction with TERM. If TERMINATION coding warrants
#   RET_TERM signaling any test scripts after current script to not run but we need to save
#   current script's result. For example, current scripts final result can be PASS however
#   restoring the token value was not successful therefore it can impact the execution of next 
#   test scripts, therefore it can warrant the use of TERM. In this caes, RET_TERM_PASS is used.
#   Therefore, RET_TERM_PASS means: current test has passed but failed in restoring the token(s')
#   or some other environmental value to a starting condition, therefore risking to affect and falsify
#   the result.

RET_TERM_PASS = RET_TERM + RET_PASS
RET_TERM_PASS_E = RET_TERM + RET_PASS_E
RET_TERM_FAIL = RET_TERM + RET_FAIL
RET_TERM_BLOCK = RET_TERM + RET_BLOCK
RET_TERM_INSPECT = RET_TERM + RET_INSPECT
RET_TERM_EXCEPT = RET_TERM + RET_EXCEPT

#   Construct easy-to-iterate list containins all test return values.

retStatus = {\
    RET_PASS: 'PASS', \
    RET_FAIL: 'FAIL', \
    RET_PASS_E: 'PASS WITH EXCEPTION AND CONTINUE', \
    RET_BLOCK: 'BLOCK', \
    RET_INSPECT: 'INSPECT', \
    RET_EXCEPT: 'EXCEPTION', \
    RET_TERM_PASS: 'PASS AND TERMINATE', \
    RET_TERM_FAIL: 'FAIL AND TERMINATE', \
    RET_TERM_PASS_E: 'PASS WITH EXCEPTION AND TERMINATE', \
    RET_TERM_BLOCK: 'BLOCKED AND TERMINATE', \
    RET_TERM_INSPECT: 'INSPECT AND TERMINATE', \
    RET_TERM_EXCEPT: 'EXCEPTION AND TERMINATE' \
}

#   Maximum number of lines to read for global tmp file.

CONFIG_GLOBAL_TMP_MAX_LINES = 400

#   Define text colors.

class colors:
    HEADER = '\033[95m'
    BOLD = '\033[1m'
    BLUE = '\033[94m'
    GREEN = '\033[92m'
    FAIL = '\033[91m'
    WARNING = '\033[93m'
    LIBLUE = '\033[96m'
    WHITEBOLD = '\033[97m'
    ENDC = '\033[97m'
    LIGREEN = '\033[96m'
    PURPLE = '\033[95m'
    GRAY = '\033[90m'

#   Used in certain print function, the amount of text width cutoff to align the caller and function
#   name part of the printout.

CONFIG_CALLER_FILE_CUTOFF_LEN = 20
COFNIG_CALLER_FUNCTION_CUTOFF_LEN = 25

#   Error levels are used to lower or raise the severity of certain messages.
#   e.g. Certain error messages can be interpreted as error that is leading to 
#   a real error condition. However the same error message can also be used 
#   simply as a check purpose in which case the failure condition simply
#   imply the check rather than error. Since each error and warning messages
#   are color-coded, it might be preferably to display the error messages 
#   that are not worthy of implying a serious or fatal error in a warning 
#   color code or some other colors. The following definition will encode several
#   levels of error levels that can customize the error levels of certain printouts.

ERROR_LEVEL_PRINT = 1           # normal informational messages. 
ERROR_LEVEL_WARN_MINOR = 2      # minor level warning messages.
ERROR_LEVEL_WARN = 3            # warning messages.
ERROR_LEVEL_ERROR = 4           # error messages.
ERROR_LEVEL_FATAL = 5           # fatal error messages. 

#   Obsolete class. Delete after testing.

class clsGlobalConfig:
#   clsConfigList = {}
    clsPid = None
    clsConfigFile = None

#   Checks function arguments.
#   Outputs attention style message where the output is enclosed in double hashed space.
#   input:
#   - clientFcn - client function whose arguments will be checked.
#   return:
#   - wrapper function object back to caller 

def argChecker(clientFunc):

    # define wrapper function inside which introduces iterating over arguments (variable)
    # and prints the argument if it has value and None! if it doesn't"
    # After that client function is called.

    def wrapper(*params):
        counter = 0
        debug = 0

        for i in params:
            if i == None:
                printWarn("param" + str(counter) + " is None!")
            else:
                printDbg("param" + str(counter) + " is OK: " + str(i))

            counter += 1            
        time.sleep(1)
        clientFunc(*params)

    return wrapper

#   Prepares, processes command line arguments.
#   The pSysArgv is the all values of sys.argv as it is. This is what the
#   users of particular enters in the command line.
#   The pSupportedArgv is the tuple list of supported arguments in dictionary format passed
#   on by scripts and therefore its contents are specific to a particular script's implementation.
#   However following universal convention are used throughout to enable uniform processing
#   for any or almost any types, variants and number of switches and arguments.
#   Dictionary key holds the name of the argument in the naming convention: "--<argName>"
#   Dictionary value holds the tuple list object, serving as an initial indicator implying how many 
#   values the arg needs (refer to examples below):
#   Current implementation divides the tuple values as two types in general:
#   1. (<value1>, <value2>) - implies literal processing, which means the supportedArgs lists out the
#   actual values that user can enter in the command line for that particular switches and no other
#   allowed value. Example is if particular scripts --drive switches accepts only C: D: E: drive 
#   letters and any other drive letters are not allowed.
#   2. ('__range__', <list>) - implies that there is no restrictions on the values provided by user
#   for particular switch, however the <list> contains the range of number of allowed values for that 
#   particular switch. This is useful in case, it is not known what value user will be putting at the 
#   command line or there too many possibilities for particular variable.
#   For example, if --ip switches requires one value which is IPv4 address but the range of valid
#   IPv4 is too many to listed out as literal however --ip will accept only one value, in this case
#   supported switch is declared as: ('__range__', 1)
#   if --ip allows up from one to up to 5 different IP address following, (as in the case:
#   --ip 10.24.11.0 10.0.0.1 12.0.0.1 192.168.0.1 172.17.0.0
#   then it should be declared and passed as:
#   ('__range__', 1, 2, 3, 4, 5)
#   or possible:
#   ('__range__', range(0, 5))
#
#   pSupportedArgs typical examples are below but not limited to:
#   -------------------------------------------------------------------------------------
#   Arg name     Initial indicator of value count, tuple list type   Valid user input
#   -------------------------------------------------------------------------------------
#   '--timeout': (1,),  (needs one and only one value for argument)   
#                                                           -> user input --timeout 100
#   '--timeout': ()     (needs no value)                    -> user input --timeout 
#   '--timeout': (1,2,3)(needs somewhere between 1 to 3 values, inclusive)
#                                                           -> user input --timeout 1    
#                                                           -> user input --timeout 1 100 2    
#                                                           -> user input --timeout 1 100
#   !!! This is currently unimplementable, needs to look!!! Currently it is not supported.
#   '--timeout': [1:CONFIG_MAX_ARG_VALUES] (needs somewhere between 1 to CONFIG_MAX_ARG_VALUES
#   number of values, which effectively implies "greater than 1")
#                                                           -> user input --timeout 1    
#                                                           -> user input --timeout 1 100 2
#                                                           -> user input --timeout 1 100
#                                                           -> user input --timeout 1 2 3 4 5 6 7   
#                                                           -> user input --timeout 1 2 3 4 5 6 7 10 11
#   -------------------------------------------------------------------------------------
#   Once the arguments are processes and validated, the values of the dictionary will be populated
#   with the user supplied values from the command line. If user has not provided the argument, its
#   corresponding dictionary value remain unchanged, tuple list type, implying that the caller script 
#   will not process it.
#
#   input:
#   - pSysArgv - sys.argv() as it is.
#   - pSupportedArgs - dictionary type consisting of tuple list of argument that particular calling scripts supports.
#       - key holds the name of the argument.
#       - tuple list type holding indication of how many values are needed for the argument
#        (for details, refer to function header)
#   - helpStringArr - if user input --help argument, use this string to display the help message.
#   output:
#   - pSupportedArgs' values are populated with the validated command line argument values as list type.
#   - if particular argument is not provided by user, value remains unchanged as list tuple type.
#   - EXIT_NO_ERR if --help is supplied.
#     EXIT_ERR on any error condition.

def prepArgs(pSysArgv, pSupportedArgs, helpStringArr):
    debug = 0
    idx = 0                         # holds the index of supported argument being processed during the loop.
    RANGE_CHECK = 1                 # if this value is assigned, each switch's child parameter will be checked against the range specified in the supportedArgs.
    VALUE_CHECK = 2                 # if this value is assigned, literal values in the supportedArgs will be checked against. 
    argCheckFlag = VALUE_CHECK 
    values = None
    firstSwitchCheck = 0

    if debug:
        printDbg("pSysArgv: ")
        printSeq(pSysArgv, 4)
        printDbg("pSupportedArgs: ")
        printSeq(pSupportedArgs, 4)

    # Iterate through each command line args-.

    for i in range(0, len(pSysArgv)):
        argCheckFlag = VALUE_CHECK 
        printDbg("------------------------------------------------ ---", debug)
        #printDbg(pSysArgc[i]: " + str(pSysArgv[i]), debug)
        pSysArgv_i = pSysArgv[i]

        if debug:
            printVars([i, firstSwitchCheck, pSysArgv[i]])

        if re.search("--", pSysArgv[i]):
            printDbg("Processing " + str(pSysArgv[i]), debug)
            printDbg("From : " + str(pSysArgv), debug)

            # If this is help, then display help and quit. 
        
            if pSysArgv[i] == '--help':
                printHelp(helpStringArr)
                return EXIT_NO_ERR

            # Make sure it is supported arg.

            if not pSysArgv[i] in pSupportedArgs.keys():
                printErr(pSysArgv[i] + " is not supported arguments.")
                printErr("Supported args: ")
                printSeq(pSupportedArgs, 4)
                return EXIT_ERR

            # Get the index of matching arg from supportedArg dictionary,

            idx = pSupportedArgs.keys().index(pSysArgv[i])

            printDbg("idx (index of " + str(pSysArgv[i]) + " in pSupportedArgs.keys(): " + str(idx), debug)

            if not type(pSupportedArgs.values()[idx]) == tuple:
                printErr("Wrong type for supported Arg values: ")
                print type(pSupportedArgs.values()[idx])
                return EXIT_ERR

            # If value is none then this specific parameters takes no values,
            # If not none, process further.

            if len(pSupportedArgs.values()[idx]) == 0:
                printDbg("No values are needed for " + str(pSysArgv[i]), debug)
                pSupportedArgs[pSysArgv[i]] = 1
            else:
            
                # sysArgvCurrToNextDashDash will extract first two letters of all arguments from 
                # sysArgv's members ranging from i+1 (since i-th one we know already contains --)
                # This is used in finding a next occurrence of argument starting with -- so that
                # all values between current argument and next occurrence will be assumed for values 
                # for current arg. 
                
                sysArgvCurrToNextDashDash = [j[0:2] for j in pSysArgv[i+1:]]
                sysArgvCurrToNext = [j[:] for j in pSysArgv[i+1:]]
                printDbg("sysArgvCurrToNextDashDash: " + str(sysArgvCurrToNextDashDash), debug)

                if '--' in sysArgvCurrToNextDashDash:
                
                    # Once it is found (the next occurrence), the idxEnd will hold the index of sysArgvCurrToNextDashDash
                    # Note that idxEnd is not the index from the beginning of all arguments, the idxEnd is relative to 
                    # current iteration of sysArgv whereas idxEndAbs will find the absolute index value from all the way 
                    # to the beginning of sysArgv. 
                    # If no argument with -- is found as next occurrence, we assume all the args between current one to the 
                    # end of sysargv are considered values. 
                
                    idxEnd = sysArgvCurrToNextDashDash.index('--')
                    idxEndAbs = idxEnd + i + 1
                    printDbg("Further occurrence of -- found, taking its index: " + str(idxEnd), debug)
                else:
                    idxEnd = len(sysArgvCurrToNextDashDash)
                    idxEndAbs = len(pSysArgv)
                    printDbg("No further occurrence of -- found, taking index of last member of sysArgvCurrToNextDashDash", debug)

                printDbg("idxEnd: "  + str(idxEnd), debug)
                printDbg("idxEndAbs: "  + str(idxEndAbs), debug)

                currParamSuppValues = pSupportedArgs.values()[idx]

                printVar(currParamSuppValues, debug)

                if currParamSuppValues[0] == '__range__':
                    printDbg("argCheckFlag: Range check flag is set.", debug)
                    currParamSuppValues = currParamSuppValues[1:]
                    argCheckFlag = RANGE_CHECK
                
                else:
                    printDbg("argCheckFlag: not changed.")

                printVar(currParamSuppValues, debug)
                pSupportedArgs.values()[idx] = currParamSuppValues

                # Check to see if the values are in the accepted range of values. This check is only done with non-integer values.

                if len(sysArgvCurrToNext[:idxEnd]) == 0 and len(currParamSuppValues) != 0:
                    printErr("1. " + pSysArgv[i] + " accepts range of values: ")
                    printErr(str(currParamSuppValues))
                    return EXIT_ERR

                for m in sysArgvCurrToNext[:idxEnd]:
                    if argCheckFlag == VALUE_CHECK:
                        printDbg("Literal value check...", debug)

                        if not m in currParamSuppValues:
                        
                            # Special parameters where user input does not have to match the pre-defined values for specific param-s. 
                            
                            print "Checking if ", m, " in the range..."
                            printErr(pSysArgv[i] + ":" + str(m) + " is not in the accepted range of values: ")
                            print currParamSuppValues
                            return EXIT_ERR
                    elif argCheckFlag == RANGE_CHECK:
                        printDbg("Range check...", debug)

                        if not (idxEnd) in currParamSuppValues:
                            printErr("2. " + pSysArgv[i] + " must be followed by following number of value: ")
                            printErr(str(list(currParamSuppValues)))
                            return EXIT_ERR
                        else:
                            if debug:
                                printDbg("Range check OK: idxEnd" + str(idxEnd) + " is within ", debug)
                                printN(currParamSuppValues)

                    else:
                        printErr("Unsupported values for argCheckFlag: " + str(argCheckFlag))
                        return EXIT_ERR
                    
                    values = pSysArgv[i+1:idxEndAbs]
                
                    printDbg("values/pSysArgv[" + str(i+1) + ":" + str(idxEndAbs) +  "]: ", debug)
                    print values

                pSupportedArgs[pSysArgv[i]] = values

                if debug:
                    printDbg("pSupportedArgs at the end of this loop: ", debug)
                    print pSupportedArgs
        else:
            printDbg("Not starting with --, doing add'l check.", debug)

            # If current keyword does not start withh --, then we need to do an add'l check, before skipping:
            # making sure it is not first time checking the current switch.
            # Making sure it is not first switch after location or ucsm ip.
            # make sure it is IP.
            # make sure it is location.

            if i == 0:
                printDbg("1. Skipping " + str(pSysArgv[i]), debug)
            elif firstSwitchCheck == 1:
                printDbg("2. Skipping " + str(pSysArgv[i]), debug)
            elif validateIp(pSysArgv[i], ERROR_LEVEL_WARN_MINOR):
                printDbg("3. Skipping " + str(pSysArgv[i]), debug)
            elif validateBladeLoc(pSysArgv[i], ERROR_LEVEL_WARN_MINOR):
                printDbg("4. Skipping " + str(pSysArgv[i]), debug)
            else:
                printDbg("Previous switch check.")
                if validateIp(pSysArgv[i-1], ERROR_LEVEL_WARN_MINOR) or validateBladeLoc(pSysArgv[i-1], ERROR_LEVEL_WARN_MINOR):
                    printErr("Expected switch (starting with --) after blade location or UCSM IP: ")
                    return EXIT_ERR

                firstSwitchCheck += 1

    if debug:
        printDbg("Returning pSupportedArgs: ")
        printSeq(pSupportedArgs, 4)
    return pSupportedArgs

#   Outputs attention style message where the output is enclosed in double hashed space.
#   input:
#   - pStr - string to be printed. If string is hugely long, it is split into smaller pieces
#   and resulted in multiple lines of output.

def printAttn(pStr):
    pStrSplit = None
    os.system('clear')
    LINE_SIZE = 80

    if not pStr:
        printErr("pStr is empty!")
        return EXIT_ERR

    printN(colors.WARNING + "================================================================================" + colors.ENDC)

    for i in range(0, len(pStr), LINE_SIZE):
        printN(colors.WARNING + str(pStr[i:i+LINE_SIZE]))

    printN(colors.WARNING + "================================================================================" + colors.ENDC)
    printN(colors.ENDC + "...")

#   Outputs the certain output enclosed in single line. This is mostly used in debugging purpose to see 
#   the distinct start and end of certain string. BA signify before and after.
#   - pStr - string to be enclosed and printed.

def printWithBarSingleBA(pStr):
    printBarSingle()
    printDbg(pStr)
    printBarSingle()
    return SUCCESS

#   Reads gpu pin of bmc. This function will return only 1 or 0 as a valid return.
#   In any error case, it returns EXIT_ERR
#   input:
#   - pBmcConsole - either telnet or ssh connection to bmc.
#   - pGpioFile - full path of gpio path.
#   return:     
#   - content of gpio file, either 1 or 0 as int.
#   - EXIT_ERR - on any error condition.

def bmcGpioRead(pBlade, pBmcConsole, pGpioFile):
    debug = 0

    for i in range(0, 6):
        ret1 = pBlade.bmcSendCommand(pBmcConsole, "cat " + pGpioFile)
        
        if debug: 
            printDbg("return as split: ")
            print ret1.split()
        try:
            if re.search("telnetlib", pBmcConsole.__module__):
                ret1 = int(ret1.split()[-7])
            else:
                ret1 = int(ret1.split()[-6])
            printDbg("gpio " + str(pGpioFile) + " stat: " + str(ret1))
            return ret1
        except Exception as msg:
            printWarn("Retry " + str(i) + ". Error translating the gpio content.")
            time.sleep(3)
    return EXIT_ERR


#   Parse memory struct.
#   Generic function for parsing the in-memory structure, based on predefined format.
#   input:      
#   - pBlade - blade instance object.
#   - pAddress - starting address for in-memory struct.
#   - pPcie_inst - pcie instance object holding ssh connection.
#   - pNames - list containing names of struct.
#   - pSizes - list containing size of each struct field.
#   - pOffset - list containins offset of each struct field. (might be redundant).
#   - optional, if supplied must be absolute offset from the beginning of the struct.
#   return:     
#   - dictionary object containing the parsed structure names and values.
#   - EXIT_ERR if any error occurred.

def parseMemStruct(pBlade, pAddress, pPcie_inst, pNames, pSizes, pOffsets = None):
    debug = 0

    # currFieldsize and currFieldSizeRead will contain field size but latter will contain the actual 
    # read which can be less if bytes in the field are more than 4 bytes long. Therefore currFieldSize
    # is used to hold the actual field size and used for walk through.

    currFieldSize = None
    currFieldSizeRead = None
    currOffset = None

    index1 = None
    lDictStructData = {}
    token1Size = None
    token2Multiplier = None

    # Validate inputs.

    if pOffsets == None:
        printWarn("Currently the function still requires pOffsets. This is not necessarily error condition.\
        Future implementation needed to not require pOffsets.")
        return EXIT_ERR

    if len(pNames) != len(pSizes) or len(pNames) != len(pOffsets):
        printErr("Following list must be same size: pNames, pSizes, pOffsets: " + \
            str(pNames) + ", " + str(pSizes) + ", " + str(pOffsets))

        return EXIT_ERR

    currOffset = 0
    lAddressToRead = str(hex(int(pAddress, 16)))

    # For each offsets, iterate through.

    for i in range(0, len(pSizes)):
        if debug:
            printBarSingle()
            printDbg("Processing pNames, pSize: " + str(pNames[i]) + ", " + str(pSizes[i]), debug)

        # pSizes list can either have integer (static) value or string (for field whose size
        # depends on other field in the same list). For the case of integer, it is simple: use the field size
        # If this is string the format is "<fieldSize|multiplier>" is supported. For example if field size is
        # "KeySize * 4" then look for same field name in the acmStrcutFieldNames and use its index to obtain the
        # field size from pSizes list and multiply with multipler value.
        # It is possible other string format is supported but currently not supported (future enhancement possible
        # to support these formats:
        # <fieldsize> +/- <value>
        # <fieldsize> / <value.>
        # Note: This section should be separate function so that it can be used to parse not just ACM but other
        # memory structures.

        # If integer, set currFieldSize to it.

        if type(pSizes[i]) == int:
            printDbg("current field size type is int", debug)
            currFieldSize = pSizes[i]
            currFieldSizeRead = pSizes[i]
            printDbg("1. currFieldSize is set to: "  + str(currFieldSize), debug)

        # If string, split by * first token signifies field name whose index is to be used to index to proper
        # field offset.
    
        elif type(pSizes[i]) == str:
            printDbg("current field size type is str", debug)

            # Determine the field index in pSizes based on acmStructFileNames[].
            # Determine the field multiplier.

            token1Size = pSizes[i].split("*")[0]
            token2Multiplier = int(pSizes[i].split("*")[1])

            try:
                index1 = pNames.index(token1Size)
                printDbg("found index of field name: " + str(index1))

                if index1 == None:
                    printErr("Unable to find index for " + str(token1Size))
                    return EXIT_ERR

            except ValueError:
                printErr("Unable to determine the index of " + str(pSizes[i]))
                return EXIT_ERR

            # Use index to extract the corresponding value from pOffsets.

            currFieldSize = token2Multiplier * pSizes[index1]
            pSizes[i] = token2Multiplier * pSizes[index1]
            printDbg("2. currFieldSize is set to: "  + str(currFieldSize), debug)

        else:
            printDbg("Unknown field size: " + str(type(pSizes[i])))
            return EXIT_ERR

        if currFieldSizeRead > 4:
            printWarn("Current implementation: Showing only first 4 bytes for " + str(pNames[i]))
            currFieldSizeRead = 4

        printDbg("lAddressToRead just before reading: " + str(lAddressToRead), debug)

        if currFieldSizeRead == 4:
            lDictStructData[pNames[i]] = pPcie_inst.read_mem_dword(pBlade, lAddressToRead)
        elif currFieldSizeRead == 2:
            lDictStructData[pNames[i]] = pPcie_inst.read_mem_word(pBlade, lAddressToRead)
        elif currFieldSizeRead == 1:
            lDictStructData[pNames[i]] = pPcie_inst.read_mem_byte(pBlade, lAddressToRead)
        else:
            printErr("Unsupported field size. It should be either 4, 2 or 1:" + str(currFieldSizeRead))
            return EXIT_ERR

        # Update lAddressToRead to point to next field, it holds absolute address of address to read.
        # currOffsets updated to point past current field, it holds distance of currOffset relative 
        # to struct address.

        currOffset += pSizes[i]
        #lAddressToRead = str(hex(int(lAddressToRead, 16) + currOffset))
        lAddressToRead = str(hex(int(lAddressToRead, 16) + pSizes[i]))

        printDbg("currOffset is set to: " + str(currOffset), debug)

    printDbg("Struct data collected: ", debug)

    for i in pNames:
        print i + ": " + lDictStructData[i]

    return lDictStructData

# Prints single or double bar to summary log file.
# req:  None.
# input: None
# output: None.

def printBarSingleInfo():
    printInfoToFile('-----------------------------------')
    printN('-----------------------------------')

def printBarDoubleInfo():
    printInfoToFile('===================================')

# Prints single or double bar to L-2 log file.
# req:  None.
# input: None
# output: None.

def printBarSingle():
    printN('-----------------------------------')

def printBarDouble():
    printN('===================================')

# Sleep for one hour however giving option to use to get out of sleep state any time
# by ctrl-c. 
# req:      None
# input:    None
# return:   None

def sleepWithInt():
    debug = 0
    printDbg("sleeping for 1 hour. press CTRL-C to get out of loop")
    i = 0

    while(i < 3600):
        try:
            time.sleep(3)
            i += 3
        except KeyboardInterrupt:
            printDbg("keyboard int, breaking")
            break

#   Helper function for obtaining the configuration level for the specific configuration parameters
#   in getConfigLevel function. This function is the actual scan line-by-line implementation
#   of it and is called by getConfigLevel() function, as such this function should not be 
#   called directly.
#   Configuration levels are one of three levels:
#   CONFIG_LEVEL_STRICT
#   CONFIG_LEVEL_MEDIUM
#   CONFIG_LEVEL_LOOSE and return value of this function will be one of these values if the pKey
#   configuration belonged to one of these levels. Otherwise EXIT_ERR is returned.
#
#   req:      
#   - pFpConfig - file pointer must be pointing past GLOBAL_CONFIGURATION title. Otherwise it is unpredictable.
#   input:    
#   - pKey - configuration setting whose levels are being determined.
#   - pFpConfig - file pointer
# output:   
#   - if levels are found, then value of the level: CONFIG_LEVEL_<STRICT|MEDIUM|LOOSE>
#     EXIT_ERR - if configuration level can not be determined or any other error.

def getHelpConfigLevel(pFpConfig, pKey):
    line = 1
    CONFIG_REQUIREMENT_MAX_LINES = 400
    counter = 0
    debug = 0
    debugL2 = 0
    configLevel = EXIT_ERR

    printDbg("entry", debug)
    printDbg("pKey: " + str(pKey) + " fileName being read: " + str(pFpConfig.name), debug)

    # For each line, grap the line if it start with CONFIG substring and divide into its key an value pair.

    while line:
        line = pFpConfig.readline()

        printDbg("line: " + str(counter), debugL2)

        if re.search("CONFIG_LEVEL", line):
            configLevel = line.split("=")[-1]
            printDbg("current configLevel is set to " + str(configLevel), debug)

        if re.search(pKey, line):
            printDbg("found the " + str(pKey) + " at the level: " + str(configLevel), debug)

            try:
                int(configLevel)
                printDbg("configLevel is successfully determined: " + str(configLevel), debug)    
                return int(configLevel)
    
            except TypeError:
                printErr("configLevel " + str(configLevel) + " can not be converted to integer")
                return EXIT_ERR

        counter += 1

        if counter > CONFIG_REQUIREMENT_MAX_LINES:
            printErr("Unable to find configuration level for " + str(pKey))
            return EXIT_ERR

#   Obtain the configuration level for the specific configuration parameters.
# 
#   Configuration levels are one of the three levels:
#   CONFIG_LEVEL_STRICT
#   CONFIG_LEVEL_MEDIUM
#   CONFIG_LEVEL_LOOSE
#   Currently it will look up in api/config.txt file for level.
#   input:    
#   - pKey - configuration setting whose levels are being determined.
#   output:   
#   - if levels are found, then value of the level.
#     EXIT_ERR - if configuration settings does not belong to any of the level.

def getConfigLevel(pKey):
    debug = 0
    debugL2 = 0
    fpConfigName = None
    fpConfig = None
    CONFIG_MAX_EMPTY_LINES = 10
    configLevel = None

    printDbg("Entered: key:" + str(pKey), debug)
    PYTHON_ROOT_DIR = os.environ.get('CURRENT_TREE')
    globalConfigList = {}

    # Get the config file name from global tmp config file and open it.

    fpConfigName = "api/config.txt"

    if re.search("UCSB|N20|BASE", fpConfigName):
        fpConfig = open(PYTHON_ROOT_DIR + "api/configuration/" + fpConfigName, 'r')
    else:
        fpConfig = open(PYTHON_ROOT_DIR + fpConfigName, 'r')

    if fpConfig == None:
        printErr("file system error: unable to open file api/config.txt")
        return EXIT_ERR

    while 1:
        line = fpConfig.readline()
    
        if line == "" or line == None:
            printDbg("..skip(E)Line..", debugL2)
            counter += 1
    
            if counter > CONFIG_MAX_EMPTY_LINES:
                printWarnMinor("More than " + str(CONFIG_MAX_EMPTY_LINES) + " empty lines are read\
                when setting " + str(pItem) + ". Reduce the number of empty lines from " + str(fpConfigName))
                return EXIT_ERR

        # Skip commented out line.

        elif re.search("^#", line):
            printDbg("..skip(#)Line..", debugL2)
            continue
        else:
            counter = 0
    
            # If GLOBAL_CONFIGURATION section is found, use getConfigDictionary() to fetch the      
            # configuration setting value pairs in dictionary format.        
    
            if re.search("GLOBAL_CONFIGURATION", line):
                printDbg("Determining config level...1", debug)
                configLevel = getHelpConfigLevel(fpConfig, pKey)
                printDbg("config level returned: " + str(configLevel), debug)
                break            
    if configLevel == 1:
        printDbg(str(pKey) + " is CONFIG_STRICT", debug)
    elif configLevel == 2:
        printDbg(str(pKey) + " is CONFIG_MEDIUM", debug)
    elif configLevel == 3:
        printDbg(str(pKey) + " is CONFIG_LOOSE", debug)
    else:
        printErr("Invalid configuration level for " + str(pKey) + ". (configLevel= " + str(configLevel) + ". Error determining config level.")
        return EXIT_ERR
    return configLevel
            
#   MSR register read function, upon successful read of particular MSR, it returns the msr value of the register.
#   when there are multiple cores (likely in all cases), the result is based on strict setting
#   req:      
#   - the blade must have been booted to efi shell prior to running this function.
#   input:    
#   - pBmcSsh - ssh connection to kvm
#   - pMsr - msr register
#   - pStrict = 1 - all cores must return same result to return the value, otherwise EXIT_ERR (default)
#               0 - first core value will be returned with regardless of rest.
#   return:   
#   - list <msrValue, msrCount>
#           msrValue in 64-bit int value "NNNNNNNNNNNNNNNN" in hex if read is successful (refer to pStrict input parameter)
#           msrCount - type int, number of CPU, notice this is note the number of cores rather cores*socket*threads.
#           EXIT_ERR if any failure.

def read_msr(pBmcSsh, pMsr, pStrict = 1):
    debug = 1
    debugL2 = 0
    outpTokens = None
    outpTokens1 = []
    threadsErr = []

    if validateFcnInput([pBmcSsh, pMsr]) == EXIT_ERR:
        printErr("invalid function input.")
        return EXIT_ERR

    outp = cli_with_ret(pBmcSsh, "msr " + str(pMsr), "", "efiShell")

    if outp == None:
        printErr("MSR read fail.")
        return EXIT_ERR

    printDbg("outp:\n" + str(outp), debug)
    printDbg("1. outp: ", debug)
    printPexBa(pBmcSsh, 1, outp)

    outpTokens = outp.strip().split('\n')

    flag1 = 0
    flag2 = 0
    counter = 0

    # extract the lines past ===== into outpTokens1 list.

    for i in range(0, len(outpTokens)):
        printDbg("\noutput line " + str(i) + ": " + str(outpTokens[i]), debugL2)

        if re.search("=================", outpTokens[i]) and flag1 == 0:
            flag1 = 1
            printDbg("flag1 0->1 @ idx: " + str(i) + ", line: " + str(outpTokens[i]), debug)
        
        if flag1:
            if not re.search("=================", outpTokens[i]) and flag2 == 0:
                flag2 = 1
                printDbg("flag2 0->1 @ idx: " + str(i) + ", line: " + str(outpTokens[i]), debug)

        if flag1 == 1 and flag2 == 1:
            outpTokens1.append(outpTokens[i])
        else:    
            printDbg("skipping." + str(outpTokens[i]), debug)
   
    printDbg("2. outpTokens1: ", debug)
    
    if debugL2:
        for i in outpTokens1:
            print outpTokens1

    lenOutpTokens = len(outpTokens)
    lenOutpTokens1 = len(outpTokens1)

    try:
        printVars([lenOutpTokens, lenOutpTokens1], debug)
        printDbg("outpTokens1[0]: " + str(outpTokens1[0]), debug)
    except IndexError:
        printErr("Error parsing MSR output")
        return EXIT_ERR

    msrCore0Value = outpTokens1[0].split()[3].strip() + outpTokens1[0].split()[4].strip()

    printVar(msrCore0Value, debug)

    # if strict flag is set, then every core's value must be same.
    # If not, then return error.

    threadsErr = []

    if pStrict:
        for i in outpTokens1:
            if i.split()[3] != outpTokens1[0].split()[3] or \
                     i.split()[4] != outpTokens1[0].split()[4]:
                printErr("This thread has different value than first core/thread: " )
                printErr(outpTokens1[0])
                printErr(i)
                threadsErr.append(outpTokens1[0])

            if len(threadsErr) != 0:
                printErr("Threads whose value does not match thread 0")
                return EXIT_ERR

    printDbg("Returning: ", debug)
    print  [msrCore0Value, lenOutpTokens1]
    return [msrCore0Value, lenOutpTokens1]

#   Implements uniform management of test return values, maintains the return status value across various test phases. 
#   Through out test script execution, various condition requires to change/update the FINAL return state, 
#   while still need to continue running the script.
#   Particular script can consist of multiple sub-tests and situation can arise where one of them fails 
#   subsequent ones will need to continue. This naturally will create a situation where there is a several subtests
#   or phases of test with differing results as well as one final result. 
#   The final test result must be the "lowest common denominator" of all subtest results, that is worst one
#   to prevent any test escape. 
#   If certain sub test or phases of test fails or blocked, some subsequent part of test may attempt to lift the 
#   final result into a "better" i.e.  PASS or PASS_E result which should not be allowed which implies the danger of "test escape". 
#   The management of final test result setting in the actual script body can result in unmanageable convoluted code, therefore any test 
#   scripts code that wants to update the final result should instead call this function with desire result value. Within this 
#   function any logic needed to determine the permissibility of converring return status from one value to desired value is 
#   implemented.

#   Following rules are implemented.
#   RET_STATUS = holds must up-to-date return status to be returned back to driver.
#   if RET_STATUS equal RET_TERM, it can not change to any other status.
#   if RET_STATUS equals RET_BLOCK, it can not change to any other status except RET_TERM.
#   if RET_STATUS equals RET_FAIL, it can be only changed to RET_BLOCK or RET_TERM.
#   if RET_STATUS equals RET_PASS_E (pass w exemption), it can only be changed to RET_BLOCK, RET_FAIL and RET_TERM
#   if RET_STATUS equal RET_PASS, it can be changed to any state
#   The overall function of this rule is that once particular state is degraded to lower status, (PASS to FAIL),
#   it can not be put back into better state (FAIL to PASS_E), otherwise test escape condition will happen.
#
#   req:        None
#   input:      
#   - pStatCurr - current status.
#   - pStatToSet - status to update the current status to.
#   return      
#   - updated status after running through rules.

def setReturnStat(pStatCurr, pStatToSet):
    debug = 0
    caller = None

    caller  = inspect.stack()[2][1]

    # Check both values are pre-defined value in retStatus list.

    if not pStatCurr in retStatus:
        printErr("current status is not defined, leaving current status unchanged, rest of test execution might not be valid.")
        return pStatCurr

    if not pStatToSet in retStatus:
        printErr("current status is not defined, leaving current status unchanged, rest of test execution might not be valid.")
        return pStatCurr
       
    printDbg("Entered: pStatCurr, pStatToSet: " + str(pStatCurr) + "/" + str(retStatus[pStatCurr]) + ", " + str(pStatToSet) + "/" + str(retStatus[pStatToSet]))
    printDbg("Caller: " + str(caller), debug)

#   If the return status is intended to be elevated to a "better" result, this is not allowed since it will cause a potential false positive
#   result. "Better" result has a smaller integer values than "worse". For more information, refer to a definition of retStatus list.

#   Otherwise assign the return status to a desired status. If desired status is RET_TERM then offset the return status with RET_TERM value.

    if pStatToSet < pStatCurr:
        printErr(retStatus[pStatCurr] + " -> " + retStatus[pStatToSet] + " is not allowed.")
        return pStatCurr
    else:
        if pStatToSet == RET_TERM:

            # Before offsetting with RET_TERM, we need to determine whether the return value is already one of the RET_TERM_<VALUE>
            # values, in which case, we need to subtract the RET_TERM to obtain "non-RET_TERM" values first before adding again.

            if pStatCurr >= RET_TERM:
                pStatCurr -= RET_TERM

            printInfoToFile("setting " + retStatus[pStatCurr] + " -> " + retStatus[pStatToSet + pStatCurr] + ".")
            return (pStatToSet + pStatCurr)
        else:
            printInfoToFile("setting " + retStatus[pStatCurr] + " -> " + retStatus[pStatToSet] + ".")
            return pStatToSet

    # Unable to restore blade to default state, in which it will affect the next script execution.

    try:        
        printDbg("Invalid state transition requested: " + str(retStatus[pStatCurr]) + " -> " + str(retStatus[pStatToSet]) + ". returning RET_STATUS unchanged.", debug)
    except TypeError:
        printErr("Can not map the status")

    return pStatCurr        

#   Delete the key/value pair in the config file created run-time during script execution time. 
#   Entries must be in the format <pItem><delimiter><value> in the pFileName. It is primarily
#   used for managing the runtime settings in the  temporary run time global configuration file 
#   This function server as a helper function for the user-interfacing function called delGlobalTmp.
#   delRunTimeVar should be carefully used by user as it requires user to manage his/her own file in which 
#   settings are managed in a key delimiter value pairs. If the temporary run time global configuration file
#   settings are managed, then it should use delGlobalTmp() wrapper which in turn calls this function.
#   req:      None
#   input:    
#   - pFileName - filename from which item to be deleted.
#   - pValidDelim - delimiter for the line to be considered valid.
#   - pItem - item to be deleted output: SUCCESS - item is found and deleted.
#   return:
#   -  EXIT_ERR - items is found and failed to delete for any reason or any other reason 
#      than conditions specified in SUCCESS, EXIT_WARN case.
#      EXIT_WARN - items if not found.

def delRunTimeVar(pFileName, pItem, pValidDelim = "="):
    debug = 0
    counter = None
    stat = None

#   Open temporary file in the log and record the flag that pcie setup is done.

#   Open temporary file in the log and record the flag that pcie setup is done.
#   1. remove the <pFileName>.bak with force.
#   2. mv <pFileName> to <pFileName.bak>.
#   3. open the <pFileName.bak> file for read.
#   4. open the non-existing <pFileName> for append.

    os.system("rm -rf " + pFileName + ".bak")
    os.system("mv " + pFileName + " " \
                  + pFileName + ".bak")
    tmpGlobalFileNameIn = pFileName + ".bak"
    tmpGlobalFileNameOut = pFileName
    tmpGlobalFpOut = open(tmpGlobalFileNameOut, 'a')

    try:
        tmpGlobalFpIn = open(tmpGlobalFileNameIn, 'r')
    except IOError:
        printWarn("file: " + str(tmpGlobalFileNameIn) + " does not exist. Nothing to delete.")
        return SUCCESS

#   Init counter and assume success.

    counter = 0
    stat = SUCCESS

#   Start searching for matching lines and process if found. If not found, return EXIT_WARN.

    while 1:
        line = tmpGlobalFpIn.readline()
        counter += 1

        if counter > 100:
            printDbg("more than 100 lines read. Can not find the item.")
            stat = EXIT_WARN
            break 

        if line == None:
            printErr("line = None. Can not find the item.")
            stat = EXIT_WARN
            break

        if line == "":
            printErr("line = 0-length line. Can not find the item.")
            stat = EXIT_WARN
            break

        if re.search(pItem + "=", line):
            printDbg("found item, removing matching line from temp log file", debug)

            if debug:
                line == "x=x"
            else:
                line == ""
            break

        printDbg("writing " + str(line), debug)
        tmpGlobalFpOut.write(line)

    tmpGlobalFpIn.close()
    tmpGlobalFpOut.close()
    return stat

#   Delete the value in the  temporary global config file created run-time during
#   script execution time. The naming format is: tmp.global.<pid>.log.
#   entries must be in the convetion: <pItem>=<value>.
#   req:      None
#   input:    
#   - pItem - item to be deleted
#   output:   
#   - SUCCESS - item is found and deleted.
#     EXIT_ERR - items is found and failed to delete for any reason or any other reason 
#     than conditions specified in SUCCESS, EXIT_WARN case.
#     EXIT_WARN - items if not found.

def delGlobalTmp(pItem):
    debug = 0
    counter = None
    stat = None

    lFileName = PYTHON_ROOT_DIR + "/log/tmp.global." + str(os.getpid()) + ".log"
    return delRunTimeVar(lFileName, pItem)

#   Sets the key/value pair in the config file created run-time during script execution time. 
#   Entries must be in the format <pItem><delimiter><value> in the pFileName. It is primarily
#   used for managing the runtime settings in the  temporary run time global configuration file 
#   This function server as a helper function for the user-interfacing function called setGlobalTmp.
#   setRunTimeVar should be carefully used by user as it requires user to manage his/her own file in which 
#   settings are managed in a key delimiter value pairs. If the temporary run time global configuration file
#   settings are managed, then it should use setGlobalTmp() wrapper which in turn calls this function.
#   req:      None
# 
#   Global tmp file has special processing privilege. 
#   req:      None.
#   input:    
#   - pFileName - file name to be updated, it must be full absolute path of file.
#           /<dir1>/<dir2>/<fileName>=<pFileName>
#   - pItem - substring for search value.
#   - pValue - value to be set for pItem
#   - pValidDelim - substring for testing the line as valid.
#           create  = 1 - if item to be set is not found, then append it.
#                   = 0 - if item to be set is not found, then do not append (no change at all).
#   return:   
#   - EXIT_ERR if any failure.
#   - SUCCESS - if update is successful.

def setRunTimeVar(pFileName, pItem, pValue, pValidDelim= "=", pCreate=1):
    debug = 0
    debugL2 = 0
    counter = None
    lineWrite = None
    stat = None
    buffer = ""
    itemFound = 0
    line = EXIT_ERR

    if validateFcnInput([pFileName, pValidDelim, pItem, pValue]) == EXIT_ERR:
        printErr("input validation has failed.")
        return EXIT_ERR
        
    printDbg("Entered: pFileName: " + str(pFileName) + ", pItem | pValue: " + str(pItem) + " | " + str(pValue), debug)

#   Open temporary file in the log and record the flag that pcie setup is done.
#   1. remove the <pFileName>.bak with force.
#   2. mv <pFileName> to <pFileName.bak>.
#   3. open the <pFileName.bak> file for read.
#   4. open the non-existent <pFileName> for append.

    os.system("rm -rf " + pFileName + ".bak")
    os.system("mv " + pFileName + " " \
                  + pFileName + ".bak")
    tmpGlobalFileNameIn = pFileName + ".bak"
    tmpGlobalFileNameOut = pFileName
    tmpGlobalFpOut = open(tmpGlobalFileNameOut, 'a')

    try:
        tmpGlobalFpIn = open(tmpGlobalFileNameIn, 'r')
    except IOError:
        if debug:
            printDbg("File " + str(tmpGlobalFileNameIn) + " does not exist. Will append immediately.")

        tmpGlobalFpOut.write(pItem + "=" + pValue + "\n")            
        tmpGlobalFpOut.close()

        if debug:
            printDbg("Done updating.")

        return SUCCESS

    counter = 0
    lineWrite = pItem + pValidDelim + pValue + "\n"
    stat = EXIT_ERR
    updateDone = None

    # Read through the file and update it. Start an inf.loop with while 1.

    lineTest = " "

    while 1:
        line = tmpGlobalFpIn.readline().strip() + "\n"

        printDbg("\n------------\ncounter: " + str(counter) + ", line read: \n" + str(line), debug)
        printDbg("Line len: " + str(len(line)), debug)

        try:
            printDbg("Hex: " + ":".join("{:02x}".format(ord(c)) for c in line), debugL2)
        except Exception as msg:
            printDbg("Exception occurred, empty or invalid char in the line read. Terminating...", debugL2)

            if debug:
                print msg

        # Search for valid delimiter. The format must always be pItem=pValue.
        # 1. If current line is valid
        #   a. if the line is searched Entered: then replace.
        #   b. if not searched Entered: then continue looping.
        # 2. if current line is not valid, then immediately add.
        # This means, anytime invalid line is found, it is considered "end of valid section"
        # and anything after this "invalid line" will be discarded and ignored even though
        # it is considered to be valid. Therefore it is important that no invalid line exists
        # between valid lines. 

        if itemFound == 0:

            # Check if current line is valid (contains delimiter).

            if re.search(pValidDelim, line):
                printDbg("Valid line with " + pValidDelim + " sign", debug)

                # If searched item is found (the one to be updated), then replace with lineWrite.
                # and append to buffer, otherwise just add line and append to buffer.
    
                if re.search(pItem + pValidDelim, line):
                    printDbg("Found the item " + str(pItem) + " in the log file", debug)
                    printDbg("Line being written(replacing with): " + str(lineWrite), debug)
                    buffer += lineWrite
                    stat = SUCCESS
                    itemFound = 1
                else:
    
                    # Valid line containing = but not the one being searched, move on.
    
                    printDbg("Valid line with = but not the one being searched", debug)
                    buffer += line
    
            # THIS COMMENT MIGHT BE WRONG!!!
            # If valid line is not found, (line with delimiter is not found).        
            # Either with create=1 or 0, we set success because once invalid line is 
            # encountered, it is either add or not. Everything past it must be copied.
            # THIS COMMENT MIGHT BE WRONG!!!

            # Reached here if current line is not valid. Strictly speaking, following the logic, it should only possible
            # to reach here after reading all valid line. If pCreate = 1, then we append the new item, otherwise 
            # we just add line whatever it is.
    
            else:
            
                # If entry is not found and creating new one is specified then add otherwise
                # exit with error.
    
                printDbg("Encountered invalid line (EOF), inserting new line: " + str(lineWrite) + " ending.", debug)
    
                if pCreate == 1:
                    buffer += lineWrite
                else:
                    printDbg("pCreate=0, not Creating.")
                    buffer += line

                stat = SUCCESS
                break
        else:
        
            # if found before, then do not do anything except copy the line into buffer.    

            printDbg("Item was already found, canceling the update", debug)

            if line.strip() != "" and line.strip() != None:
                printDbg("simply copying non-empty line.", debug)
                buffer += line 
            else:
                printDbg("decided to not add empty line.", debug)

        # Protection from infinite loop.

        counter += 1
        
        if counter > CONFIG_GLOBAL_TMP_MAX_LINES:
            printDbg("More than " + str(CONFIG_GLOBAL_TMP_MAX_LINES) +" lines has been read \
                when setting " + str(pItem) + " to " + str(pValue) + ", more than that is not supported.", debug)
            break

        if line == None or line == "":
            printDbg("Reached the end of file", debug)
            break

    printDbg("Buffer just before updating:\n---------------\n" + str(buffer) + "\n----------------\n", debug)
    printDbg("All done. Closing both fpOut/In file", debug)
    tmpGlobalFpOut.write(buffer)
    tmpGlobalFpOut.close()
    tmpGlobalFpIn.close()

    return stat

#   Sets the value in the  temporary global config file created run-time during
#   script execution time. The naming format is: tmp.global.<pid>.log.
#   entries must be in the convetion: <pItem>=<value>.
#   req:      None
#   input:    
#   - pItem - item to be set <pItem>=<pValue>
#   - pValue - value to be set for pItem <pItem>=<pValue>
#   - pCreate = if set, will append if entry is not found.
#   return:   
#   - EXIT_ERR - if pCreate=0 and pItem is not found or any other error condition.

def setGlobalTmp(pItem, pValue, pCreate=1):
    debug = 0
    debugL2 = 0
    counter = None
    lineWrite = None
    stat = None
    buffer = ""

    printDbg("Entered: pItem | pValue: " + str(pItem) + " | " + str(pValue), debug)

    return setRunTimeVar(PYTHON_ROOT_DIR + "/log/tmp.global." + str(os.getpid()) + ".log", \
        pItem, pValue, "=", pCreate)

#   Reads and returns the key/value pair in the config file created run-time during script execution time. 
#   Entries must be in the format <pItem><delimiter><value> in the pFileName. It is primarily
#   used for managing the runtime settings in the  temporary run time global configuration file 
#   This function server as a helper function for the user-interfacing function called getGlobalTmp.
#   setRunTimeVar should be carefully used by user as it requires user to manage his/her own file in which 
#   settings are managed in a key delimiter value pairs. If the temporary run time global configuration file
#   settings are managed, then it should use getGlobalTmp() wrapper which in turn calls this function.
#   req:      None.
#   input:    
#   - pFileName - file name to search inside.
#   - pItem - item to be returned <pItem>=<value>
#   - pDelimValue - delimiter value. 
#   output:   
#   - value corresponds to the input pItem if pItem is found successfully.
#           EXIT_ERR - if pItem value is not found or any other error except below.
#           string: "api/config.txt" if IOerror and pFileName is globalTmpLog.

def getRunTimeVar(pFileName, pItem, pDelimValue = "="):
    debug = 0
    counter = None
    stat = None
    tmpGlobalFileName = pFileName

    stat = SUCCESS

    printDbg("entered: ", debug)
    printVars([pFileName, pItem], debug)

    # Open file and search for entry.

    try:
        fpGlobalFp = open(tmpGlobalFileName, 'r')
        printDbg("opening " + str(tmpGlobalFileName), debug)

        counter = 0

        # Read line by line and search the entry for each line.
    
        while 1:
            line = fpGlobalFp.readline().strip()
            #printDbg("line: " + str(line), debug)            

            if re.search(pItem + pDelimValue, line):
                itemValue = line.split(pDelimValue)[1]
                printDbg("found/returning item, value: " + str(pItem) + ", " + str(itemValue), debug)
                fpGlobalFp.close()
                return itemValue
            
            counter += 1

            # Protection from infinite loop.
    
            if counter > CONFIG_GLOBAL_TMP_MAX_LINES:
                if pItem.strip() != "printwarning" and pItem.strip() != "printerr":
                    printWarnMinor("Could not find the item: " + str(pItem))
                stat = EXIT_ERR
                break

        fpGlobalFp.close()
        return stat           

    # if file can not be opened for any reason, return default api/config.txt file.

    except IOError:
        printDbg("("+str(pItem)+"): file IO error when opening: " + str(pFileName))

        if pFileName == PYTHON_ROOT_DIR + "/log/tmp.global." + str(os.getpid()) + ".log":
            #printWarn("Can't open blade specific setting file when retrieving:  " + str(pItem) + "!. !RETURNING DEFAULT API/CONFIG FILE for " + str(pItem) + " If happens from TEST SCRIPT -> error condition!")
            time.sleep(3)
            return "api/config.txt"
        else:
            #printErr("IO Error encountered")
            return EXIT_ERR

# Returns the value from the temporary global config file created run-time during
# script execution time. The naming format is: tmp.global.<pid>.log.
# Entries must be in the format <pItem>=<value>.
# req:      
# - None.
# input:    
# - pItem - item to be returned <pItem>=<value>
# output:   
# - value corresponds to the input pItem if pItem is found successfully.
#   EXIT_ERR - if pItem value is not found.

def getGlobalTmp(pItem):
    debug = 0
    counter = None
    stat = None
    tmpGlobalFileName = PYTHON_ROOT_DIR + "/log/tmp.global." + str(os.getpid()) + ".log"

    printDbg("====================", debug)
    printDbg("entered, looking for " + str(pItem), debug)

    return getRunTimeVar(tmpGlobalFileName, pItem)

#   Run-time function that reads api/config.txt in api and returns any the requirements string array back to the calling 
#   function based on pReqKeys. Blade location specific config.*.txt entries are ignored. Only the entries in the api/config.txt
#   is maintained to contain the latest requirement list. This function itself merely seaches for REQUIREMENT_LIST 
#   section by scanning each line and  one it finds it calls getReqListHelp function to do a individual gathering of each 
#   requirement list entries. 
#   req:      
#   - None.
#   input:    
#   - pReqKeys - list of test scripts requirements values.
#   Return    
#   - dictionary containing the keys:values that matches given keys. 
#   EXIT_ERR for any error conditions encounered.

def getReqList(pReqKeys):
    debug = 0
    debugL2 = 0
    fpConfig = None
    fpConfigName = None

    printDbg("Entered: ", debug)

    if re.search("tools", os.getcwd()):
        printWarnMinor("Calling from tools folder, not test script, N/A")
        return SUCCESS

    if debug:
        printSeq(pReqKeys)

    PYTHON_ROOT_DIR = os.environ.get('CURRENT_TREE')

    fpConfigName = getGlobalTmp("configFile")

    # For gathering requirement, use the main config file not blade specific config file!

    fpConfigName = "api/config.txt"

    if fpConfigName == None:
        printErr("Unable to find configFile line from tmpGlobalFile")
        return EXIT_ERR

    if re.search("UCSB|N20|BASE", fpConfigName):
        fpConfig = open(PYTHON_ROOT_DIR + "api/configuration/" + fpConfigName, 'r')
    else:
        fpConfig = open(fpConfigName, 'r')

    if fpConfig == None:
        printErr("file system error: unable to open file api/config.txt")
        return EXIT_ERR

    while 1:
        line = fpConfig.readline()
    
        if line == "" or line == None:
            printDbg("..skip(E)Line..", debugL2)
            counter += 1
    
            # If too many empty lines are encountered (>100), leave. 

            if counter > 100:
                quit()

        # Ignore the commented out line. 

        elif re.search("^#", line):
            printDbg("..skip(#)Line..", debugL2)
            continue
        else:

            # if non-empty line is encountered, reset the empty line counter to 0.
            counter = 0
    
            # Get the requirement list dictionary if "REQUIREMENT_LIST" separator line is found. 
            # The process of obtaining is done by getReqListHelp which does the heavy lifting.
    
            if re.search("REQUIREMENT_LIST", line):
                printDbg("Processing Requirement list...", debug)
                requirementDict = getReqListHelp(fpConfig, pReqKeys)
                return requirementDict
    
    printErr("config.txt has no REQUIREMENT LIST section.")
    return EXIT_ERR

#   Called from getReqList() function, it assists prerequisites gathering by collecting the corresponding entries from api/config.txt
#   and returns it in the dictionary format based on the available keys in pKeyReq. Since this is a help function used by getReqList()
#   function, it is not advised to call this function directly. 

#   req:    
#   - pFpConfig file pointer must be pointing the REQUIREMENT_LIST section in the api/config.txt file.
#   input:  
#   - pFpConfig - file pointer to config.txt.
#   - pKeyReq - list of keys to match
#   return: matching Requirement list dictionary
#   - EXIT_ERR - if any error occurred.

def getReqListHelp(pFpConfig, pKeyReq):
    line = 1
    CONFIG_REQUIREMENT_MAX_LINES = 400
    counter = 0
    debug = 0
    debugL2 = 0
    printDbg("entry", debug)
    lGlobalConfigList = {}
    
    matchingReq = {}

    while line:
        line = pFpConfig.readline()

        if re.search("^REQ_", line):
            key = line.split('=')[0].strip()
            val = line.split('=')[1].strip()

            if key in pKeyReq:
                printDbg("adding " + key + " / " + val, debug)
                matchingReq[key] = val

        elif re.search("--------------", line):
            printDbg("Reached the end of REQUIREMENT LIST section. Leaving...", debug)
            break
        
        counter += 1

        if counter > CONFIG_REQUIREMENT_MAX_LINES:
            print "Warning! over " + str(CONFIG_REQUIREMENT_MAX_LINES) + " lines read from REQUIREMENT_LIST section. Leaving."
            return EXIT_ERR

    printDbg("Requirement list fetching is complete: ", debug)

    return matchingReq

#   Validates that given executable is available on efi shell mapped pFsX file system drive passed on as parameter.
#   If not found, it will iterate through available file system mapped drives until the one designed as fs<pLastFsx> 
#   until it is found or return EXIT_ERR if not found in the range: fs<0>:fs<pLastFsx>
#   If multiple copies exist in different efi mapped fs drive, then first occurrence is returned. 
#   req:    
#   - None
#   input:  
#   - pBmcSsh - bmc sol console
#   - pFsX - fsX drive on which to try out the command
#   - pCommandPath - commad to try out
#   - pLastFix - if not found on pFsx, last Fs drive to iterate from fs0, to see if it can be found.
#   return:
#   - fsX efi drive where command is found.
#     EXIT_ERR - if not found, returns EXIT_ERR

def validateEfiPath(pBmcSsh, pFsX, pCommandPath, pLastFsx=10):
    debug = 0

    printDbg("Entered: pFsX/pCommandPath/pLastFsx: " + str(pFsX) + "/" + str(pCommandPath) + "/" + str(pLastFsx), debug)

    try:
        int(pLastFsx)
    except ValueError:
        printErr("Invalid pLastFsx value, setting back to 10")
        pLastFix = 10

    if not re.search("fs[0-9]*:", str(pFsX)):
        printErr("Invalid format for pFsX: " + str(pFsX))
        return EXIT_ERR
    
    if not re.search("^\\\\.*.efi", str(pCommandPath)):
        printErr("Invalid format for pCommandPath: " + str(pCommandPath))
        return EXIT_ERR

    output1 = cli_with_ret(pBmcSsh, pFsX + pCommandPath, "", "efiShell")
    
    # if command is available return the fsX drive, otherwise scan through other fsX to see if it can
    # find the file. If search is exhausted return command not available value.

    if output1 != EXIT_ERR_CMD_NOT_AVAIL:
        printDbg(str(pCommandPath) + " is available on " + str(pFsX))
        return pFsX
    else:
        for i in range(0, int(pLastFsx)):
            pFsI = pFsX[:2] + str(i) + ":"
            printDbg("trying on " + pFsI)
            output1 = cli_with_ret(pBmcSsh, pFsI + pCommandPath, "", "efiShell")

            if output1 != EXIT_ERR_CMD_NOT_AVAIL:
                printDbg("found the command on " + pFsI)
                return pFsI
        
        printErr("unable to find "+ pCommandPath + " on Fs0 - Fs" + str(i))
        return EXIT_ERR_CMD_NOT_AVAIL

#   Validates location of blade string to be in x/y format
#   The chassis number up to 8 is supported. If chassis No. is higher than
#   this function needs to be enhanced.
#   pReq:
#   - None.
#   pLoc
#   - blade location info.
#   return
#   - EXIT_ERROR if format is not correct.
#     SUCCESS if format matches x/y.

def validateBladeLoc(pBladeLoc, pErrLevel = ERROR_LEVEL_ERROR):
    debug = 0

    if debug:
        printVar(pBladeLoc)

    if pBladeLoc == None:
        printErr("pBladeLoc is None.", -1, pErrLevel)
        return EXIT_ERR

    bladeLoc = pBladeLoc.split('/')

    try:
        if int(bladeLoc[0]) > 9 or int(bladeLoc[0]) < 1  or int(bladeLoc[1]) > 8 or int(bladeLoc[1]) < 1:
            printErr("1. location is not in x/y format or wrong chassis or blade No.", -1, pErrLevel)
            return EXIT_ERR
    except Exception as msg:
        printErr("Error parsing location", -1, pErrLevel)
        printErr(msg)
        return EXIT_ERR

    return SUCCESS

#   Validates that the blade location format is x/y format
#   This is already defined in ucs.py  therefore, it should be removed.
#   Currently it is set as a wrapper call to other function.
#   req:    None.
#   input:  
#   - pLoc - input blade location to be checked for
#   return: 
#   - SUCCESS - if it follows the x/y format.
#     EXIT_ERR - for any error.
#   note: blade location can be up to 8.
#         chassis location: assuming up to 8. If more than 8 chassis is in FI, then function needs to be 
#         enhanced.

def validateLoc(pLoc, pErrLevel = ERROR_LEVEL_ERROR):
    return validateBladeLoc(pLoc, pErrLevel)

#   Validates the parameter is following the IPv4 address format. This function can be further improved
#   to validate the ip address format.
#   req:    None.
#   input:  
#   - pIp - ip address to be checked for format.
#   return: 
#   - SUCCESS - if it follows the address format (note further implementation needed for maximum 3 digits in each octet)
#     EXIT_ERR - if not follows the address format.

def validateIp(pIp, pErrLevel = ERROR_LEVEL_ERROR):
    octets = None
    octet = None
    debug = 0

    if debug:
        printVars([pIp, pErrLevel])

    if pIp == None:
        return None

    octets = pIp.split(".")

    # Separate into 4 octets, if not 4 then fail.
    
    if len(octets) != 4:
        printErr("1. Not valid IP address.", -1, pErrLevel)
        return EXIT_ERR

    for i in octets:
        
        # Octet must be able to be converted into an integer.

        try:
            octet = int(i)
        except TypeError:
            printErr("2. Not valid IP address.", -1, pErrLevel)
            return EXIT_ERR
        except ValueError:
            printErr("2. IP address contains non-integer.")
            return EXIT_ERR

        # Octet must be within 0 and 255 inclusive, mask and subnet address are valid.

        if octet < 0 or octet > 255:
            printErr("3. Not valid IP address.", -1, pErrLevel)
            return EXIT_ERR

    return SUCCESS

#   Used for showing unencrypted password on to the console for debugging password. This function should 
#   be used sparingly if CONFIG_SHOW_PW is set to 1. By default it is set to 0 in which case the returned
#   password is just series of wildcards instead of actual password.
#   By default, no function will show password on to the console.
#   req:    None.
#   input:  pPw - password to show.
#   return: unencrypted pass if CONFIG_SHOW_PW=1, otherwise stars.

def getUnEncPw(pPw):

    option = None

    #!!WARNING only set this flag to 1  for exceptional circumstances for heavy debugging!!!
    # CONFIG_SHOW_PW = 1 will not only show the entire password during the script execution
    # log files can potentially contain the unencrypted password!
    # If set to 1 as an added precuation, script will not continue without asking user to confirm.
    
    CONFIG_SHOW_PW = 0
    
    if CONFIG_SHOW_PW:
        print "WARNING!!!, CONFIG_SHOW_PW = 1, meaning unencrypted password might be shown during script execution and"
        print "also log files will contain unencrypted password. Press 1 to confirm and continue. After done executing script"
        print "make sure clean up the log file!!! This option is purely for debugging purpose!!!"

        while 1:
            option = raw_input("Choose 1 for continue, 2 for setting the CONFIG_SHOW_PW back to 0 and return astericks only: ")

            if option == "1":
                return pPw
            elif option == "2":
                CONFIG_SHOW_PW = 0
                return "*****"
            else:
                print "Invalid option."
        return pPw
    else:
        return "****"

#   Print the string with bar before and after for isolated view.
#   req:    None
#   input:  pConsole - pexpect child object.
#   return: EXIT_ERR if pConsole is None
#           SUCCESS otherwise.

def printWithBar(pMsg):
    printDbg("\n-------------------\n" + str(pMsg) +\
             "\n-------------------\n")
    return SUCCESS

def printWithBarToFile(pMsg):
    printWithBar(pMsg)
    printToFile("- Info: " + pMsg)
    printToFile("\n----------------\n" + pMsg + \
                "\n----------------\n")
    
    return SUCCESS

#   Print pConsole output with bars before and after them in order to 
#   clearly distinguish the pConsole output at given time.
#   req:    
#   - None
#   input:  
#   - pConsole - pexpect child object.
#   return: 
#   - EXIT_ERR if pConsole is None.
#     SUCCESS otherwise.

def printPexBa(pConsole, pDebugL2 = None, pCustomMessage = None):
    if pConsole == None:
        printErr("pConsole is None.")
        return EXIT_ERR

    if re.search("telnetlib", pConsole.__module__):
        printWarn("pConsole is a telnet. Can not continue.")
        return EXIT_ERR

    if pCustomMessage:
        printDbg(pCustomMessage, pDebugL2)
    printDbg("\nlen before|after: " + str(len(str(pConsole.before))) + "|" + str(len(str(pConsole.after))), pDebugL2)
    printDbg("Before:\n--------\n" + (pConsole.before) + "\n--------\nAfter:\n--------\n" + \
        str(pConsole.after) + "\n--------", pDebugL2)
    
    return SUCCESS
    
#   Print string with no new line.
#   req:    
#   - None
#   input:  
#   - pString - string to print.
#   return: 
#   - None

def printNnl(pString):

    if pString != None:
        sys.stdout.write(pString)
    else:
        sys.stdout.write(".None?.")
      
#   Print dictionary properties of an object in a more descriptive format.
#   req:    
#   - None.
#   input:  
#   - pObject - class whose members to print.
#   return: None.

def printDic(pObject):
    printN("-----------------------")
    printN("instance type: " + pObject.__class__.__name__)
    printN(pObject.__dict__)
    printN("-----------------------")

#   Reads global configuration file that reads and returns the given settings in pKey parameter.
#   
#   input:    
#   - pKey - entry in the config.txt that starts with CONFIG_*. 
#   Return:   
#   - dictionary key value based on pKey parameter, 
#     EXIT_ERR if key is not found or any other error.
    
def getGlobal(pKey):
    debug = 0
    debugL2 = 0
    fpConfigName = None
    fpConfigName3 = None
    fpConfig = None
    CONFIG_MAX_EMPTY_LINES = 10
    configLevel = None

    printDbg("============================", debug)
    printDbg("Entered: key:" + str(pKey), debug)

    PYTHON_ROOT_DIR = os.environ.get('CURRENT_TREE')
    globalConfigList = {}

    # Determine the config level. When determining configLevel it always use api/config.txt 
    # for reference.

    configLevel = getConfigLevel(pKey)

    # If it is == 1, must use blade specific configuration file and must not read (ignore) from global config file.
    # If it is == 2, will read both global and blade-specific config file, the blade-bound config file takes priority if value exists on both.
    # If it is == 3, must read global config file.
    # Set the config file names to iterate through. 

    if re.search("tools", os.getcwd()):
        fpConfigName = None
        fpConfigName3 = PYTHON_ROOT_DIR + "api/config.txt"
    else:        
        if configLevel == 1:
            fpConfigName = getGlobalTmp("configFile")
            fpConfigName3 = None
        elif configLevel == 2:
            fpConfigName = getGlobalTmp("configFile")
            fpConfigName3 = PYTHON_ROOT_DIR + "api/config.txt"
        elif configLevel == 3:
            fpConfigName = None
            fpConfigName3 = PYTHON_ROOT_DIR + "api/config.txt"
        else:
            fpConfigName = None
            fpConfigname3 = None
            printErr("Unable to determine the config level, therefore exiting.")
            return EXIT_ERR

    printDbg("configLevel: " + str(configLevel), debug)
    printDbg("config file var-s: " + str(fpConfigName) + " : " + str(fpConfigName3), debug)

    # Adjust the path.

    if fpConfigName:
        if re.search("UCSB|N20|BASE", fpConfigName):
            fpConfigName = PYTHON_ROOT_DIR + "api/configuration/" + fpConfigName
        else:
            fpConfigName = PYTHON_ROOT_DIR + fpConfigName

    # For each config file(s) in the list, search for the pKey configuration value reading line by line.

    counter1 = 0

    for i in (fpConfigName, fpConfigName3):
        if i:
            printDbg("Opening " + str(i).split('/')[-1], debug)

            try:
                if re.search("UCSB|N20|BASE", i):
                    fpConfig = open(i, 'r')
                else:
                    fpConfig = open(i, 'r')
            except Exception as msg:
                printErr("Unable to open blade specific config file.")
                printDbg(str(msg)) 
                return EXIT_ERR

            # infinite loop reading each line. Break if configuration is found or exhausted with max read amount.

            while 1:
                line = fpConfig.readline()
            
                if line == "" or line == None:
                    printDbg("..skip(E)Line..", debugL2)
                    counter += 1
            
                    if counter > CONFIG_MAX_EMPTY_LINES:
                        printWarnMinor("More than " + str(CONFIG_MAX_EMPTY_LINES) + " empty lines are read\
                        when reading " + str(pKey) + ". Reduce the number of empty lines from " + str(fpConfigName))

                        # if this is generic config file, must exit.

                        if counter1 > 0:
                            printDbg("Exiting.")
                            return EXIT_ERR
                        else:
                            break
        
                # Skip commented out line.
        
                elif re.search("^#", line):
                    printDbg("..skip(#)Line..", debugL2)
                    continue
                else:
                    counter = 0

                    # If dedicated section title is found, start the actual search. The getConfigDictionary does the heavy lifting.
            
                    if re.search("GLOBAL_CONFIGURATION", line):
                        printDbg("Processing Global variables...", debug)
                        globalConfigList = getConfigDictionary(fpConfig)
        
                        # If desired dictionary key is in the returned dictionary, return it, otherwise, exit with error.
        
                        if pKey in globalConfigList:
                            printDbg("found the key : value: " + pKey + ":", debug)
                            printDbg(str(globalConfigList[pKey]), debug)
        
                            return globalConfigList[pKey]
                        else:
                            printErr("Can not find the key: " + pKey + " in " + str(i))
    
            printErr(str(i) + " has no GLOBAL_CONFIGURATION section or " + str(pKey) + " was not found in the file: " + str(i))

            if counter1 > 0:
                printDbg("Exiting.", debug)
                return EXIT_ERR
        else:
            printDbg("Skipping." + str(fpConfigName), debug)
        counter1 += 1
    
#   Strips the cid server's challenge string off its enclosing starlines and returns the string itself.
#   input:
#   - pRawString  - cid output of challenge or response string
#   return      
#   - cid output of challenge or response string stripped off its enclosing starred line.
#     EXIT_ERR if it did not found the * chars.

def extractChallengeString(pRawString):
    debug = 0
    if validateFcnInput([pRawString]) == EXIT_ERR:
        printErr("Empty string.")
        return EXIT_ERR

    ret1 = pRawString
    nakedString = ""
    phase1 = 0
    phase2 = 0
    phase3 = 0

    printDbg("Unprocessed/raw string: \n----\n" + str(pRawString) + "\n----\n", debug)

    # Following is the logic to strip the rawstring from its enclosing starred line.

    for i in ret1:
        if i == "*":
            phase1 = 1

        if phase1 == 1 and i != "*":
            phase2 = 1

        if phase1 == 1 and phase2 == 1 and i == "*":
            phase3 = 1

        if phase2 == 1 and phase3 == 0:
            nakedString += i

    if nakedString == "":
        return EXIT_ERR
   
    printDbg(str(nakedString), debug)
    return nakedString

#   Helper function for searching the configuration value from GLOBAL_CONFIGURATION section of config file.
#   req:        
#   - pFpConfig pointer must be pointing at GLOBAL_CONFIGURATION setting prior to calling this function.
#   input:      
#   - pFpConfig   file pointer to config.txt and instance of clsGlobalConfig
#   return:     
#   - globalConfig dictionary, containing all key pairs from GLOBAL_CONFIGURATION section.
#     EXIT_ERR if error.

def getConfigDictionary(pFpConfig):
    line = 1
    CONFIG_REQUIREMENT_MAX_LINES = 400
    counter = 0
    debug = 0
    debugL2 = 0
    printDbg("entry", debug)
    lGlobalConfigList = {}
    val = None

    # For each line, perform a grep search the line if it start with CONFIG substring and divide into its key an value pair if found.

    while line:
        line = pFpConfig.readline()

        if re.search("CONFIG", line):
            if re.search("=", line):
                key = line.split('=')[0].strip()
                val = line.split('=')[1].strip()
            else:
                printDbg("invalid line with no =: " + str(line), debug)
                continue
            try:
                val = int(val)
                printDbg(key + ": is converted to integer", debug)
            except TypeError:
                printDbg(key + ": can not be converted to integer", debug)
            except ValueError:
                printDbg(key + ": can not be converted to integer", debug)

            lGlobalConfigList[key] = val
        elif re.search("--------------", line):
            printDbg("Reached the end of GLOBAL_CONFIGURATION section. Leaving...", debug)
            break
        
        counter += 1

        if counter > CONFIG_REQUIREMENT_MAX_LINES:
            print "Warning! over " + str(CONFIG_REQUIREMENT_MAX_LINES) + " lines read from GLOBAL_CONFIGURATION section. Leaving..."
            return EXIT_ERR

    printDbg("Global configuration is complete: ", debug)

    for key in lGlobalConfigList:
        printDbg(key + ": " + str(lGlobalConfigList[key]), debug)

    return lGlobalConfigList

#   Prints sequence type to summary log file in a more descriptive format.
#   req:    
#   - None.
#   input:  
#   - pSeq - sequence to print.
#   - pWidth - number of items in a sequence to print per single row. 
#   return: None

def fprintSeq(pSeq, pWidth = None):
    printSeq(pSeq, pWidth, 1)

#   Prints the sequence types: dictionary, list tuple in an adjustable column-widht format 
#   instead of continuing to end current line for more readibility.
#   req:    
#   - None
#   input:  
#   - pSeq -  List, tuple or dictionary data type to print.
#   - pWidth - number of items in the sequence to print per single row.
#   - pToFile - if set to 1, output to summary log file
#             - if set to 0, then do not output to summary file.
#   return: 
#   - None

def printSeq(pSeq, pWidth = None, pToFile = 0):
    debug = 0

    if pToFile:
        printToFile("seq size: " + str(len(pSeq)))

    if pSeq == None:
        printErr("no sequence")
        return EXIT_NO_ERR

    printDbg("seq size: " + str(len(pSeq)))

    PRINT_SEQ_COL_WIDTH = None
    PRINT_SEQ_COL_WIDTH = getGlobal('CONFIG_PRINT_SEQ_COL_WIDTH')

    if PRINT_SEQ_COL_WIDTH == EXIT_ERR:
        PRINT_SEQ_COL_WIDTH = 10

    if pWidth:
        PRINT_SEQ_COL_WIDTH =  pWidth

    printDbg("PRINT_SEQ_COL_WIDTH: " + str(PRINT_SEQ_COL_WIDTH), debug)

    counter = 0
    tmpSeq = None
    printDbg("len: " + str(len(pSeq)), debug)

    # If sequence type is a dictionary.

    if type(pSeq) == dict:
        printDbg("sequence is dictionary", debug)
        tmpSeq = {}
        counter = 0

        # For each item of a sequence.

        # Print sequence, separate lines by PRINT_SEQ_COL_WIDTH. 
        
        for i in range(0, len(pSeq) / PRINT_SEQ_COL_WIDTH):
            keys =   pSeq.keys()[i*PRINT_SEQ_COL_WIDTH:i*PRINT_SEQ_COL_WIDTH + PRINT_SEQ_COL_WIDTH]
            values = pSeq.values()[i*PRINT_SEQ_COL_WIDTH:i*PRINT_SEQ_COL_WIDTH + PRINT_SEQ_COL_WIDTH]
            str1 = ""

            for j in range(0, len(keys)):
                str1 += str(keys[j]) + ": "  + str(values[j]) + ", "
        
            print str1

            if pToFile:
                printToFile(str(str1))

        if len(pSeq) % PRINT_SEQ_COL_WIDTH  != 0:
            keys =   pSeq.keys()[len(pSeq) - len(pSeq) % PRINT_SEQ_COL_WIDTH:]
            values = pSeq.values()[len(pSeq) - len(pSeq) % PRINT_SEQ_COL_WIDTH:]
            str1 = ""

            for j in range(0, len(keys)):
                str1 += str(keys[j]) + ": "  + str(values[j]) + ", "
        
            print str1

            if pToFile:
                printToFile(str(str1))

    # If sequence type is a list or tuple.
    
    elif type(pSeq) == list or type(pSeq) == tuple:
        printDbg("sequence is list or tuple", debug)
        tmpSeq = ""

        # Print sequence, separate lines by PRINT_SEQ_COL_WIDTH. 
        
        for i in range(0, len(pSeq) / PRINT_SEQ_COL_WIDTH):
            print pSeq[i*PRINT_SEQ_COL_WIDTH:i*PRINT_SEQ_COL_WIDTH + PRINT_SEQ_COL_WIDTH]

            if pToFile:
                printToFile(str(pSeq[i*PRINT_SEQ_COL_WIDTH:i*PRINT_SEQ_COL_WIDTH + PRINT_SEQ_COL_WIDTH]))

        if len(pSeq) % PRINT_SEQ_COL_WIDTH  != 0:
            print pSeq[len(pSeq) - len(pSeq) % PRINT_SEQ_COL_WIDTH:]

            if pToFile:
                printToFile(str(pSeq[len(pSeq) - len(pSeq) % PRINT_SEQ_COL_WIDTH:]))

    return SUCCESS

#   printToFileL1, L2 and L3 functions implements printout to varying level of log level.
#   None of the L1-L3 will write to summary log (L0) as L0 is the most concise brief information of test result.
#   At L0 level (summary log): most but critical outputs are suppressed to keep to the log file to minimum.
#   At L1 level, it is similar to L0 but certain minor warnings are not suppressed.
#   At L2 level, printDbg outputs are output. If debug is passed, outputs are written too.
#   At L3 level, it is similar to L2 but all debug suppressed messages (regardless of debug switch to printDbg) are written.
#   Note L3 design has not been implemented yet.

#   input:
#   - pString - string to be output.
#   - pLoopNo - same output may repeat for large No. of times, flooding the log, pLoopNo controls it.\
#               Refer to printToFile for pre-defined values.
#   - pCallingstring - optional caller string.

def printToFileL1(pString, pLoopNo = -1, pCallerString = None):
    tmpLogFileNameL1 = PYTHON_ROOT_DIR + "/log/tmp." + str(os.getpid()) + ".V-L1.log"
    printToFile(pString, pLoopNo, pCallerString, tmpLogFileNameL1)

def printToFileL2(pString, pLoopNo = -1, pCallerString = None):
    tmpLogFileNameL2 = PYTHON_ROOT_DIR + "/log/tmp." + str(os.getpid()) + ".V-L2.log"
    printToFile(pString, pLoopNo, pCallerString, tmpLogFileNameL2)

def printToFileL3(pString, pLoopNo = -1, pCallerString = None):
    tmpLogFileNameL3 = PYTHON_ROOT_DIR + "/log/tmp." + str(os.getpid()) + ".V-L3.log"
    printToFile(pString, pLoopNo, pCallerString, tmpLogFileNameL3)

#   Prints the message to script output file name. This function should be reserved for more critical messages that
#   worthy of being logged in the summary log file log/*.log. The same output is also written to all more verbose levels of log files: L1, L2...
#   req:        
#   - None
#   input:      
#   - pString -  message to print. Do not include escape characters in the string: (, ), *, otherwise result is unpredictable.
#   - pLoopNo - if set with positive number, specify that loopNo is attached to the line with same content as pString instead of pString itself to prevent flooding 
#       - if set with negative number, force putting the string into file even it is repeated.
#       - if None - prints nothing and remains silent.
#       - if set with positive number, (considering to be obsolete) - put a ? question mark with noLoop instead of putting actual input string, this way flooding is still avoided. 
#       script that passes noLoop == None == default, should implement passing correct LoopNo.
#   return  
#  - SUCCESS - on successfull output.
#    EXIT_ERR - if any failure occurs.

def printToFile(pString, pLoopNo = -1, pCallerString = None, verboseFileNames = None):
    debug = 0
    pStringSearch = None
    lString = None
    tmpLogFileName = None
    tmpLogFileList = None

    if debug:
        print "----------------------------------------"
        print "printToFile: entered: pString: ", pString.strip(), ", pLoopNo: ", pLoopNo, ", pCallerString: ", pCallerString

    # Declare name of log files with varying verbosity.

    tmpLogFileName1 = PYTHON_ROOT_DIR + "/log/tmp." + str(os.getpid()) + ".summary.log"
    tmpLogFileNameL1 = PYTHON_ROOT_DIR + "/log/tmp." + str(os.getpid()) + ".V-L1.log"
    tmpLogFileNameL2 = PYTHON_ROOT_DIR + "/log/tmp." + str(os.getpid()) + ".V-L2.log"

    # Construct list of log file names to output to. The rule of populating the list is that the verbosity of log files will 
    # increase to the right of list. Otherwise log hierarchy will be cause the log file verbose in chaotic state.
    # The minimum verbosity level is specified in verboseFileNames parameter passed as an input to this function.
    # I f matching index is found and the output will be broadcast to all files with equal or greater verbosity.

    tmpLogFileList = [tmpLogFileName1, tmpLogFileNameL1, tmpLogFileNameL2]

    if verboseFileNames:
        try:
            idx = tmpLogFileList.index(verboseFileNames)
        except Exception as msg:
            printErr("verboseFileNames must be included in tmpLogFileList\
            \n verboseFileNames: " + str(verboseFileNames) + ", \
            \n tmpLogFileList: " + str(tmpLogFileList))
            print msg
            return EXIT_ERR
        tmpLogFileList = tmpLogFileList[idx:]

    if debug:
        print "tmpLogFileList: ", tmpLogFileList
    
    # Determine file name and function name of caller.

    for tmpLogFileName in tmpLogFileList:
        if debug:
            print "Processing for :", tmpLogFileName

        if pCallerString == None:
            caller1Fcn = inspect.stack()[1][3]
            caller1File = inspect.stack()[1][1].split('/')[-1]
            lString = "    " + pString + " (" + caller1File + "/" + caller1Fcn + ")"
        else:
            lString = "    " + pString + " (" + str(pCallerString)+ ")"
    
        # Open the summary log file for append, if file doesn't exists then create one.
    
        if not os.path.exists(tmpLogFileName):
            if debug:
                print "printToFile: tmp file does not exist, opening for writing", debug

            os.system("touch tmpLogFileName")
            fpTempLog = open(tmpLogFileName, 'w')
        else:
            if debug:
                print "printToFile: tmp file does exist, opening for read/writing"

            # pLoopNo is positive, then write ": <pLoopNo>"
            # pLoopNo is None, then write ": noLoopNo? "
            # pLoopNo is negative, then simply write the actual reason.

            if debug:
                print "printToFile: pLoopNo processing..."

            fpTempLog = open(tmpLogFileName, 'r+')
            bufferAllLines = fpTempLog.read()

            if pLoopNo > 0:

                # Determine if it has been printed before (controls flooding).
    
                if debug:
                    print "Determining whether it has been printed before..."
                    print "bufferAllLines: ", bufferAllLines
                    print "lString: ", lString
    
                for buffer in bufferAllLines:
                    try:
                        searchPattern = re.sub("\(|\)|\\\\",".", str(lString))
                        #if re.search(searchPattern, buffer, re.DOTALL):
                        #    print "printTofile: tmp duplicate entry"
    
                    except Exception as exception1:
                        print "printToFile!!!: Exception occurred during lString re.sub..."
                        print "printToFile: ", lString
                        print exception1
                        return EXIT_ERR
    
                printDbg("writing loopNo instead of input string", debug)
                fpTempLog.write(" : " + str(pLoopNo))
            elif pLoopNo == None:
                printDbg("writing loopNo ? ", debug)
#               fpTempLog.write(" : noLoopNo? ")
            elif pLoopNo < 0:

                if debug:
                    print "printToFile: writing actual input string "

                fpTempLog.write("\n" + lString)

                fpTempLog.close()
                return SUCCESS
    
        fpTempLog = open(tmpLogFileName, 'a')
    
        if fpTempLog:
            fpTempLog.write("\n" + lString)
            fpTempLog.close()

    return SUCCESS

#   Suppressable minor warning function to reduce cluttering in summary log. The output is written to log level L-1 and levels of 
#   more verbosity.
#   The printout from this function can be turned off using CONFIG_WARNING_MINOR flag.
#   pString     - warning message to print. Do not include escape characters in the string: (, ), *, otherwise result is unpredictable.
#   req:    
#   - None
#   input:  
#   - pString - error message to print. Do not include escape characters in the string: (, ), *, otherwise result is unpredi$
#   - pLoopNo - used for same purpose as the one used in printToFile()
#   return: 
#   - None

def printWarnMinor(pString, pLoopNo=-1):
    caller1Fcn = inspect.stack()[1][3]
    caller1File = inspect.stack()[1][1].split('/')[-1]
    print colors.WARNING + "    " + caller1File + "/" + caller1Fcn + ":WARN: " + str(pString) + colors.ENDC
    printToFileL1("- Warning: " + pString, pLoopNo, " (" + caller1File + "/" + caller1Fcn + ")")
    printToFileL2("- Warning: " + pString, pLoopNo, " (" + caller1File + "/" + caller1Fcn + ")")

#   Following 3 functions, printWarn, printErr, printFail will respectively print warning, error and fail messages
#   outputting to log file. Warning is not necessarily error message and potentially can be error condition.
#   Error is a definite error but not necessarily causes the script end result to fail although possibly so.
#   Fail is a definite failure message that certainly causes script end result to fail. 
#
#   input:
#   req:    
#   - None
#   input: 
#   -  pString - error message to print. Do not include escape characters in the string: (, ), *, otherwise result is unpredictable.
#   -  pLoopNo - used for same purpose as the one used in printToFile()
#   return: 
#   - None

def printWarn(pString, pLoopNo=-1):
    warnConfigName = None
    
    # If not being called either of this functions: getGLobalTmp, getGlobal, getRunTimeVar, then
    # set the warnConfigName by getting it global temp file. Once warnConfigName is set in the global tmp
    # file, the warning can be recorded in test result log file based on its value.

    if not re.search("tools", os.getcwd()):
        warnConfigName = getGlobalTmp("printwarning")

    caller1Fcn = inspect.stack()[1][3]
    caller1File = inspect.stack()[1][1].split('/')[-1]
    #print "    " + caller1File + "/" + caller1Fcn + ":WARN: " + str(pString)
    print colors.WARNING + "    " + (caller1File[0:CONFIG_CALLER_FILE_CUTOFF_LEN-1]).ljust(CONFIG_CALLER_FILE_CUTOFF_LEN, \
        ".") + "/" + caller1Fcn.ljust(COFNIG_CALLER_FUNCTION_CUTOFF_LEN, ".") + \
        ": WARN: " + str(pString) + colors.ENDC

    if str(warnConfigName):
        if not re.search("no", str(warnConfigName)):
            printToFile("- Warning: " + pString, pLoopNo, " (" + caller1File + "/" + caller1Fcn + ")")
            printToFileL2("- Warning: " + pString, pLoopNo, " (" + caller1File + "/" + caller1Fcn + ")")

def printErr(pString, pLoopNo=-1, pErrLevel = ERROR_LEVEL_ERROR):
    errConfigName = None

    if not re.search("tools", os.getcwd()):
        errConfigName = getGlobalTmp("printerr")

    caller1Fcn = inspect.stack()[1][3]
    caller1File = inspect.stack()[1][1].split('/')[-1]

    if pErrLevel == ERROR_LEVEL_WARN_MINOR:
        print colors.WARNING + "    " + (caller1File[0:COFNIG_CALLER_FUNCTION_CUTOFF_LEN-1]).ljust(CONFIG_CALLER_FILE_CUTOFF_LEN, \
            ".") + "/" + caller1Fcn.ljust(COFNIG_CALLER_FUNCTION_CUTOFF_LEN, ".") + \
            ": WARN : " + str(pString) + colors.ENDC
    else:
        print colors.FAIL + "    " + (caller1File[0:COFNIG_CALLER_FUNCTION_CUTOFF_LEN-1]).ljust(CONFIG_CALLER_FILE_CUTOFF_LEN, \
            ".") + "/" + caller1Fcn.ljust(COFNIG_CALLER_FUNCTION_CUTOFF_LEN, ".") + \
            ": ERR : " + str(pString) + colors.ENDC
    
    if str(errConfigName):
        if not re.search("no", str(errConfigName)):
            printToFile("- Error: " + str(pString).ljust(130), pLoopNo, " (" + caller1File + "/" + caller1Fcn + ")")
            printToFileL2("- Error: " + str(pString).ljust(130), pLoopNo, " (" + caller1File + "/" + caller1Fcn + ")")
        
def printFail(pString, pLoopNo=-1):
    caller1Fcn = inspect.stack()[1][3]
    caller1File = inspect.stack()[1][1].split('/')[-1]
    print "    " + caller1File + "/" + caller1Fcn + ":FAIL: " + str(pString)
    printToFile("- Fail: " + pString, pLoopNo, " (" + caller1File + "/" + caller1Fcn + ")")

#   Prints variable name along its values.
#   req:    
#   - None.
#   input:  
#   - pVarArray - array of variables to print its variable name and value.
#   - debug - if set, display the debug output otherwise suppress.
#   - pOnePerLine - if set (default, print one variable per line, otherwise will print sequentially)
#   return: 
#   - None.

def printVars(pVarArray, pDebug = 1, pOnePerLine = 1):
    debugL2 = 0
    debugL3 = 0
    debug = 1
    finalOut = {}
    caller1Fcn = None
    caller1File = None
    stack14 = None

    if pDebug == 0 or pDebug == None:
        return 

    if type(pVarArray) != list:
        printErr("Not an array type")        
        return 

    caller1Fcn = inspect.stack()[1][3]
    caller1File = inspect.stack()[1][1].split('/')[-1]
    stack14 = inspect.stack()[1][4]
        
    lVarNames = None
    lVarValues = pVarArray

    if pVarArray == None:
        printDbg("pVar is NONE")
        return EXIT_ERR

    # Stack14 has variable name from calling function. strip all special chars
    # and extract variable name.

    printDbg("1. stack14 original: " + str(stack14), debug)
    lVarNames = re.sub("[ \n\'\]\[\\\\]", "", str(stack14))
    
    printDbg("2. lVarNames after removing char-s: " + str(lVarNames), debug)

    try:
        lVarNames = re.search(".*\((.*)\)", lVarNames).group(1).split(',')
    except IndexError:
        printDbg("index error.")
    except AttributeError:
        printDbg("attribute error.")

    printDbg("3. lVarNames after removing extracting var names: " + str(lVarNames), debug)

    if debug:
        print lVarNames
        printDbg("lVarnames len: " + str(len(lVarNames)) + ", pVarArray len " + str(len(pVarArray)), debug)

    for i in range(0, len(pVarArray)):
        finalOut[lVarNames[i]] = str(pVarArray[i])

    if debug:
        print "    " + str(caller1File) + "/" + str(caller1Fcn) + ": "

    if pOnePerLine:
        for i in range(0, len(finalOut)):
            if debug:
                printDbg("Variable name: " + str(finalOut.keys()[i]) + ", value: " + str(finalOut.values()[i]))
            else:
                printN(str(finalOut.keys()[i]) + ": " + str(finalOut.values()[i]))

    else:
        print "printing in one line: "
        print finalOut

    # This is purely debugging section, in order to print most of the caller stack information.
    # Caller stack seems to be 2-d array. If particular index does not exist, print indexerror after 
    # throwing exception.

    if debugL3:
        printDbg("stack info:")

        for i in range(0, 5):
            for j in range(0, 5):
                try:
                    printDbg(str(i) + ", " + str(j) + ": " + str(inspect.stack()[i][j]))
                except IndexError:
#                   printDbg(str(i) + ", " + str(j) + ": IndexError.")
                    i = i

#   Prints variable name along its values.
#   This is mostly used in informational message and to do a "grep INFO:"
#   req:    
#   - None.
#   input:  
#   - debug - if set, display the debug output otherwise suppress.
#   return: 
#   - variable name as string if successful.
#   - EXIT_ERR on any failure.

def printVar(pVar, pDebug = 1, pToFile = 1):
    debugL2 = 0
    debug = 0

    printDbg("entered...", debug)
    caller1Fcn = None
    caller1File = None
    stack14 = None

    if pDebug:
        caller1Fcn = inspect.stack()[1][3]
        caller1File = inspect.stack()[1][1].split('/')[-1]
        stack14 = inspect.stack()[1][4]
        
    lVarName = None
    lVarValue = pVar

    if pVar == None:
        printDbg("pVar is NONE")
        return EXIT_ERR

    # Stack14 has variable name from calling function. strip all special chars
    # and extract variable name.

    lVarName = re.sub("[ \n\'\]\[\\\\]", "", str(stack14))
    
    try:
        lVarName = re.search(".*\((.*)\)", lVarName).group(1).split(',')[0]
        if pDebug:
            printDbg("Variable name: " + str(lVarName) + ", value: " +  str(pVar))
        else:
            printN(str(lVarName) + ": " +  str(pVar))
    except AttributeError:
        printDbg("re.search error. [1][4]:" + str(stack14) + ", lVarName: " + str(lVarName))
        #return EXIT_ERR
    except IndexError:
        printDbg("group error. [1][4]:" + str(stack14))
        return EXIT_ERR

    #print "    " + caller1File + "/" + caller1Fcn + ": " + str(lVarName) + ": " + str(lVarValue)

    # This is purely debugging section, in order to print most of the caller stack information.
    # Caller stack seems to be 2-d array. If particular index does not exist, print indexerror after 
    # throwing exception.

    if debugL2:
        printDbg("stack info:")

        for i in range(0, 5):
            for j in range(0, 5):
                try:
                    printDbg(str(i) + ". " + str(j) + ": " + str(inspect.stack()[i][j]))
                except IndexError:
#                   printDbg(str(i) + ". " + str(j) + ": IndexError.")
                    i = i

    return lVarName

#   This is similar to printToFile except it appends - info: phrase to front of string and 
#   caller's modules and file name at the back of string..
#   input:
#   - pString - string to printed.
#   - pLoopNo - flood control.
#   return:
#   - None.

def printInfo(pString, pLoopNo = -1):
    caller1Fcn = inspect.stack()[1][3]
    caller1File = inspect.stack()[1][1].split('/')[-1]
    print colors.LIBLUE + "    " + caller1File + "/" + caller1Fcn + ":INFO: " + str(pString) + colors.ENDC
    printToFileL2("- Info: " + pString,     pLoopNo, " (" + caller1File + "/" + caller1Fcn + ")")

#   Exactly same function as printDbg but without any prefix in front.
#   input: 
#   - pStrin - string to be printed.
#   - debug - if set, display the debug message otherwise suppress.
#   return:
#   - None

#   This is similar to printToFile except it appends - info: phrase to front of string and 
#   caller's modules and file name at the back of string..
#   input:
#   - pString - string to printed.
#   - pLoopNo - flood control.
#   return:
#   - None.

def printInfoToFile(pString, pLoopNo = -1):
    caller1Fcn = inspect.stack()[1][3]
    caller1File = inspect.stack()[1][1].split('/')[-1]
    print colors.LIBLUE + "    " + caller1File + "/" + caller1Fcn + ":INFO: " + str(pString) + colors.ENDC
    printToFile("- Info: " + pString,     pLoopNo, " (" + caller1File + "/" + caller1Fcn + ")")

#   Exactly same function as printDbg but without any prefix in front.
#   input: 
#   - pStrin - string to be printed.
#   - debug - if set, display the debug message otherwise suppress.
#   return:
#   - None

def printN(pString, debug=1, pToFile = 1):
    if debug:
        caller1Fcn = inspect.stack()[1][3]
        caller1File = inspect.stack()[1][1].split('/')[-1]
        caller2File = (caller1File[0:CONFIG_CALLER_FILE_CUTOFF_LEN-1]).ljust(CONFIG_CALLER_FILE_CUTOFF_LEN, ".")
        caller2Fcn =  caller1Fcn.ljust(COFNIG_CALLER_FUNCTION_CUTOFF_LEN, ".")
        print "     " + str(pString)

        if pToFile:
            printToFileL2(str(pString), -1)

#   Print debug type of message, along with called py file and called function preceding the message.
#   The output is broadcast to the log level 2 and all other lower levels with more verbosity. 
#   input: 
#   - debug - if set, display the debug message otherwise suppress.
#   return:
#   - None

def printDbg(pString, debug=1):
    if debug:

        caller1Fcn = inspect.stack()[1][3]
        caller1File = inspect.stack()[1][1].split('/')[-1]

        caller2File = (caller1File[0:CONFIG_CALLER_FILE_CUTOFF_LEN-1]).ljust(CONFIG_CALLER_FILE_CUTOFF_LEN, ".")
        caller2Fcn =  caller1Fcn.ljust(COFNIG_CALLER_FUNCTION_CUTOFF_LEN, ".")
        print "    " + caller2File + \
            "/" + caller2Fcn + ": DBG: " + str(pString)

        printToFileL2(pString.ljust(130), -1, " (" + caller2File + "/" + caller2Fcn + ")")

#   Validates the sys.argv type inputs.
#   req:    
#   - None.
#   input:  
#   - pArgv - argument list
#   - pInputMap - list of 1s and 0s, 1 - mandatory, 0 - optional.
#   - pReqCodeList - list of requirement codes
#   return  
#   - EXIT_NO_ERR   if --help message display is requested,
#   - EXIT_ERR      if parameter violation is found.
#   - SUCCESS       if everything is OK.
#   - Dict          if --req is passed.

def validateArgvInput(pArrInput):
    counter = None
    stat = None

    counter = 0
    stat = SUCCESS

    for i in pArrInput:
        if i == None:
            printErr("arg" + str(i) + ": None.")
            stat = EXIT_ERR

    return stat

#   Validates the function inputs passed as array type. 
#   It iterates over each argument and make sure it is not None.
#   If any of the input specified in pArrInput is None, exit with error
#   indicating the index of failure in the list.
#   req:    
#   - None.
#   input:  
#   - pArgv - argument list
#   return:
#   - SUCCESS - if everything is OK.
#   - EXIT_ERR - if parameter violation is found.

def validateFcnInput(pArrInput, pdebug = 0):
    debug = 0
    counter = None
    stat = None
    stack14 = None

    stack14 = inspect.stack()[1][4]

    counter = 0
    stat = SUCCESS

    for i in pArrInput:
        if i == None:
            printErr("arg " + str(counter) + " is None /0 to " + str(len(pArrInput)-1) + "/")
            stat = EXIT_ERR
        counter += 1

    if stat == EXIT_ERR:
        printDbg("caller[1][4]: " + str(stack14))
    return stat

#   Validates the sys.argv type inputs to test scripts. With list of requirement is passed,
#   
#   If --req and --help switches are given, process it appropriately.
#   req:        
#   - None.
#   input:      
#   - pArgv - argument list
#   - pInputMap   - bitmap of check indicators to perform: list of 1s and 0s, 1 - mandatory, 0 - optional.
#   - pReqCodeList - requirement list codes for particular test script, gathers the descriptive string value
#   for each code. This parameter is applicable only if sys.arg[3] is --req.
#   return      EXIT_NO_ERR   if --help message display is requested, 
#               EXIT_ERR      if parameter violation is found. 
#               SUCCESS       if everything is OK.
#               List of requirement in dictionary format if --req is specified in sys.argv[3].

def validateInput(pArgv, pInputMap, pHelpStringArr, pReqCodeList = None):
    debug = 0
    
    printDbg("pArgv len:     " + str(len(pArgv)), debug)
    printDbg("pInputMap len: " + str(len(pInputMap)),debug)
    printDbg("pArgv value:",  debug)
    print pArgv
    printDbg("pInputMap value:", debug)
    print pInputMap

    # if No. of arguments is 1, then it must be either --help or --req.
    # In case of help, display help message, otherwise gather requirement
    # for this particular script and return it. In case of any other parameter
    # not supported, return error.
    # this might be obsolete. 

    if int(len(pArgv)-1) == 1:
        try:  
            if (pArgv[1]) == "--help":
                printHelp(pHelpStringArr)
                return EXIT_NO_ERR
        except IndexError:
            printWarn("index error for argv!")
            return EXIT_ERR

    # compare pArgv length agains the pInput length to make sure pArgv length matches 
    # the number of pInputs.
    # pArgv is the arguments supplied by user or driver of the test script.
    # pInput is the pre-defined mandatory and optional argument map that user of this script must abide to.
    # Since pArgv's first member is always test script name, comparison is always:
    # len(pArgv) - 1 vs. len(pInputMap)

    if len(pArgv) - 1  > len(pInputMap):
        printHelp(pHelpStringArr)
        printWarn("Max inputs are " + str(len(pInputMap)) + ", " \
        + str(len(pArgv) - 1) + " inputs are entered.")
        return EXIT_ERR

    if len(pArgv) - 1 < sum(pInputMap):
        printHelp(pHelpStringArr)
        printWarn("Min inputs are " + str(sum(pInputMap)) + ", " \
        + str(len(pArgv) - 1) + " inputs are entered.")
        return EXIT_ERR

    # If --req is passed process it here.

    try:
        if pArgv[3] == "--req":
            printDbg("Getting requirements.", debug)
            reqListDict = getReqList(pReqCodeList)
            return reqListDict
    except IndexError:
        printDbg("No requirement list requested", debug)
    
    if debug:
        TIMEOUT_VALIDATE_INPUT = int(getGlobal('CONFIG_TIMEOUT_VALIDATE_INPUT'))
        if TIMEOUT_VALIDATE_INPUT == None:
            TIMEOUT_VALIDATE_INPUT = 5
        time.sleep(TIMEOUT_VALIDATE_INPUT)

    # Record fi mgmt ip to global tmp file.

    try:
        if setGlobalTmp("fi-mgmt-ip", pArgv[1]) == EXIT_ERR:
            printWarn("unable to set the fi-mgmt-ip in global. This might cause issue at times.")
    except IndexError:
        printWarn("no fi ip specified. This might cause issue at times.")

    return SUCCESS

#   PrintHelp message in a pre-formatted way at the start of each test scripts.
#   req:    
#   - None.
#   input:  
#   - pHelpString  - help message to display, new line is substituted by colon.
#   return: 
#   - None

def printHelp(pHelpString):
    os.system("clear")
    helpString = pHelpString.split("::")

    for i in helpString:
        print "  " + i

#   Display prerequisites list in a preformatted way.
#   Prerequites are the list of requirements that are needed to be met before
#   running particular test script. Each scripts has its own unique set of requirements.
#   req:        
#   - None.
#   input:      
#   - pPreReqList - prerequisite message to display, double colon means new line.
#   return:     
#   - None

def dispPrereq(pPreReqList):
    TIMEOUT_PREREQ_DISPLAY = int(getGlobal('CONFIG_TIMEOUT_PREREQ_DISPLAY'))

    if TIMEOUT_PREREQ_DISPLAY == EXIT_ERR:
        TIMEOUT_PREREQ_DISPLAY = 10

    preReqList = None
    print "  ===================================================================================="
    print "  SCRIPT: " + str(sys.argv[0])
    print "  ------------------------------------------------------------------------------------"
    print "  Before running verify following requirements are met:"
    print "  ------------------------------------------------------------------------------------"

    preReqDict = getReqList(pPreReqList)

    for i in preReqDict:
        print "     - " + preReqDict[i]

    print ""
    print "  If all above prereq is met, wait and do nothing and test will resume in 30 seconds."
    print "  If any of the prereq is not, test will likely end prematurely, therefore use Ctrl+C "
    print "  to terminate and satisfy all requirement above and restart the test."
    print "  Use --help parameter for usage."
    print "  ------------------------------------------------------------------------------------"
    time.sleep(TIMEOUT_PREREQ_DISPLAY)

#   Send linux shell command over ssh connection. 
#   req:
#   Ssh session must be established before and must be handed over with as pSsh parameter.
#   input:
#   - pSsh - ssh handle
#   - pCommand - command name
#   return:
#   - SUCCESS if success, None is failure.

#   Note: this function is not used, instead cli_with_ret is used. 
#   currently commented out, after a while if no impact is seen it should be deleted:
#   4.18.2016: use by collect.ts.py - consider using cli_with_ret instead and after that
#   this function can be made obsolete.
 
def sshSendCmdLinux(pSsh, pCommand, pUser, pPw):
    debug = 0
    debugL2 = 0
    index1 = None

    pwShowDbg = getUnEncPw(pPw)

    printDbg("command/user/pw: " + pCommand + "/" + pUser + "/" +  pwShowDbg, debug)

    pSsh.sendline(pCommand)
    time.sleep(3)

    while 1:
        if debug == 1:
            printDbg("==========", debugL2)
            printDbg("loop:", debugL2)

        index1 = pSsh.expect(['.*rsa', '.*\[.*\].*$', '.*authenticity', '\(local-mgmt\)#',\
                             'password:', 'denied','No such file', '.*write-protected.*', \
                             pexpect.EOF, pexpect.TIMEOUT], timeout=30)

        printPexBa(pSsh, debug)    
        printDbg("" + str(index1), debug) 

        if index1 == 0:
            printDbg("" + str(index1) + " sending rsa passphrase...")
            pSsh.sendline("")            
        elif index1 == 1 or index1 == 3:
            printDbg("returned to cmd", debug)
            return SUCCESS
        elif index1 == 2:
            printDbg("hash verification response", debug)
            pSsh.sendline("yes")
        elif index1 == 4:
            printDbg("password request...", debug)
            pSsh.sendline(pPw)
            time.sleep(5)
        elif index1 == 5:
            printErr("bad password")
            return EXIT_ERR
        elif index1 == 6:
            printErr("file not found")
            return EXIT_ERR
        elif index1 == 7:
            printDbg("rm confirmation... sending yes")
            pSsh.sendline('y')
        elif index1 == 8:
            printErr("EOF reached. Something wrong...")
            return EXIT_ERR
        elif index1 == 9:
            printErr("timeout reached. Something wrong...")
            return EXIT_ERR
        else:
            printDbg("captured unknown response: " + str(index1))
            return EXIT_ERR

#   Login to Linux-like telnet terminal connection.
#   Can handle the rsa (will pass empty passphrase!, password response and denied response.
#   req:    
#   - None.
#   input:  
#   - pIp - ip address of the telnet server
#   - pPort - port number of the telnet server
#   - pUser - username
#   - pPw - password
#   return: 
#   - telnet handle if successfull, EXIT_ERR if failed to connect
#   Note: currently, this function is not working and broken. 

def telnetLogin(pIp, pPort, pUserName=None, pPw=None, pTimeOut=30):
    debug = 0
    debugL2 = 0
    index1 = None 
    prompt1 = '\[.*\]#'
    lTelnet = None
    lTimeOut = int(pTimeOut)

    printWarn("This function is currently broken. Use it at your own risk till fix is made!")

    lLoginCmd = "telnet " + str(pIp) + " " + str(pPort)

    if pIp == "127.0.0.1":
        printErr("can not login to local 127.0.0.1 address")
        return EXIT_ERR

    printDbg("ip/port/timeout: " + str(pIp) + "/" + str(pPort) + "/" + str(pTimeOut), debug)
    printDbg("telnet command: " + lLoginCmd, debug)

    lTelnet = pexpect.spawn(lLoginCmd)

    printDbg("spawned telnet process",debug)

    while 1:
        printDbg("telnet expect..")
        index1 = lTelnet.expect([pexpect.TIMEOUT, pexpect.EOF, 'cmc.*login:', 'cmc.*#', \
        'refused','invalid','No route to host', 'savbu.*>','Escape character is.*\]'], int(pTimeOut))
        time.sleep(1)

        if debugL2:
            printDbg("==========", debugL2)
            printDbg("in the loop...", debugL2)
            printDbg("----------", debugL2)
            printDbg("lTelnet.before: " + str(lTelnet.before), debugL2)
            printDbg("----------", debugL2)
            printDbg("lTelnet.after: " + str(lTelnet.after), debugL2)
            printDbg("----------", debugL2)

        printDbg("pexpect.idx: " + str(index1), debug) 

        if index1 == 0:
            printDbg("Error. unable to login to " + pIp + " for " + str(pTimeOut) + " seconds. Giving up.", debug)            
            lTelnet.close()
            return EXIT_ERR
        elif index1 == 1:            
            printDbg("Error. EOF received", debug)            
            return SUCCESS
        elif index1 == 2:            
            printDbg("CMC (IOM) login, sending username/password", debug)
            lTelnet.sendline("root")
            lTelnet.expect("Password:")
            lTelnet.sendline("cmc")
        elif index1 == 3:
            printDbg("OK. The  console is opened.", debug)
            currTimeStr = strftime("%Y-%m-%d.%H-%M-%S", gmtime())
            fout = file('sublog/telnet.' + sys.argv[0] + "." + pIp  + ".log",'w')
            lTelnet.logfile = fout
            #lTelnet.logfile = sys.stdout    
            return  lTelnet
        elif index1 == 4:
            printErr("connection refused to " + str(pIp))
            lTelnet.close()
            return EXIT_ERR
        elif index1 == 5:
            printErr("invalid password for " + str(pIp))
            lTelnet.close()
            return EXIT_ERR
        elif index1 == 6:
            printErr("No route to host, IP not pingable. " + str(pIp) + " " + str(index1))
            lTelnet.close()
            return EXIT_ERR
        elif index1 == 7:
            printDbg("found terminal server (serial server)", debug)
            return lTelnet
        elif index1 == 8:
            printDbg("Login successful", debug)
            lTelnet.sendline('\n')
            time.sleep(1)
            lTelnet.sendline('\n')
            time.sleep(1)
        else:
            printDbg("Error. Unknown index. Should have never reached here. index1: " + str(index1))

    printErr("should never reach here. Closing the telnet")
    lTelnet.close()
    return EXIT_ERR

#   This function will do a reconnect to the same ssh console by disconnecting it first.
#   This is done to have a "flush" effect on any remaining chars at the pexpect pipe which was causing the issue
#   until more elegant way is found and hoping it can give some resolution.
#   req:    
#   - None.
#   input:  
#   - pSsh - ssh handle
#   - pIp - ip address 
#   - pUser - user 
#   - pPw - password
#   - pTimeOut - time out value
#   return: 
#   - new handle to ssh of reconnect is successful.
#   EXIT_ERR if reconnection is failed.

def sshReConnect(pSsh, pIp, pUser = "admin", pPw = None, pTimeOut = 30):
    debug = 0
    debugL2 = 0
    lSshNew = None

    if pPw == None:
        getPw("LIN_PW_DEFAULT")

    lSshNew = sshLoginLinux(lSshNew, pIp, pUser, pPw)

    if lSshNew:
        printWarn("Re-connected to " + str(pIp) + ", returning new ssh handle.")
        #pSsh.close()
        return lSshNew
    else:
        printWarn("Unable to connect to " + str(pIp) + ", returning original ssh handle.")
        return pSsh

#   Login to Linux operating system (and possibly other OS) 
#   over out-of-band kvm ssh sol connection.
#   This is different than logging in using in-band host OS IP connection.
#   req:        
#   - pBmcSsh object have SSH connection established to KVM SSH.
#   - Host O/S (Linux) has enabled console direction enabled.
#   - The function is specifically tested with Linux RedHat 6.x Server specifically.
#   It has not been tested extensively with other types of Linux flavors and versions.
#   input:      
#   - pBmcSsh - ssh connection to blade.
#   return:     
#   - SUCCESS - if login is successful or not needed (upon detection of shell prompt)
#     EXIT_ERR - if login is failed or any other error.

def sshLoginOob(pBmcSsh, pRetry = 20):
    debug = 0
    debugL2 = 0
    CONFIG_MAX_LINUX_LOGIN_RETRY = 2

    linuxUser = getGlobal('CONFIG_EFI_LIN_USER')
    linuxPw = getGlobal('CONFIG_EFI_LIN_PW')

    if validateFcnInput([pBmcSsh]) == EXIT_ERR:
        return EXIT_ERR

    if linuxUser == None:
        linuxUser = 'root'

    if linuxPw == None:
        linuxPw = getPw("LIN_PW_DEFAULT")
    
    if pRetry == None:
        pRetry = 20

    printDbg("linux user and password is set to: " + str(linuxUser) + ", " + str(linuxPw), debug)
    printDbg("sending enter key...", debug)

    # Send one CR char with 20 retry to give time in case it is still booting.

    stat = cli_with_ret(pBmcSsh, "\r", "login:", "linuxShell", pRetry)

    printDbg("return:\n----\n" + str(stat) + "\n----", debug)

    if stat == EXIT_ERR:
        printErr("Unable to detect login prompt")
        return EXIT_ERR

    # Scan to see bash like command line is available.

    if re.search("\[root@.*\]", stat):
        printDbg("it appears login is not needed.", debug)
    else:
        printDbg("it appears login is needed.", debug)
        counter = 0

        # Attempt to login using username and password several times.

        while 1:
            printDbg("sending username", debug)
            stat = cli_with_ret(pBmcSsh, linuxUser, "Password:", "linuxShell")
            printDbg("return:\n----\n" + str(stat) + "\n----", debug)

            time.sleep(30)
    
            printDbg("sending password", debug)
            stat = cli_with_ret(pBmcSsh, linuxPw, "#", "linuxShell")
            printDbg("return:\n----\n" + str(stat) + "\n----", debug)

            if stat == EXIT_ERR:
                printErr("login failure is detected, retrying...")
                counter += 1

                if counter > CONFIG_MAX_LINUX_LOGIN_RETRY:
                    printErr("Timeout retrying, unable to login to linux shell.")
                    return EXIT_ERR
            else:
                printDbg("Login success", debug)
                return SUCCESS

#   This function is used to login to KVM console over ssh connection to linux like console.
#   Most typical usage of this function is to login to KVM connection using its management IP
#   or login to linux OS using its host OS IP.
#   req:    
#   - None.
#   input:      
#   - pSsh - ssh handle (should be unneeded)
#   - pIp - ip address of the ssh server of linux
#   - pUser - username
#   - pPw - password
#   return: 
#   - SSH handle if successfull, 
#     EXIT_ERR if failed to connect

def sshLogin(pIp, pUser, pPw, pTimeOut=180):
    debug = 0
    debugL2 = 0
    index1 = None 
    counterPassword = None
    pSsh = None
    pPwShow = getUnEncPw(pPw)
    CONFIG_WAIT_DELETE_KNOWN_HOST = 5
    CONFIG_TRY_MAX_EOF = 2

    if validateFcnInput([pIp, pUser, pTimeOut], debug) == EXIT_ERR:
        return EXIT_ERR
    
    if pIp == "127.0.0.1":
        printErr("can not login to local 127.0.0.1 address")
        return EXIT_ERR

    lLoginCmd = "ssh -l " + pUser + " " + pIp

    printDbg("ssh command: " + lLoginCmd, debug)
    pSsh = pexpect.spawn(lLoginCmd)

    counterPassword = 0
    counterEof = 0

    while 1:
        index1 = pSsh.expect([\
            pexpect.TIMEOUT,\
            pexpect.EOF, \
            '.*rsa', \
            '.*\[.*\].*$',\
            '.*ssword.*', \
            '.*authenticity.*', \
            '.*#', \
#!!!! experiment  '.*CISCO Serial Over LAN.*',\
            '.*CISCO Serial Over LAN',\
            '.*No route to host.*',\
            '.*passphrase.*',\
            '.*No free SOL sessions available.*',\
            '\[.*\]#',\
            '.*refused.*'\
            ], timeout=pTimeOut)

        if debugL2:
            printDbg("==========", debugL2)
            printDbg("in the loop...", debugL2)
            printDbg("----------", debugL2)
            printDbg("pSsh.before: " + str(pSsh.before), debugL2)
            printDbg("----------", debugL2)
            printDbg("pSsh.after: " + str(pSsh.after), debugL2)
            printDbg("----------", debugL2)

        printDbg("pexpect.idx: " + str(index1), debug) 
    
        # process pexpect return patterns and do an appropriate action based on it.

        if index1 == 0:
            printErr("Timeout. 1. Unable to login to " + pIp + " for " + str(pTimeOut) + " seconds. Giving up.")
            return EXIT_ERR

        # EOF is likely caused by a mismatching public key. This mostly happens when server re-inserted, image is upgrade.
        # Note that this is not a secure practice to replace the old key with new one as this can also be a caused
        # by a MIM attack.

        if index1 == 1:
            printWarn("Connection to : " + str(pIp))
            printWarn("EOF: Possible reasons: ")
            printWarn("1. Different public key in ~/.ssh/knownhost or too many UCSM sessions.")
            printWarn("2. No SSH service on target IP, non-existing IP.")

            '''
            printWarn("if you think, otherwise will continue after few seconds wait.")
            printWarn("Renaming ~/.ssh/known_hosts in to ~/.ssh/known_hosts.bak in " + str(CONFIG_WAIT_DELETE_KNOWN_HOST) + " seconds. If not desired, use ctrl+C to terminate.")
            time.sleep(CONFIG_WAIT_DELETE_KNOWN_HOST)
            os.system("rm -rf ~/.ssh/known_hosts.bak")
            os.system("mv ~/.ssh/known_hosts ~/.ssh/known_hosts.bak")
            '''

            printErr("console outputs:")
            printErr("----------",)
            printErr("pSsh.before: " + str(pSsh.before))
            printErr("----------")
            printErr("pSsh.after: " + str(pSsh.after))
            printErr("----------")

            pSsh.close()
            pSsh = pexpect.spawn(lLoginCmd)
            counterEof += 1
            
            if counterEof > CONFIG_TRY_MAX_EOF:
                printErr("EOF error " + str(CONFIG_TRY_MAX_EOF) + ". Giving up.")
                return EXIT_ERR

            continue
        elif index1 == 2:
            printDbg("sending rsa passphrase...", debug)
            pSsh.sendline("")
        elif index1 == 3 or index1 == 6:
            printDbg("OK. The  console is opened.", debug)
            currTimeStr = strftime("%Y-%m-%d.%H-%M-%S", gmtime())
            fout = file('sublog/' + sys.argv[0] + "." + pIp  + ".log",'w+')
            pSsh.logfile = fout
            #pSsh.logfile = sys.stdout    
            printDbg("returning pSsh value: " + str(pSsh), debugL2)
            return pSsh
        elif index1 == 4:            
            printDbg("sending password...", debug)
            pSsh.sendline(pPw)
            counterPassword += 1
        
            if counterPassword > 3:
                printErr("password denied 3 times. Giving up.")

        elif index1 == 5:
            printDbg("login confirmation with hash", debug)
            pSsh.sendline("yes")
        elif index1 == 7:
            printDbg("OK. logged onto Cisco BMC SOL console: " + str(index1), debug)
            currTimeStr = strftime("%Y-%m-%d.%H-%M-%S", gmtime())
        
            # Create log file name, that name of file name should be unique and identificable,
            # it could ip address of ssh connection, time it was created.

            fout = file('sublog/' + sys.argv[0] + ".bmcsol." + pIp + "." + currTimeStr + ".log",'w')

            # Put bmc sol log filename into tmp run-time log file.

            tmpLogFileName = PYTHON_ROOT_DIR + "/log/tmp." + str(os.getpid()) + ".log"
            tmpFp = open("tmpLogFileName", 'w')
            tmpFp.write("BMC SOL log: " + str(fout.name))
            tmpFp.close()
        
            pSsh.logfile = fout
            break
        elif index1 == 8:
            printErr("No route to host, IP not pingable: " + str(pIp) + ", " + str(index1))
            return EXIT_ERR

        # This has not been working yet!!!! Need to fix. This happens when device or service 
        # generates different private and public key identification, and when the prior shell
        # client sends obsolete public key for identification, the server refuses to accept.
        # Currently workaround is to manually delete the ~/.ssh/known_hosts.

#        elif index1 == 10:
#            printDbg("permanently added to list of known hosts. " + str(index1), debug)
#            pSsh.expect("ssword")
#            pSsh.sendline(pPw)
        elif index1 == 9:
            printDbg("sending empty passphrase" + str(index1), debug)
            pSsh.sendline('\r')
        elif index1 == 10:
            printErr("no free SOL sessions available. " + str(index1))
            return EXIT_ERR
        elif index1 == 12:
            printErr("connection is refused.")
            return EXIT_ERR
        else:
            printDbg("Error. Unknown index. Should have never reached here. index1: " + str(index1))
    return pSsh   

# This function is used to login to Linux system over ssh connection.
# This function should phase out soon and all call instance to this function in the test framework
# should be replaced by sshLogin() function. 
# pSsh - ssh handle (should be unneeded)
# pIp - ip address of the ssh server of linux
# pUser - username
# pPw - password
# return - telnet handle if successfull, EXIT_ERR if failed to connect

def sshLoginLinux(pSsh, pIp, pUser, pPw, pTimeOut=60):
    debug = 0
    debugL2 = 0
    index1 = None 
    counterPassword = None
    CONFIG_WAIT_DELETE_KNOWN_HOST = 5
    counterEof = None
    CONFIG_TRY_MAX_EOF = 3

    printWarnMinor("THIS FUNCTION IS DEPRECATED AND SHOULD BE REPLACED BY def sshLogin")

    pPwShow = getUnEncPw(pPw)

    if pIp == None:
        printErr("pIp is none.")
        return EXIT_ERR

    if pIp == "127.0.0.1":
        printErr("can not login to local 127.0.0.1 address")
        return EXIT_ERR

    if pUser == None:
        printErr("pUser is none.")
        return EXIT_ERR

    if pPw == None:
        printErr("pPw is none.")
        return EXIT_ERR

    lLoginCmd = "ssh -l " + pUser + " " + pIp

    printDbg("ip, user, pw, timeOut: " + pIp + " : " + pUser + " : *** " + pPwShow + " : " + str(pTimeOut), debug)
    printDbg("ssh command: " + lLoginCmd, debug)
    pSsh = pexpect.spawn(lLoginCmd)

    counterPassword = 0
    counterEof = 0

    while 1:
        index1 = pSsh.expect([\
            pexpect.TIMEOUT,\
            pexpect.EOF, \
            '.*rsa', \
            '.*\[.*\].*$',\
            '.*ssword.*', \
            '.*authenticity.*', \
            '.*#', \
            '.*CISCO Serial Over LAN.*',\
            '.*No route to host.*',\
            '.*passphrase.*',\
            '.*No free SOL sessions available.*',\
            '\[.*\]#',\
            'Connection refused'\
            ], pTimeOut)
        if debugL2:
            printDbg("==========", debugL2)
            printDbg("in the loop...", debugL2)
            printDbg("----------", debugL2)
            printDbg("pSsh.before: " + str(pSsh.before), debugL2)
            printDbg("----------", debugL2)
            printDbg("pSsh.after: " + str(pSsh.after), debugL2)
            printDbg("----------", debugL2)

        printDbg("pexpect.idx: " + str(index1), debug) 
    
        if index1 == 0:
            printErr("Timeout. unable to login to " + pIp + " for " + str(pTimeOut) + " seconds. Giving up.")
            printErr("It is possible that if server is re-inserted or new server is in place, pkey is different.")
            printErr("1. Try removing or moving the known_hosts the file to different place:  /<userHomeDir>/.ssh/known_hosts")

#       if index1 == 1:
#           printErr("EOF. unable to login to " + pIp + ". Giving up.")

        if index1 == 1:
            printWarn("EOF: Possible reasons: different public key in ~/.ssh/knownhost or too many UCSM sessions.")
            printWarn("This can also be caused by man in the middle attack. Press CRTL+F to terminate the script ")
            printWarn("if you think, otherwise will continue after few seconds wait.")
            printWarn("Renaming ~/.ssh/known_hosts in to ~/.ssh/known_hosts.bak in " + \
                str(CONFIG_WAIT_DELETE_KNOWN_HOST) + " seconds. If not desired, use ctrl+C to terminate.")

            time.sleep(CONFIG_WAIT_DELETE_KNOWN_HOST)
            os.system("rm -rf ~/.ssh/known_hosts.bak")
            os.system("mv ~/.ssh/known_hosts ~/.ssh/known_hosts.bak")
            pSsh.close()
            pSsh = pexpect.spawn(lLoginCmd)
            counterEof += 1

            if counterEof > CONFIG_TRY_MAX_EOF:
                printErr("EOF error " + str(CONFIG_TRY_MAX_EOF) + ". Giving up.")
                return EXIT_ERR

            return EXIT_ERR
        elif index1 == 2:
            printDbg("sending rsa passphrase...", debug)
            pSsh.sendline("")
        elif index1 == 3 or index1 == 6:
            printDbg("OK. The  console is opened.", debug)
            currTimeStr = strftime("%Y-%m-%d.%H-%M-%S", gmtime())
            fout = file('sublog/' + sys.argv[0] + "." + pIp  + ".log",'w')
            pSsh.logfile = fout
            #pSsh.logfile = sys.stdout    

            printDbg("before and after: ", debugL2)
            printDbg("\n----------------\n" + str(pSsh.before) + "\n----------------\n", debugL2)
            printDbg("\n----------------\n" + str(pSsh.after) + "\n----------------\n", debugL2)
            printDbg("returning pSsh value: " + str(pSsh), debugL2)
            return pSsh
            break
        elif index1 == 4:            
            printDbg("sending password...", debug)
            pSsh.sendline(pPw)
            counterPassword += 1
        
            if counterPassword > 3:
                printErr("password denied 3 times. Giving up.")

        elif index1 == 5:
            printDbg("login confirmation with hash", debug)
            pSsh.sendline("yes")
        elif index1 == 7:
            printDbg("OK. logged onto Cisco BMC SOL console: " + str(index1), debug)
            currTimeStr = strftime("%Y-%m-%d.%H-%M-%S", gmtime())
        
            fout = file('sublog/' + sys.argv[0] + ".bmcsol." + pIp + "." + currTimeStr + ".log",'w')

            # put bmc sol log filename into tmp log

            tmpLogFileName = PYTHON_ROOT_DIR + "/log/tmp." + str(os.getpid()) + ".log"
            tmpFp = open("tmpLogFileName", 'w')
            tmpFp.write("BMC SOL log: " + str(fout.name))
            tmpFp.close()
        
            pSsh.logfile = fout
            #pSsh.logfile = sys.stdout    
            break
        elif index1 == 8:
            printErr("No route to host, IP not pingable. " + str(pIp) + ", " + str(index1))
            return EXIT_ERR
#        elif index1 == 10:
#            printDbg("permanently added to list of known hosts. " + str(index1), debug)
#            pSsh.expect("ssword")
#            pSsh.sendline(pPw)
        elif index1 == 9:
            printDbg("sending empty passphrase" + str(index1), debug)
            pSsh.sendline('\r')
        elif index1 == 10:
            printErr("no free SOL sessions available. " + str(index1))
            return EXIT_ERR
        elif index1 == 12:
            printErr("Connection is refused to " + str(pIp) + " for unknown reason.")
            printErr("----------")
            printErr("pSsh.before: " + str(pSsh.before), debugL2)
            printErr("----------")
            printErr("pSsh.after: " + str(pSsh.after), debugL2)
            printErr("----------")
            return EXIT_ERR
        else:
            printDbg("Error. Unknown index. Should have never reached here. index1: " + str(index1))
    return pSsh   

#   Find hostname of FI based on IP address of its UCSM management and perform initialization.
#   In order to do so, it scans through FI list file which has 1-to-1 mapping of hostname to 
#   IP address along with other optional information i.e. FI-A/FI-B/IOM-A/IOM-B telnet access
#   information. If it does not find the mapping in the file, it will attempt to login to UCSM
#   using the IP address and will try to find the name of hostname of FI.
#   Following enhancement is not implemented:
#   If the mapping is not found, this function will give user an option (with timeout) to add
#   the entries to the FI list file. However separate file (FI list add-on) is used for this purpose.
#  
#   input:    pFi - IP address of FI.
#   return -  SUCCESS if found the FI entries from the list file. 
#           EXIT_ERR - if any failure resulting not being able to find the FI.

def setFi(pFi):
    debug = 0
    debugL2 = 0
    fpBase = None
    fpAddOn = None

    line = 1
    counter = 0
    fiListFile = None
    fiListPath = None
    fiListFileAddOn = None
    fiListPathAddOn = None

    printDbg("Entered: mgmtIp: " + str(pFi.mgmtIp), debug)
    printDbg("opening FI list file", debug)

    # Open the fiListFile, this is maintained under git repository.

    fiListFile = getGlobal('CONFIG_PATH_FI_LIST_NEW')

    if fiListFile == None:
        printErr("can not find FI list file")
        return EXIT_ERR
    else:
        fiListPath = PYTHON_ROOT_DIR + "/" + fiListFile

    printDbg("fi list path: " + str(fiListPath))
    fpBase  = open(fiListPath, 'r')
    
    if fpBase == None:
        printErr("Error. Unable to open/find the file name: " + str(FiListUsFileName))
        return EXIT_ERR

    # Attempt to open the fi list add-on file.
    # Failure to open this file should not result in error condition. If FI is not found in
    # any of these files, this function still manage to temporarily login to specified IP
    # address and try to identify this is UCS FI from the prompt. 

    fiListFileAddOn = getGlobal('CONFIG_PATH_FI_LIST_ADD_ON')
    
    if fiListFileAddOn == None:
        printWarn("can not find FI list add-on file")
    else:
        fiListPathAddOn = PYTHON_ROOT_DIR + "/" + fiListFileAddOn
    printDbg("fi list add-on path: " + str(fiListPathAddOn), debug)

    try:
        fpAddon  = open(fiListPathAddOn, 'w')
    except IOError:
        printDbg("add-on fi list file does not exist.")
    except TypeError:
        printDbg("add-on fi list file does not exist.")

    # Search the FI entry from both base and add-on list file.

    for fp in (fpBase, fpAddOn):
        if fp:
            printDbg("scanning " + str(fp.name))

            while line:
                counter+=1
                line = fp.readline()
        
                printDbg("line content: ", debugL2)
                printDbg("line content: ", debug)
                printDbg(line, debugL2)
                printDbg(str(re.sub(" ","", line)), debug)
        
                lineTokens=line.split('/')
        
                for i in range (0, len(lineTokens)):
                    lineTokens[i].strip()
        
                # Skip commented line.
        
                if re.search("^#", line):
                    continue
                
                # Skip empty lines
        
                if len(line.strip()) < 2:
                    continue
            
                # If found matching IP.
        
                if pFi.mgmtIp == lineTokens[1].strip():
                    printDbg("found matching hostname: " + lineTokens[0], debug)
        
                    # Set the FI class instance other properties, i.e. FI-A/FI-B and IOM-A/IOM-B access info if exist.
        
                    if lineTokens[0].strip():
                        pFi.hostName = lineTokens[0].strip()
                        pFi.hostNameBase = lineTokens[0].strip()
                    if lineTokens[1].strip():
                        pFi.mgmtIpA = lineTokens[2].strip()
                    if lineTokens[2].strip():
                        pFi.mgmtIpB = lineTokens[3].strip()
                    if lineTokens[3].strip():
                        pFi.cmcIpA = lineTokens[4].strip()
                    if lineTokens[4].strip():
                        pFi.cmcIpB = lineTokens[5].strip()
                    
                    #printDic(pFi)
                    return pFi

    # At this stage, FI is not found in any of the list files, attempt to login manually.
        
    printDbg("Attempting to login to FI...")
    ucsmSshTmp = sshLogin(pFi.mgmtIp, pFi.user, pFi.password)

    if ucsmSshTmp == None:
        printErr("Can not login to " + str(pFi.mgmtIp) + ".")
        return EXIT_ERR

    # Once login is completed, try to see if the prompt ends with either -A or -B then it is likely
    # to be FI. It is possible devices other than UCS FI can shell prompt ending wiwth -A/-B so
    # this is not bulletproof method for ID-ing the UCS FI. Better method can be implemented.

    try:
        pFi.hostName = re.search("\n(.*)-[A-B]#", ucsmSshTmp.after, re.MULTILINE).group(1)
        printDbg("hostName: " + str(pFi.hostName))
        ucsmSshTmp.close()

        '''
        os.system('clear')
        choice = raw_input(\
        '\n--------------------------------------------------------------------\
        \nNo fabric-interconnect found for ' + pFi.mgmtIp + ' , do you wish to \
        \nadd to test bed database?\n\
        \n--------------------------------------------------------------------\
        \nChoosing no will result in error condition) Y/N: ')
    
        if choice == "Yes" or choice == "Y" or choice == "yes" or choice == "y":
            printDbg("chose to add to database.")

        #Oahu            / 10.193.225.61     / - / - / - / -
        #fpAddOn.write(pFi.hostName.ljust(17) + "/ " + pFi.mgmtIp.ljust(20) + "/ - / - / - / -")
        #fpAddOn.close()
        '''

        printDic(pFi)
        return pFi

    except AttributeError:
        ucsmSshTmp.close()
        printErr("Unable to extract hostname from UCSM after login.")
        return EXIT_ERR

    else:
        printDbg("chose not to add to database.")
        return EXIT_ERR

#   Sends cli command to different types of shell and returns back the response.
#   Currently supported types of shells are:
#   1. UCS FI shell
#   2. efiShell
#   3. linuxshell
#   Other types of shell can be added by enhancing the function.
#   req:    
#   - ssh connection to FI must be established prior to invoking this function.
#   input:  
#   - pConsole - ssh handle
#   - pCommand - command to issue
#   - pPrompt - ucsm hostname
#   - pShellType 1. ucsmCli(default), 2. efiShell 3. linuxShell. 
#   return: 
#   - output of cli command
#   - EXIT_ERR for any error condition.
#   - EXIT_PASSWD if terminal (linux) asks for password.
#   - UCSM FI:  no ucsm fi specific return values.
#               efiShell: no efishell specific return values.
#               linuxShell: no linuxshell specific return values.

def cli_with_ret(pConsole, pCommand, pPrompt, pShellType='ucsmCli', pWaitCycles = None, pConsoleB4After = None):
    debugCli = 0
    debugL2 = 0 
    debug = 0

    debugL2Ucsm = 0
    debugL2Efi = 0
    debugL2LinuxShell = 0

    debugUcsm = 0
    debugEfi = 0
    debugLinuxShell = 0

    index1 = None
    pCommandSubs = None
    finalOutIndex1 = ""
    counterTimeOut = None
    counter = None
    counterFlushTimeout = None

    if  (pShellType == 'efiShell' and debugEfi == 1) or \
        (pShellType == 'ucsmCli' and debugUcsm == 1) or \
        debug == 1:
        printDbg("\n================\nEntered: pCommand/pConsolePrompt/pShellType: " + str(pCommand) + "/" + str(pPrompt) + "/" + str(pShellType), debug)

    if pConsole == None:
        printErr("pConsole is None")
        return EXIT_ERR
  
    if  (pShellType == 'efiShell' and debugEfi == 1) or \
        (pShellType == 'ucsmCli' and debugUcsm == 1) or \
        debug == 1:
        printDbg("pCommand: " + pCommand, debug)

    LOCAL_CONFIG_CMD_WAIT_CYCLES = 4

    if pWaitCycles:
        LOCAL_CONFIG_CMD_WAIT_CYCLES = pWaitCycles

    if  (pShellType == 'efiShell' and debugEfi == 1) or \
        (pShellType == 'ucsmCli' and debugUcsm == 1) or \
        debug == 1:
        printDbg("LOCAL_CONFIG_CMD_WAIT_CYCLES is set to: " + str(LOCAL_CONFIG_CMD_WAIT_CYCLES), debug)

    # Handle the ucsm command. 

    if pShellType == 'ucsmCli':
        expectString = ".*" + pPrompt + ".*#"
    
        printDbg("expectString: " + expectString, debugUcsm)

        if debugL2Ucsm:
            printDbg("checking if flush needed...")
            printDbg("\n1.before:\n----\n" + (pConsole.before) + "\n----\nafter:\n----\n" + str(pConsole.after) + "\n----")

        # Flush the pexpect child buffer if there is any item left

        counterFlushTimeout = 0

#       Not sure why it is causing error.
#       printDbg("pConsole.after len-s: " + str(len(pConsole.after)))

        try:
            while len(str(pConsole.after).strip()) > 0:
                printDbg("pConsole.after len: " + str(len(pConsole.after)), debugUcsm)
                printDbg("pConsole after and before not empty. need to flush before sending command.", debugUcsm)
    
                index1 = pConsole.expect([pexpect.TIMEOUT, pexpect.EOF, ".*"],  re.DOTALL)
    
                if index1 == 0:
                    printWarn("TIMEOUT encountered trying to flush. Result is unpredictable.")
                    break                
                elif index1 == 1:
                    printErr("child closed (EOF) while flushing the ssh handle.")
                    return EXIT_ERR
                else:
                    if debugUcsm:
                        printDbg("captured string: ")
                        printDbg("\n2.console flush loop: before:\n----\n" + (pConsole.before) + "\n----\n\
                        console flush loop: after:\n----\n" + str(pConsole.after) + "\n----", debugL2Ucsm)
            
                    counterFlushTimeout += 1
    
        except Exception as exception1:
            printDbg("exception occurred.", debugUcsm)
            return EXIT_ERR

        printDbg("sending command...", debugL2Ucsm)

        counterTimeOut = 0

        # Send the actual command and handle the interaction with the shell.
    
        while 1:
            pConsole.sendline(pCommand)
            index1 = pConsole.expect([pexpect.TIMEOUT, pexpect.EOF, expectString],  timeout=30)

            if index1 == 0:
                printWarn("ucsmCli: TIMEOUT or EOL, can not return back to command line, trying again...")
                counterTimeOut += 1 
            
                if counterTimeOut > LOCAL_CONFIG_CMD_WAIT_CYCLES:
                    printErr("ucsmCli: Unable to return back to command line, TIMEOUT " + str(LOCAL_CONFIG_CMD_WAIT_CYCLES) + " times with command: " + str(pCommand))
                    return EXIT_ERR
            elif index1 == 1:
                printDbg("EOL encountered.")
                return EXIT_ERR
            elif index1 == 2:
                finalOutB4 = pConsole.before
                finalOut = pConsole.after

                printDbg("\n3.before:\n----\n" + (pConsole.before) + "\n----\nafter:\n----\n" + str(pConsole.after) + "\n----", debugL2Ucsm)
                finalOut = re.sub(pPrompt + ".*#", "", finalOut, 1)
                finalOut = re.sub("\n\n", "\n", finalOut, 1)
                finalOut = re.sub("\n\n", "\n", finalOut, 1)
                finalOut = re.sub("\n\n", "\n", finalOut, 1)
                pCommandSubs = re.sub("\(|\)|\\\\|\^|\"|\|",".", pCommand)
                printDbg("ucsmCli: pCommandSubs: " + str(pCommandSubs), debugUcsm)

                finalOut = re.sub(pCommandSubs + ".*\n", "", finalOut, 1)

                printDbg("ucsmCli: after resub:\n-- -- -- -- --\n" + str(finalOut) + "\n-- -- -- -- --\n", debugL2Ucsm)

                printDbg("ucsmCli: Successfully returned back to command line", debugUcsm)
                return finalOut                
            else:
                printWarn("Unknown index, should have never reached here: " + str(index1))

    # Handle the efishell command.

    elif pShellType == 'efiShell':
        printDbg("efishell: sending efi shell command", debugEfi)
        printDbg("pCommand: " + pCommand, debugEfi)

        # If prompt is None or empty, attempt using the default prompt.

        if pPrompt == "" or pPrompt == None:
            printDbg("efishell: Unknown efi prompt, using default prompt", debugEfi)
            expectString = "Shell>|fs[0-9]..>"
        else:
            printDbg("efishell: prompts is set to " + str(pPrompt), debugEfi)
            expectString = pPrompt

        finalOutReSub = None

        if debugCli:
            printDbg("efishell: expectString: " + expectString, debugEfi)

        counter = 0
        counterFlushTimeout = 0

        # Flush the pexpect child buffer if there is any string left over.

        if debugEfi:
            printDbg("checking if flush needed...")
            printDbg("\npConsole.before:\n-----------------\n" + str(pConsole.before) + \
            "\n---------------\npConsole.after:\n" + str(pConsole.after) + \
            "\n---------------\n")

        pConsole.send('\n')

        while len(str(pConsole.after)) > 0:
            printDbg("pConsole after and before not empty. need to flush before sending command.", debugEfi)

            #           this is giving error. Need to work on this. 
            '''
            Traceback (most recent call last):
              File "./driver.py", line 485, in <module>
                configureStartTest(fpConfig)
              File "./driver.py", line 289, in configureStartTest
                stat = callablePy(mainArgv, instGlobalConfig)
              File "/pythondev/git.repo.satool/tests/pcie/pcieCheckSvid.py", line 234, in main
                if (blade.bootEfiShell(fi.ucsmSsh, sp.bmcSsh)) == None:
              File "/pythondev/git.repo.satool/api/ucs.py", line 1534, in bootEfiShell
                efiShellBoConfigFile = getGlobal("CONFIG_EFI_SHELL_BOOT_ORDER")
              File "/pythondev/git.repo.satool/api/saLibrary.py", line 2943, in cli_with_ret
                printDbg("pConsole.before:after len-s: " + str(len(pConsole.before)) + " : " + str(len(pConsole.after)), debugEfi)
            TypeError: object of type 'type' has no len()
            
            '''
            #           printDbg("pConsole.before:after len-s: " + str(len(pConsole.before)) + " : " + str(len(pConsole.after)), debugEfi)

            index1 = pConsole.expect([pexpect.TIMEOUT, pexpect.EOF, ".*"],  re.DOTALL)

            if index1 == 0:
                printWarn("TIMEOUT encountered trying to flush. Result is unpredictable.")
                break                
            elif index1 == 1:
                printErr("child closed (EOF) while flushing the ssh handle.")
                return EXIT_ERR
            else:
                if debugEfi:
                    printDbg("captured string: ")
                    printDbg("\n----\n" + (pConsole.before) + "\n----\n" + str(pConsole.after))
        
                counterFlushTimeout += 1

                '''
                if counterFlushTimeout > 5:
                    printDbg("Flushing takes too long, giving up, rest of string is unpredictable.\
                    possible reason is somehing left in the buffer while at the same time, some activity\
                    on server causing more feed to buffer.")
                '''

        # Send the actual command and interaction part.
        
        printDbg("sending actual command", debugEfi)

        pConsole.sendline(pCommand)
        pConsole.sendline('\r')

        # While loop will handle the various responses from the shell and attempt to do the appropriate 
        # response based on it.    
        
        while 1:
            index1 = pConsole.expect(\
                [pexpect.TIMEOUT, \
                pexpect.EOF, \
                '.*to continue.*to exit.*to change mode.*',\
                '.*is not recognized as an internal or external command.*' + expectString.split('|')[0] + ".*", \
                '.*is not recognized as an internal or external command.*' + expectString.split('|')[1] + ".*", \
                '.*Press ESC in.*continue\..*' + expectString.split('|')[0] + ".*",\
                '.*Press ESC in.*continue\..*' + expectString.split('|')[1] + ".*",\
                '.*Press any key to stop the EFI SCT running.*' + expectString.split('|')[0] + ".*",\
                '.*Press any key to stop the EFI SCT running.*' + expectString.split('|')[1] + ".*",\
                '.*Error\..*' + expectString.split('|')[0] + ".*",\
                '.*Error\..*' + expectString.split('|')[1] + ".*",\
                expectString], timeout=20)

            printDbg("efishell: index1: " + str(index1), debugEfi)

            finalOutB4 = pConsole.before
            finalOutAfter = pConsole.after
            finalOut = finalOutB4
        
            printDbg("\npConsole.before:\n-----------------\n" + str(finalOutB4) + \
            "\n---------------\npConsole.after:\n" + str(finalOutAfter) + \
            "\n---------------\n", debugCli)

            printDbg("efishell: 1. FinalOut before filter: " + str(finalOut), debugCli)
            printDbg("efishell: filtering garbage chars...", debugEfi)
            finalOut = re.sub(".\[[0-9]m", "", finalOut)
            finalOut = re.sub(".\[.*H", "", finalOut)

            printDbg("efishell: 2. FinalOut after filter: " + str(finalOut), debugCli)

            if index1 == 1:
                printErr("efishell: EOF encountered, child has died. Check the connection.")
                return EXIT_ERR
            elif index1 == 0:
                printDbg("efishell: " + str(counter) + ". Extending time for cmd: " + str(pCommand), debugEfi)
                printDbg("LOCAL_CONFIG_CMD_WAIT_CYCLES: "  + str(LOCAL_CONFIG_CMD_WAIT_CYCLES), debugEfi)

                if counter > LOCAL_CONFIG_CMD_WAIT_CYCLES:
                    printWarnMinor("efishell: Retry exhausted, giving up with command: " + str(pCommand) +  ". Caller scripts remaining tasks could be unpredictable!")
                    if debugEfi:
                        printPexBa(pConsole, 1)
                    return EXIT_ERR

                time.sleep(10)
                counter += 1

                '''
                elif index1 == 2:
                printDbg("efishell: user input is requested or blue screen of boot error, pressing Enter")
                finalOutIndex1 = str(finalOutIndex1) + str(finalOutB4)
#               finalOutIndex1 = re.sub("Enter to continue.*for help.", "", finalOutIndex1)
                printDbg("efishell: current output: " + str(finalOutIndex1))
                pConsole.sendline('\r')
                time.sleep(3)
                '''

            elif index1 == 3 or index1 == 4:
                printErr("efishell: Unable to find the command or application: " + str(pCommand.split()[0]) )
                print pCommand.split()
                return EXIT_ERR
            elif index1 == 5 or index1 == 6 or index1 == 7 or index1 == 8:
                printDbg("efishell: Skipping any startup program..")
                pConsole.sendline('\r')
            elif index1 == 9 or index1 == 10:
                printErr("efishell: Error executing the command: " + str(pCommand))
                printErr("efishell: Result: " + str(finalOut))
                return EXIT_ERR
            elif index1 == 11:
                finalOut = str(finalOutIndex1) + finalOut
                printDbg("efishell: 3. FinalOut at index 10: " + str(finalOut), debugCli)
                printDbg("efishell: ---------------------", debugL2Efi)
                printDbg("efishell: 4. FinalOut at exit:" + str(finalOut), debugCli)
                print finalOut
                printDbg("efishell: ---------------------", debugL2Efi)
                time.sleep(1)
                return finalOut
            else:
                printDbg("efishell: unknown index1: " + str(index1), debugEfi)

    # Handle the linuxshell type command.
   
    elif pShellType == 'linuxShell':
        printDbg("-------------------", debugLinuxShell)
        printDbg("linuxShell: sending linux shell command: " + str(pCommand), debugLinuxShell)

        expectString = pPrompt
        finalOutReSub = None

        if debugCli:
            printDbg("linuxShell: expectString: " + expectString, debugLinuxShell)

        if debugL2Ucsm:
            printDbg("checking if flush needed...")
            printPexBa(pConsole, debugLinuxShell)

        # Flush the pexpect child buffer if there is any item left

        counterFlushTimeout = 0

#       not sure why it is causing error 
#       printDbg("pConsole.after len-s: " + str(len(pConsole.after)))

        while len(str(pConsole.after).strip()) > 0:
            # fucking causing error.
            #printDbg("pConsole.after len: " + str(len(pConsole.after)), debugLinuxShell)  
            #printDbg("pConsole after and before not empty. need to flush before sending command.", debugLinuxShell)

            index1 = pConsole.expect([pexpect.TIMEOUT, pexpect.EOF, ".*"],  re.DOTALL)

            if index1 == 0:
                printWarn("TIMEOUT encountered trying to flush. Result is unpredictable.")
                break                
            elif index1 == 1:
                printErr("child closed (EOF) while flushing the ssh handle.")
                return EXIT_ERR
            else:
                if debugLinuxShell:
                    printDbg("captured string: ")
                    printPexBa(pConsole, debugLinuxShell, 'console flush loop: ')
        
                counterFlushTimeout += 1

        '''
                if counterFlushTimeout > 5:
                    printDbg("Flushing takes too long, giving up, rest of string is unpredictable.\
                    possible reason is somehing left in the buffer while at the same time, some activity\
                    on server causing more feed to buffer.")
        '''

        printDbg("sending command..." + str(pCommand), debugLinuxShell)
            
        counter = 0

        # While loop to interact with the shell based on response.

        while 1:
            pConsole.sendline(pCommand + '\r')
            time.sleep(1)

            index1 = pConsole.expect([\
                pexpect.TIMEOUT, \
                pexpect.EOF, \
                expectString, \
                "-bash.*No such file or directory", \
                "-bash.*command not found", \
                "Login incorrect",\
                "password:"\
                ],  re.DOTALL)

            printDbg("index1: " + str(index1), debugLinuxShell) 

            if index1 == 0:
                printDbg("linuxShell: can not return back to command line")
                counter += 1
            
                if counter > LOCAL_CONFIG_CMD_WAIT_CYCLES:
                    printErr("linuxShell: timeout reached trying to return back to linux shell.")

                    '''
                    try:
                        pCommand = "\r"
                        #pPrompt = "IBMC-SLOT\[.*\] #"
                        printDbg("linuxShell: pCommand: " + str(int(pCommand[0])))
                        pCommand = chr(int(pCommand[0]))            
                        continue
                    except ValueError:
                        printDbg("linuxShell: Unable to translate the Ctrl+C")
                        return EXIT_ERR
                    '''
                    return EXIT_ERR
            elif index1 == 1:
                printDbg("linuxshell: EOF, the ssh console might have been closed.")
                return EXIT_ERR
            elif index1 == 4 or index1 == 3:
                printErr("linuxshell: no such file or command. Did you type correctly?: " + str(pCommand))
                return EXIT_ERR
            elif index1 == 5:
                printErr("login failure detected")
                printPexBa(pConsole, debugL2Linux)
                return EXIT_ERR
            elif index1 == 6:
                printDbg("password needed:")
                return EXIT_PASSWD
            elif index1 == 2:
                finalOutB4 = pConsole.before
                finalOutAfter = pConsole.after
                finalOut = finalOutB4

                if pConsoleB4After == 1:
                    finalOut = finalOutAfter

                printPexBa(pConsole, debugL2LinuxShell, "index 2 capture case:")

                # replace escape char in the command with . so that can be searched.
                # TEST well with various script before merging to MASTER!!!!
            
                printDbg("linuxShell: pCommand before subs: \n===\n" + str(pCommand) + "\n===", debugL2LinuxShell)
                pCommand = re.sub( "\||\|\)|\(|\\\\", ".", str(pCommand)).strip()
                printDbg("linuxShell: pCommand after  subs: \n===\n" + str(pCommand) + "\n===", debugL2LinuxShell)
                finalOut = re.sub(pCommand, "", finalOut)
    
                return finalOut
    else:
        printErr("Unknown shell type. Can not continue.")
        return EXIT_ERR

#   print Help for BIOS update script. It is only for bios.update.py only.
#   input:  
#   - None.
#   return: 
#   - None.

def printBiosUpdateHelp():
    printBarDouble()
    printDbg("Usage: ")
    printDbg(sys.argv[0] + " <UCS FI IP> <chassis/slot> <BIOS image> <TFTP-server-ip> <switches>")
    printBarDouble()
    printDbg("i.e.:")
    printDbg("Update BIOS on 10.0.0.1 1/5 blade with BIOS B200M3.blob.zip default TFTP server.")
    printDbg(sys.argv[0] + " 10.0.0.1 1/5 B200M3.blob.zip ")
    printBarSingle()
    printDbg("Update BIOS on 10.0.0.1 1/5 blade with BIOS B200M3.blob.zip from TFTP server 10.0.0.2 forcing to activate.")
    printDbg(sys.argv[0] + " 10.0.0.1 1/5 B200M3.blob.zip 10.0.0.2 force-activate")
    printBarSingle()
    printDbg("Update BIOS on 10.0.0.1 1/5 blade with BIOS B200M3.blob.zip from TFTP server 10.0.0.2 forcing to not activate.")
    printDbg(sys.argv[0] + " 10.0.0.1 1/5 B200M3.blob.zip 10.0.0.2 force-noactivate")
    printBarDouble()

