# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['ptera']

package_data = \
{'': ['*']}

install_requires = \
['contextvars>=2.4,<3.0']

setup_kwargs = {
    'name': 'ptera',
    'version': '0.2.5',
    'description': 'Call graph addressing library.',
    'long_description': '\n# Ptera\n\n**Note**: This is super alpha. A lot of the features are implemented very inefficiently and the error reporting is not very good. That will be fixed in due time, and then this note will disappear into the mists of git history.\n\n\n## What is Ptera?\n\nPtera is a set of powerful tools to query or tweak the values of variables from a program.\n\n* **Keep your program clean**: Queries can be defined outside of your main function, so there is no need to pollute your code with logging or debug code.\n* **Debug and analyze across scopes**: Easily write queries that collect variables at various points in the call stack, or even across different calls. Then, you can analyze them all together.\n* **Tag variables and functions**: Categorize parts of your program to make more general queries.\n\n\n## Example 1\n\nTake the following function, which estimates whether a point `c` in the complex pane belongs to the Mandelbrot set or not (repeat this for a range of real/imag values to draw a pretty monochrome fractal).\n\n```python\nfrom ptera import tooled\n\nMAX_ITER = 100\n\n@tooled\ndef mandelbrot(real, imag):\n    c = real + imag * 1j\n    z = 0\n    for i in range(MAX_ITER):\n        z = z * z + c\n        if abs(z) > 2:\n            return False\n    return True\n```\n\nPtera allows you, among other things, to look at the values taken by the variable `z` through the course of the function. This can be very interesting!\n\n```python\nmandelbrot_zs = mandelbrot.using(zs="mandelbrot > z")\n\nprint(mandelbrot_zs(0.25, 0).zs.map("z"))\n# -> Converges to 0.5\n\nprint(mandelbrot_zs(-1, 0).zs.map("z"))\n# -> Oscillates between 0 and -1\n\nprint(mandelbrot_zs(-0.125, 0.75).zs.map("z"))\n# -> 3-way oscillation between 0.01-0.006j, -0.125+0.75j and -0.67+0.56j\n```\n\n(Depending on which bulb of the fractal you are looking at, the iteration will converge on a cycle with a finite period, which can be arbitrarily large, which you can sort of visualize [like this](https://en.wikipedia.org/wiki/Mandelbrot_set#/media/File:Logistic_Map_Bifurcations_Underneath_Mandelbrot_Set.gif). But enough about that.)\n\nAll the values are collected together and can be mapped in any way you\'d like. You can also capture multiple variables together, for instance:\n\n```python\nmandelbrot_zis = mandelbrot.using(zis="mandelbrot(i) > z")\n\nprint(mandelbrot_zis(0.25, 0).zis.map(lambda z, i=None: (i, z)))\n# [(None, 0), (0, 0.25), (1, 0.3125), ..., (999, 0.499)]\n```\n\n**Tweaking variables**\n\nYou can also "tweak" variables. For example, you can change the value of `MAX_ITER` within the call:\n\n```python\nmandelbrot_short = mandelbrot.tweaking({"MAX_ITER": 10})\nmandelbrot_long = mandelbrot.tweaking({"MAX_ITER": 1000})\n\nprint(mandelbrot_short(0.2501, 0))  # True\nprint(mandelbrot_long(0.2501, 0))   # False\n```\n\nAs you can see, both versions of the function use their own version of `MAX_ITER`, without interference.\n\nNote however that Ptera will add significant overhead to a function like `mandelbrot` because all of its operations are cheap compared to the cost of tracking everything with Ptera. Only the body of functions decorated with `@tooled` will be slower, however, and if the bulk of the time of the program is spent in non-decorated functions, the overhead should be acceptable.\n\n\n## Example 2\n\nSuppose you have a simple PyTorch model with a few layers and a `tanh` activation function:\n\n```python\nclass MLP(torch.nn.Module):\n    def __init__(self):\n        super(MLP, self).__init__()\n        self.linear1 = torch.nn.Linear(784, 250)\n        self.linear2 = torch.nn.Linear(250, 100)\n        self.linear3 = torch.nn.Linear(100, 10)\n\n    def forward(self, inputs):\n        h1 = torch.tanh(self.linear1(inputs))\n        h2 = torch.tanh(self.linear2(h1))\n        h3 = self.linear3(h2)\n        return torch.log_softmax(h3, dim=1)\n\ndef step(model, optimizer, inputs, targets):\n    optimizer.zero_grad()\n    output = model(Variable(inputs).float())\n    loss = torch.nn.CrossEntropyLoss()(output, Variable(targets))\n    loss.backward()\n    optimizer.step()\n\ndef fit(model, data, epochs):\n    optimizer = torch.optim.SGD(model.parameters(), lr=0.1)\n    for epoch in range(epochs):\n        for batch_idx, (inputs, targets) in enumerate(data):\n            step(model, optimizer, inputs, targets)\n\nif __name__ == "__main__":\n    ...\n    fit(model, data, epochs)\n```\n\nYou want to know whether the activations on the layer `h2` tend to saturate, meaning that they are very close to -1 or 1. Therefore, you would like to log the percentage of the values in the `h2` matrix that have an absolute value greater than 0.99. You would only like to check every 100 iterations or so, though.\n\nHere is how to do this with ptera. Comments indicate all the changes you need to make:\n\n```python\nfrom ptera import tooled, Overlay\nfrom ptera.tools import every\n\nclass MLP(torch.nn.Module):\n    def __init__(self):\n        super(MLP, self).__init__()\n        self.linear1 = torch.nn.Linear(784, 250)\n        self.linear2 = torch.nn.Linear(250, 100)\n        self.linear3 = torch.nn.Linear(100, 10)\n\n    # Decorate this function\n    @tooled\n    def forward(self, inputs):\n        h1 = torch.tanh(self.linear1(inputs))\n        h2 = torch.tanh(self.linear2(h1))\n        h3 = self.linear3(h2)\n        return torch.log_softmax(h3, dim=1)\n\n# Decorate this function too\n@tooled\ndef step(model, optimizer, inputs, targets):\n    optimizer.zero_grad()\n    output = model(Variable(inputs).float())\n    loss = torch.nn.CrossEntropyLoss()(output, Variable(targets))\n    loss.backward()\n    optimizer.step()\n\ndef fit(model, data, epochs):\n    optimizer = torch.optim.SGD(model.parameters(), lr=0.1)\n    for epoch in range(epochs):\n        for batch_idx, (inputs, targets) in enumerate(data):\n            step(model, optimizer, inputs, targets)\n\nif __name__ == "__main__":\n    ...\n\n    # Create an overlay\n    overlay = Overlay()\n\n    # Set up a listener for the variable of MLP.forward named h2, but only\n    # within a call to step where the batch_idx variable is a multiple of 100.\n    # * The notation `x ~ f` means that we only trigger when f(x) == True.\n    # * The >> operator means arbitrary nesting.\n    # * The > operator means direct nesting\n    @overlay.on("step(batch_idx ~ every(100)) >> MLP.forward > h2")\n    def check_saturation(batch_idx, h2):\n        sat = float((h2.abs() > 0.99).float().mean())\n        print(sat)\n        return sat\n\n    # Call fit within a with block\n    with overlay as results:\n        fit(model, data, epochs)\n\n    # results.check_saturation contains a list of all the return values of the\n    # listener\n    print(results.check_saturation)\n```\n\nThe interface is a bit different from the Mandelbrot example, because in this example the function we are calling, `fit`, is not decorated with `@tooled`. This is fine, however, because the overlay (which also has methods like `use`, `tweak`, etc.) will apply to everything that\'s going on inside the `with` block, and the data accumulated by the various listeners and plugins will be put in the `results` data structure.\n\n\n## Overlays\n\nAn *overlay* is a collection of plugins that operate over the variables of ptera-decorated functions. It is used roughly like this:\n\n```python\noverlay = Overlay()\n\n# Use plugins. A plugin can be a simple query string, which by default\n# creates a Tap plugin.\noverlay.use(p1=plugin1, p2=plugin2, ...)\n\n# Call a function every time the query is triggered\n@overlay.on(query)\ndef p3(var1, var2, ...):\n    do_something(var1, var2, ...)\n\n# Change the value of a variable\noverlay.tweak({query: value, ...})\n\n# Change the value of a variable, but using a function which can depend on\n# other collected variables.\noverlay.rewrite({query: rewriter, ...})\n\n# Call the function while the overlay is active\nwith overlay as results:\n    func()\n\n# Do something with the results\ndo_something(results.p1)  # Set by plugin1\ndo_something(results.p2)  # Set by plugin2\ndo_something(results.p3)  # List of the return values of on(...)\n...\n\n# The "Tap" plugin, which is the default when giving a query string to use,\n# puts a Collector instance in its field in results. The main method you will\n# use is map:\nresults.p1.map()                    # List of dictionaries with all captures\nresults.p1.map("x")                 # List for variable "x"\nresults.p1.map("x", "y")            # List of tuples of values for x and y\nresults.p1.map(lambda x, y: x + y)  # Call the function on captured variables\n\n# map_all:\n# Each capture is a list of values. For example, the query `f(!x) > g(y)` will\n# trigger for every value of "x" because of the "!", but if g is called\n# multiple times, Ptera may list all of the values for "y". If that is the case,\n# `map` will error out and you must use `map_all`.\nresults.p1.map_all()\n\n# map_full:\n# Each capture is a Capture object. This can be useful with a query\n# such as `$v:SomeTag` which captures any variable annotated with SomeTag, but\n# regardless of the variable\'s actual name. The value will be provided under\n# the name "v", but the actual name will be in the capture.name field.\n# results.p1.map("x") <=> results.p1.map_full(lambda x: x.value)\n# results.p1.map_all("x") <=> results.p1.map_full(lambda x: x.values)\nresults.p1.map_full()\n```\n\n\n## Query language\n\nHere is some code annotated with queries that will match various variables. The queries are not exhaustive, just examples.\n\n* The semicolon ";" is used to separate queries and it is not part of any query.\n* The hash character "#" *is* part of the query if there is no space after it, otherwise it starts a comment.\n\n```python\nfrom ptera import ptera, tag\n\nAnimal = tag.Animal\nThing = tag.Thing\n\n@tooled\ndef art(a, b):               # art > a ; art > b ; art(!a, b) ; art(a, !b)\n\n    a1: Animal = bee(a)      # a1 ; art > a1 ; art(!a1) ; art > $x\n                             # a1:Animal ; $x:Animal\n                             # art(!a1) > bee > d  # Focus on a1, also includes d\n                             # art > bee  # This refers to the bee function\n                             # * > a1 ; *(!a1)\n\n    a2: Thing = cap(b)       # a2 ; art > a2 ; art(!a2) ; art > $x\n                             # a2:Thing ; $x:Thing\n\n    return a1 + a2           # art > #value ; art(#value as art_result)\n                             # art() as art_result\n                             # art > $x\n\n\n@tooled\ndef bee(c):\n    c1 = c + 1               # bee > c1 ; art >> c1 ; art(a2) > bee > c1\n                             # bee > c1 as xyz\n\n    return c1                # bee > #value ; bee(c) as bee_value\n\n\n@tooled\ndef cap(d: Thing & int):     # cap > d ; $x:Thing ; $x:int ; cap > $x\n                             # art(bee(c)) > cap > d\n    return d * d\n```\n\n* The `!` operator marks the **focus** of the query. There will be one result for each time the focus is triggered, and when using `tweak` or `rewrite` the focus is what is being tweaked or rewritten.\n  * Other variables are supplemental information, available along with the focus in query results. They can also be used to compute a value for the focus *if* they are available by the time the focus is reached.\n  * The nesting operators `>` and `>>` automatically set the focus to the right hand side if the rhs is a single variable and the operator is not inside `(...)`.\n* The wildcard `*` stands in for any function.\n* The `>>` operator represents **deep nesting**. For example, `art >> c1` encompasses the pattern `art > bee > c1`.\n  * In general, `a >> z` encompasses `a > z`, `a > b > z`, `a > b > c > z`, `a > * > z`, and so on.\n* A function\'s return value corresponds to a special variable named `#value`.\n* `$x` will match any variable name. Getting the variable name for the capture is possible but requires the `map_full` method. For example:\n  * Query: `art > $x`\n  * Getting the names: `results.map_full(lambda x: x.name) == ["a1", "a2", "#value"]`\n  * Other fields accessible from `map_full` are `value`, `names` and `values`, the latter two being needed if multiple results are captured together.\n* Variable annotations are preserved and can be filtered on, using the `:` operator. However, *Ptera only recognizes tags* created using `ptera.Tag("XYZ")` or `ptera.tag.XYZ`. It will not filter over types.\n* `art(bee(c)) > cap > d` triggers on the variable `d` in calls to `cap`, but it will *also* include the value of `c` for all calls to `bee` inside `art`.\n  * If there are multiple calls to `bee`, all values of `c` will be pooled together, and it will be necessary to use `map_all` to retrieve the values (or `map_full`).\n',
    'author': 'Olivier Breuleux',
    'author_email': 'breuleux@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/breuleux/ptera',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
