# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['sklearn_instrumentation', 'sklearn_instrumentation.instrumentation']

package_data = \
{'': ['*']}

install_requires = \
['numpy', 'scikit-learn']

setup_kwargs = {
    'name': 'sklearn-instrumentation',
    'version': '0.1.1',
    'description': 'scikit-learn instrumentation tooling',
    'long_description': 'sklearn-instrumentation\n=======================\n\n|actions| |rtd| |pypi| |pyversions|\n\n.. |actions| image:: https://github.com/crflynn/sklearn-instrumentation/workflows/build/badge.svg\n    :target: https://github.com/crflynn/sklearn-instrumentation/actions\n\n.. |rtd| image:: https://img.shields.io/readthedocs/sklearn-instrumentation.svg\n    :target: http://sklearn-instrumentation.readthedocs.io/en/latest/\n\n.. |pypi| image:: https://img.shields.io/pypi/v/sklearn-instrumentation.svg\n    :target: https://pypi.python.org/pypi/sklearn-instrumentation\n\n.. |pyversions| image:: https://img.shields.io/pypi/pyversions/sklearn-instrumentation.svg\n    :target: https://pypi.python.org/pypi/sklearn-instrumentation\n\n\nGeneralized instrumentation tooling for scikit-learn models. ``sklearn_instrumentation`` allows instrumenting the ``sklearn`` package and any scikit-learn compatible packages with estimators and transformers inheriting from ``sklearn.base.BaseEstimator``.\n\nInstrumentation applies decorators to methods of ``BaseEstimator``-derived classes or instances. By default the instrumentor applies instrumentation to the following methods (except when they are properties):\n\n* fit\n* predict\n* predict_proba\n* transform\n* _fit\n* _predict\n* _predict_proba\n* _transform\n\n**sklearn-instrumentation** supports instrumentation of full sklearn-compatible packages, as well as recursive instrumentation of models (metaestimators like ``Pipeline``, or even single estimators like ``RandomForestClassifier``)\n\nPackage instrumentation\n-----------------------\n\nInstrument any sklearn compatible package that has ``BaseEstimator``-derived classes.\n\n.. code-block:: python\n\n    from sklearn_instrumentation import SklearnInstrumentor\n\n    instrumentor = SklearnInstrumentor(decorator=my_instrumentation)\n    instrumentor.instrument_packages(["sklearn", "xgboost", "lightgbm"])\n\n\nFull example:\n\n.. code-block:: python\n\n    import logging\n\n    from sklearn.datasets import load_iris\n    from sklearn.decomposition import PCA\n    from sklearn.ensemble import RandomForestClassifier\n    from sklearn.pipeline import FeatureUnion\n    from sklearn.pipeline import Pipeline\n    from sklearn.preprocessing import StandardScaler\n\n    from sklearn_instrumentation import SklearnInstrumentor\n    from sklearn_instrumentation.instrumentation.logging import time_elapsed_logger\n\n    logging.basicConfig(level=logging.INFO)\n\n    # Create an instrumentor and instrument sklearn\n    instrumentor = SklearnInstrumentor(decorator=time_elapsed_logger)\n    instrumentor.instrument_packages(["sklearn"])\n\n    # Create a toy model for classification\n    ss = StandardScaler()\n    pca = PCA(n_components=3)\n    rf = RandomForestClassifier()\n    classification_model = Pipeline(\n        steps=[\n            (\n                "fu",\n                FeatureUnion(\n                    transformer_list=[\n                        ("ss", ss),\n                        ("pca", pca),\n                    ]\n                ),\n            ),\n            ("rf", rf),\n        ]\n    )\n    X, y = load_iris(return_X_y=True)\n\n    # Observe logging\n    classification_model.fit(X, y)\n    # INFO:root:Pipeline.fit starting.\n    # INFO:root:Pipeline._fit starting.\n    # INFO:root:StandardScaler.fit starting.\n    # INFO:root:StandardScaler.fit elapsed time: 0.0006406307220458984 seconds\n    # INFO:root:StandardScaler.transform starting.\n    # INFO:root:StandardScaler.transform elapsed time: 0.0001430511474609375 seconds\n    # INFO:root:PCA._fit starting.\n    # INFO:root:PCA._fit elapsed time: 0.0006711483001708984 seconds\n    # INFO:root:Pipeline._fit elapsed time: 0.0026731491088867188 seconds\n    # INFO:root:BaseForest.fit starting.\n    # INFO:root:BaseForest.fit elapsed time: 0.1768970489501953 seconds\n    # INFO:root:Pipeline.fit elapsed time: 0.17983102798461914 seconds\n\n    # Observe logging\n    classification_model.predict(X)\n    # INFO:root:Pipeline.predict starting.\n    # INFO:root:FeatureUnion.transform starting.\n    # INFO:root:StandardScaler.transform starting.\n    # INFO:root:StandardScaler.transform elapsed time: 0.00024509429931640625 seconds\n    # INFO:root:_BasePCA.transform starting.\n    # INFO:root:_BasePCA.transform elapsed time: 0.0002181529998779297 seconds\n    # INFO:root:FeatureUnion.transform elapsed time: 0.0012080669403076172 seconds\n    # INFO:root:ForestClassifier.predict starting.\n    # INFO:root:ForestClassifier.predict_proba starting.\n    # INFO:root:ForestClassifier.predict_proba elapsed time: 0.013531208038330078 seconds\n    # INFO:root:ForestClassifier.predict elapsed time: 0.013692140579223633 seconds\n    # INFO:root:Pipeline.predict elapsed time: 0.015219926834106445 seconds\n\n    # Remove instrumentation\n    instrumentor.uninstrument_packages(["sklearn"])\n\n    # Observe no logging\n    classification_model.predict(X)\n\n\nMachine learning model instrumentation\n--------------------------------------\n\nInstrument any sklearn compatible trained estimator or metaestimator.\n\n.. code-block:: python\n\n    from sklearn_instrumentation import SklearnInstrumentor\n\n    instrumentor = SklearnInstrumentor(decorator=my_instrumentation)\n    instrumentor.instrument_estimator(estimator=my_ml_pipeline)\n\n\n.. code-block:: python\n\n    import logging\n\n    from sklearn.datasets import load_iris\n    from sklearn_instrumentation import SklearnInstrumentor\n    from sklearn_instrumentation.instrumentation.logging import time_elapsed_logger\n    from sklearn.ensemble import RandomForestClassifier\n\n    logging.basicConfig(level=logging.INFO)\n\n    # Train a classifier\n    X, y = load_iris(return_X_y=True)\n    rf = RandomForestClassifier()\n\n    rf.fit(X, y)\n\n    # Create an instrumentor which decorates BaseEstimator methods with\n    # logging output when entering and exiting methods, with time elapsed logged\n    # on exit.\n    instrumentor = SklearnInstrumentor(decorator=time_elapsed_logger)\n\n    # Apply the decorator to all BaseEstimators in each of these libraries\n    instrumentor.instrument_estimator(rf)\n\n    # Observe the logging output\n    rf.predict(X)\n    # INFO:root:ForestClassifier.predict starting.\n    # INFO:root:ForestClassifier.predict_proba starting.\n    # INFO:root:ForestClassifier.predict_proba elapsed time: 0.014165163040161133 seconds\n    # INFO:root:ForestClassifier.predict elapsed time: 0.014327764511108398 seconds\n\n    # Remove the decorator from all BaseEstimators in each of these libraries\n    instrumentor.uninstrument_estimator(rf)\n\n    # No more logging\n    rf.predict(X)\n\n\nInstrumentation\n---------------\n\nThe package comes with a handful of decorators which log information about ``X`` or timing of execution. You can create your own instrumentation just by creating a decorator, following this pattern\n\n.. code-block:: python\n\n    from functools import wraps\n\n\n    def my_instrumentation(func, **dkwargs):\n        """Wrap an estimator method with instrumentation.\n\n        :param func: The method to be instrumented.\n        :param dkwargs: Decorator kwargs, which can be passed to the\n            decorator at decoration time. For estimator instrumentation\n            this allows different parametrizations for each ml model.\n        """\n        @wraps(func)\n        def wrapper(*args, **kwargs):\n            """Wrapping function.\n\n            :param args: The args passed to methods, typically\n                just ``X`` and/or ``y``\n            :param kwargs: The kwargs passed to methods, usually\n                weights or other params\n            """\n            # Code goes here before execution of the estimator method\n            retval = func(*args, **kwargs)\n            # Code goes here after execution of the estimator method\n            return retval\n\n        return wrapper\n\n\nTo pass kwargs for different ml models:\n\n.. code-block:: python\n\n    instrumentor = SklearnInstrumentor(decorator=my_instrumentation)\n\n    instrumentor.instrument_estimator(estimator=ml_model_1, decorator_kwargs={"name": "awesome_model"})\n    instrumentor.instrument_estimator(estimator=ml_model_2, decorator_kwargs={"name": "better_model"})\n\n',
    'author': 'flynn',
    'author_email': 'crf204@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/crflynn/sklearn-instrumentation',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6.1,<4.0.0',
}


setup(**setup_kwargs)
