"""
Tests for Replicator Dynamics
"""
import numpy as np
import pytest
from hypothesis import given, settings
from hypothesis.strategies import integers
from hypothesis.extra.numpy import arrays

from nashpy.learning.replicator_dynamics import (
    get_derivative_of_fitness,
    replicator_dynamics,
    get_derivative_of_asymmetric_fitness,
    asymmetric_replicator_dynamics,
)


@given(M=arrays(np.int8, (3, 3)))
def test_property_get_derivative_of_fitness(M):
    t = 0
    x = np.zeros(M.shape[1])
    derivative_of_fitness = get_derivative_of_fitness(x, t, M)

    assert len(derivative_of_fitness) == len(x)


def test_get_derivative_of_fitness():
    M = np.array([[3, 2, 3], [4, 1, 1], [2, 3, 1]])

    x_values = (
        np.array([1, 0, 0]),
        np.array([1 / 2, 1 / 2, 0]),
        np.array([0, 1 / 4, 3 / 4]),
        np.array([1 / 5, 2 / 5, 2 / 5]),
        np.array([1 / 2, 0, 1 / 2]),
        np.array([2 / 4, 1 / 4, 1 / 4]),
    )
    derivative_values = (
        np.array([0, 0, 0]),
        np.array([0, 0, 0]),
        np.array([0.0, -0.09375, 0.09375]),
        np.array([0.128, -0.144, 0.016]),
        np.array([0.375, 0.0, -0.375]),
        np.array([0.125, 0.0, -0.125]),
    )

    for x_value, expected_derivative in zip(x_values, derivative_values):
        derivative = get_derivative_of_fitness(x=x_value, t=0, A=M)
        assert np.allclose(derivative, expected_derivative), x_value


@given(M=arrays(np.int8, (3, 3)))
def test_property_of_output_dimension_for_games_of_size_3(M):
    xs = replicator_dynamics(M)
    assert all(len(x) == 3 for x in xs)


@given(M=arrays(np.int8, (4, 4)))
def test_property_of_output_dimension_for_games_of_size_4(M):
    xs = replicator_dynamics(M)
    assert all(len(x) == 4 for x in xs)


def test_replicator_dynamics_example_1():
    M = np.array([[3, 2], [4, 1]])
    y0 = np.array([0.9, 0.1])
    timepoints = np.linspace(0, 10, 100)

    expected_xs_over_time = np.array(
        [
            [0.9, 0.1],
            [0.89256013, 0.10743987],
            [0.88479436, 0.11520564],
            [0.87671801, 0.12328199],
            [0.86834987, 0.13165013],
            [0.8597121, 0.1402879],
            [0.8508299, 0.1491701],
            [0.8417312, 0.1582688],
            [0.83244622, 0.16755378],
            [0.82300701, 0.17699299],
            [0.81344687, 0.18655313],
            [0.80379989, 0.19620011],
            [0.79410031, 0.20589969],
            [0.78438204, 0.21561796],
            [0.77467816, 0.22532184],
            [0.76502043, 0.23497957],
            [0.75543894, 0.24456106],
            [0.74596174, 0.25403826],
            [0.73661466, 0.26338534],
            [0.72742107, 0.27257893],
            [0.7184018, 0.2815982],
            [0.70957507, 0.29042493],
            [0.70095654, 0.29904346],
            [0.69255932, 0.30744068],
            [0.6843941, 0.3156059],
            [0.67646927, 0.32353073],
            [0.66879107, 0.33120893],
            [0.66136378, 0.33863622],
            [0.65418987, 0.34581013],
            [0.64727021, 0.35272979],
            [0.64060422, 0.35939578],
            [0.63419006, 0.36580994],
            [0.62802483, 0.37197517],
            [0.62210466, 0.37789534],
            [0.61642492, 0.38357508],
            [0.61098033, 0.38901967],
            [0.60576506, 0.39423494],
            [0.60077288, 0.39922712],
            [0.59599723, 0.40400277],
            [0.59143133, 0.40856867],
            [0.58706824, 0.41293176],
            [0.5829009, 0.4170991],
            [0.57892225, 0.42107775],
            [0.57512523, 0.42487477],
            [0.57150283, 0.42849717],
            [0.56804814, 0.43195186],
            [0.56475435, 0.43524565],
            [0.56161481, 0.43838519],
            [0.558623, 0.441377],
            [0.55577261, 0.44422739],
            [0.55305749, 0.44694251],
            [0.55047167, 0.44952833],
            [0.5480094, 0.4519906],
            [0.54566512, 0.45433488],
            [0.54343348, 0.45656652],
            [0.54130932, 0.45869068],
            [0.53928768, 0.46071232],
            [0.53736381, 0.46263619],
            [0.53553315, 0.46446685],
            [0.53379131, 0.46620869],
            [0.53213411, 0.46786589],
            [0.53055754, 0.46944246],
            [0.52905777, 0.47094223],
            [0.52763113, 0.47236887],
            [0.52627413, 0.47372587],
            [0.52498342, 0.47501658],
            [0.52375581, 0.47624419],
            [0.52258827, 0.47741173],
            [0.52147788, 0.47852212],
            [0.52042188, 0.47957812],
            [0.51941764, 0.48058236],
            [0.51846265, 0.48153735],
            [0.51755449, 0.48244551],
            [0.51669091, 0.48330909],
            [0.51586971, 0.48413029],
            [0.51508885, 0.48491115],
            [0.51434634, 0.48565366],
            [0.51364031, 0.48635969],
            [0.51296897, 0.48703103],
            [0.51233064, 0.48766936],
            [0.51172369, 0.48827631],
            [0.51114658, 0.48885342],
            [0.51059785, 0.48940215],
            [0.51007612, 0.48992388],
            [0.50958005, 0.49041995],
            [0.50910838, 0.49089162],
            [0.50865992, 0.49134008],
            [0.50823353, 0.49176647],
            [0.50782813, 0.49217187],
            [0.50744267, 0.49255733],
            [0.50707619, 0.49292381],
            [0.50672775, 0.49327225],
            [0.50639645, 0.49360355],
            [0.50608147, 0.49391853],
            [0.50578199, 0.49421801],
            [0.50549726, 0.49450274],
            [0.50522655, 0.49477345],
            [0.50496916, 0.49503084],
            [0.50472445, 0.49527555],
            [0.50449178, 0.49550822],
        ],
    )

    xs = replicator_dynamics(y0=y0, timepoints=timepoints, A=M)
    assert np.allclose(xs, expected_xs_over_time)


def test_replicator_dynamics_example_2():
    M = np.array([[3, 2], [4, 1]])
    y0 = np.array([0.65, 0.35])
    timepoints = np.linspace(0, 10, 100)

    expected_xs_over_time = np.array(
        [
            [0.65, 0.35],
            [0.64323298, 0.35676702],
            [0.63671861, 0.36328139],
            [0.63045449, 0.36954551],
            [0.62443708, 0.37556292],
            [0.61866205, 0.38133795],
            [0.61312433, 0.38687567],
            [0.60781832, 0.39218168],
            [0.60273792, 0.39726208],
            [0.5978767, 0.4021233],
            [0.59322796, 0.40677204],
            [0.5887848, 0.4112152],
            [0.58454021, 0.41545979],
            [0.58048714, 0.41951286],
            [0.57661851, 0.42338149],
            [0.57292729, 0.42707271],
            [0.56940652, 0.43059348],
            [0.56604936, 0.43395064],
            [0.56284908, 0.43715092],
            [0.55979911, 0.44020089],
            [0.55689305, 0.44310695],
            [0.5541247, 0.4458753],
            [0.551488, 0.448512],
            [0.54897713, 0.45102287],
            [0.54658644, 0.45341356],
            [0.5443105, 0.4556895],
            [0.54214408, 0.45785592],
            [0.54008213, 0.45991787],
            [0.53811982, 0.46188018],
            [0.53625251, 0.46374749],
            [0.53447575, 0.46552425],
            [0.53278527, 0.46721473],
            [0.53117701, 0.46882299],
            [0.52964705, 0.47035295],
            [0.52819167, 0.47180833],
            [0.5268073, 0.4731927],
            [0.52549054, 0.47450946],
            [0.52423814, 0.47576186],
            [0.52304699, 0.47695301],
            [0.52191414, 0.47808586],
            [0.52083678, 0.47916322],
            [0.5198122, 0.4801878],
            [0.51883785, 0.48116215],
            [0.51791129, 0.48208871],
            [0.5170302, 0.4829698],
            [0.51619235, 0.48380765],
            [0.51539564, 0.48460436],
            [0.51463806, 0.48536194],
            [0.51391769, 0.48608231],
            [0.51323273, 0.48676727],
            [0.51258143, 0.48741857],
            [0.51196214, 0.48803786],
            [0.51137331, 0.48862669],
            [0.51081343, 0.48918657],
            [0.51028109, 0.48971891],
            [0.50977494, 0.49022506],
            [0.50929368, 0.49070632],
            [0.50883611, 0.49116389],
            [0.50840105, 0.49159895],
            [0.5079874, 0.4920126],
            [0.5075941, 0.4924059],
            [0.50722017, 0.49277983],
            [0.50686464, 0.49313536],
            [0.5065266, 0.4934734],
            [0.50620521, 0.49379479],
            [0.50589964, 0.49410036],
            [0.50560912, 0.49439088],
            [0.5053329, 0.4946671],
            [0.50507027, 0.49492973],
            [0.50482058, 0.49517942],
            [0.50458318, 0.49541682],
            [0.50435748, 0.49564252],
            [0.50414288, 0.49585712],
            [0.50393885, 0.49606115],
            [0.50374487, 0.49625513],
            [0.50356044, 0.49643956],
            [0.5033851, 0.4966149],
            [0.50321838, 0.49678162],
            [0.50305988, 0.49694012],
            [0.50290918, 0.49709082],
            [0.50276591, 0.49723409],
            [0.50262969, 0.49737031],
            [0.50250018, 0.49749982],
            [0.50237704, 0.49762296],
            [0.50225997, 0.49774003],
            [0.50214867, 0.49785133],
            [0.50204285, 0.49795715],
            [0.50194224, 0.49805776],
            [0.50184658, 0.49815342],
            [0.50175564, 0.49824436],
            [0.50166917, 0.49833083],
            [0.50158696, 0.49841304],
            [0.50150881, 0.49849119],
            [0.5014345, 0.4985655],
            [0.50136385, 0.49863615],
            [0.50129668, 0.49870332],
            [0.50123281, 0.49876719],
            [0.5011721, 0.4988279],
            [0.50111437, 0.49888563],
            [0.50105949, 0.49894051],
        ]
    )

    xs = replicator_dynamics(y0=y0, timepoints=timepoints, A=M)
    assert np.allclose(xs, expected_xs_over_time)


def test_replicator_dynamics_example_3_default_y0():
    M = np.array([[8, 2], [5, 3]])
    timepoints = np.linspace(0, 10, 100)

    expected_xs_over_time = np.array(
        [
            [5.00000000e-01, 5.00000000e-01],
            [5.26546322e-01, 4.73453678e-01],
            [5.55724197e-01, 4.44275803e-01],
            [5.87511410e-01, 4.12488590e-01],
            [6.21728489e-01, 3.78271511e-01],
            [6.57990146e-01, 3.42009854e-01],
            [6.95672143e-01, 3.04327857e-01],
            [7.33912742e-01, 2.66087258e-01],
            [7.71667086e-01, 2.28332914e-01],
            [8.07818764e-01, 1.92181236e-01],
            [8.41329566e-01, 1.58670434e-01],
            [8.71386505e-01, 1.28613495e-01],
            [8.97500124e-01, 1.02499876e-01],
            [9.19528237e-01, 8.04717629e-02],
            [9.37629038e-01, 6.23709623e-02],
            [9.52172012e-01, 4.78279882e-02],
            [9.63640143e-01, 3.63598573e-02],
            [9.72547714e-01, 2.74522863e-02],
            [9.79383883e-01, 2.06161168e-02],
            [9.84581519e-01, 1.54184812e-02],
            [9.88504989e-01, 1.14950108e-02],
            [9.91450448e-01, 8.54955249e-03],
            [9.93652534e-01, 6.34746643e-03],
            [9.95293742e-01, 4.70625842e-03],
            [9.96514084e-01, 3.48591598e-03],
            [9.97419907e-01, 2.58009292e-03],
            [9.98091404e-01, 1.90859574e-03],
            [9.98588714e-01, 1.41128551e-03],
            [9.98956761e-01, 1.04323949e-03],
            [9.99228997e-01, 7.71002583e-04],
            [9.99430288e-01, 5.69712057e-04],
            [9.99579078e-01, 4.20922069e-04],
            [9.99689037e-01, 3.10962897e-04],
            [9.99770287e-01, 2.29713463e-04],
            [9.99830315e-01, 1.69684731e-04],
            [9.99874662e-01, 1.25338089e-04],
            [9.99907422e-01, 9.25784618e-05],
            [9.99931619e-01, 6.83805402e-05],
            [9.99949493e-01, 5.05073861e-05],
            [9.99962695e-01, 3.73049637e-05],
            [9.99972447e-01, 2.75532065e-05],
            [9.99979649e-01, 2.03505730e-05],
            [9.99984969e-01, 1.50308175e-05],
            [9.99988898e-01, 1.11015649e-05],
            [9.99991801e-01, 8.19919968e-06],
            [9.99993944e-01, 6.05565505e-06],
            [9.99995527e-01, 4.47259259e-06],
            [9.99996697e-01, 3.30326369e-06],
            [9.99997560e-01, 2.43964910e-06],
            [9.99998198e-01, 1.80194467e-06],
            [9.99998669e-01, 1.33084681e-06],
            [9.99999018e-01, 9.82166934e-07],
            [9.99999275e-01, 7.25194883e-07],
            [9.99999464e-01, 5.35772717e-07],
            [9.99999604e-01, 3.96066315e-07],
            [9.99999707e-01, 2.92673377e-07],
            [9.99999784e-01, 2.16251201e-07],
            [9.99999840e-01, 1.59970163e-07],
            [9.99999882e-01, 1.18489534e-07],
            [9.99999912e-01, 8.79198227e-08],
            [9.99999935e-01, 6.47958220e-08],
            [9.99999952e-01, 4.76360370e-08],
            [9.99999965e-01, 3.46731961e-08],
            [9.99999974e-01, 2.55665790e-08],
            [9.99999981e-01, 1.86146820e-08],
            [9.99999986e-01, 1.36425118e-08],
            [9.99999990e-01, 1.00816167e-08],
            [9.99999992e-01, 7.55059409e-09],
            [9.99999994e-01, 5.67732943e-09],
            [9.99999996e-01, 4.28158816e-09],
            [9.99999997e-01, 3.20976917e-09],
            [9.99999998e-01, 2.40345206e-09],
            [9.99999998e-01, 1.78669125e-09],
            [9.99999999e-01, 1.33286584e-09],
            [9.99999999e-01, 9.89591714e-10],
            [9.99999999e-01, 7.40089560e-10],
            [9.99999999e-01, 5.51209294e-10],
            [1.00000000e00, 3.28925518e-10],
            [1.00000000e00, 1.11214696e-10],
            [1.00000000e00, -7.15664780e-11],
            [1.00000000e00, -2.19418003e-10],
            [1.00000000e00, -3.32339878e-10],
            [1.00000000e00, -4.10332104e-10],
            [1.00000000e00, -4.53394682e-10],
            [1.00000000e00, -4.61527610e-10],
            [9.99999999e-01, -4.34730889e-10],
            [9.99999999e-01, -3.73004520e-10],
            [9.99999999e-01, -3.39039249e-10],
            [9.99999999e-01, -3.64704692e-10],
            [9.99999999e-01, -3.81253172e-10],
            [9.99999999e-01, -3.88684691e-10],
            [9.99999999e-01, -3.86999249e-10],
            [9.99999999e-01, -3.76196845e-10],
            [9.99999999e-01, -3.56277479e-10],
            [9.99999999e-01, -3.27241152e-10],
            [9.99999999e-01, -2.89087864e-10],
            [9.99999999e-01, -2.41817614e-10],
            [9.99999998e-01, -2.02072563e-10],
            [9.99999998e-01, -1.86998011e-10],
            [9.99999998e-01, -1.71923460e-10],
        ]
    )

    xs = replicator_dynamics(timepoints=timepoints, A=M)
    assert np.allclose(xs, expected_xs_over_time)


def test_replicator_dynamics_game_size_3_example_1():
    M = np.array([[3, 2, 3], [4, 1, 1], [2, 3, 1]])
    y0 = np.array([0.2, 0.1, 0.7])
    timepoints = np.linspace(0, 20, 100)

    expected_xs_over_time = np.array(
        [
            [0.2, 0.1, 0.7],
            [0.25084045, 0.09789735, 0.6512622],
            [0.30680235, 0.096512, 0.59668565],
            [0.36555987, 0.09596192, 0.53847822],
            [0.4244435, 0.09636333, 0.47919317],
            [0.48091315, 0.097819, 0.42126785],
            [0.53292835, 0.10041113, 0.36666052],
            [0.57912, 0.10419922, 0.31668078],
            [0.61877662, 0.10922098, 0.2720024],
            [0.6517203, 0.11549383, 0.23278586],
            [0.67814881, 0.12301537, 0.19883582],
            [0.69849303, 0.13176241, 0.16974455],
            [0.71330936, 0.14168906, 0.14500158],
            [0.72320749, 0.15272473, 0.12406778],
            [0.72880646, 0.16477305, 0.10642048],
            [0.73070961, 0.17771243, 0.09157796],
            [0.72949106, 0.19139871, 0.07911023],
            [0.72568835, 0.2056699, 0.06864175],
            [0.719798, 0.22035258, 0.05984942],
            [0.71227239, 0.23526931, 0.0524583],
            [0.70351757, 0.25024617, 0.04623626],
            [0.69389181, 0.26511966, 0.04098853],
            [0.68370534, 0.27974226, 0.03655241],
            [0.67322108, 0.29398645, 0.03279247],
            [0.66265685, 0.30774686, 0.02959629],
            [0.65218849, 0.32094081, 0.0268707],
            [0.64195385, 0.33350758, 0.02453857],
            [0.63205732, 0.34540661, 0.02253606],
            [0.62257449, 0.35661514, 0.02081038],
            [0.61355672, 0.36712552, 0.01931776],
            [0.60503556, 0.37694249, 0.01802195],
            [0.59702658, 0.38608059, 0.01689283],
            [0.58953289, 0.39456178, 0.01590533],
            [0.58254803, 0.40241341, 0.01503856],
            [0.57605847, 0.40966651, 0.01427502],
            [0.5700456, 0.41635432, 0.01360008],
            [0.5644874, 0.42251121, 0.01300139],
            [0.55935967, 0.42817177, 0.01246856],
            [0.55463707, 0.43337015, 0.01199279],
            [0.55029384, 0.43813956, 0.0115666],
            [0.54630443, 0.44251196, 0.01118362],
            [0.54264384, 0.44651774, 0.01083841],
            [0.53928799, 0.4501857, 0.01052632],
            [0.53621383, 0.45354285, 0.01024332],
            [0.53339955, 0.45661449, 0.00998596],
            [0.5308246, 0.45942417, 0.00975123],
            [0.52846971, 0.46199374, 0.00953656],
            [0.5263169, 0.46434344, 0.00933966],
            [0.52434948, 0.46649195, 0.00915857],
            [0.52255195, 0.46845648, 0.00899157],
            [0.52091, 0.47025287, 0.00883714],
            [0.51941039, 0.47189566, 0.00869395],
            [0.51804096, 0.47339821, 0.00856083],
            [0.51679051, 0.47477274, 0.00843675],
            [0.51564876, 0.47603045, 0.0083208],
            [0.51460626, 0.47718159, 0.00821216],
            [0.51365436, 0.47823552, 0.00811011],
            [0.51278515, 0.47920082, 0.00801403],
            [0.51199138, 0.48008529, 0.00792334],
            [0.5112664, 0.48089607, 0.00783753],
            [0.51060417, 0.48163968, 0.00775615],
            [0.50999914, 0.48232207, 0.0076788],
            [0.50944625, 0.48294864, 0.00760511],
            [0.50894089, 0.48352435, 0.00753476],
            [0.50847885, 0.48405369, 0.00746746],
            [0.50805628, 0.48454077, 0.00740295],
            [0.50766968, 0.48498934, 0.00734099],
            [0.50731584, 0.48540279, 0.00728137],
            [0.50699186, 0.48578424, 0.0072239],
            [0.50669508, 0.4861365, 0.00716841],
            [0.50642309, 0.48646217, 0.00711475],
            [0.50617366, 0.48676357, 0.00706277],
            [0.50594481, 0.48704285, 0.00701234],
            [0.50573469, 0.48730195, 0.00696337],
            [0.50554164, 0.48754263, 0.00691573],
            [0.50536414, 0.48776652, 0.00686933],
            [0.50520082, 0.48797508, 0.0068241],
            [0.5050504, 0.48816965, 0.00677995],
            [0.50491176, 0.48835143, 0.00673681],
            [0.50478384, 0.48852154, 0.00669463],
            [0.50466569, 0.48868099, 0.00665333],
            [0.50455645, 0.48883068, 0.00661287],
            [0.50445532, 0.48897146, 0.00657321],
            [0.50436161, 0.48910409, 0.00653429],
            [0.50427466, 0.48922926, 0.00649607],
            [0.50419387, 0.4893476, 0.00645853],
            [0.5041187, 0.48945968, 0.00642162],
            [0.50404866, 0.48956602, 0.00638531],
            [0.50398331, 0.48966711, 0.00634958],
            [0.50392223, 0.48976337, 0.0063144],
            [0.50386505, 0.48985519, 0.00627975],
            [0.50381145, 0.48994295, 0.0062456],
            [0.50376111, 0.49002695, 0.00621194],
            [0.50371375, 0.49010751, 0.00617874],
            [0.50366912, 0.49018489, 0.00614599],
            [0.50362698, 0.49025935, 0.00611367],
            [0.50358714, 0.4903311, 0.00608177],
            [0.50354938, 0.49040035, 0.00605027],
            [0.50351355, 0.49046729, 0.00601915],
            [0.50347948, 0.4905321, 0.00598842],
        ]
    )

    xs = replicator_dynamics(y0=y0, A=M, timepoints=timepoints)
    assert np.allclose(xs, expected_xs_over_time)


def test_replicator_dynamics_game_size_3_example_default_timepoints():
    M = np.array([[3, 2, 3], [4, 1, 1], [2, 3, 1]])
    y0 = np.array([0.2, 0.1, 0.7])

    expected_x_1 = np.array([[0.20237066, 0.09988063, 0.69774871]])
    expected_x_1000 = np.array([[0.52171238, 0.46937475, 0.00891287]])
    xs = replicator_dynamics(y0=y0, A=M)
    assert np.allclose(xs[1], expected_x_1)
    assert np.allclose(xs[-1], expected_x_1000)
    assert len(xs) == 1000


def test_replicator_dynamics_game_size_3_example_2():
    M = np.array([[3, 2, 3], [4, 1, 1], [2, 3, 1]])
    y0 = np.array([0.5, 0.1, 0.4])
    timepoints = np.linspace(0, 10, 100)

    expected_xs_over_time = np.array(
        [
            [0.5, 0.1, 0.4],
            [0.52559968, 0.10135984, 0.37304048],
            [0.5497745, 0.10301946, 0.34720604],
            [0.57240754, 0.10498432, 0.32260814],
            [0.59342086, 0.10725857, 0.29932057],
            [0.61277116, 0.10984506, 0.27738379],
            [0.63044466, 0.11274538, 0.25680996],
            [0.64645211, 0.11595993, 0.23758796],
            [0.66082391, 0.11948781, 0.21968828],
            [0.67360575, 0.12332687, 0.20306738],
            [0.68485481, 0.1274736, 0.18767159],
            [0.69463657, 0.13192312, 0.17344031],
            [0.70302219, 0.13666906, 0.16030874],
            [0.71008648, 0.14170354, 0.14820998],
            [0.71590619, 0.14701708, 0.13707673],
            [0.72055883, 0.15259863, 0.12684255],
            [0.7241217, 0.1584355, 0.1174428],
            [0.72667121, 0.16451346, 0.10881533],
            [0.72828236, 0.17081672, 0.10090091],
            [0.72902836, 0.17732812, 0.09364353],
            [0.72898035, 0.18402915, 0.08699051],
            [0.72820721, 0.19090019, 0.08089259],
            [0.72677541, 0.19792069, 0.0753039],
            [0.72474884, 0.20506931, 0.07018184],
            [0.72218875, 0.21232425, 0.06548701],
            [0.71915361, 0.21966337, 0.06118302],
            [0.71569909, 0.22706454, 0.05723638],
            [0.71187792, 0.23450578, 0.05361629],
            [0.70773992, 0.24196557, 0.0502945],
            [0.7033319, 0.24942298, 0.04724512],
            [0.69869765, 0.25685792, 0.04444443],
            [0.69387795, 0.26425127, 0.04187078],
            [0.68891058, 0.27158505, 0.03950437],
            [0.68383036, 0.27884249, 0.03732715],
            [0.67866917, 0.28600818, 0.03532264],
            [0.67345609, 0.29306806, 0.03347585],
            [0.66821741, 0.3000095, 0.03177308],
            [0.66297682, 0.30682127, 0.03020191],
            [0.65775544, 0.31349356, 0.02875099],
            [0.65257204, 0.32001794, 0.02741002],
            [0.64744309, 0.3263873, 0.02616961],
            [0.64238298, 0.3325958, 0.02502123],
            [0.63740411, 0.33863879, 0.0239571],
            [0.63251708, 0.34451275, 0.02297018],
            [0.62773079, 0.35021519, 0.02205402],
            [0.62305263, 0.35574459, 0.02120278],
            [0.61848858, 0.36110029, 0.02041112],
            [0.6140434, 0.36628239, 0.01967421],
            [0.60972067, 0.37129171, 0.01898762],
            [0.60552299, 0.37612969, 0.01834732],
            [0.60145203, 0.38079834, 0.01774963],
            [0.59750868, 0.3853001, 0.01719122],
            [0.59369315, 0.38963783, 0.01666902],
            [0.59000503, 0.39381472, 0.01618024],
            [0.58644338, 0.39783429, 0.01572233],
            [0.5830068, 0.40170025, 0.01529295],
            [0.57969351, 0.40541652, 0.01488997],
            [0.57650141, 0.40898716, 0.01451143],
            [0.57342812, 0.41241633, 0.01415554],
            [0.57047106, 0.41570828, 0.01382066],
            [0.56762744, 0.41886729, 0.01350527],
            [0.56489435, 0.42189765, 0.01320799],
            [0.56226876, 0.42480368, 0.01292755],
            [0.55974758, 0.42758964, 0.01266278],
            [0.55732765, 0.43025976, 0.01241259],
            [0.5550058, 0.43281821, 0.01217599],
            [0.55277882, 0.43526911, 0.01195207],
            [0.55064354, 0.43761648, 0.01173997],
            [0.54859681, 0.43986427, 0.01153892],
            [0.54663548, 0.44201632, 0.0113482],
            [0.54475648, 0.44407639, 0.01116712],
            [0.54295678, 0.44604813, 0.01099509],
            [0.54123342, 0.44793507, 0.01083152],
            [0.53958348, 0.44974065, 0.01067587],
            [0.53800412, 0.45146821, 0.01052767],
            [0.53649261, 0.45312096, 0.01038644],
            [0.53504623, 0.45470201, 0.01025176],
            [0.5336624, 0.45621436, 0.01012324],
            [0.53233858, 0.45766092, 0.0100005],
            [0.53107233, 0.45904447, 0.0098832],
            [0.52986128, 0.46036769, 0.00977103],
            [0.52870314, 0.46163318, 0.00966368],
            [0.52759571, 0.46284341, 0.00956088],
            [0.52653686, 0.46400078, 0.00946235],
            [0.52552454, 0.46510759, 0.00936787],
            [0.52455677, 0.46616604, 0.0092772],
            [0.52363164, 0.46717823, 0.00919013],
            [0.52274734, 0.46814621, 0.00910645],
            [0.5219021, 0.46907191, 0.00902599],
            [0.52109424, 0.4699572, 0.00894857],
            [0.52032212, 0.47080386, 0.00887402],
            [0.5195842, 0.47161362, 0.00880218],
            [0.51887898, 0.47238809, 0.00873292],
            [0.51820503, 0.47312887, 0.0086661],
            [0.51756097, 0.47383744, 0.00860159],
            [0.51694549, 0.47451525, 0.00853926],
            [0.51635732, 0.47516367, 0.00847902],
            [0.51579525, 0.47578401, 0.00842074],
            [0.51525813, 0.47637754, 0.00836433],
            [0.51474485, 0.47694545, 0.0083097],
        ]
    )

    xs = replicator_dynamics(y0=y0, timepoints=timepoints, A=M)
    assert np.allclose(xs, expected_xs_over_time)


def test_replicator_dynamics_game_size_3_example_1_default_y0():
    M = np.array([[3, 2, 3], [4, 1, 1], [2, 3, 1]])
    timepoints = np.linspace(0, 10, 100)

    expected_xs_over_time = np.array(
        [
            [0.33333333, 0.33333333, 0.33333333],
            [0.34828459, 0.3262229, 0.32549251],
            [0.36315697, 0.31983211, 0.31701092],
            [0.37787298, 0.31412581, 0.30800121],
            [0.3923581, 0.30907001, 0.29857189],
            [0.40654213, 0.30463189, 0.28882598],
            [0.42036029, 0.30077978, 0.27885992],
            [0.43375425, 0.29748314, 0.26876261],
            [0.44667261, 0.29471251, 0.25861488],
            [0.45907142, 0.29243945, 0.24848913],
            [0.47091426, 0.29063654, 0.23844919],
            [0.48217224, 0.2892773, 0.22855046],
            [0.49282377, 0.28833613, 0.2188401],
            [0.50285411, 0.28778834, 0.20935755],
            [0.51225504, 0.28761002, 0.20013494],
            [0.52102418, 0.28777809, 0.19119773],
            [0.52916446, 0.28827022, 0.18256531],
            [0.53668352, 0.28906482, 0.17425166],
            [0.54359302, 0.29014101, 0.16626597],
            [0.54990812, 0.2914786, 0.15861327],
            [0.55564688, 0.2930581, 0.15129501],
            [0.5608297, 0.29486067, 0.14430963],
            [0.56547887, 0.29686812, 0.13765301],
            [0.56961809, 0.29906291, 0.131319],
            [0.57327211, 0.30142816, 0.12529973],
            [0.57646634, 0.30394761, 0.11958605],
            [0.57922655, 0.30660565, 0.1141678],
            [0.58157859, 0.30938732, 0.1090341],
            [0.58354816, 0.31227828, 0.10417356],
            [0.58516064, 0.31526484, 0.09957451],
            [0.58644089, 0.31833398, 0.09522513],
            [0.58741312, 0.32147329, 0.09111359],
            [0.58810078, 0.32467101, 0.08722821],
            [0.58852649, 0.32791604, 0.08355748],
            [0.58871195, 0.33119789, 0.08009016],
            [0.58867791, 0.33450672, 0.07681537],
            [0.58844412, 0.33783332, 0.07372256],
            [0.58802931, 0.34116907, 0.07080162],
            [0.5874512, 0.34450598, 0.06804282],
            [0.58672648, 0.34783662, 0.0654369],
            [0.58587083, 0.35115417, 0.06297501],
            [0.58489893, 0.35445233, 0.06064874],
            [0.58382451, 0.35772538, 0.05845012],
            [0.58266033, 0.36096807, 0.05637159],
            [0.58141828, 0.3641757, 0.05440602],
            [0.58010932, 0.36734401, 0.05254667],
            [0.57874361, 0.37046921, 0.05078718],
            [0.57733051, 0.37354793, 0.04912156],
            [0.57587859, 0.37657722, 0.04754419],
            [0.57439572, 0.37955451, 0.04604977],
            [0.57288908, 0.38247759, 0.04463333],
            [0.57136522, 0.38534459, 0.04329019],
            [0.56983008, 0.38815395, 0.04201597],
            [0.56828904, 0.3909044, 0.04080656],
            [0.56674693, 0.39359496, 0.03965812],
            [0.56520812, 0.39622487, 0.03856701],
            [0.56367651, 0.39879364, 0.03752985],
            [0.56215558, 0.40130095, 0.03654348],
            [0.5606484, 0.40374669, 0.03560491],
            [0.5591577, 0.40613095, 0.03471135],
            [0.55768587, 0.40845392, 0.03386021],
            [0.55623498, 0.410716, 0.03304902],
            [0.55480684, 0.41291766, 0.0322755],
            [0.55340299, 0.41505951, 0.0315375],
            [0.55202472, 0.41714228, 0.030833],
            [0.55067314, 0.41916674, 0.03016012],
            [0.54934913, 0.42113379, 0.02951708],
            [0.5480534, 0.42304437, 0.02890223],
            [0.54678652, 0.42489947, 0.02831401],
            [0.5455489, 0.42670015, 0.02775096],
            [0.54434081, 0.42844749, 0.0272117],
            [0.54316241, 0.43014264, 0.02669495],
            [0.54201376, 0.43178673, 0.02619951],
            [0.54089483, 0.43338094, 0.02572423],
            [0.53980549, 0.43492646, 0.02526805],
            [0.53874555, 0.43642449, 0.02482996],
            [0.53771475, 0.43787623, 0.02440903],
            [0.53671276, 0.43928288, 0.02400436],
            [0.53573922, 0.44064566, 0.02361512],
            [0.53479372, 0.44196575, 0.02324053],
            [0.53387581, 0.44324434, 0.02287984],
            [0.53298501, 0.44448262, 0.02253237],
            [0.53212081, 0.44568175, 0.02219745],
            [0.53128267, 0.44684287, 0.02187446],
            [0.53047007, 0.44796711, 0.02156282],
            [0.52968241, 0.4490556, 0.02126199],
            [0.52891915, 0.45010942, 0.02097143],
            [0.52817969, 0.45112965, 0.02069066],
            [0.52746344, 0.45211733, 0.02041923],
            [0.52676982, 0.4530735, 0.02015668],
            [0.52609823, 0.45399916, 0.01990261],
            [0.52544809, 0.45489529, 0.01965662],
            [0.5248188, 0.45576285, 0.01941835],
            [0.52420978, 0.45660278, 0.01918744],
            [0.52362045, 0.45741598, 0.01896357],
            [0.52305024, 0.45820334, 0.01874642],
            [0.52249859, 0.45896572, 0.0185357],
            [0.52196493, 0.45970395, 0.01833112],
            [0.52144873, 0.46041886, 0.01813241],
            [0.52094945, 0.46111123, 0.01793933],
        ]
    )

    xs = replicator_dynamics(timepoints=timepoints, A=M)
    assert np.allclose(xs, expected_xs_over_time)


def test_replicator_dynamics_game_size_4_example_1():
    M = np.array([[3, 2, 4, 2], [5, 1, 1, 3], [6, 2, 3, 2], [1, 3, 4, 7]])
    y0 = np.array([0.2, 0.2, 0.5, 0.1])
    timepoints = np.linspace(0, 10, 100)

    expected_xs_over_time = np.array(
        [
            [2.00000000e-01, 2.00000000e-01, 5.00000000e-01, 1.00000000e-01],
            [2.03014607e-01, 1.79775683e-01, 5.12598077e-01, 1.04611633e-01],
            [2.05602634e-01, 1.61119562e-01, 5.24116145e-01, 1.09161659e-01],
            [2.07780154e-01, 1.44008247e-01, 5.34544791e-01, 1.13666808e-01],
            [2.09565963e-01, 1.28397079e-01, 5.43887293e-01, 1.18149665e-01],
            [2.10980399e-01, 1.14224398e-01, 5.52156911e-01, 1.22638292e-01],
            [2.12044290e-01, 1.01415511e-01, 5.59374243e-01, 1.27165955e-01],
            [2.12777911e-01, 8.98868138e-02, 5.65564304e-01, 1.31770970e-01],
            [2.13200213e-01, 7.95489630e-02, 5.70754057e-01, 1.36496767e-01],
            [2.13328015e-01, 7.03100693e-02, 5.74969767e-01, 1.41392148e-01],
            [2.13175377e-01, 6.20781199e-02, 5.78234667e-01, 1.46511836e-01],
            [2.12753003e-01, 5.47629634e-02, 5.80566684e-01, 1.51917349e-01],
            [2.12067694e-01, 4.82778457e-02, 5.81976254e-01, 1.57678207e-01],
            [2.11121790e-01, 4.25404241e-02, 5.82464120e-01, 1.63873667e-01],
            [2.09912560e-01, 3.74735370e-02, 5.82018990e-01, 1.70594913e-01],
            [2.08431491e-01, 3.30056010e-02, 5.80614947e-01, 1.77947961e-01],
            [2.06663415e-01, 2.90707653e-02, 5.78208424e-01, 1.86057396e-01],
            [2.04585400e-01, 2.56088838e-02, 5.74734550e-01, 1.95071166e-01],
            [2.02165324e-01, 2.25653431e-02, 5.70102601e-01, 2.05166732e-01],
            [1.99360030e-01, 1.98907774e-02, 5.64190280e-01, 2.16558913e-01],
            [1.96112947e-01, 1.75407096e-02, 5.56836486e-01, 2.29509857e-01],
            [1.92351073e-01, 1.54751355e-02, 5.47832225e-01, 2.44341566e-01],
            [1.87981233e-01, 1.36580686e-02, 5.36909436e-01, 2.61451263e-01],
            [1.82885632e-01, 1.20570529e-02, 5.23727800e-01, 2.81329515e-01],
            [1.76917104e-01, 1.06426524e-02, 5.07860610e-01, 3.04579633e-01],
            [1.69895030e-01, 9.38792086e-03, 4.88782663e-01, 3.31934386e-01],
            [1.61604457e-01, 8.26788100e-03, 4.65867494e-01, 3.64260168e-01],
            [1.51803622e-01, 7.25907672e-03, 4.38409213e-01, 4.02528089e-01],
            [1.40249665e-01, 6.33936318e-03, 4.05697635e-01, 4.47713338e-01],
            [1.26757836e-01, 5.48826132e-03, 3.67191501e-01, 5.00562402e-01],
            [1.11310569e-01, 4.68839622e-03, 3.22837907e-01, 5.61163128e-01],
            [9.42142047e-02, 3.92847992e-03, 2.73531647e-01, 6.28325668e-01],
            [7.62423767e-02, 3.20736701e-03, 2.21536256e-01, 6.99014000e-01],
            [5.86232788e-02, 2.53656143e-03, 1.70448974e-01, 7.68391186e-01],
            [4.27407988e-02, 1.93705559e-03, 1.24328358e-01, 8.30993787e-01],
            [2.96454250e-02, 1.42958466e-03, 8.62637052e-02, 8.82661285e-01],
            [1.97150138e-02, 1.02426416e-03, 5.73804221e-02, 9.21880300e-01],
            [1.26941686e-02, 7.17049061e-04, 3.69516241e-02, 9.49637158e-01],
            [7.98780274e-03, 4.93669057e-04, 2.32539272e-02, 9.68264601e-01],
            [4.94920545e-03, 3.36053575e-04, 1.44088459e-02, 9.80305895e-01],
            [3.03598214e-03, 2.27083826e-04, 8.83910216e-03, 9.87897832e-01],
            [1.85064498e-03, 1.52736531e-04, 5.38817086e-03, 9.92608448e-01],
            [1.12369181e-03, 1.02433840e-04, 3.27168224e-03, 9.95502192e-01],
            [6.80655561e-04, 6.85759243e-05, 1.98177675e-03, 9.97268992e-01],
            [4.11689813e-04, 4.58593611e-05, 1.19866970e-03, 9.98343781e-01],
            [2.48785662e-04, 3.06476572e-05, 7.24362584e-04, 9.98996204e-01],
            [1.50260753e-04, 2.04735127e-05, 4.37498917e-04, 9.99391767e-01],
            [9.07242335e-05, 1.36735679e-05, 2.64152780e-04, 9.99631449e-01],
            [5.47664709e-05, 9.13076963e-06, 1.59458220e-04, 9.99776645e-01],
            [3.30562786e-05, 6.09669114e-06, 9.62467961e-05, 9.99864600e-01],
            [1.99507373e-05, 4.07058812e-06, 5.80886620e-05, 9.99917890e-01],
            [1.20411070e-05, 2.71775634e-06, 3.50589495e-05, 9.99950182e-01],
            [7.26674553e-06, 1.81448136e-06, 2.11578956e-05, 9.99969761e-01],
            [4.38544006e-06, 1.21140807e-06, 1.27686717e-05, 9.99981634e-01],
            [2.64635182e-06, 8.08751772e-07, 7.70513293e-06, 9.99988840e-01],
            [1.59658805e-06, 5.39894155e-07, 4.64863489e-06, 9.99993215e-01],
            [9.63663665e-07, 3.60440823e-07, 2.80580866e-06, 9.99995870e-01],
            [5.81458028e-07, 2.40616233e-07, 1.69297654e-06, 9.99997485e-01],
            [3.50999339e-07, 1.60627592e-07, 1.02197170e-06, 9.99998466e-01],
            [2.12105795e-07, 1.07247667e-07, 6.17568459e-07, 9.99999063e-01],
            [1.28067075e-07, 7.16010679e-08, 3.72880837e-07, 9.99999427e-01],
            [7.73743336e-08, 4.78056299e-08, 2.25283557e-07, 9.99999650e-01],
            [4.68788643e-08, 3.19497919e-08, 1.36492772e-07, 9.99999785e-01],
            [2.84194486e-08, 2.13634439e-08, 8.27462311e-08, 9.99999867e-01],
            [1.70359632e-08, 1.42530879e-08, 4.96020091e-08, 9.99999919e-01],
            [1.01359124e-08, 9.49314320e-09, 2.95117812e-08, 9.99999951e-01],
            [6.01592329e-09, 6.30408466e-09, 1.75159971e-08, 9.99999970e-01],
            [3.60293196e-09, 4.18726641e-09, 1.04903175e-08, 9.99999982e-01],
            [2.21909257e-09, 2.79929916e-09, 6.46112278e-09, 9.99999989e-01],
            [1.38252426e-09, 1.87597724e-09, 4.02536563e-09, 9.99999993e-01],
            [8.55072319e-10, 1.25591890e-09, 2.48963351e-09, 9.99999995e-01],
            [5.23161237e-10, 8.39645178e-10, 1.52323929e-09, 9.99999997e-01],
            [3.19635397e-10, 5.61528892e-10, 9.30652271e-10, 9.99999998e-01],
            [1.96274641e-10, 3.75788327e-10, 5.71474380e-10, 9.99999999e-01],
            [7.90089904e-11, 2.01486586e-10, 2.30043033e-10, 9.99999999e-01],
            [-2.75223088e-11, 4.43008850e-11, -8.01341155e-11, 1.00000000e00],
            [-1.02459537e-10, -7.11494645e-11, -2.98321786e-10, 1.00000000e00],
            [-1.45802696e-10, -1.44864462e-10, -4.24519978e-10, 1.00000000e00],
            [-1.57551783e-10, -1.76844109e-10, -4.58728693e-10, 1.00000000e00],
            [-1.37706801e-10, -1.67088404e-10, -4.00947928e-10, 1.00000000e00],
            [-1.15830837e-10, -1.52277395e-10, -3.37253748e-10, 1.00000000e00],
            [-1.27082255e-10, -1.76033398e-10, -3.70013445e-10, 1.00000000e00],
            [-1.30203828e-10, -1.87167004e-10, -3.79102236e-10, 9.99999999e-01],
            [-1.25195555e-10, -1.85678212e-10, -3.64520121e-10, 9.99999999e-01],
            [-1.12057437e-10, -1.71567024e-10, -3.26267100e-10, 9.99999999e-01],
            [-9.07894722e-11, -1.44833438e-10, -2.64343172e-10, 9.99999999e-01],
            [-6.58177915e-11, -1.29826786e-10, -1.91635490e-10, 9.99999998e-01],
            [-5.77524326e-11, -1.14827540e-10, -1.68152342e-10, 9.99999999e-01],
            [-4.96870738e-11, -9.98282933e-11, -1.44669194e-10, 9.99999999e-01],
            [-4.16217149e-11, -8.48290467e-11, -1.21186046e-10, 9.99999999e-01],
            [-3.35563561e-11, -6.98298001e-11, -9.77028982e-11, 9.99999999e-01],
            [-2.54909973e-11, -5.48305535e-11, -7.42197503e-11, 9.99999999e-01],
            [-1.74256384e-11, -3.98313069e-11, -5.07366023e-11, 9.99999999e-01],
            [-1.56239261e-11, -3.63303742e-11, -4.54907255e-11, 9.99999999e-01],
            [-1.52023171e-11, -3.53629278e-11, -4.42631659e-11, 9.99999999e-01],
            [-1.47807080e-11, -3.43954813e-11, -4.30356063e-11, 9.99999999e-01],
            [-1.43590990e-11, -3.34280348e-11, -4.18080466e-11, 9.99999999e-01],
            [-1.39374900e-11, -3.24605883e-11, -4.05804870e-11, 9.99999999e-01],
            [-1.35158810e-11, -3.14931418e-11, -3.93529274e-11, 9.99999999e-01],
            [-1.30942720e-11, -3.05256954e-11, -3.81253678e-11, 9.99999999e-01],
        ]
    )

    xs = replicator_dynamics(y0=y0, timepoints=timepoints, A=M)
    assert np.allclose(xs, expected_xs_over_time)


def test_replicator_dynamics_game_size_4_example_2():
    M = np.array([[3, 2, 4, 2], [5, 1, 1, 3], [6, 2, 3, 2], [1, 3, 4, 7]])
    y0 = np.array([0.6, 0.1, 0.2, 0.1])
    timepoints = np.linspace(0, 10, 100)
    expected_xs_over_time = np.array(
        [
            [6.00000000e-01, 1.00000000e-01, 2.00000000e-01, 1.00000000e-01],
            [5.80420179e-01, 1.02112104e-01, 2.26438063e-01, 9.10296545e-02],
            [5.60703224e-01, 1.02764556e-01, 2.53803256e-01, 8.27289637e-02],
            [5.41167086e-01, 1.01977465e-01, 2.81742410e-01, 7.51130396e-02],
            [5.22074545e-01, 9.98438659e-02, 3.09908715e-01, 6.81728742e-02],
            [5.03628230e-01, 9.65147322e-02, 3.37975461e-01, 6.18815768e-02],
            [4.85972882e-01, 9.21809929e-02, 3.65645769e-01, 5.62003554e-02],
            [4.69202136e-01, 8.70552790e-02, 3.92658949e-01, 5.10836358e-02],
            [4.53367471e-01, 8.13554217e-02, 4.18794068e-01, 4.64830393e-02],
            [4.38487551e-01, 7.52909495e-02, 4.43871296e-01, 4.23502030e-02],
            [4.24556757e-01, 6.90530159e-02, 4.67751586e-01, 3.86386415e-02],
            [4.11552390e-01, 6.28078193e-02, 4.90334955e-01, 3.53048365e-02],
            [3.99440406e-01, 5.66931176e-02, 5.11557695e-01, 3.23087817e-02],
            [3.88179692e-01, 5.08174781e-02, 5.31388695e-01, 2.96141349e-02],
            [3.77725273e-01, 4.52612790e-02, 5.49825265e-01, 2.71881840e-02],
            [3.68030476e-01, 4.00793316e-02, 5.66888559e-01, 2.50016336e-02],
            [3.59048444e-01, 3.53042210e-02, 5.82618984e-01, 2.30283506e-02],
            [3.50733127e-01, 3.09500459e-02, 5.97071747e-01, 2.12450802e-02],
            [3.43039913e-01, 2.70161764e-02, 6.10312748e-01, 1.96311622e-02],
            [3.35926019e-01, 2.34907933e-02, 6.22414927e-01, 1.81682605e-02],
            [3.29350717e-01, 2.03540183e-02, 6.33455151e-01, 1.68401144e-02],
            [3.23275448e-01, 1.75806009e-02, 6.43511642e-01, 1.56323094e-02],
            [3.17663866e-01, 1.51421124e-02, 6.52661949e-01, 1.45320720e-02],
            [3.12481831e-01, 1.30086774e-02, 6.60981407e-01, 1.35280845e-02],
            [3.07697368e-01, 1.11502875e-02, 6.68542025e-01, 1.26103192e-02],
            [3.03280608e-01, 9.53775700e-03, 6.75411745e-01, 1.17698907e-02],
            [2.99203707e-01, 8.14336365e-03, 6.81654005e-01, 1.09989243e-02],
            [2.95440763e-01, 6.94128233e-03, 6.87327517e-01, 1.02904379e-02],
            [2.91967717e-01, 5.90780631e-03, 6.92486238e-01, 9.63823834e-03],
            [2.88762258e-01, 5.02143667e-03, 6.97179477e-01, 9.03682838e-03],
            [2.85803721e-01, 4.26287314e-03, 7.01452081e-01, 8.48132511e-03],
            [2.83072986e-01, 3.61493852e-03, 7.05344689e-01, 7.96738728e-03],
            [2.80552380e-01, 3.06245306e-03, 7.08894015e-01, 7.49115128e-03],
            [2.78225583e-01, 2.59208318e-03, 7.12133159e-01, 7.04917476e-03],
            [2.76077528e-01, 2.19217981e-03, 7.15091905e-01, 6.63838682e-03],
            [2.74094319e-01, 1.85260717e-03, 7.17797029e-01, 6.25604450e-03],
            [2.72263140e-01, 1.56458788e-03, 7.20272578e-01, 5.89969385e-03],
            [2.70572180e-01, 1.32054238e-03, 7.22540141e-01, 5.56713642e-03],
            [2.69010552e-01, 1.11394543e-03, 7.24619103e-01, 5.25639953e-03],
            [2.67568225e-01, 9.39195127e-04, 7.26526869e-01, 4.96571001e-03],
            [2.66235959e-01, 7.91492823e-04, 7.28279077e-01, 4.69347140e-03],
            [2.65005238e-01, 6.66737179e-04, 7.29889781e-01, 4.43824386e-03],
            [2.63868220e-01, 5.61428279e-04, 7.31371625e-01, 4.19872636e-03],
            [2.62817677e-01, 4.72584699e-04, 7.32735997e-01, 3.97374121e-03],
            [2.61846950e-01, 3.97670564e-04, 7.33993159e-01, 3.76222045e-03],
            [2.60949904e-01, 3.34531655e-04, 7.35152371e-01, 3.56319374e-03],
            [2.60120883e-01, 2.81339801e-04, 7.36222000e-01, 3.37577783e-03],
            [2.59354674e-01, 2.36545608e-04, 7.37209613e-01, 3.19916730e-03],
            [2.58646473e-01, 1.98836743e-04, 7.38122064e-01, 3.03262621e-03],
            [2.57991847e-01, 1.67102937e-04, 7.38965569e-01, 2.87548099e-03],
            [2.57386712e-01, 1.40406606e-04, 7.39745767e-01, 2.72711416e-03],
            [2.56827297e-01, 1.17953508e-04, 7.40467791e-01, 2.58695819e-03],
            [2.56310127e-01, 9.90745170e-05, 7.41136308e-01, 2.45449105e-03],
            [2.55831991e-01, 8.32032628e-05, 7.41755574e-01, 2.32923122e-03],
            [2.55389930e-01, 6.98644087e-05, 7.42329471e-01, 2.21073444e-03],
            [2.54981210e-01, 5.86559924e-05, 7.42861544e-01, 2.09858959e-03],
            [2.54603308e-01, 4.92394766e-05, 7.43355037e-01, 1.99241593e-03],
            [2.54253894e-01, 4.13300061e-05, 7.43812916e-01, 1.89186027e-03],
            [2.53930817e-01, 3.46872608e-05, 7.44237902e-01, 1.79659435e-03],
            [2.53632090e-01, 2.91099509e-05, 7.44632487e-01, 1.70631297e-03],
            [2.53355875e-01, 2.44261534e-05, 7.44998968e-01, 1.62073123e-03],
            [2.53100479e-01, 2.04943791e-05, 7.45339443e-01, 1.53958396e-03],
            [2.52864331e-01, 1.71936064e-05, 7.45655852e-01, 1.46262303e-03],
            [2.52645987e-01, 1.44237080e-05, 7.45949973e-01, 1.38961674e-03],
            [2.52444104e-01, 1.20990188e-05, 7.46223449e-01, 1.32034782e-03],
            [2.52257447e-01, 1.01486469e-05, 7.46477791e-01, 1.25461296e-03],
            [2.52084872e-01, 8.51182213e-06, 7.46714395e-01, 1.19222110e-03],
            [2.51925320e-01, 7.13869064e-06, 7.46934548e-01, 1.13299311e-03],
            [2.51777814e-01, 5.98662919e-06, 7.47139439e-01, 1.07676038e-03],
            [2.51641450e-01, 5.02038939e-06, 7.47330165e-01, 1.02336449e-03],
            [2.51515390e-01, 4.20987155e-06, 7.47507744e-01, 9.72656120e-04],
            [2.51398863e-01, 3.53027631e-06, 7.47673112e-01, 9.24494781e-04],
            [2.51291152e-01, 2.96001271e-06, 7.47827141e-01, 8.78747738e-04],
            [2.51191595e-01, 2.48172331e-06, 7.47970634e-01, 8.35290020e-04],
            [2.51099580e-01, 2.08057621e-06, 7.48104336e-01, 7.94003505e-04],
            [2.51014541e-01, 1.74422498e-06, 7.48228938e-01, 7.54776690e-04],
            [2.50935955e-01, 1.46226491e-06, 7.48345078e-01, 7.17504179e-04],
            [2.50863336e-01, 1.22585116e-06, 7.48453351e-01, 6.82086278e-04],
            [2.50796237e-01, 1.02769432e-06, 7.48554306e-01, 6.48428765e-04],
            [2.50734243e-01, 8.61444851e-07, 7.48648453e-01, 6.16442360e-04],
            [2.50676969e-01, 7.22075292e-07, 7.48736266e-01, 5.86042659e-04],
            [2.50624061e-01, 6.05224449e-07, 7.48818184e-01, 5.57149616e-04],
            [2.50575191e-01, 5.07304773e-07, 7.48894614e-01, 5.29687483e-04],
            [2.50530054e-01, 4.25231575e-07, 7.48965936e-01, 5.03584414e-04],
            [2.50488370e-01, 3.56536646e-07, 7.49032501e-01, 4.78772412e-04],
            [2.50449879e-01, 2.99072838e-07, 7.49094635e-01, 4.55186947e-04],
            [2.50414339e-01, 2.50787707e-07, 7.49152643e-01, 4.32766625e-04],
            [2.50381527e-01, 2.09990839e-07, 7.49206810e-01, 4.11453246e-04],
            [2.50351237e-01, 1.75740049e-07, 7.49257395e-01, 3.91191937e-04],
            [2.50323280e-01, 1.46960312e-07, 7.49304642e-01, 3.71930334e-04],
            [2.50297478e-01, 1.22681547e-07, 7.49348781e-01, 3.53618629e-04],
            [2.50273668e-01, 1.02313601e-07, 7.49390020e-01, 3.36209829e-04],
            [2.50251700e-01, 8.53321199e-08, 7.49428555e-01, 3.19659221e-04],
            [2.50231434e-01, 7.12334339e-08, 7.49464570e-01, 3.03924199e-04],
            [2.50212740e-01, 5.95800286e-08, 7.49498236e-01, 2.88964386e-04],
            [2.50195501e-01, 4.99771822e-08, 7.49529708e-01, 2.74741555e-04],
            [2.50179604e-01, 4.19815889e-08, 7.49559135e-01, 2.61219233e-04],
            [2.50164947e-01, 3.51538946e-08, 7.49586655e-01, 2.48362728e-04],
            [2.50151436e-01, 2.93685983e-08, 7.49612395e-01, 2.36139268e-04],
            [2.50138985e-01, 2.44612938e-08, 7.49636473e-01, 2.24517619e-04],
        ]
    )

    xs = replicator_dynamics(y0=y0, timepoints=timepoints, A=M)
    assert np.allclose(xs, expected_xs_over_time)


def test_replicator_dynamics_game_size_4_default_y0_example_1():
    M = np.array([[3, 2, 4, 2], [5, 1, 1, 3], [6, 2, 3, 2], [1, 3, 4, 7]])
    timepoints = np.linspace(0, 10, 100)

    expected_xs_over_time = np.array(
        [
            [2.50000000e-01, 2.50000000e-01, 2.50000000e-01, 2.50000000e-01],
            [2.41719166e-01, 2.35735156e-01, 2.53873636e-01, 2.68672042e-01],
            [2.32546984e-01, 2.21360179e-01, 2.55769777e-01, 2.90323060e-01],
            [2.22291827e-01, 2.06871196e-01, 2.55261786e-01, 3.15575190e-01],
            [2.10731484e-01, 1.92233270e-01, 2.51864185e-01, 3.45171061e-01],
            [1.97619659e-01, 1.77375409e-01, 2.45043522e-01, 3.79961410e-01],
            [1.82707977e-01, 1.62190468e-01, 2.34255381e-01, 4.20846174e-01],
            [1.65796132e-01, 1.46547147e-01, 2.19029301e-01, 4.68627420e-01],
            [1.46825571e-01, 1.30325960e-01, 1.99128685e-01, 5.23719784e-01],
            [1.26021893e-01, 1.13492486e-01, 1.74796565e-01, 5.85689056e-01],
            [1.04051013e-01, 9.62074113e-02, 1.47031763e-01, 6.52709813e-01],
            [8.20816740e-02, 7.89301100e-02, 1.17717959e-01, 7.21270257e-01],
            [6.16068147e-02, 6.24180199e-02, 8.93556151e-02, 7.86619550e-01],
            [4.39995702e-02, 4.75413069e-02, 6.43425051e-02, 8.44116618e-01],
            [3.00443142e-02, 3.49797125e-02, 4.41856313e-02, 8.90790342e-01],
            [1.97703777e-02, 2.50057355e-02, 2.91865133e-02, 9.26037374e-01],
            [1.26494877e-02, 1.74854076e-02, 1.87200451e-02, 9.51145060e-01],
            [7.93309603e-03, 1.20349937e-02, 1.17584783e-02, 9.68273432e-01],
            [4.90797650e-03, 8.19467468e-03, 7.28166293e-03, 9.79615686e-01],
            [3.00928572e-03, 5.54018743e-03, 4.46734830e-03, 9.86983179e-01],
            [1.83444440e-03, 3.72835899e-03, 2.72426145e-03, 9.91712935e-01],
            [1.11413336e-03, 2.50170133e-03, 1.65492179e-03, 9.94729244e-01],
            [6.75070430e-04, 1.67550183e-03, 1.00287691e-03, 9.96646551e-01],
            [4.08428511e-04, 1.12084196e-03, 6.06805960e-04, 9.97863924e-01],
            [2.46874544e-04, 7.49242911e-04, 3.66801756e-04, 9.98637081e-01],
            [1.49135183e-04, 5.00608887e-04, 2.21588946e-04, 9.99128667e-01],
            [9.00578701e-05, 3.34385059e-04, 1.33812730e-04, 9.99441744e-01],
            [5.43699910e-05, 2.23313206e-04, 8.07866707e-05, 9.99641530e-01],
            [3.28200257e-05, 1.49118817e-04, 4.87665696e-05, 9.99769295e-01],
            [1.98098290e-05, 9.95680219e-05, 2.94351134e-05, 9.99851187e-01],
            [1.19560257e-05, 6.64791103e-05, 1.77653131e-05, 9.99903800e-01],
            [7.21588878e-06, 4.43855409e-05, 1.07220167e-05, 9.99937677e-01],
            [4.35460889e-06, 2.96335830e-05, 6.47047541e-06, 9.99959541e-01],
            [2.62795277e-06, 1.97844865e-05, 3.90485418e-06, 9.99973683e-01],
            [1.58581836e-06, 1.32085926e-05, 2.35635571e-06, 9.99982849e-01],
            [9.57098845e-07, 8.81857690e-06, 1.42214629e-06, 9.99988802e-01],
            [5.77181917e-07, 5.88651878e-06, 8.57630601e-07, 9.99992679e-01],
            [3.48017960e-07, 3.92912327e-06, 5.17117549e-07, 9.99995206e-01],
            [2.10432900e-07, 2.62354883e-06, 3.12680828e-07, 9.99996853e-01],
            [1.27160084e-07, 1.75168432e-06, 1.88946318e-07, 9.99997932e-01],
            [7.69138599e-08, 1.17045061e-06, 1.14285800e-07, 9.99998638e-01],
            [4.67857994e-08, 7.82807429e-07, 6.95187050e-08, 9.99999101e-01],
            [2.81940356e-08, 5.22518381e-07, 4.18933223e-08, 9.99999407e-01],
            [1.66353720e-08, 3.46827263e-07, 2.47183847e-08, 9.99999612e-01],
            [9.92969426e-09, 2.31604309e-07, 1.47544662e-08, 9.99999744e-01],
            [5.86008245e-09, 1.52926389e-07, 8.70745685e-09, 9.99999833e-01],
            [3.61260541e-09, 1.01921785e-07, 5.36794489e-09, 9.99999889e-01],
            [2.35820495e-09, 6.89433682e-08, 3.50403969e-09, 9.99999925e-01],
            [1.54512934e-09, 4.67336874e-08, 2.29589660e-09, 9.99999949e-01],
            [9.91566874e-10, 3.15653821e-08, 1.47336228e-09, 9.99999966e-01],
            [6.23213391e-10, 2.13755295e-08, 9.26028545e-10, 9.99999977e-01],
            [3.89169337e-10, 1.45088069e-08, 5.78264090e-10, 9.99999985e-01],
            [2.44232046e-10, 9.82444655e-09, 3.62902722e-10, 9.99999990e-01],
            [1.55958046e-10, 6.64441716e-09, 2.31736960e-10, 9.99999993e-01],
            [1.01833748e-10, 4.50886059e-09, 1.51314038e-10, 9.99999995e-01],
            [6.61625623e-11, 3.05585415e-09, 9.83104808e-11, 9.99999997e-01],
            [4.22829112e-11, 2.06460089e-09, 6.28278830e-11, 9.99999998e-01],
            [2.66805156e-11, 1.40000880e-09, 3.96443958e-11, 9.99999999e-01],
            [1.67673488e-11, 9.49807666e-10, 2.49144894e-11, 9.99999999e-01],
            [1.05528316e-11, 6.42239872e-10, 1.56803796e-11, 9.99999999e-01],
            [2.03838562e-12, 2.61718511e-10, 3.02882291e-12, 1.00000000e00],
            [-4.66662938e-12, -4.40298241e-11, -6.93411260e-12, 1.00000000e00],
            [-9.48563948e-12, -2.72252392e-10, -1.40946462e-11, 1.00000000e00],
            [-1.24186447e-11, -4.22949193e-10, -1.84527780e-11, 1.00000000e00],
            [-1.34656449e-11, -4.96120226e-10, -2.00085079e-11, 1.00000000e00],
            [-1.26266403e-11, -4.91765491e-10, -1.87618360e-11, 1.00000000e00],
            [-9.90163080e-12, -4.09884990e-10, -1.47127621e-11, 1.00000000e00],
            [-9.10831756e-12, -3.98815250e-10, -1.35339828e-11, 9.99999999e-01],
            [-9.61446910e-12, -4.33990960e-10, -1.42860696e-11, 9.99999999e-01],
            [-9.68711978e-12, -4.48077852e-10, -1.43940204e-11, 9.99999999e-01],
            [-9.32626962e-12, -4.41075926e-10, -1.38578353e-11, 9.99999999e-01],
            [-8.53191859e-12, -4.12985183e-10, -1.26775142e-11, 9.99999999e-01],
            [-7.30406672e-12, -3.63805622e-10, -1.08530573e-11, 9.99999999e-01],
            [-5.64271399e-12, -2.93537243e-10, -8.38446446e-12, 9.99999999e-01],
            [-3.76042064e-12, -2.46757395e-10, -5.58757547e-12, 9.99999998e-01],
            [-3.34138439e-12, -2.20697822e-10, -4.96493302e-12, 9.99999999e-01],
            [-2.92234814e-12, -1.94638250e-10, -4.34229056e-12, 9.99999999e-01],
            [-2.50331189e-12, -1.68578677e-10, -3.71964811e-12, 9.99999999e-01],
            [-2.08427564e-12, -1.42519105e-10, -3.09700565e-12, 9.99999999e-01],
            [-1.66523938e-12, -1.16459532e-10, -2.47436319e-12, 9.99999999e-01],
            [-1.24620313e-12, -9.03999598e-11, -1.85172074e-12, 9.99999999e-01],
            [-8.29582639e-13, -6.44890104e-11, -1.23266784e-12, 9.99999999e-01],
            [-8.11410998e-13, -6.30915593e-11, -1.20566680e-12, 9.99999999e-01],
            [-7.93239357e-13, -6.16941082e-11, -1.17866576e-12, 9.99999999e-01],
            [-7.75067715e-13, -6.02966571e-11, -1.15166472e-12, 9.99999999e-01],
            [-7.56896074e-13, -5.88992060e-11, -1.12466368e-12, 9.99999999e-01],
            [-7.38724433e-13, -5.75017549e-11, -1.09766264e-12, 9.99999999e-01],
            [-7.20552791e-13, -5.61043038e-11, -1.07066160e-12, 9.99999999e-01],
            [-7.02381150e-13, -5.47068527e-11, -1.04366056e-12, 9.99999999e-01],
            [-6.84209509e-13, -5.33094016e-11, -1.01665952e-12, 9.99999999e-01],
            [-6.66037867e-13, -5.19119505e-11, -9.89658484e-13, 9.99999999e-01],
            [-6.47866226e-13, -5.05144994e-11, -9.62657445e-13, 9.99999999e-01],
            [-6.29694585e-13, -4.91170483e-11, -9.35656406e-13, 9.99999999e-01],
            [-6.11522943e-13, -4.77195972e-11, -9.08655366e-13, 9.99999999e-01],
            [-5.93351302e-13, -4.63221461e-11, -8.81654327e-13, 9.99999999e-01],
            [-5.75179661e-13, -4.49246950e-11, -8.54653288e-13, 9.99999999e-01],
            [-5.57008019e-13, -4.35272439e-11, -8.27652249e-13, 9.99999999e-01],
            [-5.38836378e-13, -4.21297928e-11, -8.00651210e-13, 9.99999999e-01],
            [-5.20664737e-13, -4.07323417e-11, -7.73650170e-13, 9.99999999e-01],
            [-5.02493095e-13, -3.93348906e-11, -7.46649131e-13, 9.99999999e-01],
        ]
    )

    xs = replicator_dynamics(timepoints=timepoints, A=M)
    assert np.allclose(xs, expected_xs_over_time)


def test_replicator_dynamics_with_incorrect_inputs():
    """
    Test that if an incorrect starting value is given, an error is raised
    """
    M = np.array([[3, 2, 4, 2], [5, 1, 1, 3], [6, 2, 3, 2], [1, 3, 4, 7]])
    y0 = np.array([1, 0, 0])
    with pytest.raises(ValueError):
        replicator_dynamics(y0=y0, A=M)


@given(A=arrays(np.int8, (3, 2)), B=arrays(np.int8, (3, 2)))
def test_property_get_derivative_of_asymmetric_fitness(A, B):
    """
    Property-based test of get_derivative_of_asymmetric_fitness for a 3x2 game

    Parameters
    ----------
    A : array
        2 dimensional list/array representing the payoff matrix for a
        the row player in a game.
    B : array
        2 dimensional list/array representing the payoff matrix for a
        the column player in a game.
    """
    t = 0
    x = np.ones(A.shape[1] + A.shape[0])
    derivative_of_fitness = get_derivative_of_asymmetric_fitness(x, t, A, B)

    assert len(derivative_of_fitness) == len(x)


def test_get_derivative_of_asymmetric_fitness_example():
    """
    Test for the asymmetric derivative of fitness function
    """
    M = np.array([[3, 2, 3], [4, 1, 1], [2, 3, 1]])
    N = np.array([[1, 2, 3], [3, 2, 1], [2, 1, 3]])

    x_values = (
        np.array([1, 0, 0, 1, 0, 0]),
        np.array([1 / 2, 1 / 2, 0, 1 / 2, 1 / 2, 0]),
        np.array([0, 1 / 4, 3 / 4, 0, 1 / 4, 3 / 4]),
        np.array([1 / 5, 2 / 5, 2 / 5, 1 / 5, 2 / 5, 2 / 5]),
        np.array([1 / 2, 0, 1 / 2, 1 / 2, 0, 1 / 2]),
        np.array([2 / 4, 1 / 4, 1 / 4, 2 / 4, 1 / 4, 1 / 4]),
    )
    derivative_values = (
        np.array([0, 0, 0, 0, 0, 0]),
        np.array([0, 0, 0, 0, 0, 0]),
        np.array([0.0, -0.09375, 0.09375, 0.0, -0.234375, 0.234375]),
        np.array([0.128, -0.144, 0.016, 0.048, -0.144, 0.096]),
        np.array([0.375, 0.0, -0.375, -0.375, 0.0, 0.375]),
        np.array([0.125, 0.0, -0.125, -0.09375, -0.046875, 0.140625]),
    )

    for x_value, expected_derivative in zip(x_values, derivative_values):
        derivative = get_derivative_of_asymmetric_fitness(x=x_value, t=0, A=M, B=N)
        assert np.allclose(derivative, expected_derivative), x_value


@settings(max_examples=10)
@given(
    A=arrays(np.int8, (4, 2), elements=integers(0, 100)),
    B=arrays(np.int8, (4, 2), elements=integers(0, 100)),
)
def test_property_of_output_dimension_for_asymmetric_games_of_size_4_2(A, B):
    """
    Property-based test of asymmetric_replicator_dynamics for a 4x2 game

    Parameters
    ----------
    A : array
        2 dimensional list/array representing the payoff matrix for a
        the row player in a game.
    B : array
        2 dimensional list/array representing the payoff matrix for a
        the column player in a game.
    """
    xs1, xs2 = asymmetric_replicator_dynamics(A, B)
    assert all(len(x) == 4 for x in xs1)
    assert all(len(x) == 2 for x in xs2)


@given(A=arrays(np.int8, shape=(2, 2), elements=integers(1, 5)))
def test_equivalence_between_symmetric_and_asymmetric_replicator_dynamics(A):
    """
    Tests that when we have two populations with identical strategies then the
    output of the asymmetric_replicator_dynamics for both populations is the
    same as using just one population in replicator_dynamics. The test is
    carried out for 2x2 matrices with elements from 1-5

    Note that the test hypothesis can find cases where this test can fail for
    larger elements or larger matrix sizes. One potenetial reason for this might
    be the fact that scipy.odeint() is a deprecated function.

    Parameters
    ----------
    A : array
        2 dimensional list/array representing the payoff matrix for a
        the row player in a game.
    """
    B = A.transpose()

    symmetric_xs = replicator_dynamics(A)
    asymmetric_row_xs, asymmetric_col_xs = asymmetric_replicator_dynamics(A, B)

    assert np.allclose(asymmetric_row_xs, asymmetric_col_xs, atol=1e-3)
    assert np.allclose(symmetric_xs, asymmetric_row_xs, atol=1e-3)
    assert np.allclose(symmetric_xs, asymmetric_col_xs, atol=1e-3)


def test_asymmetric_replicator_dynamics_size_2_3_default_values():
    """
    Test the asymmetric replicator dynamics function for a 2x3 game by using
    the default values
    """
    A = np.array([[1, 2, 3], [4, 5, 6]])
    B = np.array([[7, 8, 9], [10, 11, 12]])

    xs_A, xs_B = asymmetric_replicator_dynamics(A, B)

    assert np.allclose(xs_A[1], np.array([0.49249308, 0.50750692]), atol=1e-5)
    assert np.allclose(xs_A[-1], np.array([9.33624531e-14, 1]), atol=1e-5)
    assert np.allclose(
        xs_B[1], np.array([0.33000229, 0.3333222, 0.33667551]), atol=1e-5
    )
    assert np.allclose(
        xs_B[-1],
        np.array([2.04812640e-09, 4.53898590e-05, 9.99954607e-01]),
        atol=1e-5,
    )


def test_asymmetric_replicator_dynamics_size_2_3_given_timepoints():
    """
    Test the asymmetric replicator dynamics function for a 2x3 game and not
    using the default timepoints
    """
    timepoints = np.linspace(0, 100, 100)
    A = np.array([[1, 1, 2], [2, 3, 2]])
    B = np.array([[1, 2, 2], [2, 1, 3]])

    xs_A, xs_B = asymmetric_replicator_dynamics(A, B, timepoints=timepoints)

    assert np.allclose(xs_A[1], np.array([0.30904906, 0.69095094]))
    assert np.allclose(xs_B[1], np.array([0.2196786, 0.1771107, 0.6032107]))
    assert np.allclose(xs_A[-1], np.array([0.2, 0.8]))
    assert np.allclose(xs_B[-1], np.array([-6.57013390e-14, 2.92761632e-17, 1]))


def test_asymmetric_replicator_dynamics_size_4_6_given_x0_y0():
    """
    Test the asymmetric replicator dynamics function for a 4x6 game by
    specifying values for x0 and y0
    """
    A = np.array(
        [
            [1, 20, 23, 21, 15, 4],
            [9, 29, 0, 14, 19, 27],
            [22, 28, 30, 12, 3, 25],
            [5, 16, 8, 17, 11, 18],
        ]
    )
    B = np.array(
        [
            [11, 39, 27, 15, 36, 35],
            [1, 31, 2, 18, 10, 19],
            [21, 38, 8, 24, 40, 32],
            [22, 37, 25, 7, 30, 0],
        ]
    )
    x0 = np.array([0.5, 0.2, 0.2, 0.1])
    y0 = np.array([0.4, 0.1, 0.1, 0.1, 0.2, 0.1])

    xs_A, xs_B = asymmetric_replicator_dynamics(A, B, x0=x0, y0=y0)

    assert np.allclose(
        xs_A[1], np.array([0.48729326, 0.20349646, 0.21191178, 0.0972985])
    )
    assert np.allclose(
        xs_A[-1],
        np.array([-2.50483397e-15, 9.99977992e-01, 2.20078313e-05, 1.18367977e-17]),
    )
    assert np.allclose(
        xs_B[1],
        np.array(
            [
                0.36455939,
                0.11688505,
                0.096508,
                0.09537898,
                0.22015362,
                0.10651496,
            ]
        ),
    )
    assert np.allclose(
        xs_B[-1],
        np.array(
            [
                4.58211507e-12,
                1.00000000e00,
                8.73932312e-12,
                1.58763628e-18,
                -1.22965529e-14,
                -9.91094095e-17,
            ]
        ),
    )
