# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['tsflex',
 'tsflex.chunking',
 'tsflex.features',
 'tsflex.features.segmenter',
 'tsflex.processing',
 'tsflex.utils']

package_data = \
{'': ['*']}

install_requires = \
['dill>=0.3.4,<0.4.0',
 'multiprocess>=0.70.12,<0.71.0',
 'numpy>=1.21.5,<2.0.0',
 'pandas>=1.3.5,<2.0.0',
 'tqdm>=4.62.3,<5.0.0']

setup_kwargs = {
    'name': 'tsflex',
    'version': '0.2.3.5',
    'description': 'Toolkit for flexible processing & feature extraction on time-series data',
    'long_description': '# <p align="center"> <a href="https://predict-idlab.github.io/tsflex"><img alt="tsflex" src="https://raw.githubusercontent.com/predict-idlab/tsflex/main/docs/_static/logo.png" width="66%"></a></p>\n\n[![PyPI Latest Release](https://img.shields.io/pypi/v/tsflex.svg)](https://pypi.org/project/tsflex/)\n[![Conda Latest Release](https://img.shields.io/conda/vn/conda-forge/tsflex?label=conda)](https://anaconda.org/conda-forge/tsflex)\n[![support-version](https://img.shields.io/pypi/pyversions/tsflex)](https://img.shields.io/pypi/pyversions/tsflex)\n[![codecov](https://img.shields.io/codecov/c/github/predict-idlab/tsflex?logo=codecov)](https://codecov.io/gh/predict-idlab/tsflex)\n[![Code quality](https://img.shields.io/lgtm/grade/python/github/predict-idlab/tsflex?label=code%20quality&logo=lgtm)](https://lgtm.com/projects/g/predict-idlab/tsflex/context:python)\n[![Downloads](https://pepy.tech/badge/tsflex)](https://pepy.tech/project/tsflex)\n[![PRs Welcome](https://img.shields.io/badge/PRs-welcome-brightgreen.svg?)](http://makeapullrequest.com)\n[![Documentation](https://github.com/predict-idlab/tsflex/actions/workflows/deploy-docs.yml/badge.svg)](https://github.com/predict-idlab/tsflex/actions/workflows/deploy-docs.yml)\n[![Testing](https://github.com/predict-idlab/tsflex/actions/workflows/test.yml/badge.svg)](https://github.com/predict-idlab/tsflex/actions/workflows/test.yml)\n\n<!-- ![Downloads](https://img.shields.io/conda/dn/conda-forge/tsflex?logo=anaconda) -->\n\n> *tsflex* is a toolkit for _**flex**ible **t**ime **s**eries_ [processing](https://predict-idlab.github.io/tsflex/processing) & [feature extraction](https://predict-idlab.github.io/tsflex/features), that is efficient and makes few assumptions about sequence data.\n\n#### Useful links\n\n- [Paper (preprint)](https://arxiv.org/abs/2111.12429)\n- [Documentation](https://predict-idlab.github.io/tsflex/)\n- [Example (machine learning) notebooks](https://github.com/predict-idlab/tsflex/tree/main/examples)\n\n#### Installation\n\n| | command|\n|:--------------|:--------------|\n| [**pip**](https://pypi.org/project/tsflex/) | `pip install tsflex` | \n| [**conda**](https://anaconda.org/conda-forge/tsflex) | `conda install -c conda-forge tsflex` |\n\n## Usage\n\n_tsflex_ is built to be intuitive, so we encourage you to copy-paste this code and toy with some parameters!\n\n### <a href="https://predict-idlab.github.io/tsflex/features/#getting-started">Feature extraction</a>\n\n```python\nimport pandas as pd; import numpy as np; import scipy.stats as ss\nfrom tsflex.features import MultipleFeatureDescriptors, FeatureCollection\nfrom tsflex.utils.data import load_empatica_data\n\n# 1. Load sequence-indexed data (in this case a time-index)\ndf_tmp, df_acc, df_ibi = load_empatica_data([\'tmp\', \'acc\', \'ibi\'])\n\n# 2. Construct your feature extraction configuration\nfc = FeatureCollection(\n    MultipleFeatureDescriptors(\n          functions=[np.min, np.mean, np.std, ss.skew, ss.kurtosis],\n          series_names=["TMP", "ACC_x", "ACC_y", "IBI"],\n          windows=["15min", "30min"],\n          strides="15min",\n    )\n)\n\n# 3. Extract features\nfc.calculate(data=[df_tmp, df_acc, df_ibi], approve_sparsity=True)\n```\n\nNote that the feature extraction is performed on multivariate data with varying sample rates.\n| signal | columns | sample rate |\n|:-------|:-------|------------------:|\n| df_tmp | ["TMP"]| 4Hz |\n| df_acc | ["ACC_x", "ACC_y", "ACC_z" ]| 32Hz |\n| df_ibi | ["IBI"]| irregularly sampled |\n\n### <a href="https://predict-idlab.github.io/tsflex/processing/index.html#getting-started">Processing</a>\n[Working example in our docs](https://predict-idlab.github.io/tsflex/processing/index.html#working-example)\n\n## Why tsflex? ✨\n\n* `Flexible`:\n    * handles multivariate/multimodal time series\n    * versatile function support\n      => **integrates** with many packages for:\n      * processing (e.g., [scipy.signal](https://docs.scipy.org/doc/scipy/reference/tutorial/signal.html), [statsmodels.tsa](https://www.statsmodels.org/stable/tsa.html#time-series-filters))\n      * feature extraction (e.g., [numpy](https://numpy.org/doc/stable/reference/routines.html), [scipy.stats](https://docs.scipy.org/doc/scipy/reference/tutorial/stats.html), [seglearn](https://dmbee.github.io/seglearn/feature_functions.html)¹, [tsfresh](https://tsfresh.readthedocs.io/en/latest/text/list_of_features.html)¹, [tsfel](https://tsfel.readthedocs.io/en/latest/descriptions/feature_list.html)¹)\n    * feature extraction handles **multiple strides & window sizes**\n* `Efficient`:<br>\n  * view-based operations for processing & feature extraction => extremely **low memory peak** & **fast execution time**<br>\n    * see: [feature extraction benchmark visualization](https://predict-idlab.github.io/tsflex/#benchmark)\n* `Intuitive`:<br>\n  * maintains the sequence-index of the data\n  * feature extraction constructs interpretable output column names\n  * intuitive API\n* `Few assumptions` about the sequence data:\n  * no assumptions about sampling rate\n  * able to deal with multivariate asynchronous data<br>i.e. data with small time-offsets between the modalities\n* `Advanced functionalities`:\n  * apply [FeatureCollection.**reduce**](https://predict-idlab.github.io/tsflex/features/index.html#tsflex.features.FeatureCollection.reduce) after feature selection for faster inference\n  * use **function execution time logging** to discover processing and feature extraction bottlenecks\n  * embedded [SeriesPipeline](http://predict-idlab.github.io/tsflex/processing/#tsflex.processing.SeriesPipeline.serialize) & [FeatureCollection](https://predict-idlab.github.io/tsflex/features/index.html#tsflex.features.FeatureCollection.serialize) **serialization**\n  * time series [**chunking**](https://predict-idlab.github.io/tsflex/chunking/index.html)\n\n¹ These integrations are shown in [integration-example notebooks](https://github.com/predict-idlab/tsflex/tree/main/examples).\n## Future work 🔨\n\n* scikit-learn integration for both processing and feature extraction<br>\n  **note**: is actively developed upon [sklearn integration](https://github.com/predict-idlab/tsflex/tree/sklearn_integration) branch.\n* Support time series segmentation (exposing under the hood strided-rolling functionality) - [see this issue](https://github.com/predict-idlab/tsflex/issues/15)\n* Support for multi-indexed dataframes\n\n=> Also see the [enhancement issues](https://github.com/predict-idlab/tsflex/issues?q=is%3Aissue+is%3Aopen+label%3Aenhancement+)\n## Contributing 👪\n\nWe are thrilled to see your contributions to further enhance `tsflex`.<br>\nSee [this guide](CONTRIBUTING.md) for more instructions on how to contribute.\n\n## Referencing our package\n\nIf you use `tsflex` in a scientific publication, we would highly appreciate citing us as:\n\n```bibtex\n@article{vanderdonckt2021tsflex,\n    author = {Van Der Donckt, Jonas and Van Der Donckt, Jeroen and Deprost, Emiel and Van Hoecke, Sofie},\n    title = {tsflex: flexible time series processing \\& feature extraction},\n    journal = {SoftwareX},\n    year = {2021},\n    url = {https://github.com/predict-idlab/tsflex},\n    publisher={Elsevier}\n}\n```\n\nLinkt to the preprint paper: https://arxiv.org/abs/2111.12429\n\n---\n\n<p align="center">\n👤 <i>Jonas Van Der Donckt, Jeroen Van Der Donckt, Emiel Deprost</i>\n</p>\n',
    'author': 'Jonas Van Der Donckt, Jeroen Van Der Donckt, Emiel Deprost',
    'author_email': None,
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/predict-idlab/tsflex',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7.1,<3.11',
}


setup(**setup_kwargs)
