"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../../../lib");
describe('HttpProxyIntegration', () => {
    test('default', () => {
        const stack = new core_1.Stack();
        const api = new lib_1.HttpApi(stack, 'HttpApi');
        new lib_1.HttpRoute(stack, 'HttpProxyRoute', {
            httpApi: api,
            integration: new lib_1.HttpProxyIntegration({
                url: 'some-target-url',
            }),
            routeKey: lib_1.HttpRouteKey.with('/pets'),
        });
        expect(stack).toHaveResource('AWS::ApiGatewayV2::Integration', {
            IntegrationType: 'HTTP_PROXY',
            IntegrationUri: 'some-target-url',
            PayloadFormatVersion: '1.0',
            IntegrationMethod: 'ANY',
        });
    });
    test('method option is correctly recognized', () => {
        const stack = new core_1.Stack();
        const api = new lib_1.HttpApi(stack, 'HttpApi');
        new lib_1.HttpRoute(stack, 'HttpProxyRoute', {
            httpApi: api,
            integration: new lib_1.HttpProxyIntegration({
                url: 'some-target-url',
                method: lib_1.HttpMethod.PATCH,
            }),
            routeKey: lib_1.HttpRouteKey.with('/pets'),
        });
        expect(stack).toHaveResource('AWS::ApiGatewayV2::Integration', {
            IntegrationMethod: 'PATCH',
        });
    });
    test('custom payload format version is allowed', () => {
        const stack = new core_1.Stack();
        const api = new lib_1.HttpApi(stack, 'HttpApi');
        new lib_1.HttpIntegration(stack, 'HttpInteg', {
            payloadFormatVersion: lib_1.PayloadFormatVersion.custom('99.99'),
            httpApi: api,
            integrationType: lib_1.HttpIntegrationType.HTTP_PROXY,
            integrationUri: 'some-target-url',
        });
        expect(stack).toHaveResource('AWS::ApiGatewayV2::Integration', {
            IntegrationType: 'HTTP_PROXY',
            IntegrationUri: 'some-target-url',
            PayloadFormatVersion: '99.99',
        });
    });
    test('HttpIntegration without payloadFormatVersion is allowed', () => {
        const stack = new core_1.Stack();
        const api = new lib_1.HttpApi(stack, 'HttpApi');
        new lib_1.HttpIntegration(stack, 'HttpInteg', {
            httpApi: api,
            integrationType: lib_1.HttpIntegrationType.HTTP_PROXY,
            integrationUri: 'some-target-url',
        });
        expect(stack).toHaveResource('AWS::ApiGatewayV2::Integration', {
            IntegrationType: 'HTTP_PROXY',
            IntegrationUri: 'some-target-url',
        });
    });
});
describe('CORS', () => {
    test('CORS Configuration is correctly configured.', () => {
        const stack = new core_1.Stack();
        new lib_1.HttpApi(stack, 'HttpApi', {
            corsPreflight: {
                allowCredentials: true,
                allowHeaders: ['Authorization'],
                allowMethods: [lib_1.HttpMethod.GET, lib_1.HttpMethod.HEAD, lib_1.HttpMethod.OPTIONS, lib_1.HttpMethod.POST],
                allowOrigins: ['*'],
                maxAge: core_1.Duration.seconds(36400),
            },
        });
        expect(stack).toHaveResource('AWS::ApiGatewayV2::Api', {
            CorsConfiguration: {
                AllowCredentials: true,
                AllowHeaders: ['Authorization'],
                AllowMethods: ['GET', 'HEAD', 'OPTIONS', 'POST'],
                AllowOrigins: ['*'],
                MaxAge: 36400,
            },
        });
    });
    test('CorsConfiguration is ABSENT when not specified.', () => {
        const stack = new core_1.Stack();
        new lib_1.HttpApi(stack, 'HttpApi');
        expect(stack).toHaveResource('AWS::ApiGatewayV2::Api', {
            CorsConfiguration: assert_1.ABSENT,
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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