"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
// import { expect, haveResource, haveResourceLike } from '@aws-cdk/assert';
const aws_certificatemanager_1 = require("@aws-cdk/aws-certificatemanager");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../../lib");
const domainName = 'example.com';
const certArn = 'arn:aws:acm:us-east-1:111111111111:certificate';
describe('DomainName', () => {
    test('create domain name correctly', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.DomainName(stack, 'DomainName', {
            domainName,
            certificate: aws_certificatemanager_1.Certificate.fromCertificateArn(stack, 'cert', certArn),
        });
        // THEN
        expect(stack).toHaveResource('AWS::ApiGatewayV2::DomainName', {
            DomainName: 'example.com',
            DomainNameConfigurations: [
                {
                    CertificateArn: 'arn:aws:acm:us-east-1:111111111111:certificate',
                    EndpointType: 'REGIONAL',
                },
            ],
        });
    });
    test('import domain name correctly', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const dn = new lib_1.DomainName(stack, 'DomainName', {
            domainName,
            certificate: aws_certificatemanager_1.Certificate.fromCertificateArn(stack, 'cert', certArn),
        });
        // WHEN
        const imported = lib_1.DomainName.fromDomainNameAttributes(stack, 'dn', {
            domainName: dn.domainName,
            regionalDomainName: dn.regionalDomainName,
            regionalHostedZoneId: dn.regionalHostedZoneId,
        });
        // THEN;
        expect(imported.domainName).toEqual(dn.domainName);
        expect(imported.regionalDomainName).toEqual(dn.regionalDomainName);
        expect(imported.regionalHostedZoneId).toEqual(dn.regionalHostedZoneId);
    });
    test('addStage with domainNameMapping', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new lib_1.HttpApi(stack, 'Api', {
            createDefaultStage: true,
        });
        // WHEN
        const dn = new lib_1.DomainName(stack, 'DN', {
            domainName,
            certificate: aws_certificatemanager_1.Certificate.fromCertificateArn(stack, 'cert', certArn),
        });
        api.addStage('beta', {
            stageName: 'beta',
            autoDeploy: true,
            domainMapping: {
                domainName: dn,
                mappingKey: 'beta',
            },
        });
        expect(stack).toHaveResourceLike('AWS::ApiGatewayV2::DomainName', {
            DomainName: 'example.com',
            DomainNameConfigurations: [
                {
                    CertificateArn: 'arn:aws:acm:us-east-1:111111111111:certificate',
                    EndpointType: 'REGIONAL',
                },
            ],
        });
        expect(stack).toHaveResourceLike('AWS::ApiGatewayV2::ApiMapping', {
            ApiId: {
                Ref: 'ApiF70053CD',
            },
            DomainName: 'example.com',
            Stage: 'beta',
            ApiMappingKey: 'beta',
        });
    });
    test('api with defaultDomainMapping', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const dn = new lib_1.DomainName(stack, 'DN', {
            domainName,
            certificate: aws_certificatemanager_1.Certificate.fromCertificateArn(stack, 'cert', certArn),
        });
        // WHEN
        new lib_1.HttpApi(stack, 'Api', {
            createDefaultStage: true,
            defaultDomainMapping: {
                domainName: dn,
                mappingKey: '/',
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::ApiGatewayV2::DomainName', {
            DomainName: 'example.com',
            DomainNameConfigurations: [
                {
                    CertificateArn: 'arn:aws:acm:us-east-1:111111111111:certificate',
                    EndpointType: 'REGIONAL',
                },
            ],
        });
        expect(stack).toHaveResourceLike('AWS::ApiGatewayV2::ApiMapping', {
            ApiId: {
                Ref: 'ApiF70053CD',
            },
            DomainName: 'example.com',
            Stage: '$default',
        });
    });
    test('throws when defaultDomainMapping enabled with createDefaultStage disabled', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const dn = new lib_1.DomainName(stack, 'DN', {
            domainName,
            certificate: aws_certificatemanager_1.Certificate.fromCertificateArn(stack, 'cert', certArn),
        });
        const t = () => {
            new lib_1.HttpApi(stack, 'Api', {
                createDefaultStage: false,
                defaultDomainMapping: {
                    domainName: dn,
                    mappingKey: '/',
                },
            });
        };
        // WHEN/THEN
        expect(t).toThrow('defaultDomainMapping not supported with createDefaultStage disabled');
    });
});
//# sourceMappingURL=data:application/json;base64,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