"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../../lib");
describe('HttpApi', () => {
    test('default', () => {
        const stack = new core_1.Stack();
        const api = new lib_1.HttpApi(stack, 'api');
        expect(stack).toHaveResource('AWS::ApiGatewayV2::Api', {
            Name: 'api',
            ProtocolType: 'HTTP',
        });
        expect(stack).toHaveResource('AWS::ApiGatewayV2::Stage', {
            ApiId: stack.resolve(api.httpApiId),
            StageName: '$default',
            AutoDeploy: true,
        });
        expect(stack).not.toHaveResource('AWS::ApiGatewayV2::Route');
        expect(stack).not.toHaveResource('AWS::ApiGatewayV2::Integration');
        expect(api.url).toBeDefined();
    });
    test('import', () => {
        const stack = new core_1.Stack();
        const api = new lib_1.HttpApi(stack, 'api', { apiName: 'customName' });
        const imported = lib_1.HttpApi.fromApiId(stack, 'imported', api.httpApiId);
        expect(imported.httpApiId).toEqual(api.httpApiId);
    });
    test('unsetting createDefaultStage', () => {
        const stack = new core_1.Stack();
        const api = new lib_1.HttpApi(stack, 'api', {
            createDefaultStage: false,
        });
        expect(stack).not.toHaveResource('AWS::ApiGatewayV2::Stage');
        expect(api.url).toBeUndefined();
    });
    test('default integration', () => {
        const stack = new core_1.Stack();
        const httpApi = new lib_1.HttpApi(stack, 'api', {
            defaultIntegration: new lib_1.LambdaProxyIntegration({
                handler: new aws_lambda_1.Function(stack, 'fn', {
                    code: aws_lambda_1.Code.fromInline('foo'),
                    runtime: aws_lambda_1.Runtime.NODEJS_12_X,
                    handler: 'index.handler',
                }),
            }),
        });
        expect(stack).toHaveResourceLike('AWS::ApiGatewayV2::Route', {
            ApiId: stack.resolve(httpApi.httpApiId),
            RouteKey: '$default',
        });
        expect(stack).toHaveResourceLike('AWS::ApiGatewayV2::Integration', {
            ApiId: stack.resolve(httpApi.httpApiId),
        });
    });
    test('addRoutes() configures the correct routes', () => {
        const stack = new core_1.Stack();
        const httpApi = new lib_1.HttpApi(stack, 'api');
        httpApi.addRoutes({
            path: '/pets',
            methods: [lib_1.HttpMethod.GET, lib_1.HttpMethod.PATCH],
            integration: new lib_1.LambdaProxyIntegration({
                handler: new aws_lambda_1.Function(stack, 'fn', {
                    code: aws_lambda_1.Code.fromInline('foo'),
                    runtime: aws_lambda_1.Runtime.NODEJS_12_X,
                    handler: 'index.handler',
                }),
            }),
        });
        expect(stack).toHaveResourceLike('AWS::ApiGatewayV2::Route', {
            ApiId: stack.resolve(httpApi.httpApiId),
            RouteKey: 'GET /pets',
        });
        expect(stack).toHaveResourceLike('AWS::ApiGatewayV2::Route', {
            ApiId: stack.resolve(httpApi.httpApiId),
            RouteKey: 'PATCH /pets',
        });
    });
    test('addRoutes() creates the default method', () => {
        const stack = new core_1.Stack();
        const httpApi = new lib_1.HttpApi(stack, 'api');
        httpApi.addRoutes({
            path: '/pets',
            integration: new lib_1.LambdaProxyIntegration({
                handler: new aws_lambda_1.Function(stack, 'fn', {
                    code: aws_lambda_1.Code.fromInline('foo'),
                    runtime: aws_lambda_1.Runtime.NODEJS_12_X,
                    handler: 'index.handler',
                }),
            }),
        });
        expect(stack).toHaveResourceLike('AWS::ApiGatewayV2::Route', {
            ApiId: stack.resolve(httpApi.httpApiId),
            RouteKey: 'ANY /pets',
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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