"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.HttpStage = void 0;
const core_1 = require("@aws-cdk/core");
const apigatewayv2_generated_1 = require("../apigatewayv2.generated");
const api_mapping_1 = require("./api-mapping");
const DEFAULT_STAGE_NAME = '$default';
/**
 * Represents a stage where an instance of the API is deployed.
 * @resource AWS::ApiGatewayV2::Stage
 */
class HttpStage extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.stageName ? props.stageName : DEFAULT_STAGE_NAME,
        });
        new apigatewayv2_generated_1.CfnStage(this, 'Resource', {
            apiId: props.httpApi.httpApiId,
            stageName: this.physicalName,
            autoDeploy: props.autoDeploy,
        });
        this.stageName = this.physicalName;
        this.httpApi = props.httpApi;
        if (props.domainMapping) {
            new api_mapping_1.HttpApiMapping(this, `${props.domainMapping.domainName}${props.domainMapping.mappingKey}`, {
                api: props.httpApi,
                domainName: props.domainMapping.domainName,
                stage: this,
                apiMappingKey: props.domainMapping.mappingKey,
            });
        }
    }
    /**
     * Import an existing stage into this CDK app.
     */
    static fromStageName(scope, id, stageName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.stageName = stageName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * The URL to this stage.
     */
    get url() {
        const s = core_1.Stack.of(this);
        const urlPath = this.stageName === DEFAULT_STAGE_NAME ? '' : this.stageName;
        return `https://${this.httpApi.httpApiId}.execute-api.${s.region}.${s.urlSuffix}/${urlPath}`;
    }
}
exports.HttpStage = HttpStage;
//# sourceMappingURL=data:application/json;base64,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