"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.HttpRoute = exports.HttpRouteKey = exports.HttpMethod = void 0;
const core_1 = require("@aws-cdk/core");
const apigatewayv2_generated_1 = require("../apigatewayv2.generated");
const integration_1 = require("./integration");
/**
 * Supported HTTP methods
 */
var HttpMethod;
(function (HttpMethod) {
    /** HTTP ANY */
    HttpMethod["ANY"] = "ANY";
    /** HTTP DELETE */
    HttpMethod["DELETE"] = "DELETE";
    /** HTTP GET */
    HttpMethod["GET"] = "GET";
    /** HTTP HEAD */
    HttpMethod["HEAD"] = "HEAD";
    /** HTTP OPTIONS */
    HttpMethod["OPTIONS"] = "OPTIONS";
    /** HTTP PATCH */
    HttpMethod["PATCH"] = "PATCH";
    /** HTTP POST */
    HttpMethod["POST"] = "POST";
    /** HTTP PUT */
    HttpMethod["PUT"] = "PUT";
})(HttpMethod = exports.HttpMethod || (exports.HttpMethod = {}));
/**
 * HTTP route in APIGateway is a combination of the HTTP method and the path component.
 * This class models that combination.
 */
class HttpRouteKey {
    constructor(key, path) {
        this.key = key;
        this.path = path;
    }
    /**
     * Create a route key with the combination of the path and the method.
     * @param method default is 'ANY'
     */
    static with(path, method) {
        if (path !== '/' && (!path.startsWith('/') || path.endsWith('/'))) {
            throw new Error('path must always start with a "/" and not end with a "/"');
        }
        return new HttpRouteKey(`${method !== null && method !== void 0 ? method : 'ANY'} ${path}`, path);
    }
}
exports.HttpRouteKey = HttpRouteKey;
/**
 * The catch-all route of the API, i.e., when no other routes match
 */
HttpRouteKey.DEFAULT = new HttpRouteKey('$default');
/**
 * Route class that creates the Route for API Gateway HTTP API
 * @resource AWS::ApiGatewayV2::Route
 */
class HttpRoute extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.httpApi = props.httpApi;
        this.path = props.routeKey.path;
        let integration;
        const config = props.integration.bind(this);
        integration = new integration_1.HttpIntegration(this, `${this.node.id}-Integration`, {
            httpApi: props.httpApi,
            integrationType: config.type,
            integrationUri: config.uri,
            method: config.method,
            payloadFormatVersion: config.payloadFormatVersion,
        });
        const routeProps = {
            apiId: props.httpApi.httpApiId,
            routeKey: props.routeKey.key,
            target: `integrations/${integration.integrationId}`,
        };
        const route = new apigatewayv2_generated_1.CfnRoute(this, 'Resource', routeProps);
        this.routeId = route.ref;
    }
}
exports.HttpRoute = HttpRoute;
//# sourceMappingURL=data:application/json;base64,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