"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.HttpApi = void 0;
const core_1 = require("@aws-cdk/core");
const apigatewayv2_generated_1 = require("../apigatewayv2.generated");
const route_1 = require("./route");
const stage_1 = require("./stage");
/**
 * Create a new API Gateway HTTP API endpoint.
 * @resource AWS::ApiGatewayV2::Api
 */
class HttpApi extends core_1.Resource {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        const apiName = (_a = props === null || props === void 0 ? void 0 : props.apiName) !== null && _a !== void 0 ? _a : id;
        let corsConfiguration;
        if (props === null || props === void 0 ? void 0 : props.corsPreflight) {
            const { allowCredentials, allowHeaders, allowMethods, allowOrigins, exposeHeaders, maxAge, } = props.corsPreflight;
            corsConfiguration = {
                allowCredentials,
                allowHeaders,
                allowMethods,
                allowOrigins,
                exposeHeaders,
                maxAge: maxAge === null || maxAge === void 0 ? void 0 : maxAge.toSeconds(),
            };
        }
        const apiProps = {
            name: apiName,
            protocolType: 'HTTP',
            corsConfiguration,
        };
        const resource = new apigatewayv2_generated_1.CfnApi(this, 'Resource', apiProps);
        this.httpApiId = resource.ref;
        if (props === null || props === void 0 ? void 0 : props.defaultIntegration) {
            new route_1.HttpRoute(this, 'DefaultRoute', {
                httpApi: this,
                routeKey: route_1.HttpRouteKey.DEFAULT,
                integration: props.defaultIntegration,
            });
        }
        if ((props === null || props === void 0 ? void 0 : props.createDefaultStage) === undefined || props.createDefaultStage === true) {
            this.defaultStage = new stage_1.HttpStage(this, 'DefaultStage', {
                httpApi: this,
                autoDeploy: true,
                domainMapping: props === null || props === void 0 ? void 0 : props.defaultDomainMapping,
            });
        }
        if ((props === null || props === void 0 ? void 0 : props.createDefaultStage) === false && props.defaultDomainMapping) {
            throw new Error('defaultDomainMapping not supported with createDefaultStage disabled');
        }
    }
    /**
     * Import an existing HTTP API into this CDK app.
     */
    static fromApiId(scope, id, httpApiId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.httpApiId = httpApiId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Get the URL to the default stage of this API.
     * Returns `undefined` if `createDefaultStage` is unset.
     */
    get url() {
        return this.defaultStage ? this.defaultStage.url : undefined;
    }
    /**
     * Add a new stage.
     */
    addStage(id, options) {
        const stage = new stage_1.HttpStage(this, id, {
            httpApi: this,
            ...options,
        });
        return stage;
    }
    /**
     * Add multiple routes that uses the same configuration. The routes all go to the same path, but for different
     * methods.
     */
    addRoutes(options) {
        var _a;
        const methods = (_a = options.methods) !== null && _a !== void 0 ? _a : [route_1.HttpMethod.ANY];
        return methods.map((method) => new route_1.HttpRoute(this, `${method}${options.path}`, {
            httpApi: this,
            routeKey: route_1.HttpRouteKey.with(options.path, method),
            integration: options.integration,
        }));
    }
}
exports.HttpApi = HttpApi;
//# sourceMappingURL=data:application/json;base64,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