"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Filters = exports.verifyFilterable = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const constructs_1 = require("constructs");
const memoize = require("lodash.memoize"); // eslint-disable-line @typescript-eslint/no-require-imports
const core_1 = require("../core");
const types_1 = require("./types");
/**
 * Verify that store is filterable, meaning it allows destructive mutations.
 * @throws Error if store is not filterable
 */
function verifyFilterable(store) {
    if (!store.allowDestructiveMutations) {
        throw new Error("Store must allow destructive mutations to perform filtering; clone the store before applying filters using `store.clone(true)` operation and passing the cloned store to filtering operation.");
    }
}
exports.verifyFilterable = verifyFilterable;
var Filters;
(function (Filters) {
    /**
     * Prune **extraneous** nodes and edges
     * @throws Error if store is not filterable
     * @destructive
     */
    function pruneExtraneous() {
        return (store) => {
            verifyFilterable(store);
            const extraneousNodes = store.root.findAll({
                order: constructs_1.ConstructOrder.POSTORDER,
                predicate: (node) => node.isExtraneous,
            });
            // collapse all extraneous nodes to nearest non-extraneous parent, or prune the node
            for (const extraneousNode of extraneousNodes) {
                const nonExtraneousAncestor = extraneousNode.findAncestor((node) => !node.isExtraneous);
                if (nonExtraneousAncestor && !nonExtraneousAncestor.isGraphContainer) {
                    extraneousNode.mutateCollapseTo(nonExtraneousAncestor);
                }
                else {
                    extraneousNode.mutateDestroy();
                }
            }
            store.edges.forEach((edge) => {
                if (edge.isExtraneous) {
                    edge.mutateDestroy();
                }
            });
        };
    }
    Filters.pruneExtraneous = pruneExtraneous;
    /**
     * Collapses extraneous nodes to parent and cdk created nodes on themselves,
     * and prunes extraneous edges.
     *
     * This most closely represents the developers code for the current application
     * and reduces the noise one expects.
     *
     * @throws Error if store is not filterable
     * @destructive
     */
    function compact() {
        return (store) => {
            verifyFilterable(store);
            pruneExtraneous()(store);
            const cdkOwnedContainers = store.root.findAll({
                order: constructs_1.ConstructOrder.POSTORDER,
                predicate: (node) => node.hasFlag(core_1.FlagEnum.CDK_OWNED) &&
                    !node.parent?.hasFlag(core_1.FlagEnum.CDK_OWNED),
            });
            // collapse all cdk owned containers
            // NB: collapses the graph more closely mirror what developer writes, not what is auto created by cdk
            for (const cdkOwnedContainer of cdkOwnedContainers) {
                cdkOwnedContainer.mutateCollapse();
            }
            const cdkResources = store.root.findAll({
                order: constructs_1.ConstructOrder.POSTORDER,
                predicate: (node) => core_1.Graph.ResourceNode.isResourceNode(node),
            });
            // collapse all cfnResource wrapped by cdk resource
            for (const cdkResource of cdkResources) {
                if (cdkResource.isResourceWrapper) {
                    cdkResource.mutateCollapse();
                }
                else if (cdkResource.cfnResource) {
                    cdkResource.cfnResource.mutateCollapseToParent();
                }
            }
        };
    }
    Filters.compact = compact;
    function _filterNodeType(values, exclude) {
        const isMatch = memoize((input) => {
            if (input == null) {
                return false;
            }
            return (values.find((_value) => {
                if (typeof _value === "string") {
                    return input === _value;
                }
                return _value.test(input);
            }) != null);
        });
        return (store) => {
            for (const node of store.root.findAll({
                order: constructs_1.ConstructOrder.POSTORDER,
            })) {
                if (isMatch(node.nodeType) === exclude) {
                    if (node.isLeaf) {
                        node.mutateCollapseToParent();
                    }
                    else {
                        node.mutateUncluster();
                    }
                }
            }
        };
    }
    /**
     * Prune all {@link Graph.Node}s *except those matching* specified list.
     *
     * This filter targets all nodes (except root) - {@link IGraphFilter.allNodes}
     * @throws Error if store is not filterable
     * @destructive
     */
    function includeNodeType(nodeTypes) {
        return _filterNodeType(nodeTypes, false);
    }
    Filters.includeNodeType = includeNodeType;
    /**
     * Prune all {@link Graph.Node}s *matching* specified list.
     *
     * This filter targets all nodes (except root) - {@link IGraphFilter.allNodes}
     * @throws Error if store is not filterable
     * @destructive
     */
    function excludeNodeType(nodeTypes) {
        return _filterNodeType(nodeTypes, true);
    }
    Filters.excludeNodeType = excludeNodeType;
    function _filterCfnType(values, exclude) {
        const isMatch = memoize((input) => {
            if (input == null) {
                return false;
            }
            return (values.find((_value) => {
                if (typeof _value === "string") {
                    return input === _value;
                }
                return _value.test(input);
            }) != null);
        });
        return {
            strategy: types_1.FilterStrategy.PRUNE,
            node: (node) => {
                // Preserve container structure (stages, stacks, etc.)
                if (node.isCluster || node.isGraphContainer)
                    return true;
                if (core_1.Graph.CfnResourceNode.isCfnResourceNode(node) ||
                    core_1.Graph.ResourceNode.isResourceNode(node)) {
                    const match = !!node.cfnType && isMatch(node.cfnType);
                    return (match && !exclude) || (!match && exclude);
                }
                // Preserve non *Resource nodes
                return true;
            },
        };
    }
    /**
     * Prune all {@link Graph.ResourceNode} and {@link Graph.CfnResourceNode} nodes
     * *except those matching* specified list of CloudFormation types.
     * @throws Error if store is not filterable
     * @destructive
     */
    function includeCfnType(cfnTypes) {
        return _filterCfnType(cfnTypes, false);
    }
    Filters.includeCfnType = includeCfnType;
    /**
     * Prune all {@link Graph.ResourceNode} and {@link Graph.CfnResourceNode} nodes
     * *matching* specified list of CloudFormation types.
     * @throws Error if store is not filterable
     * @destructive
     */
    function excludeCfnType(cfnTypes) {
        return _filterCfnType(cfnTypes, true);
    }
    Filters.excludeCfnType = excludeCfnType;
    /**
     * Remove clusters by hoisting their children to the parent of the cluster
     * and collapsing the cluster itself to its parent.
     * @param clusterTypes
     * @throws Error if store is not filterable
     * @see {@link Graph.Node.mutateUncluster}
     * @destructive
     */
    function uncluster(clusterTypes) {
        // Use set for constant lookup
        const clusterTypesSet = new Set(clusterTypes);
        return (store) => {
            verifyFilterable(store);
            const clusters = store.root.findAll({
                predicate: (node) => {
                    if (node.isGraphContainer)
                        return false;
                    if (clusterTypesSet.size === 0)
                        return node.isCluster;
                    return clusterTypesSet.has(node.nodeType);
                },
            });
            for (const cluster of clusters) {
                cluster.mutateUncluster();
            }
        };
    }
    Filters.uncluster = uncluster;
})(Filters = exports.Filters || (exports.Filters = {}));
//# sourceMappingURL=data:application/json;base64,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