import { Graph, NodeTypeEnum } from "../core";
import { IGraphFilter, IGraphStoreFilter } from "./types";
/**
 * Verify that store is filterable, meaning it allows destructive mutations.
 * @throws Error if store is not filterable
 */
export declare function verifyFilterable(store: Graph.Store): void;
export declare namespace Filters {
    /**
     * Prune **extraneous** nodes and edges
     * @throws Error if store is not filterable
     * @destructive
     */
    function pruneExtraneous(): IGraphStoreFilter;
    /**
     * Collapses extraneous nodes to parent and cdk created nodes on themselves,
     * and prunes extraneous edges.
     *
     * This most closely represents the developers code for the current application
     * and reduces the noise one expects.
     *
     * @throws Error if store is not filterable
     * @destructive
     */
    function compact(): IGraphStoreFilter;
    /**
     * Prune all {@link Graph.Node}s *except those matching* specified list.
     *
     * This filter targets all nodes (except root) - {@link IGraphFilter.allNodes}
     * @throws Error if store is not filterable
     * @destructive
     */
    function includeNodeType(nodeTypes: (string | RegExp)[]): IGraphStoreFilter;
    /**
     * Prune all {@link Graph.Node}s *matching* specified list.
     *
     * This filter targets all nodes (except root) - {@link IGraphFilter.allNodes}
     * @throws Error if store is not filterable
     * @destructive
     */
    function excludeNodeType(nodeTypes: (string | RegExp)[]): IGraphStoreFilter;
    /**
     * Prune all {@link Graph.ResourceNode} and {@link Graph.CfnResourceNode} nodes
     * *except those matching* specified list of CloudFormation types.
     * @throws Error if store is not filterable
     * @destructive
     */
    function includeCfnType(cfnTypes: (string | RegExp)[]): IGraphFilter;
    /**
     * Prune all {@link Graph.ResourceNode} and {@link Graph.CfnResourceNode} nodes
     * *matching* specified list of CloudFormation types.
     * @throws Error if store is not filterable
     * @destructive
     */
    function excludeCfnType(cfnTypes: (string | RegExp)[]): IGraphFilter;
    /**
     * Remove clusters by hoisting their children to the parent of the cluster
     * and collapsing the cluster itself to its parent.
     * @param clusterTypes
     * @throws Error if store is not filterable
     * @see {@link Graph.Node.mutateUncluster}
     * @destructive
     */
    function uncluster(clusterTypes?: NodeTypeEnum[]): IGraphStoreFilter;
}
