"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.inferFlags = exports.extractUnresolvedReferences = exports.IGNORE_REF_PATTERN = exports.extractInspectableAttributes = exports.inferNodeProps = exports.tryGetLogicalId = exports.getConstructUUID = exports.generateConsistentUUID = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cloneDeep = require("lodash.clonedeep"); // eslint-disable-line @typescript-eslint/no-require-imports
const shorthash = require("shorthash2"); // eslint-disable-line @typescript-eslint/no-require-imports
const traverse = require("traverse"); // eslint-disable-line @typescript-eslint/no-require-imports
const cdk_internals_1 = require("../cdk-internals");
const types_1 = require("./types");
/**
 * Generate deterministic UUID based on given value and prefix.
 * @param value The value to hash as UUID
 * @param {string} [prefix=""] Optional prefix used to prevent value conflicts
 */
function generateConsistentUUID(value, prefix = "") {
    return prefix + shorthash(JSON.stringify(value));
}
exports.generateConsistentUUID = generateConsistentUUID;
/** Get UUID for a given construct */
function getConstructUUID(construct) {
    return aws_cdk_lib_1.Names.uniqueResourceName(construct, {});
}
exports.getConstructUUID = getConstructUUID;
/** Try to get *logicalId* for given construct */
function tryGetLogicalId(construct) {
    if (aws_cdk_lib_1.CfnElement.isCfnElement(construct)) {
        const stack = aws_cdk_lib_1.Stack.of(construct);
        return stack.resolve(stack.getLogicalId(construct));
    }
    return undefined;
}
exports.tryGetLogicalId = tryGetLogicalId;
/** Infer node props from construct */
function inferNodeProps(construct) {
    const uuid = getConstructUUID(construct);
    const logicalId = tryGetLogicalId(construct);
    const metadata = construct.node.metadata.filter((entry) => {
        if (entry.type === types_1.MetadataTypeEnum.LOGICAL_ID)
            return false;
        return true;
    });
    const attributes = cloneDeep(extractInspectableAttributes(construct) || {});
    const cfnType = attributes[types_1.CfnAttributesEnum.TYPE];
    if (cfnType) {
        // @ts-ignore
        delete attributes[types_1.CfnAttributesEnum.TYPE];
    }
    const cfnProps = attributes[types_1.CfnAttributesEnum.PROPS] || {};
    let tags = {};
    // normalize tags
    if (typeof cfnProps === "object" && "tags" in cfnProps) {
        const _tags = cfnProps.tags;
        // remove the tags from the attributes since we normalize
        // @ts-ignore
        delete cfnProps.tags;
        if (Array.isArray(_tags)) {
            tags = Object.fromEntries(_tags.map(({ key, value }) => [key, value]));
        }
        else {
            tags = _tags;
        }
    }
    const constructInfo = cdk_internals_1.constructInfoFromConstruct(construct);
    const flags = inferFlags(construct, constructInfo);
    return {
        uuid,
        attributes,
        metadata,
        tags,
        logicalId,
        cfnType,
        constructInfo,
        dependencies: construct.node.dependencies.map(getConstructUUID),
        unresolvedReferences: extractUnresolvedReferences(uuid, attributes),
        flags,
    };
}
exports.inferNodeProps = inferNodeProps;
/** Extract inspectable attributes from construct */
function extractInspectableAttributes(construct) {
    // check if a construct implements IInspectable
    function canInspect(inspectable) {
        return inspectable.inspect !== undefined;
    }
    const inspector = new aws_cdk_lib_1.TreeInspector();
    // get attributes from the inspector
    if (canInspect(construct)) {
        construct.inspect(inspector);
        return aws_cdk_lib_1.Stack.of(construct).resolve(inspector.attributes);
    }
    return undefined;
}
exports.extractInspectableAttributes = extractInspectableAttributes;
/** Pattern of ignored references. Those which are resolved during deploy-time. */
exports.IGNORE_REF_PATTERN = /^AWS::/;
/** Extract unresolved references from attributes for a given source */
function extractUnresolvedReferences(source, from) {
    const references = [];
    traverse(from).forEach(function () {
        switch (this.key) {
            case types_1.ReferenceTypeEnum.ATTRIBUTE: {
                const [logicalId, attribute] = this.node;
                references.push({
                    source,
                    referenceType: types_1.ReferenceTypeEnum.ATTRIBUTE,
                    target: logicalId,
                    value: attribute,
                });
                break;
            }
            case types_1.ReferenceTypeEnum.REF: {
                if (typeof this.node === "string") {
                    if (!exports.IGNORE_REF_PATTERN.test(this.node)) {
                        references.push({
                            source,
                            referenceType: types_1.ReferenceTypeEnum.REF,
                            target: this.node,
                        });
                    }
                }
                else {
                    console.warn(`Found non-string "Ref"`, this.node);
                }
                break;
            }
            case types_1.ReferenceTypeEnum.IMPORT: {
                // "Fn::ImportValue": "Ada:ExportsOutputFnGetAttCommonStackA8F9EE77OutputsAdaCommonStackCounterTable5D6ADA16ArnED1AF27F"
                // "Fn::ImportValue": "Stage-Ada:ExportsOutputFnGetAttCommonStackA8F9EE77OutputsAdaCommonStackCounterTable5D6ADA16ArnED1AF27F"
                references.push({
                    source,
                    referenceType: types_1.ReferenceTypeEnum.IMPORT,
                    // NB: remove stage - separator
                    target: this.node.replace("-", ""),
                });
                break;
            }
        }
    });
    return references;
}
exports.extractUnresolvedReferences = extractUnresolvedReferences;
/** Infer construct flags  */
function inferFlags(construct, constructInfo) {
    const flags = new Set();
    const fqn = constructInfo?.fqn;
    if (fqn && types_1.ExtraneousFqnEnum.includes(fqn)) {
        flags.add(types_1.FlagEnum.EXTRANEOUS);
    }
    if (fqn && types_1.AssetFqnEnum.includes(fqn)) {
        flags.add(types_1.FlagEnum.ASSET);
    }
    if (construct.node.children.length === 1) {
        const onlyChild = construct.node.children[0];
        if ([types_1.CdkConstructIds.RESOURCE, types_1.CdkConstructIds.DEFAULT].includes(onlyChild.node.id)) {
            flags.add(types_1.FlagEnum.RESOURCE_WRAPPER);
        }
    }
    if (construct.node.id === "Exports" && aws_cdk_lib_1.Stack.isStack(construct.node.scope)) {
        flags.add(types_1.FlagEnum.EXTRANEOUS);
    }
    if (construct.node.id.startsWith("SsmParameterValue:")) {
        flags.add(types_1.FlagEnum.EXTRANEOUS);
    }
    return Array.from(flags.values());
}
exports.inferFlags = inferFlags;
//# sourceMappingURL=data:application/json;base64,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