"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.computeGraph = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const aws_cdk_lib_1 = require("aws-cdk-lib");
const merge = require("lodash.merge"); // eslint-disable-line @typescript-eslint/no-require-imports
const uniq = require("lodash.uniq"); // eslint-disable-line @typescript-eslint/no-require-imports
const uniqBy = require("lodash.uniqby"); // eslint-disable-line @typescript-eslint/no-require-imports
const constants_1 = require("./constants");
const graph_1 = require("./graph");
const types_1 = require("./types");
const utils_1 = require("./utils");
/** List of cdk stack children used by cdk that the graph ignores */
const IGNORED_STACK_CHILDREN = [
    "CDKMetadata",
    "BootstrapVersion",
    "CheckBootstrapVersion",
];
/**
 * Computes the graph store for a given Cdk construct, usually App.
 * @internal
 */
function computeGraph(root) {
    const store = new graph_1.Graph.Store();
    // List of nodes that should not be processed (extraneous nodes detected during compute)
    // NB: example is stack template construct which cdk creates as sibling of nested stacks,
    // we care about the stack and not the marshalled construct.
    const nodesToIgnore = [];
    // List of all unresolved referenced detected during compute, which are resolved after all nodes have been stored
    const allUnresolvedReferences = [];
    // List of all unresolved dependencies detected during compute, which are resolved after all nodes have been stored
    const allUnresolvedDependencies = [];
    const visit = (construct, parent) => {
        // Do not graph the CdkGraph itself
        if (construct.node.id === constants_1.GRAPH_ID) {
            return;
        }
        // Do not graph the Tree construct (synthesizes tree.json)
        if (graph_1.Graph.AppNode.isAppNode(parent) && construct.node.id === "Tree") {
            return;
        }
        // Do not graph stack CDKMetadata, BootstrapVersion, and similar constructs
        if (graph_1.Graph.StackNode.isStackNode(parent) &&
            IGNORED_STACK_CHILDREN.includes(construct.node.id)) {
            return;
        }
        const { uuid, attributes = {}, metadata = [], tags = {}, logicalId, cfnType, constructInfo, dependencies, unresolvedReferences, flags, } = utils_1.inferNodeProps(construct);
        if (nodesToIgnore.includes(uuid)) {
            return;
        }
        // Infer the stack this construct belongs to
        let stack;
        try {
            if (construct.node.scope) {
                const stackUUID = utils_1.getConstructUUID(aws_cdk_lib_1.Stack.of(construct));
                stack = store.getStack(stackUUID);
            }
        }
        catch {
            // ignore - expected to throw if construct is not contained in a stack
        }
        const nodeProps = {
            store,
            stack,
            parent,
            uuid,
            id: construct.node.id,
            path: construct.node.path,
            attributes,
            metadata,
            tags,
            constructInfo,
            logicalId,
            cfnType,
            flags,
        };
        let node;
        switch (constructInfo?.fqn) {
            case types_1.ConstructInfoFqnEnum.APP: {
                node = new graph_1.Graph.AppNode({
                    store,
                    parent,
                    attributes,
                    metadata,
                    tags,
                    constructInfo,
                    logicalId,
                    flags,
                });
                break;
            }
            case types_1.ConstructInfoFqnEnum.STAGE: {
                node = new graph_1.Graph.StageNode(nodeProps);
                break;
            }
            case types_1.ConstructInfoFqnEnum.STACK: {
                node = new graph_1.Graph.StackNode(nodeProps);
                break;
            }
            case types_1.ConstructInfoFqnEnum.NESTED_STACK: {
                // NB: handle NestedStack<->CfnStack as single Node with NestedStack construct as source
                // https://github.com/aws/aws-cdk/blob/119c92f65bf26c3fdf4bb818a4a4812022a3744a/packages/%40aws-cdk/core/lib/nested-stack.ts#L119
                const parentStack = utils_1.inferNodeProps(construct.nestedStackParent);
                const _nestedStack = construct.node.scope.node.findChild(construct.node.id + ".NestedStack");
                const cfnStackWrapper = utils_1.inferNodeProps(_nestedStack);
                const cfnStack = utils_1.inferNodeProps(_nestedStack.node.findChild(construct.node.id + ".NestedStackResource"));
                // ignore parent scope cfn stack (template) constructs
                nodesToIgnore.push(cfnStackWrapper.uuid, cfnStackWrapper.uuid);
                node = new graph_1.Graph.NestedStackNode({
                    ...nodeProps,
                    logicalId: cfnStack.logicalId,
                    attributes: merge({}, attributes, cfnStackWrapper.attributes, cfnStack.attributes),
                    metadata: [
                        ...metadata,
                        ...(cfnStackWrapper.metadata || []),
                        ...(cfnStack.metadata || []),
                    ],
                    parentStack: store.getStack(parentStack.uuid),
                });
                // Only add uniq dependencies as wrapper and stack may duplicate
                dependencies.push(...uniq([...cfnStackWrapper.dependencies, ...cfnStack.dependencies]));
                // Only add uniq references as wrapper and stack may duplicate
                unresolvedReferences.push(...uniqBy([
                    ...cfnStackWrapper.unresolvedReferences.map((ref) => ({
                        ...ref,
                        source: uuid,
                    })),
                    ...cfnStack.unresolvedReferences.map((ref) => ({
                        ...ref,
                        source: uuid,
                    })),
                ], (v) => `${v.referenceType}::${v.source}::${v.target}::${JSON.stringify(v.value || "")}`));
                break;
            }
            case types_1.ConstructInfoFqnEnum.CFN_STACK: {
                // CfnStack always proceeds NestedStack, based on above case we merge CfnStack into
                // NestedStack (mirror developer expectations) and then ignore CfnStack.
                throw new Error(`CfnStack should be ignored by NestedStack: ${uuid}`);
            }
            case types_1.ConstructInfoFqnEnum.CFN_OUTPUT: {
                node = new graph_1.Graph.OutputNode({
                    ...nodeProps,
                    value: aws_cdk_lib_1.Stack.of(construct).resolve(construct.value),
                    description: aws_cdk_lib_1.Stack.of(construct).resolve(construct.description),
                    exportName: aws_cdk_lib_1.Stack.of(construct).resolve(construct.exportName),
                });
                // extract unresolved references from value
                unresolvedReferences.push(...utils_1.extractUnresolvedReferences(node.uuid, node.value || {}));
                break;
            }
            case types_1.ConstructInfoFqnEnum.CFN_PARAMETER: {
                const cfnParameter = construct;
                node = new graph_1.Graph.ParameterNode({
                    ...nodeProps,
                    value: aws_cdk_lib_1.Stack.of(construct).resolve(cfnParameter.value),
                    description: aws_cdk_lib_1.Stack.of(construct).resolve(cfnParameter.description),
                    parameterType: cfnParameter.type,
                });
                break;
            }
            default: {
                if (aws_cdk_lib_1.Resource.isResource(construct)) {
                    node = new graph_1.Graph.ResourceNode({
                        ...nodeProps,
                        cdkOwned: aws_cdk_lib_1.Resource.isOwnedResource(construct),
                    });
                }
                else if (cfnType) {
                    node = new graph_1.Graph.CfnResourceNode(nodeProps);
                    const ancestralResourceNode = node.findAncestor(graph_1.Graph.ResourceNode.isResourceNode, 2) || parent;
                    if (graph_1.Graph.ResourceNode.isResourceNode(ancestralResourceNode) &&
                        node.isEquivalentFqn(ancestralResourceNode)) {
                        node.addFlag(types_1.FlagEnum.WRAPPED_CFN_RESOURCE);
                        node.addFlag(types_1.FlagEnum.EXTRANEOUS);
                    }
                }
                else {
                    node = new graph_1.Graph.Node({
                        nodeType: types_1.NodeTypeEnum.DEFAULT,
                        ...nodeProps,
                    });
                    // Cdk Stack.Exports is proxy to actual Cfn exports and extraneous in the graph
                    if (construct.node.id === types_1.CdkConstructIds.EXPORTS &&
                        aws_cdk_lib_1.Stack.isStack(construct.node.scope)) {
                        node.addFlag(types_1.FlagEnum.EXTRANEOUS);
                    }
                }
            }
        }
        // Track unresolved dependencies, since nodes might not be created in the store yet
        allUnresolvedDependencies.push(...dependencies.map((dep) => [uuid, dep]));
        // Track all logicalId references in the nodes attributes
        // this will get resolved after all nodes have been stored
        allUnresolvedReferences.push(...unresolvedReferences);
        // Visit all child to compute the tree
        for (const child of construct.node.children) {
            try {
                visit(child, node);
            }
            catch (e) {
                aws_cdk_lib_1.Annotations.of(root).addWarning(`Failed to render graph for node ${child.node.path}. Reason: ${e}`);
                throw e;
            }
        }
    };
    visit(root, store.root);
    // Resolve all references - now that the tree is stored
    for (const unresolved of allUnresolvedReferences) {
        try {
            resolveReference(store, unresolved);
        }
        catch (e) {
            console.warn(e, unresolved);
            // TODO: consider saving unresolved references if become valuable.
        }
    }
    // Resolve all dependencies - now that the tree is stored
    for (const unresolved of allUnresolvedDependencies) {
        resolveDependency(store, unresolved);
    }
    return store;
}
exports.computeGraph = computeGraph;
/**
 * Resolve reference. During initial graph traversal not all nodes have been added at the time
 * a reference has been detected, as such we need to resolve all references after the graph tree
 * has been stored.
 * @internal
 */
function resolveReference(store, unresolved) {
    const source = store.getNode(unresolved.source);
    if (source.stack == null) {
        console.warn(String(source), source);
        throw new Error(`Node ${source} is not within stack`);
    }
    let target;
    switch (unresolved.referenceType) {
        case types_1.ReferenceTypeEnum.REF: {
            // ref logicalId is only unique in the stack
            target = store.findNodeByLogicalId(source.stack, unresolved.target);
            return new graph_1.Graph.Reference({
                store,
                uuid: utils_1.generateConsistentUUID(unresolved, graph_1.Graph.Reference.PREFIX),
                referenceType: types_1.ReferenceTypeEnum.REF,
                source,
                target,
            });
        }
        case types_1.ReferenceTypeEnum.IMPORT: {
            // imports already contain the stack id (stack:logicalId)
            target = store.findNodeByLogicalUniversalId(unresolved.target);
            return new graph_1.Graph.ImportReference({
                store,
                uuid: utils_1.generateConsistentUUID(unresolved, graph_1.Graph.ImportReference.PREFIX),
                source,
                target,
            });
        }
        case types_1.ReferenceTypeEnum.ATTRIBUTE: {
            const attribute = unresolved.value;
            if (attribute && attribute.startsWith("Outputs.")) {
                // Stack output reference
                const stacksToSearch = source.rootStack?.stage?.stacks || store.stacks;
                const potentialRefStacks = Object.values(stacksToSearch).filter((_stack) => _stack.logicalId === unresolved.target);
                if (potentialRefStacks.length === 1) {
                    const refStack = potentialRefStacks[0];
                    target = refStack.findOutput(attribute.replace("Outputs.", ""));
                }
                else {
                    console.warn("Failed to find logical id from attribute reference:", unresolved.target);
                    if (potentialRefStacks.length) {
                        console.warn("Found multiple matching stacks:", Object.values(potentialRefStacks).map((stack) => `${String(stack)}:${stack.logicalId || "ROOT"}`));
                    }
                    else {
                        console.warn("Available stacks:", Object.values(store.stacks).map((stack) => `${String(stack)}:${stack.logicalId || "ROOT"}`));
                    }
                    throw new Error(`Failed to find Fn::GetAtt stack for output reference "${unresolved.target}": ${potentialRefStacks.length}`);
                }
            }
            else {
                target = store.findNodeByLogicalId(source.stack, unresolved.target);
            }
            if (target) {
                return new graph_1.Graph.AttributeReference({
                    store,
                    uuid: utils_1.generateConsistentUUID(unresolved, graph_1.Graph.AttributeReference.PREFIX),
                    source,
                    target,
                    value: attribute,
                });
            }
        }
    }
    throw new Error(`Failed to resolve reference: ${JSON.stringify(unresolved)}`);
}
/**
 * Resolve dependency. During initial graph traversal not all nodes have been added at the time
 * a dependency has been detected, as such we need to resolve all dependencies after the graph tree
 * has been stored.
 * @internal
 */
function resolveDependency(store, unresolved) {
    const source = store.getNode(unresolved[0]);
    const target = store.getNode(unresolved[1]);
    return new graph_1.Graph.Dependency({
        store,
        uuid: utils_1.generateConsistentUUID(unresolved, graph_1.Graph.Dependency.PREFIX),
        source,
        target,
    });
}
//# sourceMappingURL=data:application/json;base64,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