"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CdkGraph = exports.CdkGraphContext = exports.CdkGraphArtifacts = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const chalk = require("chalk"); // eslint-disable-line @typescript-eslint/no-require-imports
const constructs_1 = require("constructs");
const fs = require("fs-extra");
const cdk_internals_1 = require("./cdk-internals");
const config_1 = require("./config");
const core_1 = require("./core");
const GRAPH_ARTIFACT_ID = "GRAPH";
/** CdkGraph core artifacts */
var CdkGraphArtifacts;
(function (CdkGraphArtifacts) {
    CdkGraphArtifacts["GRAPH_METADATA"] = "graph-metadata.json";
    CdkGraphArtifacts["GRAPH"] = "graph.json";
})(CdkGraphArtifacts = exports.CdkGraphArtifacts || (exports.CdkGraphArtifacts = {}));
/** CdkGraph context */
class CdkGraphContext {
    constructor(store, outdir) {
        this.store = store;
        this.outdir = outdir;
        /** @internal */
        this._artifacts = {};
    }
    /**
     * Get CdkGraph artifact by id
     * @throws Error is artifact does not exist
     */
    getArtifact(id) {
        const artifact = this._artifacts[id];
        if (artifact) {
            return artifact;
        }
        throw new Error(`Graph artifact ${id} does not exist`);
    }
    /** Get CdkGraph core `graph.json` artifact */
    get graphJson() {
        return this.getArtifact(GRAPH_ARTIFACT_ID);
    }
    /** Indicates if context has an artifact with *filename* defined */
    hasArtifactFile(filename) {
        return !!Object.values(this._artifacts).find((artifact) => artifact.filename === filename);
    }
    /** Get record of all graph artifacts keyed by artifact id */
    get artifacts() {
        return this._artifacts;
    }
    /**
     * Logs an artifact entry. In general this should not be called directly, as `writeArtifact` should be utilized
     * to perform writing and logging artifacts. However some plugins utilize other tools that generate the artifacts,
     * in which case the plugin would call this method to log the entry.
     * @param source The source of the artifact, such as the name of plugin
     * @param id Unique id of the artifact
     * @param filepath Full path where the artifact is stored
     * @param description Description of the artifact
     * @returns
     * @throws Error is artifact id or filename already exists
     */
    logArtifact(source, id, filepath, description) {
        if (id in this._artifacts) {
            throw new Error(`Graph artifact ${id} already defined`);
        }
        if (this.hasArtifactFile(filepath)) {
            throw new Error(`Graph artifact "${filepath}" already defined`);
        }
        const filename = path.relative(this.outdir, filepath);
        if (!(source instanceof CdkGraph)) {
            if (Object.keys(CdkGraphArtifacts).includes(id)) {
                throw new Error(`Graph artifact id ${id} is reserved`);
            }
            if (Object.values(CdkGraphArtifacts).includes(filename)) {
                throw new Error(`Graph artifact file ${filename} is reserved`);
            }
        }
        const artifact = {
            id,
            filepath,
            description,
            filename,
            source: source instanceof CdkGraph
                ? `${CdkGraph.ID}`
                : `plugin:${source.id}@${source.version}`,
        };
        this._artifacts[id] = artifact;
        console.info(chalk.cyanBright(`[CdkGraph] Artifact ${id} written to ${artifact.filename} (${artifact.source})`));
        return artifact;
    }
    /**
     * Writes artifact data to outdir and logs the entry.
     * @param source The source of the artifact, such as the name of plugin
     * @param id Unique id of the artifact
     * @param filename Relative name of the file
     * @param description Description of the artifact
     * @returns
     */
    writeArtifact(source, id, filename, data, description) {
        const filepath = path.join(this.outdir, filename);
        const artifact = this.logArtifact(source, id, filepath, description);
        fs.ensureDirSync(path.dirname(filepath));
        fs.writeFileSync(filepath, data, { encoding: "utf-8" });
        return artifact;
    }
}
exports.CdkGraphContext = CdkGraphContext;
_a = JSII_RTTI_SYMBOL_1;
CdkGraphContext[_a] = { fqn: "@aws-prototyping-sdk/cdk-graph.CdkGraphContext", version: "0.12.22" };
/**
 * CdkGraph construct is the cdk-graph framework controller that is responsible for
 * computing the graph, storing serialized graph, and instrumenting plugins per the
 * plugin contract.
 */
class CdkGraph extends constructs_1.Construct {
    constructor(root, props = {}) {
        super(root, CdkGraph.ID);
        this.root = root;
        this.config = config_1.resolveConfig();
        this.plugins = props.plugins || [];
        // TODO: verify plugin deps via semver
        // bind all plugins to this instance of the graph
        this.plugins.forEach((plugin) => {
            plugin.bind(this);
        });
        // Apply Aspect for each plugin that supports "inspect" phase
        this.plugins.forEach((plugin) => {
            if (plugin.inspect) {
                aws_cdk_lib_1.Aspects.of(this.root).add({
                    visit: plugin.inspect,
                });
            }
        });
        cdk_internals_1.addCustomSynthesis(this, {
            onSynthesize: (session) => {
                this._synthesize(session);
            },
        });
    }
    /**
     * Get the context for the graph instance.
     *
     * This will be `undefined` before construct synthesis has initiated.
     */
    get graphContext() {
        return this._context;
    }
    /** @internal */
    _synthesize(session) {
        const store = core_1.computeGraph(this.root);
        const outdir = config_1.resolveOutdir(session.outdir, this.config.outdir);
        const context = new CdkGraphContext(store, outdir);
        context.writeArtifact(this, GRAPH_ARTIFACT_ID, CdkGraphArtifacts.GRAPH, JSON.stringify(context.store.serialize(), null, 2), "Serialized graph");
        this.plugins.forEach((plugin) => {
            plugin.synthesize && plugin.synthesize(context);
        });
        fs.writeFileSync(path.join(outdir, CdkGraphArtifacts.GRAPH_METADATA), JSON.stringify({
            version: CdkGraph.VERSION,
            artifacts: context.artifacts,
        }, null, 2), { encoding: "utf-8" });
        // store context for reporting
        this._context = context;
    }
    /**
     * Asynchronous report generation. This operation enables running expensive and non-synchronous
     * report generation by plugins post synthesis.
     *
     * If a given plugin requires performing asynchronous operations or is general expensive, it should
     * utilize `report` rather than `synthesize`.
     */
    async report() {
        if (this._context == null) {
            // TODO: support deserializing pdk-graph to generate store/context
            console.warn(chalk.yellowBright("[CdkGraph] In the near future, reports will be runnable outside of cdk synth"));
            throw new Error("CdkGraph report called outside of cdk synth process");
        }
        for (const plugin of this.plugins) {
            plugin.report && (await plugin.report(this._context));
        }
    }
}
exports.CdkGraph = CdkGraph;
_b = JSII_RTTI_SYMBOL_1;
CdkGraph[_b] = { fqn: "@aws-prototyping-sdk/cdk-graph.CdkGraph", version: "0.12.22" };
/** Fixed CdkGraph construct id */
CdkGraph.ID = core_1.GRAPH_ID;
/** Current CdkGraph semantic version */
CdkGraph.VERSION = "0.0.0"; // TODO: make dynamic from package
//# sourceMappingURL=data:application/json;base64,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