# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['runtype']

package_data = \
{'': ['*']}

extras_require = \
{':python_version >= "3.6" and python_version < "3.7"': ['dataclasses']}

setup_kwargs = {
    'name': 'runtype',
    'version': '0.1.16',
    'description': 'Type dispatch and validation for run-time Python',
    'long_description': '# Runtype\n\nRuntype offers fast run-time type validation for Python, by providing utilities for multiple-dispatch and type-safe dataclasses.\n\nRuntype\'s integration with the `typing` module allows users to invoke type signatures such as `List[int]`, `Dict[str, Optional[str]]`, or `Union[str, Callable]`.\n\n## Multiple Dispatch\n\nMultiple-dispatch is a state-of-the-art technique for structuring code, that complements object-oriented programming.\n\nUnlike in OOP, where the type of the "object" (or: first argument) is always what determines the dispatch, in multiple-dispatch all the arguments decide together, according the idea of specificity: The more specific classes (i.e. subclasses) get picked before the more abstract ones (i.e. superclasses).\n\nThat means that when you need to define a logical operation that applies to several types, you can first solve the most abstract case, and then slowly add special handling for more specific types as required. If you ever found yourself writing several "isinstance" in a row, you could probably use multiple-dispatch to write better code!\n\nRuntype\'s dispatcher is fast, and will never make an arbitrary choice: in ambiguous situations it will always throw an error.\n\nAs a side-effect, it also provides type-validation to functions. Trying to dispatch with types that don\'t match, will result in a dispatch-error.\n\n## Type-Safe Dataclasses\n\nThe ability to annotate dataclasses with types has spurred the creation of many great static type-validation tools (such as `mypy`). Unfortunately, they can\'t always predict what types your dataclasses will receive.\n\nThe trouble with storing the wrong data, is that it can just sit there for a while, and by time you get the error, it\'s hard to track which component or thread put it there.\n\nRuntype provides a `dataclass` drop-in replacement to Python\'s native dataclass, that validates the types in runtime, and makes sure you\'ll see the error the moment something goes wrong, and in the right context.\n\nWhile Runtype\'s validation can add a small runtime overhead, it\'s relatively light. And because it\'s a drop-in replacement, you can always just switch the import back once you\'re done debugging.\n\n## Docs\n\nRead the docs here: https://runtype.readthedocs.io/\n\n## Install\n\n```bash\n$ pip install runtype\n```\n\nNo dependencies.\n\nRequires Python 3.6 or up.\n\n[![Build Status](https://travis-ci.org/erezsh/runtype.svg?branch=master)](https://travis-ci.org/erezsh/runtype)\n[![codecov](https://codecov.io/gh/erezsh/runtype/branch/master/graph/badge.svg)](https://codecov.io/gh/erezsh/runtype)\n\n## Examples\n\n### Isa & Subclass\n\n```python\nfrom typing import Dict\nfrom runtype import isa, issubclass\n\nisa({\'a\': 1}, Dict[str, int])      # True\nisa({\'a\': \'b\'}, Dict[str, int])    #  False\n\nissubclass(Dict[str, int], Mapping[str, int])   # True\nissubclass(Dict[str, int], Mapping[int, str])   # False\n```\n\n### Multiple Dispatch\n\n```python\nfrom runtype import Dispatch\ndp = Dispatch()\n\n@dp\ndef append(a: list, b):\n    return a + [b]\n\n@dp\ndef append(a: tuple, b):\n    return a + (b,)\n\n@dp\ndef append(a: str, b: str):\n    return a + b\n\n\nprint( append([1, 2, 3], 4) )        # prints [1, 2, 3, 4]\nprint( append((1, 2, 3), 4) )        # prints (1, 2, 3, 4)\nprint( append(\'foo\', \'bar\') )        # prints foobar\nprint( append(\'foo\', 4)     )        # raises DispatchError, no signature for (str, int)\n\n\n```\n\n### Dataclasses\n\nBasic usage:\n\n```python\n>>> from runtype import dataclass\n\n>>> @dataclass\n>>> class Point:\n...     x: int\n...     y: int\n\n>>> p = Point(2, 3)\n>>> p\nPoint(x=2, y=3)\n>>> dict(p)          # Maintains order\n{\'x\': 2, \'y\': 3}\n\n>>> p.replace(x=30)  # New instance\nPoint(x=30, y=3)\n\n>>> Point(10, 3.5)   # Actively validates types\nTraceback (most recent call last):\n    ...\nTypeError: [Point] Attribute \'y\' expected value of type <class \'int\'>, instead got 3.5\n```\n\nUsing advanced types:\n\n```python\n>>> from typing import Optional, Callable\n>>> from runtype import dataclass\n\n>>> @dataclass\n>>> class Animal:\n...     name: str\n...     make_sound: Optional[Callable] = None\n\n>>> Animal("Ant")\nAnimal(name=\'Ant\', make_sound=None)\n\n>>> Animal("Cat", lambda: print("meow"))\nAnimal(name=\'Cat\', make_sound=<function <lambda> at ...>)\n\n>>> Animal("Dog", "woof")\nTraceback (most recent call last):\n  File "<stdin>", line 1, in <module>\n  ...\n[Animal] Attribute \'make_sound\' expected value of type typing.Union[typing.Callable, NoneType], instead got \'woof\'\n```\n\n## Performance\nType verification in classes introduces a slight run-time overhead. When running in production, it\'s recommended to use the `-O` switch for Python. It will skip all `assert`s, and also skip type verification on classes by default (use the `check_types` option to adjust it manually).\n\nMultiple-dispatch caches call-signatures by default (disable at your own risk!), and should add a minimal overhead after the initial resolution. Dispatch is only 5 to 8 times slower than adding two numbers (see: [examples/benchmark\\_dispatch](examples/benchmark\\_dispatch.py)), which is negligible.\n\nRuntype is not recommended for use in functions that are called often in time-critical code (or classes that are created often).\n\n## License\n\nRuntype uses the [MIT license](LICENSE).\n\n## Donate\n\nIf you like Runtype and want to show your appreciation, you can do so at my [patreon page](https://www.patreon.com/erezsh), or [ko-fi page](https://ko-fi.com/erezsh).\n',
    'author': 'Erez Shinan',
    'author_email': 'erezshin@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/erezsh/runtype',
    'packages': packages,
    'package_data': package_data,
    'extras_require': extras_require,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
