"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cognito_1 = require("@aws-cdk/aws-cognito");
const aws_dynamodb_1 = require("@aws-cdk/aws-dynamodb");
const core_1 = require("@aws-cdk/core");
const path_1 = require("path");
const lib_1 = require("../lib");
const app = new core_1.App();
const stack = new core_1.Stack(app, 'aws-appsync-integ');
const userPool = new aws_cognito_1.UserPool(stack, 'Pool', {
    userPoolName: 'myPool',
});
const api = new lib_1.GraphQLApi(stack, 'Api', {
    name: 'demoapi',
    schemaDefinitionFile: path_1.join(__dirname, 'schema.graphql'),
    authorizationConfig: {
        defaultAuthorization: {
            authorizationType: lib_1.AuthorizationType.USER_POOL,
            userPoolConfig: {
                userPool,
                defaultAction: lib_1.UserPoolDefaultAction.ALLOW,
            },
        },
        additionalAuthorizationModes: [
            {
                authorizationType: lib_1.AuthorizationType.API_KEY,
            },
        ],
    },
});
const noneDS = api.addNoneDataSource('None', 'Dummy data source');
noneDS.createResolver({
    typeName: 'Query',
    fieldName: 'getServiceVersion',
    requestMappingTemplate: lib_1.MappingTemplate.fromString(JSON.stringify({
        version: '2017-02-28',
    })),
    responseMappingTemplate: lib_1.MappingTemplate.fromString(JSON.stringify({
        version: 'v1',
    })),
});
const customerTable = new aws_dynamodb_1.Table(stack, 'CustomerTable', {
    billingMode: aws_dynamodb_1.BillingMode.PAY_PER_REQUEST,
    partitionKey: {
        name: 'id',
        type: aws_dynamodb_1.AttributeType.STRING,
    },
    removalPolicy: core_1.RemovalPolicy.DESTROY,
});
const orderTable = new aws_dynamodb_1.Table(stack, 'OrderTable', {
    billingMode: aws_dynamodb_1.BillingMode.PAY_PER_REQUEST,
    partitionKey: {
        name: 'customer',
        type: aws_dynamodb_1.AttributeType.STRING,
    },
    sortKey: {
        name: 'order',
        type: aws_dynamodb_1.AttributeType.STRING,
    },
    removalPolicy: core_1.RemovalPolicy.DESTROY,
});
const customerDS = api.addDynamoDbDataSource('Customer', 'The customer data source', customerTable);
const orderDS = api.addDynamoDbDataSource('Order', 'The order data source', orderTable);
customerDS.createResolver({
    typeName: 'Query',
    fieldName: 'getCustomers',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbScanTable(),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultList(),
});
customerDS.createResolver({
    typeName: 'Query',
    fieldName: 'getCustomer',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbGetItem('id', 'id'),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultItem(),
});
customerDS.createResolver({
    typeName: 'Mutation',
    fieldName: 'addCustomer',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbPutItem(lib_1.PrimaryKey.partition('id').auto(), lib_1.Values.projecting('customer')),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultItem(),
});
customerDS.createResolver({
    typeName: 'Mutation',
    fieldName: 'saveCustomer',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbPutItem(lib_1.PrimaryKey.partition('id').is('id'), lib_1.Values.projecting('customer')),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultItem(),
});
customerDS.createResolver({
    typeName: 'Mutation',
    fieldName: 'saveCustomerWithFirstOrder',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbPutItem(lib_1.PrimaryKey
        .partition('order').auto()
        .sort('customer').is('customer.id'), lib_1.Values
        .projecting('order')
        .attribute('referral').is('referral')),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultItem(),
});
customerDS.createResolver({
    typeName: 'Mutation',
    fieldName: 'removeCustomer',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbDeleteItem('id', 'id'),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultItem(),
});
const ops = [
    { suffix: 'Eq', op: lib_1.KeyCondition.eq },
    { suffix: 'Lt', op: lib_1.KeyCondition.lt },
    { suffix: 'Le', op: lib_1.KeyCondition.le },
    { suffix: 'Gt', op: lib_1.KeyCondition.gt },
    { suffix: 'Ge', op: lib_1.KeyCondition.ge },
];
for (const { suffix, op } of ops) {
    orderDS.createResolver({
        typeName: 'Query',
        fieldName: 'getCustomerOrders' + suffix,
        requestMappingTemplate: lib_1.MappingTemplate.dynamoDbQuery(op('customer', 'customer')),
        responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultList(),
    });
}
orderDS.createResolver({
    typeName: 'Query',
    fieldName: 'getCustomerOrdersFilter',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbQuery(lib_1.KeyCondition.eq('customer', 'customer').and(lib_1.KeyCondition.beginsWith('order', 'order'))),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultList(),
});
orderDS.createResolver({
    typeName: 'Query',
    fieldName: 'getCustomerOrdersBetween',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbQuery(lib_1.KeyCondition.eq('customer', 'customer').and(lib_1.KeyCondition.between('order', 'order1', 'order2'))),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultList(),
});
const httpDS = api.addHttpDataSource('http', 'The http data source', 'https://aws.amazon.com/');
httpDS.createResolver({
    typeName: 'Mutation',
    fieldName: 'doPostOnAws',
    requestMappingTemplate: lib_1.MappingTemplate.fromString(`{
    "version": "2018-05-29",
    "method": "POST",
    # if full path is https://api.xxxxxxxxx.com/posts then resourcePath would be /posts
    "resourcePath": "/path/123",
    "params":{
        "body": $util.toJson($ctx.args),
        "headers":{
            "Content-Type": "application/json",
            "Authorization": "$ctx.request.headers.Authorization"
        }
    }
  }`),
    responseMappingTemplate: lib_1.MappingTemplate.fromString(`
    ## Raise a GraphQL field error in case of a datasource invocation error
    #if($ctx.error)
      $util.error($ctx.error.message, $ctx.error.type)
    #end
    ## if the response status code is not 200, then return an error. Else return the body **
    #if($ctx.result.statusCode == 200)
        ## If response is 200, return the body.
        $ctx.result.body
    #else
        ## If response is not 200, append the response to error block.
        $utils.appendError($ctx.result.body, "$ctx.result.statusCode")
    #end
  `),
});
app.synth();
//# sourceMappingURL=data:application/json;base64,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