"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Resolver = exports.MappingTemplate = exports.Values = exports.AttributeValuesStep = exports.AttributeValues = exports.PartitionKey = exports.PrimaryKey = exports.SortKeyStep = exports.PartitionKeyStep = exports.Assign = exports.KeyCondition = exports.LambdaDataSource = exports.HttpDataSource = exports.DynamoDbDataSource = exports.NoneDataSource = exports.BackedDataSource = exports.BaseDataSource = exports.GraphQLApi = exports.FieldLogLevel = exports.UserPoolDefaultAction = exports.AuthorizationType = void 0;
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const fs_1 = require("fs");
const appsync_generated_1 = require("./appsync.generated");
/**
 * enum with all possible values for AppSync authorization type
 */
var AuthorizationType;
(function (AuthorizationType) {
    /**
     * API Key authorization type
     */
    AuthorizationType["API_KEY"] = "API_KEY";
    /**
     * AWS IAM authorization type. Can be used with Cognito Identity Pool federated credentials
     */
    AuthorizationType["IAM"] = "AWS_IAM";
    /**
     * Cognito User Pool authorization type
     */
    AuthorizationType["USER_POOL"] = "AMAZON_COGNITO_USER_POOLS";
    /**
     * OpenID Connect authorization type
     */
    AuthorizationType["OIDC"] = "OPENID_CONNECT";
})(AuthorizationType = exports.AuthorizationType || (exports.AuthorizationType = {}));
/**
 * enum with all possible values for Cognito user-pool default actions
 */
var UserPoolDefaultAction;
(function (UserPoolDefaultAction) {
    /**
     * ALLOW access to API
     */
    UserPoolDefaultAction["ALLOW"] = "ALLOW";
    /**
     * DENY access to API
     */
    UserPoolDefaultAction["DENY"] = "DENY";
})(UserPoolDefaultAction = exports.UserPoolDefaultAction || (exports.UserPoolDefaultAction = {}));
/**
 * log-level for fields in AppSync
 */
var FieldLogLevel;
(function (FieldLogLevel) {
    /**
     * No logging
     */
    FieldLogLevel["NONE"] = "NONE";
    /**
     * Error logging
     */
    FieldLogLevel["ERROR"] = "ERROR";
    /**
     * All logging
     */
    FieldLogLevel["ALL"] = "ALL";
})(FieldLogLevel = exports.FieldLogLevel || (exports.FieldLogLevel = {}));
/**
 * An AppSync GraphQL API
 */
class GraphQLApi extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l;
        super(scope, id);
        this.validateAuthorizationProps(props);
        const defaultAuthorizationType = ((_b = (_a = props.authorizationConfig) === null || _a === void 0 ? void 0 : _a.defaultAuthorization) === null || _b === void 0 ? void 0 : _b.authorizationType) ||
            AuthorizationType.API_KEY;
        let apiLogsRole;
        if (props.logConfig) {
            apiLogsRole = new aws_iam_1.Role(this, 'ApiLogsRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('appsync'),
            });
            apiLogsRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSAppSyncPushToCloudWatchLogs'));
        }
        this.api = new appsync_generated_1.CfnGraphQLApi(this, 'Resource', {
            name: props.name,
            authenticationType: defaultAuthorizationType,
            ...(props.logConfig && {
                logConfig: {
                    cloudWatchLogsRoleArn: apiLogsRole ? apiLogsRole.roleArn : undefined,
                    excludeVerboseContent: props.logConfig.excludeVerboseContent,
                    fieldLogLevel: props.logConfig.fieldLogLevel
                        ? props.logConfig.fieldLogLevel.toString()
                        : undefined,
                },
            }),
            openIdConnectConfig: ((_d = (_c = props.authorizationConfig) === null || _c === void 0 ? void 0 : _c.defaultAuthorization) === null || _d === void 0 ? void 0 : _d.authorizationType) ===
                AuthorizationType.OIDC
                ? this.formatOpenIdConnectConfig(props.authorizationConfig.defaultAuthorization
                    .openIdConnectConfig)
                : undefined,
            userPoolConfig: ((_f = (_e = props.authorizationConfig) === null || _e === void 0 ? void 0 : _e.defaultAuthorization) === null || _f === void 0 ? void 0 : _f.authorizationType) ===
                AuthorizationType.USER_POOL
                ? this.formatUserPoolConfig(props.authorizationConfig.defaultAuthorization.userPoolConfig)
                : undefined,
            additionalAuthenticationProviders: ((_g = props.authorizationConfig) === null || _g === void 0 ? void 0 : _g.additionalAuthorizationModes.length) ? this.formatAdditionalAuthorizationModes(props.authorizationConfig.additionalAuthorizationModes)
                : undefined,
        });
        this.apiId = this.api.attrApiId;
        this.arn = this.api.attrArn;
        this.graphQlUrl = this.api.attrGraphQlUrl;
        this.name = this.api.name;
        if (defaultAuthorizationType === AuthorizationType.API_KEY ||
            ((_j = (_h = props.authorizationConfig) === null || _h === void 0 ? void 0 : _h.additionalAuthorizationModes) === null || _j === void 0 ? void 0 : _j.findIndex((authMode) => authMode.authorizationType === AuthorizationType.API_KEY)) !== -1) {
            const apiKeyConfig = ((_l = (_k = props.authorizationConfig) === null || _k === void 0 ? void 0 : _k.defaultAuthorization) === null || _l === void 0 ? void 0 : _l.apiKeyConfig) || {
                name: 'DefaultAPIKey',
                description: 'Default API Key created by CDK',
            };
            this.createAPIKey(apiKeyConfig);
        }
        let definition;
        if (props.schemaDefinition) {
            definition = props.schemaDefinition;
        }
        else if (props.schemaDefinitionFile) {
            definition = fs_1.readFileSync(props.schemaDefinitionFile).toString('UTF-8');
        }
        else {
            throw new Error('Missing Schema definition. Provide schemaDefinition or schemaDefinitionFile');
        }
        this.schema = new appsync_generated_1.CfnGraphQLSchema(this, 'Schema', {
            apiId: this.apiId,
            definition,
        });
    }
    /**
     * the configured API key, if present
     */
    get apiKey() {
        return this._apiKey;
    }
    /**
     * add a new dummy data source to this API
     * @param name The name of the data source
     * @param description The description of the data source
     */
    addNoneDataSource(name, description) {
        return new NoneDataSource(this, `${name}DS`, {
            api: this,
            description,
            name,
        });
    }
    /**
     * add a new DynamoDB data source to this API
     * @param name The name of the data source
     * @param description The description of the data source
     * @param table The DynamoDB table backing this data source [disable-awslint:ref-via-interface]
     */
    addDynamoDbDataSource(name, description, table) {
        return new DynamoDbDataSource(this, `${name}DS`, {
            api: this,
            description,
            name,
            table,
        });
    }
    /**
     * add a new http data source to this API
     * @param name The name of the data source
     * @param description The description of the data source
     * @param endpoint The http endpoint
     */
    addHttpDataSource(name, description, endpoint) {
        return new HttpDataSource(this, `${name}DS`, {
            api: this,
            description,
            endpoint,
            name,
        });
    }
    /**
     * add a new Lambda data source to this API
     * @param name The name of the data source
     * @param description The description of the data source
     * @param lambdaFunction The Lambda function to call to interact with this data source
     */
    addLambdaDataSource(name, description, lambdaFunction) {
        return new LambdaDataSource(this, `${name}DS`, {
            api: this,
            description,
            name,
            lambdaFunction,
        });
    }
    validateAuthorizationProps(props) {
        var _a, _b, _c, _d, _e, _f, _g;
        const defaultAuthorizationType = ((_b = (_a = props.authorizationConfig) === null || _a === void 0 ? void 0 : _a.defaultAuthorization) === null || _b === void 0 ? void 0 : _b.authorizationType) ||
            AuthorizationType.API_KEY;
        if (defaultAuthorizationType === AuthorizationType.OIDC &&
            !((_d = (_c = props.authorizationConfig) === null || _c === void 0 ? void 0 : _c.defaultAuthorization) === null || _d === void 0 ? void 0 : _d.openIdConnectConfig)) {
            throw new Error('Missing default OIDC Configuration');
        }
        if (defaultAuthorizationType === AuthorizationType.USER_POOL &&
            !((_f = (_e = props.authorizationConfig) === null || _e === void 0 ? void 0 : _e.defaultAuthorization) === null || _f === void 0 ? void 0 : _f.userPoolConfig)) {
            throw new Error('Missing default User Pool Configuration');
        }
        if ((_g = props.authorizationConfig) === null || _g === void 0 ? void 0 : _g.additionalAuthorizationModes) {
            props.authorizationConfig.additionalAuthorizationModes.forEach((authorizationMode) => {
                if (authorizationMode.authorizationType === AuthorizationType.API_KEY &&
                    defaultAuthorizationType === AuthorizationType.API_KEY) {
                    throw new Error("You can't duplicate API_KEY in additional authorization config. See https://docs.aws.amazon.com/appsync/latest/devguide/security.html");
                }
                if (authorizationMode.authorizationType === AuthorizationType.IAM &&
                    defaultAuthorizationType === AuthorizationType.IAM) {
                    throw new Error("You can't duplicate IAM in additional authorization config. See https://docs.aws.amazon.com/appsync/latest/devguide/security.html");
                }
                if (authorizationMode.authorizationType === AuthorizationType.OIDC &&
                    !authorizationMode.openIdConnectConfig) {
                    throw new Error('Missing OIDC Configuration inside an additional authorization mode');
                }
                if (authorizationMode.authorizationType ===
                    AuthorizationType.USER_POOL &&
                    !authorizationMode.userPoolConfig) {
                    throw new Error('Missing User Pool Configuration inside an additional authorization mode');
                }
            });
        }
    }
    formatOpenIdConnectConfig(config) {
        return {
            authTtl: config.tokenExpiryFromAuth,
            clientId: config.clientId,
            iatTtl: config.tokenExpiryFromIssue,
            issuer: config.oidcProvider,
        };
    }
    formatUserPoolConfig(config) {
        return {
            userPoolId: config.userPool.userPoolId,
            awsRegion: config.userPool.stack.region,
            appIdClientRegex: config.appIdClientRegex,
            defaultAction: config.defaultAction || 'ALLOW',
        };
    }
    createAPIKey(config) {
        let expires;
        if (config.expires) {
            expires = new Date(config.expires).valueOf();
            const days = (d) => Date.now() + core_1.Duration.days(d).toMilliseconds();
            if (expires < days(1) || expires > days(365)) {
                throw Error('API key expiration must be between 1 and 365 days.');
            }
            expires = Math.round(expires / 1000);
        }
        const key = new appsync_generated_1.CfnApiKey(this, `${config.name || 'DefaultAPIKey'}ApiKey`, {
            expires,
            description: config.description || 'Default API Key created by CDK',
            apiId: this.apiId,
        });
        this._apiKey = key.attrApiKey;
    }
    formatAdditionalAuthorizationModes(authModes) {
        return authModes.reduce((acc, authMode) => [
            ...acc,
            {
                authenticationType: authMode.authorizationType,
                userPoolConfig: authMode.authorizationType === AuthorizationType.USER_POOL
                    ? this.formatUserPoolConfig(authMode.userPoolConfig)
                    : undefined,
                openIdConnectConfig: authMode.authorizationType === AuthorizationType.OIDC
                    ? this.formatOpenIdConnectConfig(authMode.openIdConnectConfig)
                    : undefined,
            },
        ], []);
    }
}
exports.GraphQLApi = GraphQLApi;
/**
 * Abstract AppSync datasource implementation. Do not use directly but use subclasses for concrete datasources
 */
class BaseDataSource extends core_1.Construct {
    constructor(scope, id, props, extended) {
        var _a;
        super(scope, id);
        if (extended.type !== 'NONE') {
            this.serviceRole = props.serviceRole || new aws_iam_1.Role(this, 'ServiceRole', { assumedBy: new aws_iam_1.ServicePrincipal('appsync') });
        }
        this.ds = new appsync_generated_1.CfnDataSource(this, 'Resource', {
            apiId: props.api.apiId,
            name: props.name,
            description: props.description,
            serviceRoleArn: (_a = this.serviceRole) === null || _a === void 0 ? void 0 : _a.roleArn,
            ...extended,
        });
        this.name = props.name;
        this.api = props.api;
    }
    /**
     * creates a new resolver for this datasource and API using the given properties
     */
    createResolver(props) {
        return new Resolver(this, `${props.typeName}${props.fieldName}Resolver`, {
            api: this.api,
            dataSource: this,
            ...props,
        });
    }
}
exports.BaseDataSource = BaseDataSource;
/**
 * Abstract AppSync datasource implementation. Do not use directly but use subclasses for resource backed datasources
 */
class BackedDataSource extends BaseDataSource {
    constructor(scope, id, props, extended) {
        super(scope, id, props, extended);
        this.grantPrincipal = this.serviceRole;
    }
}
exports.BackedDataSource = BackedDataSource;
/**
 * An AppSync dummy datasource
 */
class NoneDataSource extends BaseDataSource {
    constructor(scope, id, props) {
        super(scope, id, props, {
            type: 'NONE',
        });
    }
}
exports.NoneDataSource = NoneDataSource;
/**
 * An AppSync datasource backed by a DynamoDB table
 */
class DynamoDbDataSource extends BackedDataSource {
    constructor(scope, id, props) {
        super(scope, id, props, {
            type: 'AMAZON_DYNAMODB',
            dynamoDbConfig: {
                tableName: props.table.tableName,
                awsRegion: props.table.stack.region,
                useCallerCredentials: props.useCallerCredentials,
            },
        });
        if (props.readOnlyAccess) {
            props.table.grantReadData(this);
        }
        else {
            props.table.grantReadWriteData(this);
        }
    }
}
exports.DynamoDbDataSource = DynamoDbDataSource;
/**
 * An AppSync datasource backed by a http endpoint
 */
class HttpDataSource extends BaseDataSource {
    constructor(scope, id, props) {
        super(scope, id, props, {
            httpConfig: {
                endpoint: props.endpoint,
            },
            type: 'HTTP',
        });
    }
}
exports.HttpDataSource = HttpDataSource;
/**
 * An AppSync datasource backed by a Lambda function
 */
class LambdaDataSource extends BackedDataSource {
    constructor(scope, id, props) {
        super(scope, id, props, {
            type: 'AWS_LAMBDA',
            lambdaConfig: {
                lambdaFunctionArn: props.lambdaFunction.functionArn,
            },
        });
        props.lambdaFunction.grantInvoke(this);
    }
}
exports.LambdaDataSource = LambdaDataSource;
function concatAndDedup(left, right) {
    return left.concat(right).filter((elem, index, self) => {
        return index === self.indexOf(elem);
    });
}
/**
 * Utility class to represent DynamoDB key conditions.
 */
class BaseKeyCondition {
    and(cond) {
        return new (class extends BaseKeyCondition {
            constructor(left, right) {
                super();
                this.left = left;
                this.right = right;
            }
            renderCondition() {
                return `${this.left.renderCondition()} AND ${this.right.renderCondition()}`;
            }
            keyNames() {
                return concatAndDedup(this.left.keyNames(), this.right.keyNames());
            }
            args() {
                return concatAndDedup(this.left.args(), this.right.args());
            }
        })(this, cond);
    }
    renderExpressionNames() {
        return this.keyNames()
            .map((keyName) => {
            return `"#${keyName}" : "${keyName}"`;
        })
            .join(', ');
    }
    renderExpressionValues() {
        return this.args()
            .map((arg) => {
            return `":${arg}" : $util.dynamodb.toDynamoDBJson($ctx.args.${arg})`;
        })
            .join(', ');
    }
}
/**
 * Utility class to represent DynamoDB "begins_with" key conditions.
 */
class BeginsWith extends BaseKeyCondition {
    constructor(keyName, arg) {
        super();
        this.keyName = keyName;
        this.arg = arg;
    }
    renderCondition() {
        return `begins_with(#${this.keyName}, :${this.arg})`;
    }
    keyNames() {
        return [this.keyName];
    }
    args() {
        return [this.arg];
    }
}
/**
 * Utility class to represent DynamoDB binary key conditions.
 */
class BinaryCondition extends BaseKeyCondition {
    constructor(keyName, op, arg) {
        super();
        this.keyName = keyName;
        this.op = op;
        this.arg = arg;
    }
    renderCondition() {
        return `#${this.keyName} ${this.op} :${this.arg}`;
    }
    keyNames() {
        return [this.keyName];
    }
    args() {
        return [this.arg];
    }
}
/**
 * Utility class to represent DynamoDB "between" key conditions.
 */
class Between extends BaseKeyCondition {
    constructor(keyName, arg1, arg2) {
        super();
        this.keyName = keyName;
        this.arg1 = arg1;
        this.arg2 = arg2;
    }
    renderCondition() {
        return `#${this.keyName} BETWEEN :${this.arg1} AND :${this.arg2}`;
    }
    keyNames() {
        return [this.keyName];
    }
    args() {
        return [this.arg1, this.arg2];
    }
}
/**
 * Factory class for DynamoDB key conditions.
 */
class KeyCondition {
    constructor(cond) {
        this.cond = cond;
    }
    /**
     * Condition k = arg, true if the key attribute k is equal to the Query argument
     */
    static eq(keyName, arg) {
        return new KeyCondition(new BinaryCondition(keyName, '=', arg));
    }
    /**
     * Condition k < arg, true if the key attribute k is less than the Query argument
     */
    static lt(keyName, arg) {
        return new KeyCondition(new BinaryCondition(keyName, '<', arg));
    }
    /**
     * Condition k <= arg, true if the key attribute k is less than or equal to the Query argument
     */
    static le(keyName, arg) {
        return new KeyCondition(new BinaryCondition(keyName, '<=', arg));
    }
    /**
     * Condition k > arg, true if the key attribute k is greater than the the Query argument
     */
    static gt(keyName, arg) {
        return new KeyCondition(new BinaryCondition(keyName, '>', arg));
    }
    /**
     * Condition k >= arg, true if the key attribute k is greater or equal to the Query argument
     */
    static ge(keyName, arg) {
        return new KeyCondition(new BinaryCondition(keyName, '>=', arg));
    }
    /**
     * Condition (k, arg). True if the key attribute k begins with the Query argument.
     */
    static beginsWith(keyName, arg) {
        return new KeyCondition(new BeginsWith(keyName, arg));
    }
    /**
     * Condition k BETWEEN arg1 AND arg2, true if k >= arg1 and k <= arg2.
     */
    static between(keyName, arg1, arg2) {
        return new KeyCondition(new Between(keyName, arg1, arg2));
    }
    /**
     * Conjunction between two conditions.
     */
    and(keyCond) {
        return new KeyCondition(this.cond.and(keyCond.cond));
    }
    /**
     * Renders the key condition to a VTL string.
     */
    renderTemplate() {
        return `"query" : {
            "expression" : "${this.cond.renderCondition()}",
            "expressionNames" : {
                ${this.cond.renderExpressionNames()}
            },
            "expressionValues" : {
                ${this.cond.renderExpressionValues()}
            }
        }`;
    }
}
exports.KeyCondition = KeyCondition;
/**
 * Utility class representing the assigment of a value to an attribute.
 */
class Assign {
    constructor(attr, arg) {
        this.attr = attr;
        this.arg = arg;
    }
    /**
     * Renders the assignment as a VTL string.
     */
    renderAsAssignment() {
        return `"${this.attr}" : $util.dynamodb.toDynamoDBJson(${this.arg})`;
    }
    /**
     * Renders the assignment as a map element.
     */
    putInMap(map) {
        return `$util.qr($${map}.put("${this.attr}", ${this.arg}))`;
    }
}
exports.Assign = Assign;
/**
 * Utility class to allow assigning a value or an auto-generated id
 * to a partition key.
 */
class PartitionKeyStep {
    constructor(key) {
        this.key = key;
    }
    /**
     * Assign an auto-generated value to the partition key.
     */
    is(val) {
        return new PartitionKey(new Assign(this.key, `$ctx.args.${val}`));
    }
    /**
     * Assign an auto-generated value to the partition key.
     */
    auto() {
        return new PartitionKey(new Assign(this.key, '$util.autoId()'));
    }
}
exports.PartitionKeyStep = PartitionKeyStep;
/**
 * Utility class to allow assigning a value or an auto-generated id
 * to a sort key.
 */
class SortKeyStep {
    constructor(pkey, skey) {
        this.pkey = pkey;
        this.skey = skey;
    }
    /**
     * Assign an auto-generated value to the sort key.
     */
    is(val) {
        return new PrimaryKey(this.pkey, new Assign(this.skey, `$ctx.args.${val}`));
    }
    /**
     * Assign an auto-generated value to the sort key.
     */
    auto() {
        return new PrimaryKey(this.pkey, new Assign(this.skey, '$util.autoId()'));
    }
}
exports.SortKeyStep = SortKeyStep;
/**
 * Specifies the assignment to the primary key. It either
 * contains the full primary key or only the partition key.
 */
class PrimaryKey {
    constructor(pkey, skey) {
        this.pkey = pkey;
        this.skey = skey;
    }
    /**
     * Allows assigning a value to the partition key.
     */
    static partition(key) {
        return new PartitionKeyStep(key);
    }
    /**
     * Renders the key assignment to a VTL string.
     */
    renderTemplate() {
        const assignments = [this.pkey.renderAsAssignment()];
        if (this.skey) {
            assignments.push(this.skey.renderAsAssignment());
        }
        return `"key" : {
      ${assignments.join(',')}
    }`;
    }
}
exports.PrimaryKey = PrimaryKey;
/**
 * Specifies the assignment to the partition key. It can be
 * enhanced with the assignment of the sort key.
 */
class PartitionKey extends PrimaryKey {
    constructor(pkey) {
        super(pkey);
    }
    /**
     * Allows assigning a value to the sort key.
     */
    sort(key) {
        return new SortKeyStep(this.pkey, key);
    }
}
exports.PartitionKey = PartitionKey;
/**
 * Specifies the attribute value assignments.
 */
class AttributeValues {
    constructor(container, assignments = []) {
        this.container = container;
        this.assignments = assignments;
    }
    /**
     * Allows assigning a value to the specified attribute.
     */
    attribute(attr) {
        return new AttributeValuesStep(attr, this.container, this.assignments);
    }
    /**
     * Renders the variables required for `renderTemplate`.
     */
    renderVariables() {
        return `#set($input = ${this.container})
      ${this.assignments.map(a => a.putInMap('input')).join('\n')}`;
    }
    /**
     * Renders the attribute value assingments to a VTL string.
     */
    renderTemplate() {
        return '"attributeValues": $util.dynamodb.toMapValuesJson($input)';
    }
}
exports.AttributeValues = AttributeValues;
/**
 * Utility class to allow assigning a value to an attribute.
 */
class AttributeValuesStep {
    constructor(attr, container, assignments) {
        this.attr = attr;
        this.container = container;
        this.assignments = assignments;
    }
    /**
     * Assign the value to the current attribute.
     */
    is(val) {
        this.assignments.push(new Assign(this.attr, val));
        return new AttributeValues(this.container, this.assignments);
    }
}
exports.AttributeValuesStep = AttributeValuesStep;
/**
 * Factory class for attribute value assignments.
 */
class Values {
    /**
     * Treats the specified object as a map of assignments, where the property
     * names represent attribute names. It’s opinionated about how it represents
     * some of the nested objects: e.g., it will use lists (“L”) rather than sets
     * (“SS”, “NS”, “BS”). By default it projects the argument container ("$ctx.args").
     */
    static projecting(arg) {
        return new AttributeValues('$ctx.args' + (arg ? `.${arg}` : ''));
    }
    /**
     * Allows assigning a value to the specified attribute.
     */
    static attribute(attr) {
        return new AttributeValues('{}').attribute(attr);
    }
}
exports.Values = Values;
/**
 * MappingTemplates for AppSync resolvers
 */
class MappingTemplate {
    /**
     * Create a mapping template from the given string
     */
    static fromString(template) {
        return new StringMappingTemplate(template);
    }
    /**
     * Create a mapping template from the given file
     */
    static fromFile(fileName) {
        return new StringMappingTemplate(fs_1.readFileSync(fileName).toString('UTF-8'));
    }
    /**
     * Mapping template for a result list from DynamoDB
     */
    static dynamoDbResultList() {
        return this.fromString('$util.toJson($ctx.result.items)');
    }
    /**
     * Mapping template for a single result item from DynamoDB
     */
    static dynamoDbResultItem() {
        return this.fromString('$util.toJson($ctx.result)');
    }
    /**
     * Mapping template to scan a DynamoDB table to fetch all entries
     */
    static dynamoDbScanTable() {
        return this.fromString('{"version" : "2017-02-28", "operation" : "Scan"}');
    }
    /**
     * Mapping template to query a set of items from a DynamoDB table
     *
     * @param cond the key condition for the query
     */
    static dynamoDbQuery(cond) {
        return this.fromString(`{"version" : "2017-02-28", "operation" : "Query", ${cond.renderTemplate()}}`);
    }
    /**
     * Mapping template to get a single item from a DynamoDB table
     *
     * @param keyName the name of the hash key field
     * @param idArg the name of the Query argument
     */
    static dynamoDbGetItem(keyName, idArg) {
        return this.fromString(`{"version": "2017-02-28", "operation": "GetItem", "key": {"${keyName}": $util.dynamodb.toDynamoDBJson($ctx.args.${idArg})}}`);
    }
    /**
     * Mapping template to delete a single item from a DynamoDB table
     *
     * @param keyName the name of the hash key field
     * @param idArg the name of the Mutation argument
     */
    static dynamoDbDeleteItem(keyName, idArg) {
        return this.fromString(`{"version": "2017-02-28", "operation": "DeleteItem", "key": {"${keyName}": $util.dynamodb.toDynamoDBJson($ctx.args.${idArg})}}`);
    }
    /**
     * Mapping template to save a single item to a DynamoDB table
     *
     * @param key the assigment of Mutation values to the primary key
     * @param values the assignment of Mutation values to the table attributes
     */
    static dynamoDbPutItem(key, values) {
        return this.fromString(`
      ${values.renderVariables()}
      {
        "version": "2017-02-28",
        "operation": "PutItem",
        ${key.renderTemplate()},
        ${values.renderTemplate()}
      }`);
    }
    /**
     * Mapping template to invoke a Lambda function
     *
     * @param payload the VTL template snippet of the payload to send to the lambda.
     * If no payload is provided all available context fields are sent to the Lambda function
     */
    static lambdaRequest(payload = '$util.toJson($ctx)') {
        return this.fromString(`{"version": "2017-02-28", "operation": "Invoke", "payload": ${payload}}`);
    }
    /**
     * Mapping template to return the Lambda result to the caller
     */
    static lambdaResult() {
        return this.fromString('$util.toJson($ctx.result)');
    }
}
exports.MappingTemplate = MappingTemplate;
class StringMappingTemplate extends MappingTemplate {
    constructor(template) {
        super();
        this.template = template;
    }
    renderTemplate() {
        return this.template;
    }
}
/**
 * An AppSync resolver
 */
class Resolver extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.resolver = new appsync_generated_1.CfnResolver(this, 'Resource', {
            apiId: props.api.apiId,
            typeName: props.typeName,
            fieldName: props.fieldName,
            dataSourceName: props.dataSource ? props.dataSource.name : undefined,
            kind: props.pipelineConfig ? 'PIPELINE' : 'UNIT',
            requestMappingTemplate: props.requestMappingTemplate ? props.requestMappingTemplate.renderTemplate() : undefined,
            responseMappingTemplate: props.responseMappingTemplate ? props.responseMappingTemplate.renderTemplate() : undefined,
        });
        this.resolver.addDependsOn(props.api.schema);
        if (props.dataSource) {
            this.resolver.addDependsOn(props.dataSource.ds);
        }
        this.arn = this.resolver.attrResolverArn;
    }
}
exports.Resolver = Resolver;
//# sourceMappingURL=data:application/json;base64,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