from __future__ import absolute_import
from __future__ import print_function

from builtins import object
from builtins import zip

from tsgettoolbox.odo import convert
from tsgettoolbox.odo import resource
from tsgettoolbox.odo import odo

import numpy as np
import pandas as pd

from requests import Session

from tstoolbox import tsutils

from zeep import Client
from zeep import Transport


"""
+---------+----------+-------------------------+-----------+------------+
| Product | Acronym  | Name                    | Frequency | Resolution |
|         |          |                         | (days)    | (m)        |
+=========+==========+=========================+===========+============+
| MCD12Q1 | LC       | MODIS/Terra+Aqua Land   | annual    | 500        |
|         |          | Cover (LC) Type Yearly  |           |            |
|         |          | L3 Global 500m SIN Grid |           |            |
+---------+----------+-------------------------+-----------+------------+
| MCD12Q2 | LCD      | MODIS/Terra+Aqua Land   | annual    | 500        |
|         |          | Cover Dynamics (LCD)    |           |            |
|         |          | Yearly L3 Global 500m   |           |            |
|         |          | SIN Grid                |           |            |
+---------+----------+-------------------------+-----------+------------+
| MCD43A1 | BRDF     | MODIS/Terra+Aqua        | 16        | 500        |
|         |          | BRDF/Albedo             |           |            |
|         |          | (BRDF/MCD43A1) 16-Day   |           |            |
|         |          | L3 Global 500m SIN Grid |           |            |
+---------+----------+-------------------------+-----------+------------+
| MCD43A2 | BRDF/QA  | MODIS/Terra+Aqua        | 16        | 500        |
|         |          | BRDF/Model Quality      |           |            |
|         |          | (BRDF/MCD43A2) 16-Day   |           |            |
|         |          | L3 Global 500m SIN Grid |           |            |
|         |          | V005                    |           |            |
+---------+----------+-------------------------+-----------+------------+
| MCD43A4 | NBAR     | MODIS/Terra+Aqua Nadir  | 16        | 500        |
|         |          | BRDF- Adjusted          |           |            |
|         |          | Reflectance (NBAR)      |           |            |
|         |          | 16-Day L3 Global 500m   |           |            |
|         |          | SIN Grid                |           |            |
+---------+----------+-------------------------+-----------+------------+
| MOD09A1 | SREF     | MODIS/Terra Surface     | 8         | 500        |
|         |          | Reflectance (SREF)      |           |            |
|         |          | 8-Day L3 Global 500m    |           |            |
|         |          | SIN Grid                |           |            |
+---------+----------+-------------------------+-----------+------------+
| MOD11A2 | TEMP     | MODIS/Terra Land        | 8         | 1000       |
|         |          | Surface                 |           |            |
|         |          | Temperature/Emissivity  |           |            |
|         |          | (LST) 8-Day L3 Global   |           |            |
|         |          | 1km SIN Grid            |           |            |
+---------+----------+-------------------------+-----------+------------+
| MOD13Q1 | NDVI/EVI | MODIS/Terra Vegetation  | 16        | 250        |
|         |          | Indices (NDVI/EVI)      |           |            |
|         |          | 16-Day L3 Global 250m   |           |            |
|         |          | SIN Grid [Collection 5] |           |            |
+---------+----------+-------------------------+-----------+------------+
| MOD15A2 | LAI/FPAR | Leaf Area Index (LAI)   | 8         | 1000       |
|         |          | and Fraction of         |           |            |
|         |          | Photosynthetically      |           |            |
|         |          | Active Radiation (FPAR) |           |            |
|         |          | 8-Day Composite         |           |            |
|         |          | [Collection 5]          |           |            |
+---------+----------+-------------------------+-----------+------------+
| MOD16A2 | ET       | MODIS/Terra             | 8         | 1000       |
|         |          | Evapotranspiration (ET) |           |            |
|         |          | 8-Day L4 Global         |           |            |
|         |          | Collection 5            |           |            |
+---------+----------+-------------------------+-----------+------------+
| MOD17A2 | GPP      | MODIS/Terra Gross       | 8         | 1000       |
|         |          | Primary Production      |           |            |
|         |          | (GPP) 8-Day L4 Global   |           |            |
|         |          | [Collection 5.1]        |           |            |
+---------+----------+-------------------------+-----------+------------+
| MOD17A3 | NPP      | MODIS/Terra Net Primary | annual    | 1000       |
|         |          | Production (NPP) Yearly |           |            |
|         |          | L4 Global 1km SIN Grid  |           |            |
+---------+----------+-------------------------+-----------+------------+
| MYD09A1 | SREF     | MODIS/Aqua Surface      | 8         | 500        |
|         |          | Reflectance (SREF)      |           |            |
|         |          | 8-Day L3 Global 500m    |           |            |
|         |          | SIN Grid                |           |            |
+---------+----------+-------------------------+-----------+------------+
| MYD11A2 | TEMP     | MODIS/Aqua Land Surface | 8         | 1000       |
|         |          | Temperature/Emissivity  |           |            |
|         |          | (LST)8-Day L3 Global    |           |            |
|         |          | 1km SIN Grid            |           |            |
+---------+----------+-------------------------+-----------+------------+
| MYD13Q1 | NDVI/EVI | MODIS/Aqua Vegetation   | 16        | 250        |
|         |          | Indices (NDVI/EVI)      |           |            |
|         |          | 16-Day L3 Global 1km    |           |            |
|         |          | SIN Grid                |           |            |
+---------+----------+-------------------------+-----------+------------+
| MYD15A2 | LAI/FPAR | MODIS/Aqua Leaf Area    | 8         | 1000       |
|         |          | Index (LAI) and         |           |            |
|         |          | Fraction of             |           |            |
|         |          | Photosynthetically      |           |            |
|         |          | Active Radiation (FPAR) |           |            |
|         |          | 8 Day Composite         |           |            |
+---------+----------+-------------------------+-----------+------------+
| MYD17A2 | GPP      | MODIS/Aqua Gross        | 8         | 1000       |
|         |          | Primary Production      |           |            |
|         |          | (GPP) 8 Day L4 Global   |           |            |
+---------+----------+-------------------------+-----------+------------+

+------------+---------------------------------------------------------+
| Property   | Band                                                    |
+============+=========================================================+
| MCD12Q1    |                                                         |
+------------+---------------------------------------------------------+
|            | LC_Property_1                                           |
+------------+---------------------------------------------------------+
|            | LC_Property_2                                           |
+------------+---------------------------------------------------------+
|            | LC_Property_3                                           |
+------------+---------------------------------------------------------+
|            | Land_Cover_Type_1                                       |
+------------+---------------------------------------------------------+
|            | Land_Cover_Type_2                                       |
+------------+---------------------------------------------------------+
|            | Land_Cover_Type_3                                       |
+------------+---------------------------------------------------------+
|            | Land_Cover_Type_4                                       |
+------------+---------------------------------------------------------+
|            | Land_Cover_Type_5                                       |
+------------+---------------------------------------------------------+
|            | Land_Cover_Type_1_Assessment                            |
+------------+---------------------------------------------------------+
|            | Land_Cover_Type_2_Assessment                            |
+------------+---------------------------------------------------------+
|            | Land_Cover_Type_3_Assessment                            |
+------------+---------------------------------------------------------+
|            | Land_Cover_Type_4_Assessment                            |
+------------+---------------------------------------------------------+
|            | Land_Cover_Type_5_Assessment                            |
+------------+---------------------------------------------------------+
|            | Land_Cover_Type_1_Secondary                             |
+------------+---------------------------------------------------------+
|            | Land_Cover_Type_1_Secondary_Percent                     |
+------------+---------------------------------------------------------+
| MCD12Q2    |                                                         |
+------------+---------------------------------------------------------+
|            | NBAR_EVI_Onset_Greenness_Maximum.Num_Modes_02           |
+------------+---------------------------------------------------------+
|            | NBAR_EVI_Onset_Greenness_Minimum.Num_Modes_02           |
+------------+---------------------------------------------------------+
|            | NBAR_EVI_Onset_Greenness_Maximum.Num_Modes_01           |
+------------+---------------------------------------------------------+
|            | NBAR_EVI_Onset_Greenness_Minimum.Num_Modes_01           |
+------------+---------------------------------------------------------+
|            | Onset_Greenness_Minimum.Num_Modes_02                    |
+------------+---------------------------------------------------------+
|            | Onset_Greenness_Decrease.Num_Modes_02                   |
+------------+---------------------------------------------------------+
|            | Onset_Greenness_Maximum.Num_Modes_02                    |
+------------+---------------------------------------------------------+
|            | Onset_Greenness_Increase.Num_Modes_02                   |
+------------+---------------------------------------------------------+
|            | Onset_Greenness_Minimum.Num_Modes_01                    |
+------------+---------------------------------------------------------+
|            | Onset_Greenness_Decrease.Num_Modes_01                   |
+------------+---------------------------------------------------------+
|            | Onset_Greenness_Maximum.Num_Modes_01                    |
+------------+---------------------------------------------------------+
|            | Onset_Greenness_Increase.Num_Modes_01                   |
+------------+---------------------------------------------------------+
|            | NBAR_EVI_Area.Num_Modes_01                              |
+------------+---------------------------------------------------------+
|            | NBAR_EVI_Area.Num_Modes_02                              |
+------------+---------------------------------------------------------+
| MCD43A1    |                                                         |
+------------+---------------------------------------------------------+
|            | BRDF_Albedo_Parameters_Band1.Num_Parameters_01          |
+------------+---------------------------------------------------------+
|            | BRDF_Albedo_Parameters_Band1.Num_Parameters_02          |
+------------+---------------------------------------------------------+
|            | BRDF_Albedo_Parameters_Band1.Num_Parameters_03          |
+------------+---------------------------------------------------------+
|            | BRDF_Albedo_Parameters_Band2.Num_Parameters_01          |
+------------+---------------------------------------------------------+
|            | BRDF_Albedo_Parameters_Band2.Num_Parameters_02          |
+------------+---------------------------------------------------------+
|            | BRDF_Albedo_Parameters_Band2.Num_Parameters_03          |
+------------+---------------------------------------------------------+
|            | BRDF_Albedo_Parameters_Band3.Num_Parameters_01          |
+------------+---------------------------------------------------------+
|            | BRDF_Albedo_Parameters_Band3.Num_Parameters_02          |
+------------+---------------------------------------------------------+
|            | BRDF_Albedo_Parameters_Band3.Num_Parameters_03          |
+------------+---------------------------------------------------------+
|            | BRDF_Albedo_Parameters_Band4.Num_Parameters_01          |
+------------+---------------------------------------------------------+
|            | BRDF_Albedo_Parameters_Band4.Num_Parameters_02          |
+------------+---------------------------------------------------------+
|            | BRDF_Albedo_Parameters_Band4.Num_Parameters_03          |
+------------+---------------------------------------------------------+
|            | BRDF_Albedo_Parameters_Band5.Num_Parameters_01          |
+------------+---------------------------------------------------------+
|            | BRDF_Albedo_Parameters_Band5.Num_Parameters_02          |
+------------+---------------------------------------------------------+
|            | BRDF_Albedo_Parameters_Band5.Num_Parameters_03          |
+------------+---------------------------------------------------------+
|            | BRDF_Albedo_Parameters_Band6.Num_Parameters_01          |
+------------+---------------------------------------------------------+
|            | BRDF_Albedo_Parameters_Band6.Num_Parameters_02          |
+------------+---------------------------------------------------------+
|            | BRDF_Albedo_Parameters_Band6.Num_Parameters_03          |
+------------+---------------------------------------------------------+
|            | BRDF_Albedo_Parameters_Band7.Num_Parameters_01          |
+------------+---------------------------------------------------------+
|            | BRDF_Albedo_Parameters_Band7.Num_Parameters_02          |
+------------+---------------------------------------------------------+
|            | BRDF_Albedo_Parameters_Band7.Num_Parameters_03          |
+------------+---------------------------------------------------------+
|            | BRDF_Albedo_Parameters_vis.Num_Parameters_01            |
+------------+---------------------------------------------------------+
|            | BRDF_Albedo_Parameters_vis.Num_Parameters_02            |
+------------+---------------------------------------------------------+
|            | BRDF_Albedo_Parameters_vis.Num_Parameters_03            |
+------------+---------------------------------------------------------+
|            | BRDF_Albedo_Parameters_nir.Num_Parameters_01            |
+------------+---------------------------------------------------------+
|            | BRDF_Albedo_Parameters_nir.Num_Parameters_02            |
+------------+---------------------------------------------------------+
|            | BRDF_Albedo_Parameters_nir.Num_Parameters_03            |
+------------+---------------------------------------------------------+
|            | BRDF_Albedo_Parameters_shortwave.Num_Paramete rs_01     |
+------------+---------------------------------------------------------+
|            | BRDF_Albedo_Parameters_shortwave.Num_Paramete rs_02     |
+------------+---------------------------------------------------------+
|            | BRDF_Albedo_Parameters_shortwave.Num_Paramete rs_03     |
+------------+---------------------------------------------------------+
|            | BRDF_Shape_Indicators.Num_Shape_Fields_01               |
+------------+---------------------------------------------------------+
|            | BRDF_Shape_Indicators.Num_Shape_Fields_02               |
+------------+---------------------------------------------------------+
|            | BRDF_Shape_Indicators.Num_Shape_Fields_03               |
+------------+---------------------------------------------------------+
|            | BRDF_Shape_Indicators.Num_Shape_Fields_04               |
+------------+---------------------------------------------------------+
| MCD43A2    | BRDF_Albedo_Quality                                     |
+------------+---------------------------------------------------------+
|            | BRDF_Albedo_Ancillary                                   |
+------------+---------------------------------------------------------+
|            | BRDF_Albedo_Band_Quality                                |
+------------+---------------------------------------------------------+
|            | Snow_BRDF_Albedo                                        |
+------------+---------------------------------------------------------+
| MCD43A4    | Nadir_Reflectance_Band1                                 |
+------------+---------------------------------------------------------+
|            | Nadir_Reflectance_Band2                                 |
+------------+---------------------------------------------------------+
|            | Nadir_Reflectance_Band3                                 |
+------------+---------------------------------------------------------+
|            | Nadir_Reflectance_Band4                                 |
+------------+---------------------------------------------------------+
|            | Nadir_Reflectance_Band5                                 |
+------------+---------------------------------------------------------+
|            | Nadir_Reflectance_Band6                                 |
+------------+---------------------------------------------------------+
|            | Nadir_Reflectance_Band7                                 |
+------------+---------------------------------------------------------+
| MOD09A1    |                                                         |
+------------+---------------------------------------------------------+
|            | sur_refl_day_of_year                                    |
+------------+---------------------------------------------------------+
|            | sur_refl_qc_500m                                        |
+------------+---------------------------------------------------------+
|            | sur_refl_raz                                            |
+------------+---------------------------------------------------------+
|            | sur_refl_state_500m                                     |
+------------+---------------------------------------------------------+
|            | sur_refl_szen                                           |
+------------+---------------------------------------------------------+
|            | sur_refl_vzen                                           |
+------------+---------------------------------------------------------+
|            | sur_refl_b01                                            |
+------------+---------------------------------------------------------+
|            | sur_refl_b02                                            |
+------------+---------------------------------------------------------+
|            | sur_refl_b03                                            |
+------------+---------------------------------------------------------+
|            | sur_refl_b04                                            |
+------------+---------------------------------------------------------+
|            | sur_refl_b05                                            |
+------------+---------------------------------------------------------+
|            | sur_refl_b06                                            |
+------------+---------------------------------------------------------+
|            | sur_refl_b07                                            |
+------------+---------------------------------------------------------+
| MOD11A2    |                                                         |
+------------+---------------------------------------------------------+
|            | Clear_sky_days                                          |
+------------+---------------------------------------------------------+
|            | Clear_sky_nights                                        |
+------------+---------------------------------------------------------+
|            | Day_view_angl                                           |
+------------+---------------------------------------------------------+
|            | Day_view_time                                           |
+------------+---------------------------------------------------------+
|            | Emis_31                                                 |
+------------+---------------------------------------------------------+
|            | Emis_32                                                 |
+------------+---------------------------------------------------------+
|            | Night_view_angl                                         |
+------------+---------------------------------------------------------+
|            | Night_view_time                                         |
+------------+---------------------------------------------------------+
|            | QC_Day                                                  |
+------------+---------------------------------------------------------+
|            | QC_Night                                                |
+------------+---------------------------------------------------------+
|            | LST_Day_1km                                             |
+------------+---------------------------------------------------------+
|            | LST_Night_1km                                           |
+------------+---------------------------------------------------------+
| MOD13Q1    |                                                         |
+------------+---------------------------------------------------------+
|            | 250m_16_days_blue_reflectance                           |
+------------+---------------------------------------------------------+
|            | 250m_16_days_MIR_reflectance                            |
+------------+---------------------------------------------------------+
|            | 250m_16_days_NIR_reflectance                            |
+------------+---------------------------------------------------------+
|            | 250m_16_days_pixel_reliability                          |
+------------+---------------------------------------------------------+
|            | 250m_16_days_red_reflectance                            |
+------------+---------------------------------------------------------+
|            | 250m_16_days_relative_azimuth_angle                     |
+------------+---------------------------------------------------------+
|            | 250m_16_days_sun_zenith_angle                           |
+------------+---------------------------------------------------------+
|            | 250m_16_days_view_zenith_angle                          |
+------------+---------------------------------------------------------+
|            | 250m_16_days_VI_Quality                                 |
+------------+---------------------------------------------------------+
|            | 250m_16_days_NDVI                                       |
+------------+---------------------------------------------------------+
|            | 250m_16_days_EVI                                        |
+------------+---------------------------------------------------------+
|            | 250m_16_days_composite_day_of_the_year                  |
+------------+---------------------------------------------------------+
| MOD15A2    |                                                         |
+------------+---------------------------------------------------------+
|            | FparExtra_QC                                            |
+------------+---------------------------------------------------------+
|            | FparLai_QC                                              |
+------------+---------------------------------------------------------+
|            | FparStdDev_1km                                          |
+------------+---------------------------------------------------------+
|            | LaiStdDev_1km                                           |
+------------+---------------------------------------------------------+
|            | Lai_1km                                                 |
+------------+---------------------------------------------------------+
|            | Fpar_1km                                                |
+------------+---------------------------------------------------------+
| MOD15A2GFS |                                                         |
+------------+---------------------------------------------------------+
|            | FparExtra_QC                                            |
+------------+---------------------------------------------------------+
|            | FparLai_QC                                              |
+------------+---------------------------------------------------------+
|            | FparStdDev_1km                                          |
+------------+---------------------------------------------------------+
|            | LaiStdDev_1km                                           |
+------------+---------------------------------------------------------+
|            | Lai_1km                                                 |
+------------+---------------------------------------------------------+
|            | Fpar_1km                                                |
+------------+---------------------------------------------------------+
| MOD16A2    |                                                         |
+------------+---------------------------------------------------------+
|            | ET_1km                                                  |
+------------+---------------------------------------------------------+
|            | LE_1km                                                  |
+------------+---------------------------------------------------------+
|            | PET_1km                                                 |
+------------+---------------------------------------------------------+
|            | PLE_1km                                                 |
+------------+---------------------------------------------------------+
|            | ET_QC_1km                                               |
+------------+---------------------------------------------------------+
| MOD17A2_51 |                                                         |
+------------+---------------------------------------------------------+
|            | Psn_QC_1km                                              |
+------------+---------------------------------------------------------+
|            | PsnNet_1km                                              |
+------------+---------------------------------------------------------+
|            | Gpp_1km                                                 |
+------------+---------------------------------------------------------+
| MOD17A3    |                                                         |
+------------+---------------------------------------------------------+
|            | Gpp_Npp_QC_1km                                          |
+------------+---------------------------------------------------------+
|            | Npp_1km                                                 |
+------------+---------------------------------------------------------+
|            | Gpp_1km                                                 |
+------------+---------------------------------------------------------+
| MYD09A1    |                                                         |
+------------+---------------------------------------------------------+
|            | sur_refl_day_of_year                                    |
+------------+---------------------------------------------------------+
|            | sur_refl_qc_500m                                        |
+------------+---------------------------------------------------------+
|            | sur_refl_raz                                            |
+------------+---------------------------------------------------------+
|            | sur_refl_state_500m                                     |
+------------+---------------------------------------------------------+
|            | sur_refl_szen                                           |
+------------+---------------------------------------------------------+
|            | sur_refl_vzen                                           |
+------------+---------------------------------------------------------+
|            | sur_refl_b01                                            |
+------------+---------------------------------------------------------+
|            | sur_refl_b02                                            |
+------------+---------------------------------------------------------+
|            | sur_refl_b03                                            |
+------------+---------------------------------------------------------+
|            | sur_refl_b04                                            |
+------------+---------------------------------------------------------+
|            | sur_refl_b05                                            |
+------------+---------------------------------------------------------+
|            | sur_refl_b06                                            |
+------------+---------------------------------------------------------+
|            | sur_refl_b07                                            |
+------------+---------------------------------------------------------+
| MYD11A2    |                                                         |
+------------+---------------------------------------------------------+
|            | Clear_sky_days                                          |
+------------+---------------------------------------------------------+
|            | Clear_sky_nights                                        |
+------------+---------------------------------------------------------+
|            | Day_view_angl                                           |
+------------+---------------------------------------------------------+
|            | Day_view_time                                           |
+------------+---------------------------------------------------------+
|            | Emis_31                                                 |
+------------+---------------------------------------------------------+
|            | Emis_32                                                 |
+------------+---------------------------------------------------------+
|            | Night_view_angl                                         |
+------------+---------------------------------------------------------+
|            | Night_view_time                                         |
+------------+---------------------------------------------------------+
|            | QC_Day                                                  |
+------------+---------------------------------------------------------+
|            | QC_Night                                                |
+------------+---------------------------------------------------------+
|            | LST_Day_1km                                             |
+------------+---------------------------------------------------------+
|            | LST_Night_1km                                           |
+------------+---------------------------------------------------------+
| MYD13Q1    |                                                         |
+------------+---------------------------------------------------------+
|            | 250m_16_days_blue_reflectance                           |
+------------+---------------------------------------------------------+
|            | 250m_16_days_MIR_reflectance                            |
+------------+---------------------------------------------------------+
|            | 250m_16_days_NIR_reflectance                            |
+------------+---------------------------------------------------------+
|            | 250m_16_days_pixel_reliability                          |
+------------+---------------------------------------------------------+
|            | 250m_16_days_red_reflectance                            |
+------------+---------------------------------------------------------+
|            | 250m_16_days_relative_azimuth_angle                     |
+------------+---------------------------------------------------------+
|            | 250m_16_days_sun_zenith_angle                           |
+------------+---------------------------------------------------------+
|            | 250m_16_days_view_zenith_angle                          |
+------------+---------------------------------------------------------+
|            | 250m_16_days_VI_Quality                                 |
+------------+---------------------------------------------------------+
|            | 250m_16_days_NDVI                                       |
+------------+---------------------------------------------------------+
|            | 250m_16_days_EVI                                        |
+------------+---------------------------------------------------------+
|            | 250m_16_days_composite_day_of_the_year                  |
+------------+---------------------------------------------------------+
| MYD15A2    |                                                         |
+------------+---------------------------------------------------------+
|            | FparExtra_QC                                            |
+------------+---------------------------------------------------------+
|            | FparLai_QC                                              |
+------------+---------------------------------------------------------+
|            | FparStdDev_1km                                          |
+------------+---------------------------------------------------------+
|            | LaiStdDev_1km                                           |
+------------+---------------------------------------------------------+
|            | Lai_1km                                                 |
+------------+---------------------------------------------------------+
|            | Fpar_1km                                                |
+------------+---------------------------------------------------------+
"""

# modis


class modis(object):
    def __init__(self, url, **query_params):
        self.url = url
        if query_params["startdate"] is None:
            query_params["startdate"] = pd.to_datetime("1900-01-01T00")
        else:
            query_params["startdate"] = tsutils.parsedate(query_params["startdate"])
        if query_params["enddate"] is None:
            query_params["enddate"] = pd.datetime.now()
        else:
            query_params["enddate"] = tsutils.parsedate(query_params["enddate"])
        self.query_params = query_params


@resource.register(
    r"https://modis.ornl.gov/cgi-bin/MODIS/soapservice/MODIS_soapservice.wsdl",
    priority=17,
)
def resource_modis(uri, **kwargs):
    return modis(uri, **kwargs)


def date_parser(strdates):
    return [
        pd.datetime.fromordinal(
            pd.datetime(int(i[1:5]), 1, 1).toordinal() + int(i[5:]) - 1
        )
        for i in strdates
    ]


# Function to convert from modis type to pd.DataFrame


@convert.register(pd.DataFrame, modis)
def modis_to_df(data, **kwargs):
    session = Session()
    session.verify = True
    transport = Transport(session=session)
    client = Client(wsdl=data.url, transport=transport)
    products = client.service.getproducts()
    if data.query_params["product"] not in products:
        raise ValueError(
            tsutils.error_wrapper(
                """
Available products at the current time are: {0}.

You gave {1}.
""".format(
                    products, data.query_params["product"]
                )
            )
        )
    bands = client.service.getbands(data.query_params["product"])
    if data.query_params["band"] not in bands:
        raise ValueError(
            tsutils.error_wrapper(
                """
'band' argument must be in the following list for 'product' = {0}.
{1}.

You gave me {2}.
""".format(
                    data.query_params["product"], bands, data.query_params["band"]
                )
            )
        )

    startdate = data.query_params["startdate"]
    enddate = data.query_params["enddate"]

    dates = client.service.getdates(
        float(data.query_params["latitude"]),
        float(data.query_params["longitude"]),
        data.query_params["product"],
    )
    dates = np.array(dates)

    dr = pd.DatetimeIndex([date_parser([i])[0] for i in dates])

    teststartdate = dr[0]
    if startdate <= teststartdate:
        startdate = teststartdate

    testenddate = dr[-1]
    if enddate >= testenddate:
        enddate = testenddate

    tenddate = dates[-1]
    mask = (dr >= startdate) & (dr <= enddate)
    dates = dates[mask]
    dateintervals = dates[::10].tolist()
    if dateintervals[-1] != dates[-1]:
        dateintervals.append(dates[-1])

    datelist = []
    valuelist = []
    for idate, jdate in zip(dateintervals[:-1], dateintervals[1:]):
        di = client.service.getsubset(
            float(data.query_params["latitude"]),
            float(data.query_params["longitude"]),
            data.query_params["product"],
            data.query_params["band"],
            idate,
            jdate,
            0,
            0,
        )
        try:
            testv = di["subset"]["_value_1"]
        except TypeError:
            testv = di["subset"]

        if testv is None:
            continue
        for d in testv:
            _, _, d, _, _, v = d.split(",")
            datelist.append(d)
            valuelist.append(float(v.strip()))

    df = pd.DataFrame(valuelist, index=date_parser(datelist))
    df.index.name = "Datetime"
    df.columns = [
        data.query_params["product"]
        + "_"
        + data.query_params["band"]
        + ":"
        + di["units"]
    ]
    if di["scale"] != 0:
        df = df * di["scale"]
    return df


if __name__ == "__main__":
    r = resource(
        r"https://modis.ornl.gov/cgi-bin/MODIS/soapservice/MODIS_soapservice.wsdl",
        product="MOD13Q1",
        band="250m_16_days_NDVI",
        latitude=40.0,
        longitude=-110.0,
        startdate="2002-06-01T09",
        enddate="2003-05-04T21",
    )

    as_df = odo(r, pd.DataFrame)
    print("modis")
    print(as_df)

    r = resource(
        r"https://modis.ornl.gov/cgi-bin/MODIS/soapservice/MODIS_soapservice.wsdl",
        product="MOD15A2H",
        band="LaiStdDev_500m",
        latitude=29.65,
        longitude=-82.32,
        startdate="3 years ago",
        enddate="2 years ago",
    )

    as_df = odo(r, pd.DataFrame)
    print("modis")
    print(as_df)
