"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.renderJsonPath = exports.renderList = exports.State = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const constructs_1 = require("constructs");
const fields_1 = require("../fields");
const types_1 = require("../types");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_1 = require("@aws-cdk/core");
/**
 * Base class for all other state classes
 */
class State extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.branches = [];
        this.retries = [];
        this.catches = [];
        this.choices = [];
        this.prefixes = [];
        /**
         * States with references to this state.
         *
         * Used for finding complete connected graph that a state is part of.
         */
        this.incomingStates = [];
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_StateProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, State);
            }
            throw error;
        }
        this.startState = this;
        this.comment = props.comment;
        this.inputPath = props.inputPath;
        this.parameters = props.parameters;
        this.outputPath = props.outputPath;
        this.resultPath = props.resultPath;
        this.resultSelector = props.resultSelector;
    }
    /**
     * Add a prefix to the stateId of all States found in a construct tree
     */
    static prefixStates(root, prefix) {
        const queue = [root];
        while (queue.length > 0) {
            const el = queue.splice(0, 1)[0];
            if (isPrefixable(el)) {
                el.addPrefix(prefix);
            }
            queue.push(...constructs_1.Node.of(el).children);
        }
    }
    /**
     * Find the set of states reachable through transitions from the given start state.
     * This does not retrieve states from within sub-graphs, such as states within a Parallel state's branch.
     */
    static findReachableStates(start, options = {}) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_State(start);
            jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_FindStateOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.findReachableStates);
            }
            throw error;
        }
        const visited = new Set();
        const ret = new Set();
        const queue = [start];
        while (queue.length > 0) {
            const state = queue.splice(0, 1)[0];
            if (visited.has(state)) {
                continue;
            }
            visited.add(state);
            const outgoing = state.outgoingTransitions(options);
            queue.push(...outgoing);
            ret.add(state);
        }
        return Array.from(ret);
    }
    /**
     * Find the set of end states states reachable through transitions from the given start state
     */
    static findReachableEndStates(start, options = {}) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_State(start);
            jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_FindStateOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.findReachableEndStates);
            }
            throw error;
        }
        const visited = new Set();
        const ret = new Set();
        const queue = [start];
        while (queue.length > 0) {
            const state = queue.splice(0, 1)[0];
            if (visited.has(state)) {
                continue;
            }
            visited.add(state);
            const outgoing = state.outgoingTransitions(options);
            if (outgoing.length > 0) {
                // We can continue
                queue.push(...outgoing);
            }
            else {
                // Terminal state
                ret.add(state);
            }
        }
        return Array.from(ret);
    }
    /**
     * Return only the states that allow chaining from an array of states
     */
    static filterNextables(states) {
        return states.filter(isNextable);
    }
    get id() {
        return this.node.id;
    }
    /**
     * Tokenized string that evaluates to the state's ID
     */
    get stateId() {
        return this.prefixes.concat(this.id).join('');
    }
    /**
     * Add a prefix to the stateId of this state
     */
    addPrefix(x) {
        if (x !== '') {
            this.prefixes.splice(0, 0, x);
        }
    }
    /**
     * Register this state as part of the given graph
     *
     * Don't call this. It will be called automatically when you work
     * with states normally.
     */
    bindToGraph(graph) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_StateGraph(graph);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.bindToGraph);
            }
            throw error;
        }
        if (this.containingGraph === graph) {
            return;
        }
        if (this.containingGraph) {
            // eslint-disable-next-line max-len
            throw new Error(`Trying to use state '${this.stateId}' in ${graph}, but is already in ${this.containingGraph}. Every state can only be used in one graph.`);
        }
        this.containingGraph = graph;
        this.whenBoundToGraph(graph);
        for (const incoming of this.incomingStates) {
            incoming.bindToGraph(graph);
        }
        for (const outgoing of this.outgoingTransitions({ includeErrorHandlers: true })) {
            outgoing.bindToGraph(graph);
        }
        for (const branch of this.branches) {
            branch.registerSuperGraph(this.containingGraph);
        }
        if (!!this.iteration) {
            this.iteration.registerSuperGraph(this.containingGraph);
        }
    }
    /**
     * Add a retrier to the retry list of this state
     * @internal
     */
    _addRetry(props = {}) {
        validateErrors(props.errors);
        this.retries.push({
            ...props,
            errors: props.errors ?? [types_1.Errors.ALL],
        });
    }
    /**
     * Add an error handler to the catch list of this state
     * @internal
     */
    _addCatch(handler, props = {}) {
        validateErrors(props.errors);
        this.catches.push({
            next: handler,
            props: {
                errors: props.errors ?? [types_1.Errors.ALL],
                resultPath: props.resultPath,
            },
        });
        handler.addIncoming(this);
        if (this.containingGraph) {
            handler.bindToGraph(this.containingGraph);
        }
    }
    /**
     * Make the indicated state the default transition of this state
     */
    makeNext(next) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_State(next);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.makeNext);
            }
            throw error;
        }
        // Can't be called 'setNext' because of JSII
        if (this._next) {
            throw new Error(`State '${this.id}' already has a next state`);
        }
        this._next = next;
        next.addIncoming(this);
        if (this.containingGraph) {
            next.bindToGraph(this.containingGraph);
        }
    }
    /**
     * Add a choice branch to this state
     */
    addChoice(condition, next) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_Condition(condition);
            jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_State(next);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addChoice);
            }
            throw error;
        }
        this.choices.push({ condition, next });
        next.startState.addIncoming(this);
        if (this.containingGraph) {
            next.startState.bindToGraph(this.containingGraph);
        }
    }
    /**
     * Add a paralle branch to this state
     */
    addBranch(branch) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_StateGraph(branch);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addBranch);
            }
            throw error;
        }
        this.branches.push(branch);
        if (this.containingGraph) {
            branch.registerSuperGraph(this.containingGraph);
        }
    }
    /**
     * Add a map iterator to this state
     */
    addIterator(iteration) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_StateGraph(iteration);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addIterator);
            }
            throw error;
        }
        this.iteration = iteration;
        if (this.containingGraph) {
            iteration.registerSuperGraph(this.containingGraph);
        }
    }
    /**
     * Make the indicated state the default choice transition of this state
     */
    makeDefault(def) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_State(def);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.makeDefault);
            }
            throw error;
        }
        // Can't be called 'setDefault' because of JSII
        if (this.defaultChoice) {
            throw new Error(`Choice '${this.id}' already has a default next state`);
        }
        this.defaultChoice = def;
    }
    /**
     * Render the default next state in ASL JSON format
     */
    renderNextEnd() {
        if (this._next) {
            return { Next: this._next.stateId };
        }
        else {
            return { End: true };
        }
    }
    /**
     * Render the choices in ASL JSON format
     */
    renderChoices() {
        return {
            Choices: renderList(this.choices, renderChoice),
            Default: this.defaultChoice?.stateId,
        };
    }
    /**
     * Render InputPath/Parameters/OutputPath in ASL JSON format
     */
    renderInputOutput() {
        return {
            InputPath: renderJsonPath(this.inputPath),
            Parameters: this.parameters,
            OutputPath: renderJsonPath(this.outputPath),
        };
    }
    /**
     * Render parallel branches in ASL JSON format
     */
    renderBranches() {
        return {
            Branches: this.branches.map(b => b.toGraphJson()),
        };
    }
    /**
     * Render map iterator in ASL JSON format
     */
    renderIterator() {
        if (!this.iteration) {
            throw new Error('Iterator must not be undefined !');
        }
        return {
            Iterator: this.iteration.toGraphJson(),
        };
    }
    /**
     * Render error recovery options in ASL JSON format
     */
    renderRetryCatch() {
        return {
            Retry: renderList(this.retries, renderRetry, (a, b) => compareErrors(a.errors, b.errors)),
            Catch: renderList(this.catches, renderCatch, (a, b) => compareErrors(a.props.errors, b.props.errors)),
        };
    }
    /**
     * Render ResultSelector in ASL JSON format
     */
    renderResultSelector() {
        return fields_1.FieldUtils.renderObject({
            ResultSelector: this.resultSelector,
        });
    }
    /**
     * Called whenever this state is bound to a graph
     *
     * Can be overridden by subclasses.
     */
    whenBoundToGraph(graph) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_StateGraph(graph);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.whenBoundToGraph);
            }
            throw error;
        }
        graph.registerState(this);
    }
    /**
     * Add a state to the incoming list
     */
    addIncoming(source) {
        this.incomingStates.push(source);
    }
    /**
     * Return all states this state can transition to
     */
    outgoingTransitions(options) {
        const ret = new Array();
        if (this._next) {
            ret.push(this._next);
        }
        if (this.defaultChoice) {
            ret.push(this.defaultChoice);
        }
        for (const c of this.choices) {
            ret.push(c.next);
        }
        if (options.includeErrorHandlers) {
            for (const c of this.catches) {
                ret.push(c.next);
            }
        }
        return ret;
    }
}
exports.State = State;
_a = JSII_RTTI_SYMBOL_1;
State[_a] = { fqn: "@aws-cdk/aws-stepfunctions.State", version: "1.190.0" };
/**
 * Render a choice transition
 */
function renderChoice(c) {
    return {
        ...c.condition.renderCondition(),
        Next: c.next.stateId,
    };
}
/**
 * Render a Retry object to ASL
 */
function renderRetry(retry) {
    return {
        ErrorEquals: retry.errors,
        IntervalSeconds: retry.interval && retry.interval.toSeconds(),
        MaxAttempts: retry.maxAttempts,
        BackoffRate: retry.backoffRate,
    };
}
/**
 * Render a Catch object to ASL
 */
function renderCatch(c) {
    return {
        ErrorEquals: c.props.errors,
        ResultPath: renderJsonPath(c.props.resultPath),
        Next: c.next.stateId,
    };
}
/**
 * Compares a list of Errors to move Errors.ALL last in a sort function
 */
function compareErrors(a, b) {
    if (a?.includes(types_1.Errors.ALL)) {
        return 1;
    }
    if (b?.includes(types_1.Errors.ALL)) {
        return -1;
    }
    return 0;
}
/**
 * Validates an errors list
 */
function validateErrors(errors) {
    if (errors?.includes(types_1.Errors.ALL) && errors.length > 1) {
        throw new Error(`${types_1.Errors.ALL} must appear alone in an error list`);
    }
}
/**
 * Render a list or return undefined for an empty list
 */
function renderList(xs, mapFn, sortFn) {
    if (xs.length === 0) {
        return undefined;
    }
    let list = xs;
    if (sortFn) {
        list = xs.sort(sortFn);
    }
    return list.map(mapFn);
}
exports.renderList = renderList;
/**
 * Render JSON path, respecting the special value DISCARD
 */
function renderJsonPath(jsonPath) {
    if (jsonPath === undefined) {
        return undefined;
    }
    if (jsonPath === fields_1.JsonPath.DISCARD) {
        return null;
    }
    if (!jsonPath.startsWith('$')) {
        throw new Error(`Expected JSON path to start with '$', got: ${jsonPath}`);
    }
    return jsonPath;
}
exports.renderJsonPath = renderJsonPath;
/**
 * Whether an object is a Prefixable
 */
function isPrefixable(x) {
    return typeof (x) === 'object' && x.addPrefix;
}
/**
 * Whether an object is INextable
 */
function isNextable(x) {
    return typeof (x) === 'object' && x.next;
}
//# sourceMappingURL=data:application/json;base64,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