#!/usr/bin/env python
# -*- coding: utf-8 -*-
''' SIDEX miscellaneous functions.

This module provides miscellaneous functions to setup the SIDEX server.
The behaviours of the `get`, `put`, and `delete` methods are defined.

Use the `setup` function to launch a customized SIDEX server.
'''
from flask import Flask, Response
from flask import request, url_for, render_template
import os, re, io, tarfile, logging, requests

werkzeug = logging.getLogger('werkzeug')
werkzeug.setLevel('ERROR')

app = Flask(__name__)


@app.context_processor
def override_url_for():
  ''' A helper function to construct a url with `subdir` option.

  This does nothing when the `subdir` option is not defined.
  The url is modified in case that the `subdir` is given. The modified
  `url_for` function accepts the following arguments.

  Args:
    endpoint (str):
        The relative path to the requested resource.
    *args:
        Variable length argument list.
    **options:
        Arbitrary option arguments.

  Returns:
    str: A constructed url to the requeste resource.
  '''
  def url_for_subdir(endpoint, *args, **options):
    subdir = '/{}'.format(app.subdir) if app.subdir else ''
    return subdir+url_for(endpoint, *args, **options)
  return dict(url_for=url_for_subdir)


def eprint(message, exc_info=None):
  ''' Print an error message in the log file.

  Args:
    message (str):
        The body of the error message.
    exc_info (Exception,optional):
        The Exception instance to print traceback information.
  '''
  remote = request.remote_addr
  app.logger.error(remote+':{}'.format(message), exc_info=exc_info)


def iprint(message):
  ''' Print an info message in the log file.

  Args:
    message (str):
        The body of the information message.
  '''
  remote = request.remote_addr
  app.logger.info(remote+':{}'.format(message))


def dprint(message):
  ''' Print a debug message in the log file.

  Args:
    message (str):
        The body of the debug message.
  '''
  remote = request.remote_addr
  app.logger.debug(remote+':{}'.format(message))


def invalid_path(path):
  ''' Check whether the <path> looks invalid.

  Args:
    path (str):
        The path to the file to be checked.

  Returns:
    bool:
        `true` when the <path> contains any invalid sequences.
  '''
  return '../' in path


def default_delete_function(req, local_path, **options):
  ''' Define the default behavior of the `delete` method.

  This function removes the file located at <local_path>.

  Args:
    req (flask.request):
        The request instance given by Flask.
    local_path (str):
        The absolute path to the requested resource.
    **options:
        Arbitrary option arguments.
        Currently no option is passed to this function.

  Returns:
    flask.Response:
        A response message.
  '''
  filename = os.path.basename(local_path)
  os.unlink(local_path)
  return Response('"{}" successfully deleted.\n'.format(filename))


def delete(req, target, **options):
  ''' Provide the `delete` method.

  This function is called when the `delete` method is selected.
  The <app.delete_function> is called internally. When the customized
  `delete_function` is specified, the behavior of the `delete` method
  is overridden.

  This method is disabled by default. To enable the `delete` method,
  the application should be setup with `delete_token`. The token is
  reffered to as <app.delete_token>.

  The request must contain the `token` field. When the `token` is not
  available, this always returns an error message.
  In case that the `token` does not equal to <app.delete_token>,
  an error message will be returned.

  Args:
    req (flask.request):
        A request instance generated by Flask.
    target (str):
        The path to the requested resource.
    **options:
        Arbitrary option arguments.
        Currently no option is passed to this function.

  Returns:
    flask.Response:
        A response message.
  '''
  local_path = '{}/{}'.format(app.workdir,target)
  filename = os.path.basename(local_path)
  dirname = os.path.dirname(local_path)
  emsg = lambda s: 'cannot delete "{}": {{}}.\n'.format(target).format(s)
  ## check if a valid token is given.
  if app.delete_token is None:
    eprint('disabled function "delete" called.')
    return Response(emsg('function disabled'), status=400)
  token = req.form.get('token')
  dprint('token = "{}"'.format(token))
  if app.delete_token is not None and token != app.delete_token:
    return Response(emsg('invalid token'), status=400)
  ## assert path seems valid.
  if invalid_path(target):
    eprint('invalid path: {}'.format(target))
    return Response(emsg('invalid path'), status=400)
  ## delete a file.
  try:
    return app.delete_function(req,local_path, **options)
  except FileNotFoundError as e:
    eprint(str(e))
    return Response(emsg('file not found'), status=404)
  except Exception as e:
    eprint(str(e))
    errmsg = emsg('unexpected error: {}'.format(e.__class__.__name__))
    return Response(errmsg, status=500)


def default_put_function(req, local_path, **options):
  ''' Define the default behavior of the `put` method.

  This function create a file stored in <req> at <local_path>.
  This fails when the local directory does not exist. Note that
  any existing file cannot be overwritten.

  Args:
    req (flask.request):
        The request instance given by Flask.
    local_path (str):
        The absolute path to the requested resource.
    **options:
        Arbitrary option arguments.
        Currently no option is passed to this function.

  Returns:
    flask.Response:
        A response message.
  '''
  filename = os.path.basename(local_path)
  dirname = os.path.dirname(local_path)
  req.files['payload'].save(local_path)
  return Response('successfully uploaded to "{}".\n'.format(filename))


def put(req, target, **options):
  ''' Provide the `put` method.

  This function is called when the `put` method is selected.
  The <app.put_function> is called internally. When the customized
  `put_function` is specified, the behavior of the `put` method
  is overridden.

  This method is disabled by default. To enable the `put` method,
  the application should be setup with `put_token`. The token is
  reffered to as <app.put_token>.

  The request must contain the `token` field. When the `token` is not
  available, this always returns an error message.
  In case that the `token` does not equal to <app.put_token>,
  an error message will be returned.

  Args:
    req (flask.request):
        A request instance generated by Flask.
    target (str):
        The path to the requested resource.
    **options:
        Arbitrary option arguments.
        Currently no option is passed to this function.

  Returns:
    flask.Response:
        A response message.
  '''
  local_path = '{}/{}'.format(app.workdir,target)
  filename = os.path.basename(local_path)
  emsg = lambda s: 'cannot create "{}": {{}}.\n'.format(target).format(s)
  ## check if a valid token is given.
  if app.put_token is None:
    eprint('disabled function "put" called.')
    return Response(emsg('function disabled'), status=400)
  token = req.form.get('token')
  dprint('token = "{}"'.format(token))
  if app.put_token is not None and token != app.put_token:
    return Response(emsg('invalid token'), status=400)
  ## assert path seems valid.
  if invalid_path(target):
    eprint('invalid path: {}'.format(target))
    return Response(emsg('invalid path'), status=400)
  ## overwrite is not allowed.
  if os.path.exists(local_path):
    eprint('file "{}" already exists.'.format(local_path))
    return Response(emsg('cannot overwrite files'), status=400)
  ## create a file.
  if 'payload' not in req.files:
    return Response(emsg('"payload" is required'), status=400)
  try:
    return app.put_function(req,local_path, **options)
  except FileNotFoundError as e:
    eprint(str(e))
    return Response(emsg('file not found'), status=404)
  except Exception as e:
    eprint(str(e))
    errmsg = emsg('unexpected error: {}'.format(e.__class__.__name__))
    return Response(errmsg, status=500)


def default_get_function(req, local_path, **options):
  ''' Define the default behavior of the `get` method.

  Returns the file content located at <local_path>.

  Args:
    req (flask.request):
        The request instance given by Flask.
    local_path (str):
        The absolute path to the requested resource.
    **options:
        Arbitrary option arguments.
        Currently no option is passed to this function.

  Returns:
    flask.Response:
        A response message.
  '''
  with open(local_path, 'rb') as f:
    return Response(f.read(), mimetype='application/octet-stream')


def get(req, target, **options):
  ''' Provide the `get` method.

  This function is called when the `get` method is selected.
  The <app.get_function> is called internally. When the customized
  `get_function` is specified, the behavior of the `get` method
  is overridden.

  This method can be protected by setting `get_token`. The token is
  reffered to as <app.get_token>. When <app.get_token> is defined,
  the request must contain the `token` field.
  In case that the `token` does not equal to <app.get_token>,
  an error message will be returned.

  Args:
    req (flask.request):
        A request instance generated by Flask.
    target (str):
        The path to the requested resource.
    **options:
        Arbitrary option arguments.
        Currently no option is passed to this function.

  Returns:
    flask.Response:
        A response message.
  '''
  local_path = '{}/{}'.format(app.workdir,target)
  filename = os.path.basename(local_path)
  emsg = lambda s: 'cannot access "{}": {{}}.\n'.format(target).format(s)
  ## check if a valid token is given.
  if app.get_token is not None:
    dprint('"get" function requres a token.')
    token = req.form.get('token')
    dprint('token = "{}"'.format(token))
    if token != app.get_token:
      return Response(emsg('invalid token'), status=400)
  ## assert path seems valid.
  if invalid_path(target):
    eprint('invalid path: {}'.format(target))
    return Response(emsg('invalid path'), status=500)
  ## access to file.
  try:
    return app.get_function(req, local_path, **options)
  except FileNotFoundError as e:
    eprint(str(e))
    return Response(emsg('file not found'), status=404)
  except IsADirectoryError as e:
    eprint(str(e))
    return Response(emsg('cannot obtain directory'), status=400)
  except Exception as e:
    eprint(str(e))
    errmsg = emsg('unexpected error: {}'.format(e.__class__.__name__))
    return Response(errmsg, status=500)


def default_dump_function(req, local_paths, **options):
  ''' Define the default behavior of the `dump` method.

  Returns the file content located at <local_path>.

  Args:
    req (flask.request):
        The request instance given by Flask.
    local_paths (list):
        The list of the absolute paths to the requested resources.
    **options:
        Arbitrary option arguments.
        Currently no option is passed to this function.

  Returns:
    flask.Response:
        A response message.
  '''
  buf = io.BytesIO()
  with tarfile.open(fileobj=buf, mode='w') as arv:
    for filename in local_paths: arv.add(filename)
  buf.seek(0)
  return Response(buf.read(), mimetype='application/octet-stream')


def dump(req, filelist, **options):
  ''' Provide the `dump` method.

  This function is called when the `dump` method is selected.
  The <app.dump_function> is called internally. When the customized
  `dump_function` is specified, the behavior of the `dump` method
  is overridden.

  This method can be protected by setting `get_token`. The token is
  reffered to as <app.get_token>. When <app.get_token> is defined,
  the request must contain the `token` field.
  In case that the `token` does not equal to <app.get_token>,
  an error message will be returned.

  Args:
    req (flask.request):
        A request instance generated by Flask.
    filelist (list):
        The list of the paths to the requested resources.
    **options:
        Arbitrary option arguments.
        Currently no option is passed to this function.

  Returns:
    flask.Response:
        A response message.
  '''
  local_paths = ['{}/{}'.format(app.workdir,f) for f in filelist]
  filenames = [os.path.basename(l) for l in local_paths]
  emsg = lambda s: 'cannot access resources: {}.\n'.format(s)
  ## check if a valid token is given.
  if app.get_token is not None:
    dprint('"get" function requres a token.')
    token = req.form.get('token')
    dprint('token = "{}"'.format(token))
    if token != app.get_token:
      return Response(emsg('invalid token'), status=400)
  ## assert path seems valid.
  for target in filelist:
    if invalid_path(target):
      eprint('invalid path: {}'.format(target))
      return Response(emsg('invalid path'), status=500)
  ## access to file.
  try:
    return app.dump_function(req, local_paths, **options)
  except FileNotFoundError as e:
    eprint(str(e))
    return Response(emsg('file not found'), status=404)
  except IsADirectoryError as e:
    eprint(str(e))
    return Response(emsg('cannot obtain directory'), status=400)
  except Exception as e:
    eprint(str(e))
    errmsg = emsg('unexpected error: {}'.format(e.__class__.__name__))
    return Response(errmsg, status=500)


@app.route('/<path:target>', methods=['GET',])
def access_by_get(target):
  ''' The access point of the SIDEX server.

  This function handles the GET request to the SIDEX server.
  This works only if the `token` is not specified.

  Args:
    target (str):
        The path to the requested resource.

  Returns:
    flask.Response:
        The response instance from the requested method.
  '''
  return get(request, target)


@app.route('/<path:target>', methods=['POST',])
def access_get_by_post(target):
  ''' The access point of the SIDEX server.

  This function handles the POST request to the SIDEX server.
  The `method` field is mandatory. The value should be one of `get`,
  `put` and `delete`. The corresponding function will be called.

  The `token` field is required in the `put` and `delete` methods.

  Args:
    target (str):
        The path to the requested resource.

  Returns:
    flask.Response:
        The response instance from the requested method.
  '''
  method = request.form.get('method','none').lower()
  if method == 'get':
    return get(request, target)
  elif method == 'put':
    return put(request, target)
  elif method == 'delete':
    return delete(request, target)
  else:
    return Response('invalid method: {}.\n'.format(method), status=400)


@app.route('/', methods=['GET', 'POST'])
def root():
  if request.method == 'GET':
    return Response('It works! The SIDEX server is running.\n')
  else:
    method = request.form.get('method','none').lower()
    if method == 'dump':
      filenames = request.form.getlist('filename')
      return dump(request, filenames)
    else:
      return Response('invalid method: {}.\n'.format(method), status=400)


def setup(
    workdir, subdir=None,
    get_function=default_get_function,
    put_function=default_put_function,
    delete_function=default_delete_function,
    dump_function=default_dump_function,
    get_token=None, put_token=None, delete_token=None,
    log_handler=None, log_level='INFO'):
  ''' Setup a customized SIDEX server.

  Args:
    workdir (str):
        The relative path to the working directory.
    subdir (str,optional):
        The name of the subdirectory.
    get_function (function,optional):
        The function instance called in the `get` method.
        The following arguments are required:
            - req (flask.request): The flask request instance.
            - local_path (str): The local path to the resource.
    put_function (function,optional):
        The function instance called in the `put` method.
        The following arguments are required:
            - req (flask.request): The flask request instance.
            - local_path (str): The local path to the resource.
    delete_function (function,optional):
        The function instance called in the `delete` method.
        The following arguments are required:
            - req (flask.request): The flask request instance.
            - local_path (str): The local path to the resource.
    dump_function (function,optional):
        The function instance called in the `dump` method.
        The following arguments are required:
            - req (flask.request): The flask request instance.
            - local_paths (list): The path list to the resources.
    get_token (str,optional):
        The secret token for the `get` method.
    put_token (str,optional):
        The secret token for the `put` method.
    delete_token (str,optional):
        The secret token for the `delete` method.
    log_handler (logger.Logger,optional):
        The user-defined log handler to override the default handler.
    log_level (str,optional):
        The log_level. `INFO` is given by default.
  '''
  if get_token is not None: assert len(get_token) > 0
  if put_token is not None: assert len(put_token) > 0
  if delete_token is not None: assert len(delete_token) > 0
  app.workdir = workdir
  app.get_function = get_function
  app.put_function = put_function
  app.delete_function = delete_function
  app.dump_function = dump_function
  app.get_token = get_token
  app.put_token = put_token
  app.delete_token = delete_token
  app.subdir = subdir
  app.logger.setLevel(log_level)
  if log_handler is not None:
    werkzeug.handlers = []
    app.logger.handlers = [log_handler,]
  return app
