import re
import unidecode
import textnorm
import nltk

# Download stpwords
nltk.download("stopwords")

# Import nltk for stopwords
from nltk.corpus import stopwords


def strong_normalizer(
    string, remove_numbers=True, remove_punctuations=True, remove_whitespaces=True
):
    """
    This is a very string normalizer string that remove almost anything except words

    Parameters
    ----------
    string: str
        String to get normalized
    remove_numbers: bool
        If the algorithm should remove numbers (Default: True)
    remove_punctuations: bool
        If the algorithm should remove punctuations symbols (Default: True)
    remove_whitespaces: bool
        If the algorithm should remove spaces (Default: True)

    Return
    ------
    string: str
        String already normalized
    """

    # Get stop words
    stop_words = set(stopwords.words("english"))

    # Normalize unicode
    string = textnorm.normalize_unicode(string)

    # Normalize spaces
    string = textnorm.normalize_space(string)

    # Apply unidecode to plain string
    string = unidecode.unidecode(string)

    # Lower it
    string = string.lower()

    # Remove numbers
    if remove_numbers:
        string = re.sub(r"\d+", "", string)

    # Remove all punctuation except words and space
    if remove_punctuations:
        string = re.sub(r"[^\w\s]", "", string)

    # Remove white spaces
    if remove_whitespaces:
        string = string.strip()

    # Convert string to list of words
    lst_string = [string][0].split()

    # Remove stopwords
    clean_string = ""
    for i in lst_string:
        if i not in stop_words:
            clean_string += i + " "

    # Removing last space
    string = clean_string[:-1]

    # Return result
    return string


if __name__ == "__main__":

    b64test = """U2FsdGFyIGFsIGNvbnRlbmlkbyBwcmluY2lwYWwKICAgICAgICAgICAgICAgIE1lbsO6CgogICAg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"""
    import base64

    test = base64.b64decode(b64test).decode()
    print(strong_normalizer(test))
