// Copyright 2022 DeepMind Technologies Limited
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#ifndef MUJOCO_PYTHON_CODEGEN_FUNCTION_TRAITS_H_
#define MUJOCO_PYTHON_CODEGEN_FUNCTION_TRAITS_H_

#include <tuple>

#include <mujoco.h>
#include "util/crossplatform.h"

namespace mujoco::python_traits {

struct mj_defaultVFS {
  static constexpr char name[] = "mj_defaultVFS";
  static constexpr char doc[] = "Initialize VFS to empty (no deallocation).";
  using type = void (mjVFS *);
  static constexpr auto param_names = std::make_tuple("vfs");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_defaultVFS;
  }
};

struct mj_addFileVFS {
  static constexpr char name[] = "mj_addFileVFS";
  static constexpr char doc[] = "Add file to VFS, return 0: success, 1: full, 2: repeated name, -1: failed to load.";
  using type = int (mjVFS *, const char *, const char *);
  static constexpr auto param_names = std::make_tuple("vfs", "directory", "filename");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_addFileVFS;
  }
};

struct mj_makeEmptyFileVFS {
  static constexpr char name[] = "mj_makeEmptyFileVFS";
  static constexpr char doc[] = "Make empty file in VFS, return 0: success, 1: full, 2: repeated name.";
  using type = int (mjVFS *, const char *, int);
  static constexpr auto param_names = std::make_tuple("vfs", "filename", "filesize");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_makeEmptyFileVFS;
  }
};

struct mj_findFileVFS {
  static constexpr char name[] = "mj_findFileVFS";
  static constexpr char doc[] = "Return file index in VFS, or -1 if not found in VFS.";
  using type = int (const mjVFS *, const char *);
  static constexpr auto param_names = std::make_tuple("vfs", "filename");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_findFileVFS;
  }
};

struct mj_deleteFileVFS {
  static constexpr char name[] = "mj_deleteFileVFS";
  static constexpr char doc[] = "Delete file from VFS, return 0: success, -1: not found in VFS.";
  using type = int (mjVFS *, const char *);
  static constexpr auto param_names = std::make_tuple("vfs", "filename");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_deleteFileVFS;
  }
};

struct mj_deleteVFS {
  static constexpr char name[] = "mj_deleteVFS";
  static constexpr char doc[] = "Delete all files from VFS.";
  using type = void (mjVFS *);
  static constexpr auto param_names = std::make_tuple("vfs");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_deleteVFS;
  }
};

struct mj_loadXML {
  static constexpr char name[] = "mj_loadXML";
  static constexpr char doc[] = "Parse XML file in MJCF or URDF format, compile it, return low-level model. If vfs is not NULL, look up files in vfs before reading from disk. If error is not NULL, it must have size error_sz.";
  using type = mjModel * (const char *, const mjVFS *, char *, int);
  static constexpr auto param_names = std::make_tuple("filename", "vfs", "error", "error_sz");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_loadXML;
  }
};

struct mj_saveLastXML {
  static constexpr char name[] = "mj_saveLastXML";
  static constexpr char doc[] = "Update XML data structures with info from low-level model, save as MJCF. If error is not NULL, it must have size error_sz.";
  using type = int (const char *, const mjModel *, char *, int);
  static constexpr auto param_names = std::make_tuple("filename", "m", "error", "error_sz");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_saveLastXML;
  }
};

struct mj_freeLastXML {
  static constexpr char name[] = "mj_freeLastXML";
  static constexpr char doc[] = "Free last XML model if loaded. Called internally at each load.";
  using type = void ();
  static constexpr auto param_names = std::make_tuple();

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_freeLastXML;
  }
};

struct mj_printSchema {
  static constexpr char name[] = "mj_printSchema";
  static constexpr char doc[] = "Print internal XML schema as plain text or HTML, with style-padding or &nbsp;.";
  using type = int (const char *, char *, int, int, int);
  static constexpr auto param_names = std::make_tuple("filename", "buffer", "buffer_sz", "flg_html", "flg_pad");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_printSchema;
  }
};

struct mj_step {
  static constexpr char name[] = "mj_step";
  static constexpr char doc[] = "Advance simulation, use control callback to obtain external force and control.";
  using type = void (const mjModel *, mjData *);
  static constexpr auto param_names = std::make_tuple("m", "d");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_step;
  }
};

struct mj_step1 {
  static constexpr char name[] = "mj_step1";
  static constexpr char doc[] = "Advance simulation in two steps: before external force and control is set by user.";
  using type = void (const mjModel *, mjData *);
  static constexpr auto param_names = std::make_tuple("m", "d");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_step1;
  }
};

struct mj_step2 {
  static constexpr char name[] = "mj_step2";
  static constexpr char doc[] = "Advance simulation in two steps: after external force and control is set by user.";
  using type = void (const mjModel *, mjData *);
  static constexpr auto param_names = std::make_tuple("m", "d");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_step2;
  }
};

struct mj_forward {
  static constexpr char name[] = "mj_forward";
  static constexpr char doc[] = "Forward dynamics: same as mj_step but do not integrate in time.";
  using type = void (const mjModel *, mjData *);
  static constexpr auto param_names = std::make_tuple("m", "d");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_forward;
  }
};

struct mj_inverse {
  static constexpr char name[] = "mj_inverse";
  static constexpr char doc[] = "Inverse dynamics: qacc must be set before calling.";
  using type = void (const mjModel *, mjData *);
  static constexpr auto param_names = std::make_tuple("m", "d");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_inverse;
  }
};

struct mj_forwardSkip {
  static constexpr char name[] = "mj_forwardSkip";
  static constexpr char doc[] = "Forward dynamics with skip; skipstage is mjtStage.";
  using type = void (const mjModel *, mjData *, int, int);
  static constexpr auto param_names = std::make_tuple("m", "d", "skipstage", "skipsensor");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_forwardSkip;
  }
};

struct mj_inverseSkip {
  static constexpr char name[] = "mj_inverseSkip";
  static constexpr char doc[] = "Inverse dynamics with skip; skipstage is mjtStage.";
  using type = void (const mjModel *, mjData *, int, int);
  static constexpr auto param_names = std::make_tuple("m", "d", "skipstage", "skipsensor");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_inverseSkip;
  }
};

struct mj_defaultLROpt {
  static constexpr char name[] = "mj_defaultLROpt";
  static constexpr char doc[] = "Set default options for length range computation.";
  using type = void (mjLROpt *);
  static constexpr auto param_names = std::make_tuple("opt");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_defaultLROpt;
  }
};

struct mj_defaultSolRefImp {
  static constexpr char name[] = "mj_defaultSolRefImp";
  static constexpr char doc[] = "Set solver parameters to default values.";
  using type = void (mjtNum *, mjtNum *);
  static constexpr auto param_names = std::make_tuple("solref", "solimp");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_defaultSolRefImp;
  }
};

struct mj_defaultOption {
  static constexpr char name[] = "mj_defaultOption";
  static constexpr char doc[] = "Set physics options to default values.";
  using type = void (mjOption *);
  static constexpr auto param_names = std::make_tuple("opt");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_defaultOption;
  }
};

struct mj_defaultVisual {
  static constexpr char name[] = "mj_defaultVisual";
  static constexpr char doc[] = "Set visual options to default values.";
  using type = void (mjVisual *);
  static constexpr auto param_names = std::make_tuple("vis");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_defaultVisual;
  }
};

struct mj_copyModel {
  static constexpr char name[] = "mj_copyModel";
  static constexpr char doc[] = "Copy mjModel, allocate new if dest is NULL.";
  using type = mjModel * (mjModel *, const mjModel *);
  static constexpr auto param_names = std::make_tuple("dest", "src");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_copyModel;
  }
};

struct mj_saveModel {
  static constexpr char name[] = "mj_saveModel";
  static constexpr char doc[] = "Save model to binary MJB file or memory buffer; buffer has precedence when given.";
  using type = void (const mjModel *, const char *, void *, int);
  static constexpr auto param_names = std::make_tuple("m", "filename", "buffer", "buffer_sz");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_saveModel;
  }
};

struct mj_loadModel {
  static constexpr char name[] = "mj_loadModel";
  static constexpr char doc[] = "Load model from binary MJB file. If vfs is not NULL, look up file in vfs before reading from disk.";
  using type = mjModel * (const char *, const mjVFS *);
  static constexpr auto param_names = std::make_tuple("filename", "vfs");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_loadModel;
  }
};

struct mj_deleteModel {
  static constexpr char name[] = "mj_deleteModel";
  static constexpr char doc[] = "Free memory allocation in model.";
  using type = void (mjModel *);
  static constexpr auto param_names = std::make_tuple("m");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_deleteModel;
  }
};

struct mj_sizeModel {
  static constexpr char name[] = "mj_sizeModel";
  static constexpr char doc[] = "Return size of buffer needed to hold model.";
  using type = int (const mjModel *);
  static constexpr auto param_names = std::make_tuple("m");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_sizeModel;
  }
};

struct mj_makeData {
  static constexpr char name[] = "mj_makeData";
  static constexpr char doc[] = "Allocate mjData correponding to given model. If the model buffer is unallocated the initial configuration will not be set.";
  using type = mjData * (const mjModel *);
  static constexpr auto param_names = std::make_tuple("m");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_makeData;
  }
};

struct mj_copyData {
  static constexpr char name[] = "mj_copyData";
  static constexpr char doc[] = "Copy mjData. m is only required to contain the size fields from MJMODEL_INTS.";
  using type = mjData * (mjData *, const mjModel *, const mjData *);
  static constexpr auto param_names = std::make_tuple("dest", "m", "src");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_copyData;
  }
};

struct mj_resetData {
  static constexpr char name[] = "mj_resetData";
  static constexpr char doc[] = "Reset data to defaults.";
  using type = void (const mjModel *, mjData *);
  static constexpr auto param_names = std::make_tuple("m", "d");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_resetData;
  }
};

struct mj_resetDataDebug {
  static constexpr char name[] = "mj_resetDataDebug";
  static constexpr char doc[] = "Reset data to defaults, fill everything else with debug_value.";
  using type = void (const mjModel *, mjData *, unsigned char);
  static constexpr auto param_names = std::make_tuple("m", "d", "debug_value");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_resetDataDebug;
  }
};

struct mj_resetDataKeyframe {
  static constexpr char name[] = "mj_resetDataKeyframe";
  static constexpr char doc[] = "Reset data, set fields from specified keyframe.";
  using type = void (const mjModel *, mjData *, int);
  static constexpr auto param_names = std::make_tuple("m", "d", "key");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_resetDataKeyframe;
  }
};

struct mj_stackAlloc {
  static constexpr char name[] = "mj_stackAlloc";
  static constexpr char doc[] = "Allocate array of specified size on mjData stack. Call mju_error on stack overflow.";
  using type = mjtNum * (mjData *, int);
  static constexpr auto param_names = std::make_tuple("d", "size");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_stackAlloc;
  }
};

struct mj_deleteData {
  static constexpr char name[] = "mj_deleteData";
  static constexpr char doc[] = "Free memory allocation in mjData.";
  using type = void (mjData *);
  static constexpr auto param_names = std::make_tuple("d");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_deleteData;
  }
};

struct mj_resetCallbacks {
  static constexpr char name[] = "mj_resetCallbacks";
  static constexpr char doc[] = "Reset all callbacks to NULL pointers (NULL is the default).";
  using type = void ();
  static constexpr auto param_names = std::make_tuple();

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_resetCallbacks;
  }
};

struct mj_setConst {
  static constexpr char name[] = "mj_setConst";
  static constexpr char doc[] = "Set constant fields of mjModel, corresponding to qpos0 configuration.";
  using type = void (mjModel *, mjData *);
  static constexpr auto param_names = std::make_tuple("m", "d");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_setConst;
  }
};

struct mj_setLengthRange {
  static constexpr char name[] = "mj_setLengthRange";
  static constexpr char doc[] = "Set actuator_lengthrange for specified actuator; return 1 if ok, 0 if error.";
  using type = int (mjModel *, mjData *, int, const mjLROpt *, char *, int);
  static constexpr auto param_names = std::make_tuple("m", "d", "index", "opt", "error", "error_sz");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_setLengthRange;
  }
};

struct mj_printFormattedModel {
  static constexpr char name[] = "mj_printFormattedModel";
  static constexpr char doc[] = "Print mjModel to text file, specifying format. float_format must be a valid printf-style format string for a single float value.";
  using type = void (const mjModel *, const char *, const char *);
  static constexpr auto param_names = std::make_tuple("m", "filename", "float_format");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_printFormattedModel;
  }
};

struct mj_printModel {
  static constexpr char name[] = "mj_printModel";
  static constexpr char doc[] = "Print model to text file.";
  using type = void (const mjModel *, const char *);
  static constexpr auto param_names = std::make_tuple("m", "filename");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_printModel;
  }
};

struct mj_printFormattedData {
  static constexpr char name[] = "mj_printFormattedData";
  static constexpr char doc[] = "Print mjData to text file, specifying format. float_format must be a valid printf-style format string for a single float value";
  using type = void (const mjModel *, mjData *, const char *, const char *);
  static constexpr auto param_names = std::make_tuple("m", "d", "filename", "float_format");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_printFormattedData;
  }
};

struct mj_printData {
  static constexpr char name[] = "mj_printData";
  static constexpr char doc[] = "Print data to text file.";
  using type = void (const mjModel *, mjData *, const char *);
  static constexpr auto param_names = std::make_tuple("m", "d", "filename");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_printData;
  }
};

struct mju_printMat {
  static constexpr char name[] = "mju_printMat";
  static constexpr char doc[] = "Print matrix to screen.";
  using type = void (const mjtNum *, int, int);
  static constexpr auto param_names = std::make_tuple("mat", "nr", "nc");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mju_printMat;
  }
};

struct mju_printMatSparse {
  static constexpr char name[] = "mju_printMatSparse";
  static constexpr char doc[] = "Print sparse matrix to screen.";
  using type = void (const mjtNum *, int, const int *, const int *, const int *);
  static constexpr auto param_names = std::make_tuple("mat", "nr", "rownnz", "rowadr", "colind");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mju_printMatSparse;
  }
};

struct mj_fwdPosition {
  static constexpr char name[] = "mj_fwdPosition";
  static constexpr char doc[] = "Run position-dependent computations.";
  using type = void (const mjModel *, mjData *);
  static constexpr auto param_names = std::make_tuple("m", "d");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_fwdPosition;
  }
};

struct mj_fwdVelocity {
  static constexpr char name[] = "mj_fwdVelocity";
  static constexpr char doc[] = "Run velocity-dependent computations.";
  using type = void (const mjModel *, mjData *);
  static constexpr auto param_names = std::make_tuple("m", "d");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_fwdVelocity;
  }
};

struct mj_fwdActuation {
  static constexpr char name[] = "mj_fwdActuation";
  static constexpr char doc[] = "Compute actuator force qfrc_actuation.";
  using type = void (const mjModel *, mjData *);
  static constexpr auto param_names = std::make_tuple("m", "d");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_fwdActuation;
  }
};

struct mj_fwdAcceleration {
  static constexpr char name[] = "mj_fwdAcceleration";
  static constexpr char doc[] = "Add up all non-constraint forces, compute qacc_unc.";
  using type = void (const mjModel *, mjData *);
  static constexpr auto param_names = std::make_tuple("m", "d");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_fwdAcceleration;
  }
};

struct mj_fwdConstraint {
  static constexpr char name[] = "mj_fwdConstraint";
  static constexpr char doc[] = "Run selected constraint solver.";
  using type = void (const mjModel *, mjData *);
  static constexpr auto param_names = std::make_tuple("m", "d");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_fwdConstraint;
  }
};

struct mj_Euler {
  static constexpr char name[] = "mj_Euler";
  static constexpr char doc[] = "Euler integrator, semi-implicit in velocity.";
  using type = void (const mjModel *, mjData *);
  static constexpr auto param_names = std::make_tuple("m", "d");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_Euler;
  }
};

struct mj_RungeKutta {
  static constexpr char name[] = "mj_RungeKutta";
  static constexpr char doc[] = "Runge-Kutta explicit order-N integrator.";
  using type = void (const mjModel *, mjData *, int);
  static constexpr auto param_names = std::make_tuple("m", "d", "N");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_RungeKutta;
  }
};

struct mj_invPosition {
  static constexpr char name[] = "mj_invPosition";
  static constexpr char doc[] = "Run position-dependent computations in inverse dynamics.";
  using type = void (const mjModel *, mjData *);
  static constexpr auto param_names = std::make_tuple("m", "d");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_invPosition;
  }
};

struct mj_invVelocity {
  static constexpr char name[] = "mj_invVelocity";
  static constexpr char doc[] = "Run velocity-dependent computations in inverse dynamics.";
  using type = void (const mjModel *, mjData *);
  static constexpr auto param_names = std::make_tuple("m", "d");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_invVelocity;
  }
};

struct mj_invConstraint {
  static constexpr char name[] = "mj_invConstraint";
  static constexpr char doc[] = "Apply the analytical formula for inverse constraint dynamics.";
  using type = void (const mjModel *, mjData *);
  static constexpr auto param_names = std::make_tuple("m", "d");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_invConstraint;
  }
};

struct mj_compareFwdInv {
  static constexpr char name[] = "mj_compareFwdInv";
  static constexpr char doc[] = "Compare forward and inverse dynamics, save results in fwdinv.";
  using type = void (const mjModel *, mjData *);
  static constexpr auto param_names = std::make_tuple("m", "d");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_compareFwdInv;
  }
};

struct mj_sensorPos {
  static constexpr char name[] = "mj_sensorPos";
  static constexpr char doc[] = "Evaluate position-dependent sensors.";
  using type = void (const mjModel *, mjData *);
  static constexpr auto param_names = std::make_tuple("m", "d");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_sensorPos;
  }
};

struct mj_sensorVel {
  static constexpr char name[] = "mj_sensorVel";
  static constexpr char doc[] = "Evaluate velocity-dependent sensors.";
  using type = void (const mjModel *, mjData *);
  static constexpr auto param_names = std::make_tuple("m", "d");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_sensorVel;
  }
};

struct mj_sensorAcc {
  static constexpr char name[] = "mj_sensorAcc";
  static constexpr char doc[] = "Evaluate acceleration and force-dependent sensors.";
  using type = void (const mjModel *, mjData *);
  static constexpr auto param_names = std::make_tuple("m", "d");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_sensorAcc;
  }
};

struct mj_energyPos {
  static constexpr char name[] = "mj_energyPos";
  static constexpr char doc[] = "Evaluate position-dependent energy (potential).";
  using type = void (const mjModel *, mjData *);
  static constexpr auto param_names = std::make_tuple("m", "d");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_energyPos;
  }
};

struct mj_energyVel {
  static constexpr char name[] = "mj_energyVel";
  static constexpr char doc[] = "Evaluate velocity-dependent energy (kinetic).";
  using type = void (const mjModel *, mjData *);
  static constexpr auto param_names = std::make_tuple("m", "d");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_energyVel;
  }
};

struct mj_checkPos {
  static constexpr char name[] = "mj_checkPos";
  static constexpr char doc[] = "Check qpos, reset if any element is too big or nan.";
  using type = void (const mjModel *, mjData *);
  static constexpr auto param_names = std::make_tuple("m", "d");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_checkPos;
  }
};

struct mj_checkVel {
  static constexpr char name[] = "mj_checkVel";
  static constexpr char doc[] = "Check qvel, reset if any element is too big or nan.";
  using type = void (const mjModel *, mjData *);
  static constexpr auto param_names = std::make_tuple("m", "d");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_checkVel;
  }
};

struct mj_checkAcc {
  static constexpr char name[] = "mj_checkAcc";
  static constexpr char doc[] = "Check qacc, reset if any element is too big or nan.";
  using type = void (const mjModel *, mjData *);
  static constexpr auto param_names = std::make_tuple("m", "d");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_checkAcc;
  }
};

struct mj_kinematics {
  static constexpr char name[] = "mj_kinematics";
  static constexpr char doc[] = "Run forward kinematics.";
  using type = void (const mjModel *, mjData *);
  static constexpr auto param_names = std::make_tuple("m", "d");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_kinematics;
  }
};

struct mj_comPos {
  static constexpr char name[] = "mj_comPos";
  static constexpr char doc[] = "Map inertias and motion dofs to global frame centered at CoM.";
  using type = void (const mjModel *, mjData *);
  static constexpr auto param_names = std::make_tuple("m", "d");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_comPos;
  }
};

struct mj_camlight {
  static constexpr char name[] = "mj_camlight";
  static constexpr char doc[] = "Compute camera and light positions and orientations.";
  using type = void (const mjModel *, mjData *);
  static constexpr auto param_names = std::make_tuple("m", "d");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_camlight;
  }
};

struct mj_tendon {
  static constexpr char name[] = "mj_tendon";
  static constexpr char doc[] = "Compute tendon lengths, velocities and moment arms.";
  using type = void (const mjModel *, mjData *);
  static constexpr auto param_names = std::make_tuple("m", "d");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_tendon;
  }
};

struct mj_transmission {
  static constexpr char name[] = "mj_transmission";
  static constexpr char doc[] = "Compute actuator transmission lengths and moments.";
  using type = void (const mjModel *, mjData *);
  static constexpr auto param_names = std::make_tuple("m", "d");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_transmission;
  }
};

struct mj_crb {
  static constexpr char name[] = "mj_crb";
  static constexpr char doc[] = "Run composite rigid body inertia algorithm (CRB).";
  using type = void (const mjModel *, mjData *);
  static constexpr auto param_names = std::make_tuple("m", "d");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_crb;
  }
};

struct mj_factorM {
  static constexpr char name[] = "mj_factorM";
  static constexpr char doc[] = "Compute sparse L'*D*L factorizaton of inertia matrix.";
  using type = void (const mjModel *, mjData *);
  static constexpr auto param_names = std::make_tuple("m", "d");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_factorM;
  }
};

struct mj_solveM {
  static constexpr char name[] = "mj_solveM";
  static constexpr char doc[] = "Solve linear system M * x = y using factorization:  x = inv(L'*D*L)*y";
  using type = void (const mjModel *, mjData *, mjtNum *, const mjtNum *, int);
  static constexpr auto param_names = std::make_tuple("m", "d", "x", "y", "n");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_solveM;
  }
};

struct mj_solveM2 {
  static constexpr char name[] = "mj_solveM2";
  static constexpr char doc[] = "Half of linear solve:  x = sqrt(inv(D))*inv(L')*y";
  using type = void (const mjModel *, mjData *, mjtNum *, const mjtNum *, int);
  static constexpr auto param_names = std::make_tuple("m", "d", "x", "y", "n");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_solveM2;
  }
};

struct mj_comVel {
  static constexpr char name[] = "mj_comVel";
  static constexpr char doc[] = "Compute cvel, cdof_dot.";
  using type = void (const mjModel *, mjData *);
  static constexpr auto param_names = std::make_tuple("m", "d");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_comVel;
  }
};

struct mj_passive {
  static constexpr char name[] = "mj_passive";
  static constexpr char doc[] = "Compute qfrc_passive from spring-dampers, viscosity and density.";
  using type = void (const mjModel *, mjData *);
  static constexpr auto param_names = std::make_tuple("m", "d");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_passive;
  }
};

struct mj_subtreeVel {
  static constexpr char name[] = "mj_subtreeVel";
  static constexpr char doc[] = "subtree linear velocity and angular momentum";
  using type = void (const mjModel *, mjData *);
  static constexpr auto param_names = std::make_tuple("m", "d");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_subtreeVel;
  }
};

struct mj_rne {
  static constexpr char name[] = "mj_rne";
  static constexpr char doc[] = "RNE: compute M(qpos)*qacc + C(qpos,qvel); flg_acc=0 removes inertial term.";
  using type = void (const mjModel *, mjData *, int, mjtNum *);
  static constexpr auto param_names = std::make_tuple("m", "d", "flg_acc", "result");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_rne;
  }
};

struct mj_rnePostConstraint {
  static constexpr char name[] = "mj_rnePostConstraint";
  static constexpr char doc[] = "RNE with complete data: compute cacc, cfrc_ext, cfrc_int.";
  using type = void (const mjModel *, mjData *);
  static constexpr auto param_names = std::make_tuple("m", "d");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_rnePostConstraint;
  }
};

struct mj_collision {
  static constexpr char name[] = "mj_collision";
  static constexpr char doc[] = "Run collision detection.";
  using type = void (const mjModel *, mjData *);
  static constexpr auto param_names = std::make_tuple("m", "d");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_collision;
  }
};

struct mj_makeConstraint {
  static constexpr char name[] = "mj_makeConstraint";
  static constexpr char doc[] = "Construct constraints.";
  using type = void (const mjModel *, mjData *);
  static constexpr auto param_names = std::make_tuple("m", "d");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_makeConstraint;
  }
};

struct mj_projectConstraint {
  static constexpr char name[] = "mj_projectConstraint";
  static constexpr char doc[] = "Compute inverse constaint inertia efc_AR.";
  using type = void (const mjModel *, mjData *);
  static constexpr auto param_names = std::make_tuple("m", "d");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_projectConstraint;
  }
};

struct mj_referenceConstraint {
  static constexpr char name[] = "mj_referenceConstraint";
  static constexpr char doc[] = "Compute efc_vel, efc_aref.";
  using type = void (const mjModel *, mjData *);
  static constexpr auto param_names = std::make_tuple("m", "d");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_referenceConstraint;
  }
};

struct mj_constraintUpdate {
  static constexpr char name[] = "mj_constraintUpdate";
  static constexpr char doc[] = "Compute efc_state, efc_force, qfrc_constraint, and (optionally) cone Hessians. If cost is not NULL, set *cost = s(jar) where jar = Jac*qacc-aref.";
  using type = void (const mjModel *, mjData *, const mjtNum *, mjtNum (*)[1], int);
  static constexpr auto param_names = std::make_tuple("m", "d", "jar", "cost", "flg_coneHessian");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mj_constraintUpdate);
  }
};

struct mj_addContact {
  static constexpr char name[] = "mj_addContact";
  static constexpr char doc[] = "Add contact to d->contact list; return 0 if success; 1 if buffer full.";
  using type = int (const mjModel *, mjData *, const mjContact *);
  static constexpr auto param_names = std::make_tuple("m", "d", "con");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_addContact;
  }
};

struct mj_isPyramidal {
  static constexpr char name[] = "mj_isPyramidal";
  static constexpr char doc[] = "Determine type of friction cone.";
  using type = int (const mjModel *);
  static constexpr auto param_names = std::make_tuple("m");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_isPyramidal;
  }
};

struct mj_isSparse {
  static constexpr char name[] = "mj_isSparse";
  static constexpr char doc[] = "Determine type of constraint Jacobian.";
  using type = int (const mjModel *);
  static constexpr auto param_names = std::make_tuple("m");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_isSparse;
  }
};

struct mj_isDual {
  static constexpr char name[] = "mj_isDual";
  static constexpr char doc[] = "Determine type of solver (PGS is dual, CG and Newton are primal).";
  using type = int (const mjModel *);
  static constexpr auto param_names = std::make_tuple("m");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_isDual;
  }
};

struct mj_mulJacVec {
  static constexpr char name[] = "mj_mulJacVec";
  static constexpr char doc[] = "Multiply dense or sparse constraint Jacobian by vector.";
  using type = void (const mjModel *, mjData *, mjtNum *, const mjtNum *);
  static constexpr auto param_names = std::make_tuple("m", "d", "res", "vec");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_mulJacVec;
  }
};

struct mj_mulJacTVec {
  static constexpr char name[] = "mj_mulJacTVec";
  static constexpr char doc[] = "Multiply dense or sparse constraint Jacobian transpose by vector.";
  using type = void (const mjModel *, mjData *, mjtNum *, const mjtNum *);
  static constexpr auto param_names = std::make_tuple("m", "d", "res", "vec");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_mulJacTVec;
  }
};

struct mj_jac {
  static constexpr char name[] = "mj_jac";
  static constexpr char doc[] = "Compute 3/6-by-nv end-effector Jacobian of global point attached to given body.";
  using type = void (const mjModel *, const mjData *, mjtNum *, mjtNum *, const mjtNum (*)[3], int);
  static constexpr auto param_names = std::make_tuple("m", "d", "jacp", "jacr", "point", "body");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mj_jac);
  }
};

struct mj_jacBody {
  static constexpr char name[] = "mj_jacBody";
  static constexpr char doc[] = "Compute body frame end-effector Jacobian.";
  using type = void (const mjModel *, const mjData *, mjtNum *, mjtNum *, int);
  static constexpr auto param_names = std::make_tuple("m", "d", "jacp", "jacr", "body");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_jacBody;
  }
};

struct mj_jacBodyCom {
  static constexpr char name[] = "mj_jacBodyCom";
  static constexpr char doc[] = "Compute body center-of-mass end-effector Jacobian.";
  using type = void (const mjModel *, const mjData *, mjtNum *, mjtNum *, int);
  static constexpr auto param_names = std::make_tuple("m", "d", "jacp", "jacr", "body");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_jacBodyCom;
  }
};

struct mj_jacGeom {
  static constexpr char name[] = "mj_jacGeom";
  static constexpr char doc[] = "Compute geom end-effector Jacobian.";
  using type = void (const mjModel *, const mjData *, mjtNum *, mjtNum *, int);
  static constexpr auto param_names = std::make_tuple("m", "d", "jacp", "jacr", "geom");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_jacGeom;
  }
};

struct mj_jacSite {
  static constexpr char name[] = "mj_jacSite";
  static constexpr char doc[] = "Compute site end-effector Jacobian.";
  using type = void (const mjModel *, const mjData *, mjtNum *, mjtNum *, int);
  static constexpr auto param_names = std::make_tuple("m", "d", "jacp", "jacr", "site");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_jacSite;
  }
};

struct mj_jacPointAxis {
  static constexpr char name[] = "mj_jacPointAxis";
  static constexpr char doc[] = "Compute translation end-effector Jacobian of point, and rotation Jacobian of axis.";
  using type = void (const mjModel *, mjData *, mjtNum *, mjtNum *, const mjtNum (*)[3], const mjtNum (*)[3], int);
  static constexpr auto param_names = std::make_tuple("m", "d", "jacPoint", "jacAxis", "point", "axis", "body");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mj_jacPointAxis);
  }
};

struct mj_name2id {
  static constexpr char name[] = "mj_name2id";
  static constexpr char doc[] = "Get id of object with specified name, return -1 if not found; type is mjtObj.";
  using type = int (const mjModel *, int, const char *);
  static constexpr auto param_names = std::make_tuple("m", "type", "name");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_name2id;
  }
};

struct mj_id2name {
  static constexpr char name[] = "mj_id2name";
  static constexpr char doc[] = "Get name of object with specified id, return 0 if invalid type or id; type is mjtObj.";
  using type = const char * (const mjModel *, int, int);
  static constexpr auto param_names = std::make_tuple("m", "type", "id");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_id2name;
  }
};

struct mj_fullM {
  static constexpr char name[] = "mj_fullM";
  static constexpr char doc[] = "Convert sparse inertia matrix M into full (i.e. dense) matrix.";
  using type = void (const mjModel *, mjtNum *, const mjtNum *);
  static constexpr auto param_names = std::make_tuple("m", "dst", "M");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_fullM;
  }
};

struct mj_mulM {
  static constexpr char name[] = "mj_mulM";
  static constexpr char doc[] = "Multiply vector by inertia matrix.";
  using type = void (const mjModel *, const mjData *, mjtNum *, const mjtNum *);
  static constexpr auto param_names = std::make_tuple("m", "d", "res", "vec");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_mulM;
  }
};

struct mj_mulM2 {
  static constexpr char name[] = "mj_mulM2";
  static constexpr char doc[] = "Multiply vector by (inertia matrix)^(1/2).";
  using type = void (const mjModel *, const mjData *, mjtNum *, const mjtNum *);
  static constexpr auto param_names = std::make_tuple("m", "d", "res", "vec");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_mulM2;
  }
};

struct mj_addM {
  static constexpr char name[] = "mj_addM";
  static constexpr char doc[] = "Add inertia matrix to destination matrix. Destination can be sparse uncompressed, or dense when all int* are NULL";
  using type = void (const mjModel *, mjData *, mjtNum *, int *, int *, int *);
  static constexpr auto param_names = std::make_tuple("m", "d", "dst", "rownnz", "rowadr", "colind");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_addM;
  }
};

struct mj_applyFT {
  static constexpr char name[] = "mj_applyFT";
  static constexpr char doc[] = "Apply cartesian force and torque (outside xfrc_applied mechanism).";
  using type = void (const mjModel *, mjData *, const mjtNum (*)[3], const mjtNum (*)[3], const mjtNum (*)[3], int, mjtNum *);
  static constexpr auto param_names = std::make_tuple("m", "d", "force", "torque", "point", "body", "qfrc_target");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mj_applyFT);
  }
};

struct mj_objectVelocity {
  static constexpr char name[] = "mj_objectVelocity";
  static constexpr char doc[] = "Compute object 6D velocity in object-centered frame, world/local orientation.";
  using type = void (const mjModel *, const mjData *, int, int, mjtNum (*)[6], int);
  static constexpr auto param_names = std::make_tuple("m", "d", "objtype", "objid", "res", "flg_local");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mj_objectVelocity);
  }
};

struct mj_objectAcceleration {
  static constexpr char name[] = "mj_objectAcceleration";
  static constexpr char doc[] = "Compute object 6D acceleration in object-centered frame, world/local orientation.";
  using type = void (const mjModel *, const mjData *, int, int, mjtNum (*)[6], int);
  static constexpr auto param_names = std::make_tuple("m", "d", "objtype", "objid", "res", "flg_local");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mj_objectAcceleration);
  }
};

struct mj_contactForce {
  static constexpr char name[] = "mj_contactForce";
  static constexpr char doc[] = "Extract 6D force:torque given contact id, in the contact frame.";
  using type = void (const mjModel *, const mjData *, int, mjtNum (*)[6]);
  static constexpr auto param_names = std::make_tuple("m", "d", "id", "result");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mj_contactForce);
  }
};

struct mj_differentiatePos {
  static constexpr char name[] = "mj_differentiatePos";
  static constexpr char doc[] = "Compute velocity by finite-differencing two positions.";
  using type = void (const mjModel *, mjtNum *, mjtNum, const mjtNum *, const mjtNum *);
  static constexpr auto param_names = std::make_tuple("m", "qvel", "dt", "qpos1", "qpos2");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_differentiatePos;
  }
};

struct mj_integratePos {
  static constexpr char name[] = "mj_integratePos";
  static constexpr char doc[] = "Integrate position with given velocity.";
  using type = void (const mjModel *, mjtNum *, const mjtNum *, mjtNum);
  static constexpr auto param_names = std::make_tuple("m", "qpos", "qvel", "dt");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_integratePos;
  }
};

struct mj_normalizeQuat {
  static constexpr char name[] = "mj_normalizeQuat";
  static constexpr char doc[] = "Normalize all quaterions in qpos-type vector.";
  using type = void (const mjModel *, mjtNum *);
  static constexpr auto param_names = std::make_tuple("m", "qpos");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_normalizeQuat;
  }
};

struct mj_local2Global {
  static constexpr char name[] = "mj_local2Global";
  static constexpr char doc[] = "Map from body local to global Cartesian coordinates.";
  using type = void (mjData *, mjtNum (*)[3], mjtNum (*)[9], const mjtNum (*)[3], const mjtNum (*)[4], int, mjtByte);
  static constexpr auto param_names = std::make_tuple("d", "xpos", "xmat", "pos", "quat", "body", "sameframe");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mj_local2Global);
  }
};

struct mj_getTotalmass {
  static constexpr char name[] = "mj_getTotalmass";
  static constexpr char doc[] = "Sum all body masses.";
  using type = mjtNum (const mjModel *);
  static constexpr auto param_names = std::make_tuple("m");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_getTotalmass;
  }
};

struct mj_setTotalmass {
  static constexpr char name[] = "mj_setTotalmass";
  static constexpr char doc[] = "Scale body masses and inertias to achieve specified total mass.";
  using type = void (mjModel *, mjtNum);
  static constexpr auto param_names = std::make_tuple("m", "newmass");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_setTotalmass;
  }
};

struct mj_version {
  static constexpr char name[] = "mj_version";
  static constexpr char doc[] = "Return version number: 1.0.2 is encoded as 102.";
  using type = int ();
  static constexpr auto param_names = std::make_tuple();

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_version;
  }
};

struct mj_versionString {
  static constexpr char name[] = "mj_versionString";
  static constexpr char doc[] = "Return the current version of MuJoCo as a null-terminated string.";
  using type = const char * ();
  static constexpr auto param_names = std::make_tuple();

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_versionString;
  }
};

struct mj_ray {
  static constexpr char name[] = "mj_ray";
  static constexpr char doc[] = "Intersect ray (pnt+x*vec, x>=0) with visible geoms, except geoms in bodyexclude. Return distance (x) to nearest surface, or -1 if no intersection and output geomid. geomgroup, flg_static are as in mjvOption; geomgroup==NULL skips group exclusion.";
  using type = mjtNum (const mjModel *, const mjData *, const mjtNum (*)[3], const mjtNum (*)[3], const mjtByte *, mjtByte, int, int (*)[1]);
  static constexpr auto param_names = std::make_tuple("m", "d", "pnt", "vec", "geomgroup", "flg_static", "bodyexclude", "geomid");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mj_ray);
  }
};

struct mj_rayHfield {
  static constexpr char name[] = "mj_rayHfield";
  static constexpr char doc[] = "Interect ray with hfield, return nearest distance or -1 if no intersection.";
  using type = mjtNum (const mjModel *, const mjData *, int, const mjtNum (*)[3], const mjtNum (*)[3]);
  static constexpr auto param_names = std::make_tuple("m", "d", "geomid", "pnt", "vec");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mj_rayHfield);
  }
};

struct mj_rayMesh {
  static constexpr char name[] = "mj_rayMesh";
  static constexpr char doc[] = "Interect ray with mesh, return nearest distance or -1 if no intersection.";
  using type = mjtNum (const mjModel *, const mjData *, int, const mjtNum (*)[3], const mjtNum (*)[3]);
  static constexpr auto param_names = std::make_tuple("m", "d", "geomid", "pnt", "vec");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mj_rayMesh);
  }
};

struct mju_rayGeom {
  static constexpr char name[] = "mju_rayGeom";
  static constexpr char doc[] = "Interect ray with pure geom, return nearest distance or -1 if no intersection.";
  using type = mjtNum (const mjtNum (*)[3], const mjtNum (*)[9], const mjtNum (*)[3], const mjtNum (*)[3], const mjtNum (*)[3], int);
  static constexpr auto param_names = std::make_tuple("pos", "mat", "size", "pnt", "vec", "geomtype");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mju_rayGeom);
  }
};

struct mju_raySkin {
  static constexpr char name[] = "mju_raySkin";
  static constexpr char doc[] = "Interect ray with skin, return nearest distance or -1 if no intersection, and also output nearest vertex id.";
  using type = mjtNum (int, int, const int *, const float *, const mjtNum (*)[3], const mjtNum (*)[3], int (*)[1]);
  static constexpr auto param_names = std::make_tuple("nface", "nvert", "face", "vert", "pnt", "vec", "vertid");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mju_raySkin);
  }
};

struct mjv_defaultCamera {
  static constexpr char name[] = "mjv_defaultCamera";
  static constexpr char doc[] = "Set default camera.";
  using type = void (mjvCamera *);
  static constexpr auto param_names = std::make_tuple("cam");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mjv_defaultCamera;
  }
};

struct mjv_defaultPerturb {
  static constexpr char name[] = "mjv_defaultPerturb";
  static constexpr char doc[] = "Set default perturbation.";
  using type = void (mjvPerturb *);
  static constexpr auto param_names = std::make_tuple("pert");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mjv_defaultPerturb;
  }
};

struct mjv_room2model {
  static constexpr char name[] = "mjv_room2model";
  static constexpr char doc[] = "Transform pose from room to model space.";
  using type = void (mjtNum (*)[3], mjtNum (*)[4], const mjtNum (*)[3], const mjtNum (*)[4], const mjvScene *);
  static constexpr auto param_names = std::make_tuple("modelpos", "modelquat", "roompos", "roomquat", "scn");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mjv_room2model);
  }
};

struct mjv_model2room {
  static constexpr char name[] = "mjv_model2room";
  static constexpr char doc[] = "Transform pose from model to room space.";
  using type = void (mjtNum (*)[3], mjtNum (*)[4], const mjtNum (*)[3], const mjtNum (*)[4], const mjvScene *);
  static constexpr auto param_names = std::make_tuple("roompos", "roomquat", "modelpos", "modelquat", "scn");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mjv_model2room);
  }
};

struct mjv_cameraInModel {
  static constexpr char name[] = "mjv_cameraInModel";
  static constexpr char doc[] = "Get camera info in model space; average left and right OpenGL cameras.";
  using type = void (mjtNum (*)[3], mjtNum (*)[3], mjtNum (*)[3], const mjvScene *);
  static constexpr auto param_names = std::make_tuple("headpos", "forward", "up", "scn");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mjv_cameraInModel);
  }
};

struct mjv_cameraInRoom {
  static constexpr char name[] = "mjv_cameraInRoom";
  static constexpr char doc[] = "Get camera info in room space; average left and right OpenGL cameras.";
  using type = void (mjtNum (*)[3], mjtNum (*)[3], mjtNum (*)[3], const mjvScene *);
  static constexpr auto param_names = std::make_tuple("headpos", "forward", "up", "scn");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mjv_cameraInRoom);
  }
};

struct mjv_frustumHeight {
  static constexpr char name[] = "mjv_frustumHeight";
  static constexpr char doc[] = "Get frustum height at unit distance from camera; average left and right OpenGL cameras.";
  using type = mjtNum (const mjvScene *);
  static constexpr auto param_names = std::make_tuple("scn");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mjv_frustumHeight;
  }
};

struct mjv_alignToCamera {
  static constexpr char name[] = "mjv_alignToCamera";
  static constexpr char doc[] = "Rotate 3D vec in horizontal plane by angle between (0,1) and (forward_x,forward_y).";
  using type = void (mjtNum (*)[3], const mjtNum (*)[3], const mjtNum (*)[3]);
  static constexpr auto param_names = std::make_tuple("res", "vec", "forward");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mjv_alignToCamera);
  }
};

struct mjv_moveCamera {
  static constexpr char name[] = "mjv_moveCamera";
  static constexpr char doc[] = "Move camera with mouse; action is mjtMouse.";
  using type = void (const mjModel *, int, mjtNum, mjtNum, const mjvScene *, mjvCamera *);
  static constexpr auto param_names = std::make_tuple("m", "action", "reldx", "reldy", "scn", "cam");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mjv_moveCamera;
  }
};

struct mjv_movePerturb {
  static constexpr char name[] = "mjv_movePerturb";
  static constexpr char doc[] = "Move perturb object with mouse; action is mjtMouse.";
  using type = void (const mjModel *, const mjData *, int, mjtNum, mjtNum, const mjvScene *, mjvPerturb *);
  static constexpr auto param_names = std::make_tuple("m", "d", "action", "reldx", "reldy", "scn", "pert");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mjv_movePerturb;
  }
};

struct mjv_moveModel {
  static constexpr char name[] = "mjv_moveModel";
  static constexpr char doc[] = "Move model with mouse; action is mjtMouse.";
  using type = void (const mjModel *, int, mjtNum, mjtNum, const mjtNum (*)[3], mjvScene *);
  static constexpr auto param_names = std::make_tuple("m", "action", "reldx", "reldy", "roomup", "scn");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mjv_moveModel);
  }
};

struct mjv_initPerturb {
  static constexpr char name[] = "mjv_initPerturb";
  static constexpr char doc[] = "Copy perturb pos,quat from selected body; set scale for perturbation.";
  using type = void (const mjModel *, const mjData *, const mjvScene *, mjvPerturb *);
  static constexpr auto param_names = std::make_tuple("m", "d", "scn", "pert");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mjv_initPerturb;
  }
};

struct mjv_applyPerturbPose {
  static constexpr char name[] = "mjv_applyPerturbPose";
  static constexpr char doc[] = "Set perturb pos,quat in d->mocap when selected body is mocap, and in d->qpos otherwise. Write d->qpos only if flg_paused and subtree root for selected body has free joint.";
  using type = void (const mjModel *, mjData *, const mjvPerturb *, int);
  static constexpr auto param_names = std::make_tuple("m", "d", "pert", "flg_paused");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mjv_applyPerturbPose;
  }
};

struct mjv_applyPerturbForce {
  static constexpr char name[] = "mjv_applyPerturbForce";
  static constexpr char doc[] = "Set perturb force,torque in d->xfrc_applied, if selected body is dynamic.";
  using type = void (const mjModel *, mjData *, const mjvPerturb *);
  static constexpr auto param_names = std::make_tuple("m", "d", "pert");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mjv_applyPerturbForce;
  }
};

struct mjv_averageCamera {
  static constexpr char name[] = "mjv_averageCamera";
  static constexpr char doc[] = "Return the average of two OpenGL cameras.";
  using type = mjvGLCamera (const mjvGLCamera *, const mjvGLCamera *);
  static constexpr auto param_names = std::make_tuple("cam1", "cam2");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mjv_averageCamera;
  }
};

struct mjv_select {
  static constexpr char name[] = "mjv_select";
  static constexpr char doc[] = "Select geom or skin with mouse, return bodyid; -1: none selected.";
  using type = int (const mjModel *, const mjData *, const mjvOption *, mjtNum, mjtNum, mjtNum, const mjvScene *, mjtNum (*)[3], int (*)[1], int (*)[1]);
  static constexpr auto param_names = std::make_tuple("m", "d", "vopt", "aspectratio", "relx", "rely", "scn", "selpnt", "geomid", "skinid");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mjv_select);
  }
};

struct mjv_defaultOption {
  static constexpr char name[] = "mjv_defaultOption";
  static constexpr char doc[] = "Set default visualization options.";
  using type = void (mjvOption *);
  static constexpr auto param_names = std::make_tuple("opt");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mjv_defaultOption;
  }
};

struct mjv_defaultFigure {
  static constexpr char name[] = "mjv_defaultFigure";
  static constexpr char doc[] = "Set default figure.";
  using type = void (mjvFigure *);
  static constexpr auto param_names = std::make_tuple("fig");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mjv_defaultFigure;
  }
};

struct mjv_initGeom {
  static constexpr char name[] = "mjv_initGeom";
  static constexpr char doc[] = "Initialize given geom fields when not NULL, set the rest to their default values.";
  using type = void (mjvGeom *, int, const mjtNum (*)[3], const mjtNum (*)[3], const mjtNum (*)[9], const float (*)[4]);
  static constexpr auto param_names = std::make_tuple("geom", "type", "size", "pos", "mat", "rgba");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mjv_initGeom);
  }
};

struct mjv_makeConnector {
  static constexpr char name[] = "mjv_makeConnector";
  static constexpr char doc[] = "Set (type, size, pos, mat) for connector-type geom between given points. Assume that mjv_initGeom was already called to set all other properties.";
  using type = void (mjvGeom *, int, mjtNum, mjtNum, mjtNum, mjtNum, mjtNum, mjtNum, mjtNum);
  static constexpr auto param_names = std::make_tuple("geom", "type", "width", "a0", "a1", "a2", "b0", "b1", "b2");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mjv_makeConnector;
  }
};

struct mjv_defaultScene {
  static constexpr char name[] = "mjv_defaultScene";
  static constexpr char doc[] = "Set default abstract scene.";
  using type = void (mjvScene *);
  static constexpr auto param_names = std::make_tuple("scn");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mjv_defaultScene;
  }
};

struct mjv_makeScene {
  static constexpr char name[] = "mjv_makeScene";
  static constexpr char doc[] = "Allocate resources in abstract scene.";
  using type = void (const mjModel *, mjvScene *, int);
  static constexpr auto param_names = std::make_tuple("m", "scn", "maxgeom");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mjv_makeScene;
  }
};

struct mjv_freeScene {
  static constexpr char name[] = "mjv_freeScene";
  static constexpr char doc[] = "Free abstract scene.";
  using type = void (mjvScene *);
  static constexpr auto param_names = std::make_tuple("scn");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mjv_freeScene;
  }
};

struct mjv_updateScene {
  static constexpr char name[] = "mjv_updateScene";
  static constexpr char doc[] = "Update entire scene given model state.";
  using type = void (const mjModel *, mjData *, const mjvOption *, const mjvPerturb *, mjvCamera *, int, mjvScene *);
  static constexpr auto param_names = std::make_tuple("m", "d", "opt", "pert", "cam", "catmask", "scn");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mjv_updateScene;
  }
};

struct mjv_addGeoms {
  static constexpr char name[] = "mjv_addGeoms";
  static constexpr char doc[] = "Add geoms from selected categories.";
  using type = void (const mjModel *, mjData *, const mjvOption *, const mjvPerturb *, int, mjvScene *);
  static constexpr auto param_names = std::make_tuple("m", "d", "opt", "pert", "catmask", "scn");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mjv_addGeoms;
  }
};

struct mjv_makeLights {
  static constexpr char name[] = "mjv_makeLights";
  static constexpr char doc[] = "Make list of lights.";
  using type = void (const mjModel *, mjData *, mjvScene *);
  static constexpr auto param_names = std::make_tuple("m", "d", "scn");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mjv_makeLights;
  }
};

struct mjv_updateCamera {
  static constexpr char name[] = "mjv_updateCamera";
  static constexpr char doc[] = "Update camera.";
  using type = void (const mjModel *, mjData *, mjvCamera *, mjvScene *);
  static constexpr auto param_names = std::make_tuple("m", "d", "cam", "scn");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mjv_updateCamera;
  }
};

struct mjv_updateSkin {
  static constexpr char name[] = "mjv_updateSkin";
  static constexpr char doc[] = "Update skins.";
  using type = void (const mjModel *, mjData *, mjvScene *);
  static constexpr auto param_names = std::make_tuple("m", "d", "scn");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mjv_updateSkin;
  }
};

struct mjr_defaultContext {
  static constexpr char name[] = "mjr_defaultContext";
  static constexpr char doc[] = "Set default mjrContext.";
  using type = void (mjrContext *);
  static constexpr auto param_names = std::make_tuple("con");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mjr_defaultContext;
  }
};

struct mjr_makeContext {
  static constexpr char name[] = "mjr_makeContext";
  static constexpr char doc[] = "Allocate resources in custom OpenGL context; fontscale is mjtFontScale.";
  using type = void (const mjModel *, mjrContext *, int);
  static constexpr auto param_names = std::make_tuple("m", "con", "fontscale");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mjr_makeContext;
  }
};

struct mjr_changeFont {
  static constexpr char name[] = "mjr_changeFont";
  static constexpr char doc[] = "Change font of existing context.";
  using type = void (int, mjrContext *);
  static constexpr auto param_names = std::make_tuple("fontscale", "con");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mjr_changeFont;
  }
};

struct mjr_addAux {
  static constexpr char name[] = "mjr_addAux";
  static constexpr char doc[] = "Add Aux buffer with given index to context; free previous Aux buffer.";
  using type = void (int, int, int, int, mjrContext *);
  static constexpr auto param_names = std::make_tuple("index", "width", "height", "samples", "con");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mjr_addAux;
  }
};

struct mjr_freeContext {
  static constexpr char name[] = "mjr_freeContext";
  static constexpr char doc[] = "Free resources in custom OpenGL context, set to default.";
  using type = void (mjrContext *);
  static constexpr auto param_names = std::make_tuple("con");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mjr_freeContext;
  }
};

struct mjr_uploadTexture {
  static constexpr char name[] = "mjr_uploadTexture";
  static constexpr char doc[] = "Upload texture to GPU, overwriting previous upload if any.";
  using type = void (const mjModel *, const mjrContext *, int);
  static constexpr auto param_names = std::make_tuple("m", "con", "texid");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mjr_uploadTexture;
  }
};

struct mjr_uploadMesh {
  static constexpr char name[] = "mjr_uploadMesh";
  static constexpr char doc[] = "Upload mesh to GPU, overwriting previous upload if any.";
  using type = void (const mjModel *, const mjrContext *, int);
  static constexpr auto param_names = std::make_tuple("m", "con", "meshid");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mjr_uploadMesh;
  }
};

struct mjr_uploadHField {
  static constexpr char name[] = "mjr_uploadHField";
  static constexpr char doc[] = "Upload height field to GPU, overwriting previous upload if any.";
  using type = void (const mjModel *, const mjrContext *, int);
  static constexpr auto param_names = std::make_tuple("m", "con", "hfieldid");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mjr_uploadHField;
  }
};

struct mjr_restoreBuffer {
  static constexpr char name[] = "mjr_restoreBuffer";
  static constexpr char doc[] = "Make con->currentBuffer current again.";
  using type = void (const mjrContext *);
  static constexpr auto param_names = std::make_tuple("con");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mjr_restoreBuffer;
  }
};

struct mjr_setBuffer {
  static constexpr char name[] = "mjr_setBuffer";
  static constexpr char doc[] = "Set OpenGL framebuffer for rendering: mjFB_WINDOW or mjFB_OFFSCREEN. If only one buffer is available, set that buffer and ignore framebuffer argument.";
  using type = void (int, mjrContext *);
  static constexpr auto param_names = std::make_tuple("framebuffer", "con");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mjr_setBuffer;
  }
};

struct mjr_readPixels {
  static constexpr char name[] = "mjr_readPixels";
  static constexpr char doc[] = "Read pixels from current OpenGL framebuffer to client buffer. Viewport is in OpenGL framebuffer; client buffer starts at (0,0).";
  using type = void (unsigned char *, float *, mjrRect, const mjrContext *);
  static constexpr auto param_names = std::make_tuple("rgb", "depth", "viewport", "con");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mjr_readPixels;
  }
};

struct mjr_drawPixels {
  static constexpr char name[] = "mjr_drawPixels";
  static constexpr char doc[] = "Draw pixels from client buffer to current OpenGL framebuffer. Viewport is in OpenGL framebuffer; client buffer starts at (0,0).";
  using type = void (const unsigned char *, const float *, mjrRect, const mjrContext *);
  static constexpr auto param_names = std::make_tuple("rgb", "depth", "viewport", "con");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mjr_drawPixels;
  }
};

struct mjr_blitBuffer {
  static constexpr char name[] = "mjr_blitBuffer";
  static constexpr char doc[] = "Blit from src viewpoint in current framebuffer to dst viewport in other framebuffer. If src, dst have different size and flg_depth==0, color is interpolated with GL_LINEAR.";
  using type = void (mjrRect, mjrRect, int, int, const mjrContext *);
  static constexpr auto param_names = std::make_tuple("src", "dst", "flg_color", "flg_depth", "con");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mjr_blitBuffer;
  }
};

struct mjr_setAux {
  static constexpr char name[] = "mjr_setAux";
  static constexpr char doc[] = "Set Aux buffer for custom OpenGL rendering (call restoreBuffer when done).";
  using type = void (int, const mjrContext *);
  static constexpr auto param_names = std::make_tuple("index", "con");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mjr_setAux;
  }
};

struct mjr_blitAux {
  static constexpr char name[] = "mjr_blitAux";
  static constexpr char doc[] = "Blit from Aux buffer to con->currentBuffer.";
  using type = void (int, mjrRect, int, int, const mjrContext *);
  static constexpr auto param_names = std::make_tuple("index", "src", "left", "bottom", "con");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mjr_blitAux;
  }
};

struct mjr_text {
  static constexpr char name[] = "mjr_text";
  static constexpr char doc[] = "Draw text at (x,y) in relative coordinates; font is mjtFont.";
  using type = void (int, const char *, const mjrContext *, float, float, float, float, float);
  static constexpr auto param_names = std::make_tuple("font", "txt", "con", "x", "y", "r", "g", "b");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mjr_text;
  }
};

struct mjr_overlay {
  static constexpr char name[] = "mjr_overlay";
  static constexpr char doc[] = "Draw text overlay; font is mjtFont; gridpos is mjtGridPos.";
  using type = void (int, int, mjrRect, const char *, const char *, const mjrContext *);
  static constexpr auto param_names = std::make_tuple("font", "gridpos", "viewport", "overlay", "overlay2", "con");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mjr_overlay;
  }
};

struct mjr_maxViewport {
  static constexpr char name[] = "mjr_maxViewport";
  static constexpr char doc[] = "Get maximum viewport for active buffer.";
  using type = mjrRect (const mjrContext *);
  static constexpr auto param_names = std::make_tuple("con");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mjr_maxViewport;
  }
};

struct mjr_rectangle {
  static constexpr char name[] = "mjr_rectangle";
  static constexpr char doc[] = "Draw rectangle.";
  using type = void (mjrRect, float, float, float, float);
  static constexpr auto param_names = std::make_tuple("viewport", "r", "g", "b", "a");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mjr_rectangle;
  }
};

struct mjr_label {
  static constexpr char name[] = "mjr_label";
  static constexpr char doc[] = "Draw rectangle with centered text.";
  using type = void (mjrRect, int, const char *, float, float, float, float, float, float, float, const mjrContext *);
  static constexpr auto param_names = std::make_tuple("viewport", "font", "txt", "r", "g", "b", "a", "rt", "gt", "bt", "con");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mjr_label;
  }
};

struct mjr_figure {
  static constexpr char name[] = "mjr_figure";
  static constexpr char doc[] = "Draw 2D figure.";
  using type = void (mjrRect, mjvFigure *, const mjrContext *);
  static constexpr auto param_names = std::make_tuple("viewport", "fig", "con");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mjr_figure;
  }
};

struct mjr_render {
  static constexpr char name[] = "mjr_render";
  static constexpr char doc[] = "Render 3D scene.";
  using type = void (mjrRect, mjvScene *, const mjrContext *);
  static constexpr auto param_names = std::make_tuple("viewport", "scn", "con");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mjr_render;
  }
};

struct mjr_finish {
  static constexpr char name[] = "mjr_finish";
  static constexpr char doc[] = "Call glFinish.";
  using type = void ();
  static constexpr auto param_names = std::make_tuple();

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mjr_finish;
  }
};

struct mjr_getError {
  static constexpr char name[] = "mjr_getError";
  static constexpr char doc[] = "Call glGetError and return result.";
  using type = int ();
  static constexpr auto param_names = std::make_tuple();

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mjr_getError;
  }
};

struct mjr_findRect {
  static constexpr char name[] = "mjr_findRect";
  static constexpr char doc[] = "Find first rectangle containing mouse, -1: not found.";
  using type = int (int, int, int, const mjrRect *);
  static constexpr auto param_names = std::make_tuple("x", "y", "nrect", "rect");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mjr_findRect;
  }
};

struct mjui_themeSpacing {
  static constexpr char name[] = "mjui_themeSpacing";
  static constexpr char doc[] = "Get builtin UI theme spacing (ind: 0-1).";
  using type = mjuiThemeSpacing (int);
  static constexpr auto param_names = std::make_tuple("ind");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mjui_themeSpacing;
  }
};

struct mjui_themeColor {
  static constexpr char name[] = "mjui_themeColor";
  static constexpr char doc[] = "Get builtin UI theme color (ind: 0-3).";
  using type = mjuiThemeColor (int);
  static constexpr auto param_names = std::make_tuple("ind");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mjui_themeColor;
  }
};

struct mjui_add {
  static constexpr char name[] = "mjui_add";
  static constexpr char doc[] = "Add definitions to UI.";
  using type = void (mjUI *, const mjuiDef *);
  static constexpr auto param_names = std::make_tuple("ui", "def");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mjui_add;
  }
};

struct mjui_addToSection {
  static constexpr char name[] = "mjui_addToSection";
  static constexpr char doc[] = "Add definitions to UI section.";
  using type = void (mjUI *, int, const mjuiDef *);
  static constexpr auto param_names = std::make_tuple("ui", "sect", "def");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mjui_addToSection;
  }
};

struct mjui_resize {
  static constexpr char name[] = "mjui_resize";
  static constexpr char doc[] = "Compute UI sizes.";
  using type = void (mjUI *, const mjrContext *);
  static constexpr auto param_names = std::make_tuple("ui", "con");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mjui_resize;
  }
};

struct mjui_update {
  static constexpr char name[] = "mjui_update";
  static constexpr char doc[] = "Update specific section/item; -1: update all.";
  using type = void (int, int, const mjUI *, const mjuiState *, const mjrContext *);
  static constexpr auto param_names = std::make_tuple("section", "item", "ui", "state", "con");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mjui_update;
  }
};

struct mjui_event {
  static constexpr char name[] = "mjui_event";
  static constexpr char doc[] = "Handle UI event, return pointer to changed item, NULL if no change.";
  using type = mjuiItem * (mjUI *, mjuiState *, const mjrContext *);
  static constexpr auto param_names = std::make_tuple("ui", "state", "con");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mjui_event;
  }
};

struct mjui_render {
  static constexpr char name[] = "mjui_render";
  static constexpr char doc[] = "Copy UI image to current buffer.";
  using type = void (mjUI *, const mjuiState *, const mjrContext *);
  static constexpr auto param_names = std::make_tuple("ui", "state", "con");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mjui_render;
  }
};

struct mju_error {
  static constexpr char name[] = "mju_error";
  static constexpr char doc[] = "Main error function; does not return to caller.";
  using type = void (const char *);
  static constexpr auto param_names = std::make_tuple("msg");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mju_error;
  }
};

struct mju_error_i {
  static constexpr char name[] = "mju_error_i";
  static constexpr char doc[] = "Error function with int argument; msg is a printf format string.";
  using type = void (const char *, int);
  static constexpr auto param_names = std::make_tuple("msg", "i");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mju_error_i;
  }
};

struct mju_error_s {
  static constexpr char name[] = "mju_error_s";
  static constexpr char doc[] = "Error function with string argument.";
  using type = void (const char *, const char *);
  static constexpr auto param_names = std::make_tuple("msg", "text");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mju_error_s;
  }
};

struct mju_warning {
  static constexpr char name[] = "mju_warning";
  static constexpr char doc[] = "Main warning function; returns to caller.";
  using type = void (const char *);
  static constexpr auto param_names = std::make_tuple("msg");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mju_warning;
  }
};

struct mju_warning_i {
  static constexpr char name[] = "mju_warning_i";
  static constexpr char doc[] = "Warning function with int argument.";
  using type = void (const char *, int);
  static constexpr auto param_names = std::make_tuple("msg", "i");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mju_warning_i;
  }
};

struct mju_warning_s {
  static constexpr char name[] = "mju_warning_s";
  static constexpr char doc[] = "Warning function with string argument.";
  using type = void (const char *, const char *);
  static constexpr auto param_names = std::make_tuple("msg", "text");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mju_warning_s;
  }
};

struct mju_clearHandlers {
  static constexpr char name[] = "mju_clearHandlers";
  static constexpr char doc[] = "Clear user error and memory handlers.";
  using type = void ();
  static constexpr auto param_names = std::make_tuple();

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mju_clearHandlers;
  }
};

struct mju_malloc {
  static constexpr char name[] = "mju_malloc";
  static constexpr char doc[] = "Allocate memory; byte-align on 8; pad size to multiple of 8.";
  using type = void * (size_t);
  static constexpr auto param_names = std::make_tuple("size");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mju_malloc;
  }
};

struct mju_free {
  static constexpr char name[] = "mju_free";
  static constexpr char doc[] = "Free memory, using free() by default.";
  using type = void (void *);
  static constexpr auto param_names = std::make_tuple("ptr");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mju_free;
  }
};

struct mj_warning {
  static constexpr char name[] = "mj_warning";
  static constexpr char doc[] = "High-level warning function: count warnings in mjData, print only the first.";
  using type = void (mjData *, int, int);
  static constexpr auto param_names = std::make_tuple("d", "warning", "info");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_warning;
  }
};

struct mju_writeLog {
  static constexpr char name[] = "mju_writeLog";
  static constexpr char doc[] = "Write [datetime, type: message] to MUJOCO_LOG.TXT.";
  using type = void (const char *, const char *);
  static constexpr auto param_names = std::make_tuple("type", "msg");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mju_writeLog;
  }
};

struct mj_activate {
  static constexpr char name[] = "mj_activate";
  static constexpr char doc[] = "Return 1 (for backward compatibility).";
  using type = int (const char *);
  static constexpr auto param_names = std::make_tuple("filename");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_activate;
  }
};

struct mj_deactivate {
  static constexpr char name[] = "mj_deactivate";
  static constexpr char doc[] = "Do nothing (for backward compatibility).";
  using type = void ();
  static constexpr auto param_names = std::make_tuple();

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mj_deactivate;
  }
};

struct mju_zero3 {
  static constexpr char name[] = "mju_zero3";
  static constexpr char doc[] = "Set res = 0.";
  using type = void (mjtNum (*)[3]);
  static constexpr auto param_names = std::make_tuple("res");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mju_zero3);
  }
};

struct mju_copy3 {
  static constexpr char name[] = "mju_copy3";
  static constexpr char doc[] = "Set res = vec.";
  using type = void (mjtNum (*)[3], const mjtNum (*)[3]);
  static constexpr auto param_names = std::make_tuple("res", "data");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mju_copy3);
  }
};

struct mju_scl3 {
  static constexpr char name[] = "mju_scl3";
  static constexpr char doc[] = "Set res = vec*scl.";
  using type = void (mjtNum (*)[3], const mjtNum (*)[3], mjtNum);
  static constexpr auto param_names = std::make_tuple("res", "vec", "scl");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mju_scl3);
  }
};

struct mju_add3 {
  static constexpr char name[] = "mju_add3";
  static constexpr char doc[] = "Set res = vec1 + vec2.";
  using type = void (mjtNum (*)[3], const mjtNum (*)[3], const mjtNum (*)[3]);
  static constexpr auto param_names = std::make_tuple("res", "vec1", "vec2");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mju_add3);
  }
};

struct mju_sub3 {
  static constexpr char name[] = "mju_sub3";
  static constexpr char doc[] = "Set res = vec1 - vec2.";
  using type = void (mjtNum (*)[3], const mjtNum (*)[3], const mjtNum (*)[3]);
  static constexpr auto param_names = std::make_tuple("res", "vec1", "vec2");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mju_sub3);
  }
};

struct mju_addTo3 {
  static constexpr char name[] = "mju_addTo3";
  static constexpr char doc[] = "Set res = res + vec.";
  using type = void (mjtNum (*)[3], const mjtNum (*)[3]);
  static constexpr auto param_names = std::make_tuple("res", "vec");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mju_addTo3);
  }
};

struct mju_subFrom3 {
  static constexpr char name[] = "mju_subFrom3";
  static constexpr char doc[] = "Set res = res - vec.";
  using type = void (mjtNum (*)[3], const mjtNum (*)[3]);
  static constexpr auto param_names = std::make_tuple("res", "vec");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mju_subFrom3);
  }
};

struct mju_addToScl3 {
  static constexpr char name[] = "mju_addToScl3";
  static constexpr char doc[] = "Set res = res + vec*scl.";
  using type = void (mjtNum (*)[3], const mjtNum (*)[3], mjtNum);
  static constexpr auto param_names = std::make_tuple("res", "vec", "scl");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mju_addToScl3);
  }
};

struct mju_addScl3 {
  static constexpr char name[] = "mju_addScl3";
  static constexpr char doc[] = "Set res = vec1 + vec2*scl.";
  using type = void (mjtNum (*)[3], const mjtNum (*)[3], const mjtNum (*)[3], mjtNum);
  static constexpr auto param_names = std::make_tuple("res", "vec1", "vec2", "scl");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mju_addScl3);
  }
};

struct mju_normalize3 {
  static constexpr char name[] = "mju_normalize3";
  static constexpr char doc[] = "Normalize vector, return length before normalization.";
  using type = mjtNum (mjtNum (*)[3]);
  static constexpr auto param_names = std::make_tuple("res");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mju_normalize3);
  }
};

struct mju_norm3 {
  static constexpr char name[] = "mju_norm3";
  static constexpr char doc[] = "Return vector length (without normalizing the vector).";
  using type = mjtNum (const mjtNum (*)[3]);
  static constexpr auto param_names = std::make_tuple("vec");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mju_norm3);
  }
};

struct mju_dot3 {
  static constexpr char name[] = "mju_dot3";
  static constexpr char doc[] = "Return dot-product of vec1 and vec2.";
  using type = mjtNum (const mjtNum (*)[3], const mjtNum (*)[3]);
  static constexpr auto param_names = std::make_tuple("vec1", "vec2");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mju_dot3);
  }
};

struct mju_dist3 {
  static constexpr char name[] = "mju_dist3";
  static constexpr char doc[] = "Return Cartesian distance between 3D vectors pos1 and pos2.";
  using type = mjtNum (const mjtNum (*)[3], const mjtNum (*)[3]);
  static constexpr auto param_names = std::make_tuple("pos1", "pos2");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mju_dist3);
  }
};

struct mju_rotVecMat {
  static constexpr char name[] = "mju_rotVecMat";
  static constexpr char doc[] = "Multiply vector by 3D rotation matrix: res = mat * vec.";
  using type = void (mjtNum (*)[3], const mjtNum (*)[3], const mjtNum (*)[9]);
  static constexpr auto param_names = std::make_tuple("res", "vec", "mat");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mju_rotVecMat);
  }
};

struct mju_rotVecMatT {
  static constexpr char name[] = "mju_rotVecMatT";
  static constexpr char doc[] = "Multiply vector by transposed 3D rotation matrix: res = mat' * vec.";
  using type = void (mjtNum (*)[3], const mjtNum (*)[3], const mjtNum (*)[9]);
  static constexpr auto param_names = std::make_tuple("res", "vec", "mat");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mju_rotVecMatT);
  }
};

struct mju_cross {
  static constexpr char name[] = "mju_cross";
  static constexpr char doc[] = "Compute cross-product: res = cross(a, b).";
  using type = void (mjtNum (*)[3], const mjtNum (*)[3], const mjtNum (*)[3]);
  static constexpr auto param_names = std::make_tuple("res", "a", "b");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mju_cross);
  }
};

struct mju_zero4 {
  static constexpr char name[] = "mju_zero4";
  static constexpr char doc[] = "Set res = 0.";
  using type = void (mjtNum (*)[4]);
  static constexpr auto param_names = std::make_tuple("res");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mju_zero4);
  }
};

struct mju_unit4 {
  static constexpr char name[] = "mju_unit4";
  static constexpr char doc[] = "Set res = (1,0,0,0).";
  using type = void (mjtNum (*)[4]);
  static constexpr auto param_names = std::make_tuple("res");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mju_unit4);
  }
};

struct mju_copy4 {
  static constexpr char name[] = "mju_copy4";
  static constexpr char doc[] = "Set res = vec.";
  using type = void (mjtNum (*)[4], const mjtNum (*)[4]);
  static constexpr auto param_names = std::make_tuple("res", "data");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mju_copy4);
  }
};

struct mju_normalize4 {
  static constexpr char name[] = "mju_normalize4";
  static constexpr char doc[] = "Normalize vector, return length before normalization.";
  using type = mjtNum (mjtNum (*)[4]);
  static constexpr auto param_names = std::make_tuple("res");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mju_normalize4);
  }
};

struct mju_zero {
  static constexpr char name[] = "mju_zero";
  static constexpr char doc[] = "Set res = 0.";
  using type = void (mjtNum *, int);
  static constexpr auto param_names = std::make_tuple("res", "n");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mju_zero;
  }
};

struct mju_copy {
  static constexpr char name[] = "mju_copy";
  static constexpr char doc[] = "Set res = vec.";
  using type = void (mjtNum *, const mjtNum *, int);
  static constexpr auto param_names = std::make_tuple("res", "data", "n");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mju_copy;
  }
};

struct mju_sum {
  static constexpr char name[] = "mju_sum";
  static constexpr char doc[] = "Return sum(vec).";
  using type = mjtNum (const mjtNum *, int);
  static constexpr auto param_names = std::make_tuple("vec", "n");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mju_sum;
  }
};

struct mju_L1 {
  static constexpr char name[] = "mju_L1";
  static constexpr char doc[] = "Return L1 norm: sum(abs(vec)).";
  using type = mjtNum (const mjtNum *, int);
  static constexpr auto param_names = std::make_tuple("vec", "n");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mju_L1;
  }
};

struct mju_scl {
  static constexpr char name[] = "mju_scl";
  static constexpr char doc[] = "Set res = vec*scl.";
  using type = void (mjtNum *, const mjtNum *, mjtNum, int);
  static constexpr auto param_names = std::make_tuple("res", "vec", "scl", "n");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mju_scl;
  }
};

struct mju_add {
  static constexpr char name[] = "mju_add";
  static constexpr char doc[] = "Set res = vec1 + vec2.";
  using type = void (mjtNum *, const mjtNum *, const mjtNum *, int);
  static constexpr auto param_names = std::make_tuple("res", "vec1", "vec2", "n");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mju_add;
  }
};

struct mju_sub {
  static constexpr char name[] = "mju_sub";
  static constexpr char doc[] = "Set res = vec1 - vec2.";
  using type = void (mjtNum *, const mjtNum *, const mjtNum *, int);
  static constexpr auto param_names = std::make_tuple("res", "vec1", "vec2", "n");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mju_sub;
  }
};

struct mju_addTo {
  static constexpr char name[] = "mju_addTo";
  static constexpr char doc[] = "Set res = res + vec.";
  using type = void (mjtNum *, const mjtNum *, int);
  static constexpr auto param_names = std::make_tuple("res", "vec", "n");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mju_addTo;
  }
};

struct mju_subFrom {
  static constexpr char name[] = "mju_subFrom";
  static constexpr char doc[] = "Set res = res - vec.";
  using type = void (mjtNum *, const mjtNum *, int);
  static constexpr auto param_names = std::make_tuple("res", "vec", "n");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mju_subFrom;
  }
};

struct mju_addToScl {
  static constexpr char name[] = "mju_addToScl";
  static constexpr char doc[] = "Set res = res + vec*scl.";
  using type = void (mjtNum *, const mjtNum *, mjtNum, int);
  static constexpr auto param_names = std::make_tuple("res", "vec", "scl", "n");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mju_addToScl;
  }
};

struct mju_addScl {
  static constexpr char name[] = "mju_addScl";
  static constexpr char doc[] = "Set res = vec1 + vec2*scl.";
  using type = void (mjtNum *, const mjtNum *, const mjtNum *, mjtNum, int);
  static constexpr auto param_names = std::make_tuple("res", "vec1", "vec2", "scl", "n");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mju_addScl;
  }
};

struct mju_normalize {
  static constexpr char name[] = "mju_normalize";
  static constexpr char doc[] = "Normalize vector, return length before normalization.";
  using type = mjtNum (mjtNum *, int);
  static constexpr auto param_names = std::make_tuple("res", "n");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mju_normalize;
  }
};

struct mju_norm {
  static constexpr char name[] = "mju_norm";
  static constexpr char doc[] = "Return vector length (without normalizing vector).";
  using type = mjtNum (const mjtNum *, int);
  static constexpr auto param_names = std::make_tuple("res", "n");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mju_norm;
  }
};

struct mju_dot {
  static constexpr char name[] = "mju_dot";
  static constexpr char doc[] = "Return dot-product of vec1 and vec2.";
  using type = mjtNum (const mjtNum *, const mjtNum *, const int);
  static constexpr auto param_names = std::make_tuple("vec1", "vec2", "n");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mju_dot;
  }
};

struct mju_mulMatVec {
  static constexpr char name[] = "mju_mulMatVec";
  static constexpr char doc[] = "Multiply matrix and vector: res = mat * vec.";
  using type = void (mjtNum *, const mjtNum *, const mjtNum *, int, int);
  static constexpr auto param_names = std::make_tuple("res", "mat", "vec", "nr", "nc");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mju_mulMatVec;
  }
};

struct mju_mulMatTVec {
  static constexpr char name[] = "mju_mulMatTVec";
  static constexpr char doc[] = "Multiply transposed matrix and vector: res = mat' * vec.";
  using type = void (mjtNum *, const mjtNum *, const mjtNum *, int, int);
  static constexpr auto param_names = std::make_tuple("res", "mat", "vec", "nr", "nc");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mju_mulMatTVec;
  }
};

struct mju_transpose {
  static constexpr char name[] = "mju_transpose";
  static constexpr char doc[] = "Transpose matrix: res = mat'.";
  using type = void (mjtNum *, const mjtNum *, int, int);
  static constexpr auto param_names = std::make_tuple("res", "mat", "nr", "nc");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mju_transpose;
  }
};

struct mju_mulMatMat {
  static constexpr char name[] = "mju_mulMatMat";
  static constexpr char doc[] = "Multiply matrices: res = mat1 * mat2.";
  using type = void (mjtNum *, const mjtNum *, const mjtNum *, int, int, int);
  static constexpr auto param_names = std::make_tuple("res", "mat1", "mat2", "r1", "c1", "c2");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mju_mulMatMat;
  }
};

struct mju_mulMatMatT {
  static constexpr char name[] = "mju_mulMatMatT";
  static constexpr char doc[] = "Multiply matrices, second argument transposed: res = mat1 * mat2'.";
  using type = void (mjtNum *, const mjtNum *, const mjtNum *, int, int, int);
  static constexpr auto param_names = std::make_tuple("res", "mat1", "mat2", "r1", "c1", "r2");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mju_mulMatMatT;
  }
};

struct mju_mulMatTMat {
  static constexpr char name[] = "mju_mulMatTMat";
  static constexpr char doc[] = "Multiply matrices, first argument transposed: res = mat1' * mat2.";
  using type = void (mjtNum *, const mjtNum *, const mjtNum *, int, int, int);
  static constexpr auto param_names = std::make_tuple("res", "mat1", "mat2", "r1", "c1", "c2");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mju_mulMatTMat;
  }
};

struct mju_sqrMatTD {
  static constexpr char name[] = "mju_sqrMatTD";
  static constexpr char doc[] = "Set res = mat' * diag * mat if diag is not NULL, and res = mat' * mat otherwise.";
  using type = void (mjtNum *, const mjtNum *, const mjtNum *, int, int);
  static constexpr auto param_names = std::make_tuple("res", "mat", "diag", "nr", "nc");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mju_sqrMatTD;
  }
};

struct mju_transformSpatial {
  static constexpr char name[] = "mju_transformSpatial";
  static constexpr char doc[] = "Coordinate transform of 6D motion or force vector in rotation:translation format. rotnew2old is 3-by-3, NULL means no rotation; flg_force specifies force or motion type.";
  using type = void (mjtNum (*)[6], const mjtNum (*)[6], int, const mjtNum (*)[3], const mjtNum (*)[3], const mjtNum (*)[9]);
  static constexpr auto param_names = std::make_tuple("res", "vec", "flg_force", "newpos", "oldpos", "rotnew2old");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mju_transformSpatial);
  }
};

struct mju_rotVecQuat {
  static constexpr char name[] = "mju_rotVecQuat";
  static constexpr char doc[] = "Rotate vector by quaternion.";
  using type = void (mjtNum (*)[3], const mjtNum (*)[3], const mjtNum (*)[4]);
  static constexpr auto param_names = std::make_tuple("res", "vec", "quat");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mju_rotVecQuat);
  }
};

struct mju_negQuat {
  static constexpr char name[] = "mju_negQuat";
  static constexpr char doc[] = "Conjugate quaternion, corresponding to opposite rotation.";
  using type = void (mjtNum (*)[4], const mjtNum (*)[4]);
  static constexpr auto param_names = std::make_tuple("res", "quat");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mju_negQuat);
  }
};

struct mju_mulQuat {
  static constexpr char name[] = "mju_mulQuat";
  static constexpr char doc[] = "Muiltiply quaternions.";
  using type = void (mjtNum (*)[4], const mjtNum (*)[4], const mjtNum (*)[4]);
  static constexpr auto param_names = std::make_tuple("res", "quat1", "quat2");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mju_mulQuat);
  }
};

struct mju_mulQuatAxis {
  static constexpr char name[] = "mju_mulQuatAxis";
  static constexpr char doc[] = "Muiltiply quaternion and axis.";
  using type = void (mjtNum (*)[4], const mjtNum (*)[4], const mjtNum (*)[3]);
  static constexpr auto param_names = std::make_tuple("res", "quat", "axis");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mju_mulQuatAxis);
  }
};

struct mju_axisAngle2Quat {
  static constexpr char name[] = "mju_axisAngle2Quat";
  static constexpr char doc[] = "Convert axisAngle to quaternion.";
  using type = void (mjtNum (*)[4], const mjtNum (*)[3], mjtNum);
  static constexpr auto param_names = std::make_tuple("res", "axis", "angle");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mju_axisAngle2Quat);
  }
};

struct mju_quat2Vel {
  static constexpr char name[] = "mju_quat2Vel";
  static constexpr char doc[] = "Convert quaternion (corresponding to orientation difference) to 3D velocity.";
  using type = void (mjtNum (*)[3], const mjtNum (*)[4], mjtNum);
  static constexpr auto param_names = std::make_tuple("res", "quat", "dt");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mju_quat2Vel);
  }
};

struct mju_subQuat {
  static constexpr char name[] = "mju_subQuat";
  static constexpr char doc[] = "Subtract quaternions, express as 3D velocity: qb*quat(res) = qa.";
  using type = void (mjtNum (*)[3], const mjtNum (*)[4], const mjtNum (*)[4]);
  static constexpr auto param_names = std::make_tuple("res", "qa", "qb");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mju_subQuat);
  }
};

struct mju_quat2Mat {
  static constexpr char name[] = "mju_quat2Mat";
  static constexpr char doc[] = "Convert quaternion to 3D rotation matrix.";
  using type = void (mjtNum (*)[9], const mjtNum (*)[4]);
  static constexpr auto param_names = std::make_tuple("res", "quat");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mju_quat2Mat);
  }
};

struct mju_mat2Quat {
  static constexpr char name[] = "mju_mat2Quat";
  static constexpr char doc[] = "Convert 3D rotation matrix to quaterion.";
  using type = void (mjtNum (*)[4], const mjtNum (*)[9]);
  static constexpr auto param_names = std::make_tuple("quat", "mat");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mju_mat2Quat);
  }
};

struct mju_derivQuat {
  static constexpr char name[] = "mju_derivQuat";
  static constexpr char doc[] = "Compute time-derivative of quaternion, given 3D rotational velocity.";
  using type = void (mjtNum (*)[4], const mjtNum (*)[4], const mjtNum (*)[3]);
  static constexpr auto param_names = std::make_tuple("res", "quat", "vel");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mju_derivQuat);
  }
};

struct mju_quatIntegrate {
  static constexpr char name[] = "mju_quatIntegrate";
  static constexpr char doc[] = "Integrate quaterion given 3D angular velocity.";
  using type = void (mjtNum (*)[4], const mjtNum (*)[3], mjtNum);
  static constexpr auto param_names = std::make_tuple("quat", "vel", "scale");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mju_quatIntegrate);
  }
};

struct mju_quatZ2Vec {
  static constexpr char name[] = "mju_quatZ2Vec";
  static constexpr char doc[] = "Construct quaternion performing rotation from z-axis to given vector.";
  using type = void (mjtNum (*)[4], const mjtNum (*)[3]);
  static constexpr auto param_names = std::make_tuple("quat", "vec");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mju_quatZ2Vec);
  }
};

struct mju_mulPose {
  static constexpr char name[] = "mju_mulPose";
  static constexpr char doc[] = "Multiply two poses.";
  using type = void (mjtNum (*)[3], mjtNum (*)[4], const mjtNum (*)[3], const mjtNum (*)[4], const mjtNum (*)[3], const mjtNum (*)[4]);
  static constexpr auto param_names = std::make_tuple("posres", "quatres", "pos1", "quat1", "pos2", "quat2");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mju_mulPose);
  }
};

struct mju_negPose {
  static constexpr char name[] = "mju_negPose";
  static constexpr char doc[] = "Conjugate pose, corresponding to the opposite spatial transformation.";
  using type = void (mjtNum (*)[3], mjtNum (*)[4], const mjtNum (*)[3], const mjtNum (*)[4]);
  static constexpr auto param_names = std::make_tuple("posres", "quatres", "pos", "quat");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mju_negPose);
  }
};

struct mju_trnVecPose {
  static constexpr char name[] = "mju_trnVecPose";
  static constexpr char doc[] = "Transform vector by pose.";
  using type = void (mjtNum (*)[3], const mjtNum (*)[3], const mjtNum (*)[4], const mjtNum (*)[3]);
  static constexpr auto param_names = std::make_tuple("res", "pos", "quat", "vec");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mju_trnVecPose);
  }
};

struct mju_cholFactor {
  static constexpr char name[] = "mju_cholFactor";
  static constexpr char doc[] = "Cholesky decomposition: mat = L*L'; return rank, decomposition performed in-place into mat.";
  using type = int (mjtNum *, int, mjtNum);
  static constexpr auto param_names = std::make_tuple("mat", "n", "mindiag");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mju_cholFactor;
  }
};

struct mju_cholSolve {
  static constexpr char name[] = "mju_cholSolve";
  static constexpr char doc[] = "Solve mat * res = vec, where mat is Cholesky-factorized";
  using type = void (mjtNum *, const mjtNum *, const mjtNum *, int);
  static constexpr auto param_names = std::make_tuple("res", "mat", "vec", "n");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mju_cholSolve;
  }
};

struct mju_cholUpdate {
  static constexpr char name[] = "mju_cholUpdate";
  static constexpr char doc[] = "Cholesky rank-one update: L*L' +/- x*x'; return rank.";
  using type = int (mjtNum *, mjtNum *, int, int);
  static constexpr auto param_names = std::make_tuple("mat", "x", "n", "flg_plus");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mju_cholUpdate;
  }
};

struct mju_eig3 {
  static constexpr char name[] = "mju_eig3";
  static constexpr char doc[] = "Eigenvalue decomposition of symmetric 3x3 matrix.";
  using type = int (mjtNum (*)[3], mjtNum (*)[9], mjtNum (*)[4], const mjtNum (*)[9]);
  static constexpr auto param_names = std::make_tuple("eigval", "eigvec", "quat", "mat");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mju_eig3);
  }
};

struct mju_muscleGain {
  static constexpr char name[] = "mju_muscleGain";
  static constexpr char doc[] = "Muscle active force, prm = (range[2], force, scale, lmin, lmax, vmax, fpmax, fvmax).";
  using type = mjtNum (mjtNum, mjtNum, const mjtNum (*)[2], mjtNum, const mjtNum (*)[9]);
  static constexpr auto param_names = std::make_tuple("len", "vel", "lengthrange", "acc0", "prm");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mju_muscleGain);
  }
};

struct mju_muscleBias {
  static constexpr char name[] = "mju_muscleBias";
  static constexpr char doc[] = "Muscle passive force, prm = (range[2], force, scale, lmin, lmax, vmax, fpmax, fvmax).";
  using type = mjtNum (mjtNum, const mjtNum (*)[2], mjtNum, const mjtNum (*)[9]);
  static constexpr auto param_names = std::make_tuple("len", "lengthrange", "acc0", "prm");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mju_muscleBias);
  }
};

struct mju_muscleDynamics {
  static constexpr char name[] = "mju_muscleDynamics";
  static constexpr char doc[] = "Muscle activation dynamics, prm = (tau_act, tau_deact).";
  using type = mjtNum (mjtNum, mjtNum, const mjtNum (*)[2]);
  static constexpr auto param_names = std::make_tuple("ctrl", "act", "prm");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return *reinterpret_cast<type*>(&::mju_muscleDynamics);
  }
};

struct mju_encodePyramid {
  static constexpr char name[] = "mju_encodePyramid";
  static constexpr char doc[] = "Convert contact force to pyramid representation.";
  using type = void (mjtNum *, const mjtNum *, const mjtNum *, int);
  static constexpr auto param_names = std::make_tuple("pyramid", "force", "mu", "dim");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mju_encodePyramid;
  }
};

struct mju_decodePyramid {
  static constexpr char name[] = "mju_decodePyramid";
  static constexpr char doc[] = "Convert pyramid representation to contact force.";
  using type = void (mjtNum *, const mjtNum *, const mjtNum *, int);
  static constexpr auto param_names = std::make_tuple("force", "pyramid", "mu", "dim");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mju_decodePyramid;
  }
};

struct mju_springDamper {
  static constexpr char name[] = "mju_springDamper";
  static constexpr char doc[] = "Integrate spring-damper analytically, return pos(dt).";
  using type = mjtNum (mjtNum, mjtNum, mjtNum, mjtNum, mjtNum);
  static constexpr auto param_names = std::make_tuple("pos0", "vel0", "Kp", "Kv", "dt");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mju_springDamper;
  }
};

struct mju_min {
  static constexpr char name[] = "mju_min";
  static constexpr char doc[] = "Return min(a,b) with single evaluation of a and b.";
  using type = mjtNum (mjtNum, mjtNum);
  static constexpr auto param_names = std::make_tuple("a", "b");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mju_min;
  }
};

struct mju_max {
  static constexpr char name[] = "mju_max";
  static constexpr char doc[] = "Return max(a,b) with single evaluation of a and b.";
  using type = mjtNum (mjtNum, mjtNum);
  static constexpr auto param_names = std::make_tuple("a", "b");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mju_max;
  }
};

struct mju_sign {
  static constexpr char name[] = "mju_sign";
  static constexpr char doc[] = "Return sign of x: +1, -1 or 0.";
  using type = mjtNum (mjtNum);
  static constexpr auto param_names = std::make_tuple("x");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mju_sign;
  }
};

struct mju_round {
  static constexpr char name[] = "mju_round";
  static constexpr char doc[] = "Round x to nearest integer.";
  using type = int (mjtNum);
  static constexpr auto param_names = std::make_tuple("x");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mju_round;
  }
};

struct mju_type2Str {
  static constexpr char name[] = "mju_type2Str";
  static constexpr char doc[] = "Convert type id (mjtObj) to type name.";
  using type = const char * (int);
  static constexpr auto param_names = std::make_tuple("type");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mju_type2Str;
  }
};

struct mju_str2Type {
  static constexpr char name[] = "mju_str2Type";
  static constexpr char doc[] = "Convert type name to type id (mjtObj).";
  using type = int (const char *);
  static constexpr auto param_names = std::make_tuple("str");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mju_str2Type;
  }
};

struct mju_warningText {
  static constexpr char name[] = "mju_warningText";
  static constexpr char doc[] = "Construct a warning message given the warning type and info.";
  using type = const char * (int, int);
  static constexpr auto param_names = std::make_tuple("warning", "info");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mju_warningText;
  }
};

struct mju_isBad {
  static constexpr char name[] = "mju_isBad";
  static constexpr char doc[] = "Return 1 if nan or abs(x)>mjMAXVAL, 0 otherwise. Used by check functions.";
  using type = int (mjtNum);
  static constexpr auto param_names = std::make_tuple("x");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mju_isBad;
  }
};

struct mju_isZero {
  static constexpr char name[] = "mju_isZero";
  static constexpr char doc[] = "Return 1 if all elements are 0.";
  using type = int (mjtNum *, int);
  static constexpr auto param_names = std::make_tuple("vec", "n");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mju_isZero;
  }
};

struct mju_standardNormal {
  static constexpr char name[] = "mju_standardNormal";
  static constexpr char doc[] = "Standard normal random number generator (optional second number).";
  using type = mjtNum (mjtNum *);
  static constexpr auto param_names = std::make_tuple("num2");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mju_standardNormal;
  }
};

struct mju_f2n {
  static constexpr char name[] = "mju_f2n";
  static constexpr char doc[] = "Convert from float to mjtNum.";
  using type = void (mjtNum *, const float *, int);
  static constexpr auto param_names = std::make_tuple("res", "vec", "n");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mju_f2n;
  }
};

struct mju_n2f {
  static constexpr char name[] = "mju_n2f";
  static constexpr char doc[] = "Convert from mjtNum to float.";
  using type = void (float *, const mjtNum *, int);
  static constexpr auto param_names = std::make_tuple("res", "vec", "n");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mju_n2f;
  }
};

struct mju_d2n {
  static constexpr char name[] = "mju_d2n";
  static constexpr char doc[] = "Convert from double to mjtNum.";
  using type = void (mjtNum *, const double *, int);
  static constexpr auto param_names = std::make_tuple("res", "vec", "n");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mju_d2n;
  }
};

struct mju_n2d {
  static constexpr char name[] = "mju_n2d";
  static constexpr char doc[] = "Convert from mjtNum to double.";
  using type = void (double *, const mjtNum *, int);
  static constexpr auto param_names = std::make_tuple("res", "vec", "n");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mju_n2d;
  }
};

struct mju_insertionSort {
  static constexpr char name[] = "mju_insertionSort";
  static constexpr char doc[] = "Insertion sort, resulting list is in increasing order.";
  using type = void (mjtNum *, int);
  static constexpr auto param_names = std::make_tuple("list", "n");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mju_insertionSort;
  }
};

struct mju_insertionSortInt {
  static constexpr char name[] = "mju_insertionSortInt";
  static constexpr char doc[] = "Integer insertion sort, resulting list is in increasing order.";
  using type = void (int *, int);
  static constexpr auto param_names = std::make_tuple("list", "n");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mju_insertionSortInt;
  }
};

struct mju_Halton {
  static constexpr char name[] = "mju_Halton";
  static constexpr char doc[] = "Generate Halton sequence.";
  using type = mjtNum (int, int);
  static constexpr auto param_names = std::make_tuple("index", "base");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mju_Halton;
  }
};

struct mju_strncpy {
  static constexpr char name[] = "mju_strncpy";
  static constexpr char doc[] = "Call strncpy, then set dst[n-1] = 0.";
  using type = char * (char *, const char *, int);
  static constexpr auto param_names = std::make_tuple("dst", "src", "n");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mju_strncpy;
  }
};

struct mju_sigmoid {
  static constexpr char name[] = "mju_sigmoid";
  static constexpr char doc[] = "Sigmoid function over 0<=x<=1 constructed from half-quadratics.";
  using type = mjtNum (mjtNum);
  static constexpr auto param_names = std::make_tuple("x");

  MUJOCO_ALWAYS_INLINE static type& GetFunc() {
    return ::mju_sigmoid;
  }
};

}  // namespace mujoco::python_traits

#endif  // MUJOCO_PYTHON_CODEGEN_FUNCTION_TRAITS_H_

