import pytest
from unittest.mock import patch, MagicMock

from bedrock_server_manager.api.web import (
    start_web_server_api,
    stop_web_server_api,
    get_web_server_status_api,
    create_web_ui_service,
    enable_web_ui_service,
    disable_web_ui_service,
    remove_web_ui_service,
    get_web_ui_service_status,
)


@pytest.fixture
def mock_system_linux_utils(mocker):
    """Fixture to patch system_linux_utils."""
    return mocker.patch(
        "bedrock_server_manager.core.manager_mixins.web_service_mixin.system_linux_utils"
    )


class TestWebServerLifecycle:
    def test_start_web_server_direct(self, app_context):
        app_context.manager.start_web_ui_direct = MagicMock()
        start_web_server_api(mode="direct", app_context=app_context)
        app_context.manager.start_web_ui_direct.assert_called_once_with(
            app_context, None, False, None
        )

    @patch("bedrock_server_manager.api.web.system_process_utils")
    @patch("bedrock_server_manager.api.web.PSUTIL_AVAILABLE", True)
    def test_start_web_server_detached(self, mock_system_process, app_context):
        with patch(
            "bedrock_server_manager.api.web.get_manager_instance",
            return_value=app_context.manager,
        ):
            mock_system_process.read_pid_from_file.return_value = None
            mock_system_process.launch_detached_process.return_value = 12345
            result = start_web_server_api(mode="detached", app_context=app_context)
            assert result["status"] == "success"
            assert result["pid"] == 12345

    @patch("bedrock_server_manager.api.web.system_process_utils")
    @patch("bedrock_server_manager.api.web.PSUTIL_AVAILABLE", True)
    def test_stop_web_server_api(self, mock_system_process, app_context):
        with patch(
            "bedrock_server_manager.api.web.get_manager_instance",
            return_value=app_context.manager,
        ):
            mock_system_process.read_pid_from_file.return_value = 12345
            mock_system_process.is_process_running.return_value = True
            result = stop_web_server_api(app_context=app_context)
            assert result["status"] == "success"
            mock_system_process.terminate_process_by_pid.assert_called_once_with(12345)

    @patch("bedrock_server_manager.api.web.system_process_utils")
    @patch("bedrock_server_manager.api.web.PSUTIL_AVAILABLE", True)
    def test_get_web_server_status_api_running(self, mock_system_process, app_context):
        with patch(
            "bedrock_server_manager.api.web.get_manager_instance",
            return_value=app_context.manager,
        ):
            mock_system_process.read_pid_from_file.return_value = 12345
            mock_system_process.is_process_running.return_value = True
            result = get_web_server_status_api(app_context=app_context)
            assert result["status"] == "RUNNING"
            assert result["pid"] == 12345


class TestWebServiceManagement:
    def test_create_web_ui_service_autostart(
        self, app_context, mock_system_linux_utils
    ):
        with patch(
            "bedrock_server_manager.api.web.get_manager_instance",
            return_value=app_context.manager,
        ):
            create_web_ui_service(app_context=app_context, autostart=True)
            mock_system_linux_utils.create_systemd_service_file.assert_called_once()
            mock_system_linux_utils.enable_systemd_service.assert_called_once()

    def test_enable_web_ui_service(self, app_context, mock_system_linux_utils):
        with patch(
            "bedrock_server_manager.api.web.get_manager_instance",
            return_value=app_context.manager,
        ):
            enable_web_ui_service(app_context=app_context)
            mock_system_linux_utils.enable_systemd_service.assert_called_once()

    def test_disable_web_ui_service(self, app_context, mock_system_linux_utils):
        with patch(
            "bedrock_server_manager.api.web.get_manager_instance",
            return_value=app_context.manager,
        ):
            disable_web_ui_service(app_context=app_context)
            mock_system_linux_utils.disable_systemd_service.assert_called_once()

    @pytest.mark.skip(reason="Failing due to mock issues")
    @patch("bedrock_server_manager.core.manager.os.remove")
    def test_remove_web_ui_service(
        self, mock_os_remove, app_context, mock_system_linux_utils
    ):
        with patch(
            "bedrock_server_manager.api.web.get_manager_instance",
            return_value=app_context.manager,
        ):
            mock_system_linux_utils.get_systemd_service_file_path.return_value = (
                "/etc/systemd/user/bedrock-server-manager-webui.service"
            )
            result = remove_web_ui_service(app_context=app_context)
            assert result["status"] == "success"
            mock_os_remove.assert_called_once_with(
                "/etc/systemd/user/bedrock-server-manager-webui.service"
            )

    def test_get_web_ui_service_status(self, app_context, mock_system_linux_utils):
        with patch(
            "bedrock_server_manager.api.web.get_manager_instance",
            return_value=app_context.manager,
        ):
            app_context.manager.check_web_service_exists = MagicMock(return_value=True)
            app_context.manager.is_web_service_active = MagicMock(return_value=True)
            app_context.manager.is_web_service_enabled = MagicMock(return_value=True)
            result = get_web_ui_service_status(app_context=app_context)
            assert result["status"] == "success"
            assert result["service_exists"] is True
            assert result["is_active"] is True
            assert result["is_enabled"] is True
