"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.WebApp = void 0;
const path = require("path");
const cloudfront = require("@aws-cdk/aws-cloudfront");
const origins = require("@aws-cdk/aws-cloudfront-origins");
const r53 = require("@aws-cdk/aws-route53");
const r53targets = require("@aws-cdk/aws-route53-targets");
const s3 = require("@aws-cdk/aws-s3");
const s3deploy = require("@aws-cdk/aws-s3-deployment");
const core_1 = require("@aws-cdk/core");
const constants_1 = require("../backend/shared/constants");
const caching_1 = require("../caching");
const monitored_certificate_1 = require("../monitored-certificate");
const storage_1 = require("../s3/storage");
const temp_file_1 = require("../temp-file");
const config_1 = require("./config");
const response_function_1 = require("./response-function");
class WebApp extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b, _c, _d;
        super(scope, id);
        const storageFactory = storage_1.S3StorageFactory.getOrCreate(this);
        this.bucket = storageFactory.newBucket(this, 'WebsiteBucket', {
            blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
            enforceSSL: true,
        });
        // generate a stable unique id for the cloudfront function and use it
        // both for the function name and the logical id of the function so if
        // it is changed the function will be recreated.
        // see https://github.com/aws/aws-cdk/issues/15523
        const functionId = `AddHeadersFunction${this.node.addr}`;
        const behaviorOptions = {
            compress: true,
            cachePolicy: cloudfront.CachePolicy.CACHING_OPTIMIZED,
            functionAssociations: [{
                    function: new response_function_1.ResponseFunction(this, functionId, {
                        functionName: functionId,
                    }),
                    eventType: cloudfront.FunctionEventType.VIEWER_RESPONSE,
                }],
        };
        this.distribution = new cloudfront.Distribution(this, 'Distribution', {
            defaultBehavior: { origin: new origins.S3Origin(this.bucket), ...behaviorOptions },
            domainNames: props.domain ? [props.domain.zone.zoneName] : undefined,
            certificate: props.domain ? props.domain.cert : undefined,
            defaultRootObject: 'index.html',
            errorResponses: [404, 403].map(httpStatus => ({
                httpStatus,
                responseHttpStatus: 200,
                responsePagePath: '/index.html',
            })),
            minimumProtocolVersion: cloudfront.SecurityPolicyProtocol.TLS_V1_2_2018,
        });
        // The base URL is currently the custom DNS if any was used, or the distribution domain name.
        // This needs changing in case, for example, we add support for a custom URL prefix.
        this.baseUrl = `https://${props.domain ? props.domain.zone.zoneName : this.distribution.distributionDomainName}`;
        const jsiiObjOrigin = new origins.S3Origin(props.packageData);
        this.distribution.addBehavior('/data/*', jsiiObjOrigin, behaviorOptions);
        this.distribution.addBehavior(`/${constants_1.CATALOG_KEY}`, jsiiObjOrigin, behaviorOptions);
        this.distribution.addBehavior(`/${constants_1.VERSION_TRACKER_KEY}`, jsiiObjOrigin, behaviorOptions);
        if (props.packageStats) {
            this.distribution.addBehavior(`/${props.packageStats.statsKey}`, jsiiObjOrigin, behaviorOptions);
        }
        // if we use a domain, and A records with a CloudFront alias
        if (props.domain) {
            // IPv4
            new r53.ARecord(this, 'ARecord', {
                zone: props.domain.zone,
                target: r53.RecordTarget.fromAlias(new r53targets.CloudFrontTarget(this.distribution)),
                comment: 'Created by the AWS CDK',
            });
            // IPv6
            new r53.AaaaRecord(this, 'AaaaRecord', {
                zone: props.domain.zone,
                target: r53.RecordTarget.fromAlias(new r53targets.CloudFrontTarget(this.distribution)),
                comment: 'Created by the AWS CDK',
            });
            // Monitor certificate expiration
            if ((_a = props.domain.monitorCertificateExpiration) !== null && _a !== void 0 ? _a : true) {
                const monitored = new monitored_certificate_1.MonitoredCertificate(this, 'ExpirationMonitor', {
                    certificate: props.domain.cert,
                    domainName: props.domain.zone.zoneName,
                });
                props.monitoring.addHighSeverityAlarm('ACM Certificate Expiry', monitored.alarmAcmCertificateExpiresSoon);
                props.monitoring.addHighSeverityAlarm('Endpoint Certificate Expiry', monitored.alarmEndpointCertificateExpiresSoon);
            }
        }
        // "website" contains the static react app
        const webappDir = path.join(__dirname, '..', '..', 'website');
        new s3deploy.BucketDeployment(this, 'DeployWebsite', {
            destinationBucket: this.bucket,
            distribution: this.distribution,
            prune: false,
            sources: [s3deploy.Source.asset(webappDir)],
            cacheControl: caching_1.CacheStrategy.default().toArray(),
        });
        // Generate config.json to customize frontend behavior
        const config = new config_1.WebappConfig({
            packageLinks: props.packageLinks,
            packageTags: props.packageTags,
            packageTagGroups: props.packageTagGroups,
            featuredPackages: props.featuredPackages,
            showPackageStats: (_b = props.showPackageStats) !== null && _b !== void 0 ? _b : props.packageStats !== undefined,
            featureFlags: props.featureFlags,
            categories: props.categories,
        });
        // Generate preload.js
        const preloadScript = new temp_file_1.TempFile('preload.js', (_d = (_c = props.preloadScript) === null || _c === void 0 ? void 0 : _c.bind()) !== null && _d !== void 0 ? _d : '');
        new s3deploy.BucketDeployment(this, 'DeployWebsiteConfig', {
            sources: [s3deploy.Source.asset(config.file.dir), s3deploy.Source.asset(preloadScript.dir)],
            destinationBucket: this.bucket,
            distribution: this.distribution,
            prune: false,
            cacheControl: caching_1.CacheStrategy.default().toArray(),
        });
        new core_1.CfnOutput(this, 'DomainName', {
            value: this.distribution.domainName,
            exportName: 'ConstructHubDomainName',
        });
        // add a canary that pings our home page and alarms if it returns errors.
        props.monitoring.addWebCanary('Home Page', `https://${this.distribution.domainName}`);
    }
}
exports.WebApp = WebApp;
//# sourceMappingURL=data:application/json;base64,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