"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Isolation = exports.ConstructHub = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const s3 = require("@aws-cdk/aws-s3");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const _limited_internet_access_1 = require("./_limited-internet-access");
const backend_1 = require("./backend");
const backend_dashboard_1 = require("./backend-dashboard");
const inventory_1 = require("./backend/inventory");
const license_list_1 = require("./backend/license-list");
const orchestration_1 = require("./backend/orchestration");
const package_stats_1 = require("./backend/package-stats");
const constants_1 = require("./backend/shared/constants");
const version_tracker_1 = require("./backend/version-tracker");
const repository_1 = require("./codeartifact/repository");
const domain_redirect_1 = require("./domain-redirect");
const monitoring_1 = require("./monitoring");
const package_sources_1 = require("./package-sources");
const storage_1 = require("./s3/storage");
const spdx_license_1 = require("./spdx-license");
const webapp_1 = require("./webapp");
/**
 * Construct Hub.
 */
class ConstructHub extends core_1.Construct {
    constructor(scope, id, props = {}) {
        var _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m;
        super(scope, id);
        if (props.isolateSensitiveTasks != null && props.sensitiveTaskIsolation != null) {
            throw new Error('Supplying both isolateSensitiveTasks and sensitiveTaskIsolation is not supported. Remove usage of isolateSensitiveTasks.');
        }
        const storageFactory = storage_1.S3StorageFactory.getOrCreate(this, {
            failover: props.failoverStorage,
        });
        const monitoring = new monitoring_1.Monitoring(this, 'Monitoring', {
            alarmActions: props.alarmActions,
        });
        const packageData = storageFactory.newBucket(this, 'PackageData', {
            blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
            enforceSSL: true,
            encryption: s3.BucketEncryption.S3_MANAGED,
            lifecycleRules: [
                // Abort multi-part uploads after 1 day
                { abortIncompleteMultipartUploadAfter: core_1.Duration.days(1) },
                // Transition non-current object versions to IA after 1 month
                {
                    noncurrentVersionTransitions: [
                        {
                            storageClass: s3.StorageClass.INFREQUENT_ACCESS,
                            transitionAfter: core_1.Duration.days(31),
                        },
                    ],
                },
                // Permanently delete non-current object versions after 3 months
                {
                    noncurrentVersionExpiration: core_1.Duration.days(90),
                    expiredObjectDeleteMarker: true,
                },
                // Permanently delete non-current versions of catalog.json earlier
                { noncurrentVersionExpiration: core_1.Duration.days(7), prefix: constants_1.CATALOG_KEY },
            ],
            versioned: true,
        });
        const isolation = (_b = props.sensitiveTaskIsolation) !== null && _b !== void 0 ? _b : (props.isolateSensitiveTasks ? Isolation.NO_INTERNET_ACCESS : Isolation.UNLIMITED_INTERNET_ACCESS);
        // Create an internal CodeArtifact repository if we run in network-controlled mode, or if a domain is provided.
        const codeArtifact = isolation === Isolation.NO_INTERNET_ACCESS || props.codeArtifactDomain != null
            ? new repository_1.Repository(this, 'CodeArtifact', {
                description: 'Proxy to npmjs.com for ConstructHub',
                domainName: (_c = props.codeArtifactDomain) === null || _c === void 0 ? void 0 : _c.name,
                domainExists: props.codeArtifactDomain != null,
                upstreams: (_d = props.codeArtifactDomain) === null || _d === void 0 ? void 0 : _d.upstreams,
            })
            : undefined;
        const { vpc, vpcEndpoints, vpcSubnets, vpcSecurityGroups } = this.createVpc(isolation, codeArtifact);
        const denyList = new backend_1.DenyList(this, 'DenyList', {
            rules: (_e = props.denyList) !== null && _e !== void 0 ? _e : [],
            packageDataBucket: packageData,
            packageDataKeyPrefix: constants_1.STORAGE_KEY_PREFIX,
            monitoring: monitoring,
        });
        // disable fetching package stats by default if a different package
        // source is configured
        const fetchPackageStats = (_f = props.fetchPackageStats) !== null && _f !== void 0 ? _f : (props.packageSources ? false : true);
        let packageStats;
        const statsKey = 'stats.json';
        if (fetchPackageStats) {
            packageStats = new package_stats_1.PackageStats(this, 'Stats', {
                bucket: packageData,
                monitoring,
                logRetention: props.logRetention,
                objectKey: statsKey,
            });
        }
        const versionTracker = new version_tracker_1.VersionTracker(this, 'VersionTracker', {
            bucket: packageData,
            monitoring,
            logRetention: props.logRetention,
        });
        const orchestration = new orchestration_1.Orchestration(this, 'Orchestration', {
            bucket: packageData,
            codeArtifact,
            denyList,
            logRetention: props.logRetention,
            monitoring,
            vpc,
            vpcEndpoints,
            vpcSubnets,
            vpcSecurityGroups,
        });
        // rebuild the catalog when the deny list changes.
        denyList.prune.onChangeInvoke(orchestration.catalogBuilder.function);
        const packageTagsSerialized = (_h = (_g = props.packageTags) === null || _g === void 0 ? void 0 : _g.map((config) => {
            return {
                ...config,
                condition: config.condition.bind(),
            };
        })) !== null && _h !== void 0 ? _h : [];
        this.ingestion = new backend_1.Ingestion(this, 'Ingestion', {
            bucket: packageData,
            codeArtifact,
            orchestration,
            logRetention: props.logRetention,
            monitoring,
            packageLinks: props.packageLinks,
            packageTags: packageTagsSerialized,
            reprocessFrequency: props.reprocessFrequency,
        });
        const licenseList = new license_list_1.LicenseList(this, 'LicenseList', {
            licenses: (_j = props.allowedLicenses) !== null && _j !== void 0 ? _j : [
                ...spdx_license_1.SpdxLicense.apache(),
                ...spdx_license_1.SpdxLicense.bsd(),
                ...spdx_license_1.SpdxLicense.cddl(),
                ...spdx_license_1.SpdxLicense.epl(),
                spdx_license_1.SpdxLicense.ISC,
                ...spdx_license_1.SpdxLicense.mit(),
                spdx_license_1.SpdxLicense.MPL_2_0,
            ],
        });
        const webApp = new webapp_1.WebApp(this, 'WebApp', {
            domain: props.domain,
            monitoring,
            packageData,
            packageLinks: props.packageLinks,
            packageTags: packageTagsSerialized,
            packageTagGroups: props.packageTagGroups,
            featuredPackages: props.featuredPackages,
            packageStats,
            featureFlags: props.featureFlags,
            categories: props.categories,
            preloadScript: props.preloadScript,
        });
        const sources = new core_1.Construct(this, 'Sources');
        const packageSources = ((_k = props.packageSources) !== null && _k !== void 0 ? _k : [new package_sources_1.NpmJs()]).map((source) => source.bind(sources, {
            baseUrl: webApp.baseUrl,
            denyList,
            ingestion: this.ingestion,
            licenseList,
            monitoring,
            queue: this.ingestion.queue,
            repository: codeArtifact,
        }));
        const inventory = new inventory_1.Inventory(this, 'InventoryCanary', { bucket: packageData, logRetention: props.logRetention, monitoring });
        new backend_dashboard_1.BackendDashboard(this, 'BackendDashboard', {
            packageData,
            dashboardName: props.backendDashboardName,
            packageSources,
            ingestion: this.ingestion,
            inventory,
            orchestration,
            denyList,
            packageStats,
            versionTracker,
        });
        // add domain redirects
        if (props.domain) {
            for (const redirctSource of (_l = props.additionalDomains) !== null && _l !== void 0 ? _l : []) {
                new domain_redirect_1.DomainRedirect(this, `Redirect-${redirctSource.hostedZone.zoneName}`, {
                    source: redirctSource,
                    targetDomainName: (_m = props.domain) === null || _m === void 0 ? void 0 : _m.zone.zoneName,
                });
            }
        }
        else {
            if (props.additionalDomains && props.additionalDomains.length > 0) {
                throw new Error('Cannot specify "domainRedirects" if a domain is not specified');
            }
        }
    }
    get grantPrincipal() {
        return this.ingestion.grantPrincipal;
    }
    get ingestionQueue() {
        return this.ingestion.queue;
    }
    createVpc(isolation, codeArtifact) {
        if (isolation === Isolation.UNLIMITED_INTERNET_ACCESS) {
            return { vpc: undefined, vpcEndpoints: undefined, vpcSubnets: undefined };
        }
        const subnetType = isolation === Isolation.NO_INTERNET_ACCESS
            ? ec2.SubnetType.ISOLATED
            : ec2.SubnetType.PRIVATE_WITH_NAT;
        const vpcSubnets = { subnetType };
        const vpc = new ec2.Vpc(this, 'VPC', {
            enableDnsHostnames: true,
            enableDnsSupport: true,
            // Provision no NAT gateways if we are running ISOLATED (we wouldn't have a public subnet)
            natGateways: subnetType === ec2.SubnetType.ISOLATED ? 0 : undefined,
            // Pre-allocating PUBLIC / PRIVATE / INTERNAL subnets, regardless of use, so we don't create
            // a whole new VPC if we ever need to introduce subnets of these types.
            subnetConfiguration: [
                // If there is a PRIVATE subnet, there must also have a PUBLIC subnet (for NAT gateways).
                { name: 'Public', subnetType: ec2.SubnetType.PUBLIC, reserved: subnetType === ec2.SubnetType.ISOLATED },
                { name: 'Private', subnetType: ec2.SubnetType.PRIVATE_WITH_NAT, reserved: subnetType === ec2.SubnetType.ISOLATED },
                { name: 'Isolated', subnetType: ec2.SubnetType.ISOLATED, reserved: subnetType !== ec2.SubnetType.ISOLATED },
            ],
        });
        core_1.Tags.of(vpc.node.defaultChild).add('Name', vpc.node.path);
        const securityGroups = subnetType === ec2.SubnetType.PRIVATE_WITH_NAT
            ? _limited_internet_access_1.createRestrictedSecurityGroups(this, vpc)
            : undefined;
        // Creating the CodeArtifact endpoints only if a repository is present.
        const codeArtifactEndpoints = codeArtifact && {
            codeArtifactApi: vpc.addInterfaceEndpoint('CodeArtifact.API', {
                privateDnsEnabled: false,
                service: new ec2.InterfaceVpcEndpointAwsService('codeartifact.api'),
                subnets: vpcSubnets,
                securityGroups,
            }),
            codeArtifact: vpc.addInterfaceEndpoint('CodeArtifact', {
                privateDnsEnabled: true,
                service: new ec2.InterfaceVpcEndpointAwsService('codeartifact.repositories'),
                subnets: vpcSubnets,
                securityGroups,
            }),
        };
        // We'll only use VPC endpoints if we are configured to run in an ISOLATED subnet.
        const vpcEndpoints = {
            ...codeArtifactEndpoints,
            // This is needed so that ECS workloads can use the awslogs driver
            cloudWatchLogs: vpc.addInterfaceEndpoint('CloudWatch.Logs', {
                privateDnsEnabled: true,
                service: ec2.InterfaceVpcEndpointAwsService.CLOUDWATCH_LOGS,
                subnets: vpcSubnets,
                securityGroups,
            }),
            // These are needed for ECS workloads to be able to pull images
            ecrApi: vpc.addInterfaceEndpoint('ECR.API', {
                privateDnsEnabled: true,
                service: ec2.InterfaceVpcEndpointAwsService.ECR,
                subnets: vpcSubnets,
                securityGroups,
            }),
            ecr: vpc.addInterfaceEndpoint('ECR.Docker', {
                privateDnsEnabled: true,
                service: ec2.InterfaceVpcEndpointAwsService.ECR_DOCKER,
                subnets: vpcSubnets,
                securityGroups,
            }),
            // This is needed (among others) for CodeArtifact registry usage
            s3: vpc.addGatewayEndpoint('S3', {
                service: ec2.GatewayVpcEndpointAwsService.S3,
                subnets: [vpcSubnets],
            }),
            // This is useful for getting results from ECS tasks within workflows
            stepFunctions: vpc.addInterfaceEndpoint('StepFunctions', {
                privateDnsEnabled: true,
                service: ec2.InterfaceVpcEndpointAwsService.STEP_FUNCTIONS,
                subnets: vpcSubnets,
                securityGroups,
            }),
        };
        // The S3 access is necessary for the CodeArtifact Repository and ECR Docker
        // endpoints to be used (they serve objects from S3).
        vpcEndpoints.s3.addToPolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: ['s3:GetObject'],
            resources: [
                // The in-region CodeArtifact S3 Bucket
                ...codeArtifact ? [`${codeArtifact.s3BucketArn}/*`] : [],
                // The in-region ECR layer bucket
                `arn:aws:s3:::prod-${core_1.Stack.of(this).region}-starport-layer-bucket/*`,
            ],
            // It doesn't seem we can constrain principals for these grants (unclear
            // which principal those calls are made from, or if that is something we
            // could name here).
            principals: [new aws_iam_1.AnyPrincipal()],
            sid: 'Allow-CodeArtifact-and-ECR',
        }));
        return { vpc, vpcEndpoints, vpcSubnets, vpcSecurityGroups: securityGroups };
    }
}
exports.ConstructHub = ConstructHub;
_a = JSII_RTTI_SYMBOL_1;
ConstructHub[_a] = { fqn: "construct-hub.ConstructHub", version: "0.3.256" };
/**
 * How possibly risky operations (such as doc-generation, which requires
 * installing the indexed packages in order to trans-literate sample code) are
 * isolated to mitigate possible arbitrary code execution vulnerabilities in and
 * around `npm install` or the transliterator's use of the TypeScript compiler.
 */
var Isolation;
(function (Isolation) {
    /**
     * No isolation is done whatsoever. The doc-generation process still is
     * provisioned with least-privilege permissions, but retains complete access
     * to internet.
     *
     * While this maximizes the chances of successfully installing packages (and
     * hence successfully generating documentation for those), it is also the
     * least secure mode of operation.
     *
     * We advise you only consider using this isolation mode if you are hosting a
     * ConstructHub instance that only indexes trusted packages (including
     * transitive dependencies).
     */
    Isolation[Isolation["UNLIMITED_INTERNET_ACCESS"] = 0] = "UNLIMITED_INTERNET_ACCESS";
    /**
     * The same protections as `UNLIMITED_INTERNET_ACCESS`, except outbound
     * internet connections are limited to IP address ranges corresponding to
     * hosting endpoints for npmjs.com.
     */
    Isolation[Isolation["LIMITED_INTERNET_ACCESS"] = 1] = "LIMITED_INTERNET_ACCESS";
    /**
     * The same protections as `LIMITED_INTERNET_ACCESS`, except all remaining
     * internet access is removed. All traffic to AWS service endpoints is routed
     * through VPC Endpoints, as the compute nodes are jailed in a completely
     * isolated VPC.
     *
     * This is the most secure (and recommended) mode of operation for
     * ConstructHub instances.
     */
    Isolation[Isolation["NO_INTERNET_ACCESS"] = 2] = "NO_INTERNET_ACCESS";
})(Isolation = exports.Isolation || (exports.Isolation = {}));
//# sourceMappingURL=data:application/json;base64,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