"use strict";
var __classPrivateFieldGet = (this && this.__classPrivateFieldGet) || function (receiver, privateMap) {
    if (!privateMap.has(receiver)) {
        throw new TypeError("attempted to get private field on non-instance");
    }
    return privateMap.get(receiver);
};
var __classPrivateFieldSet = (this && this.__classPrivateFieldSet) || function (receiver, privateMap, value) {
    if (!privateMap.has(receiver)) {
        throw new TypeError("attempted to set private field on non-instance");
    }
    privateMap.set(receiver, value);
    return value;
};
var _externalConnections, _repositoryNpmEndpoint, _publishingRepositoryNpmEndpoint, _s3BucketArn;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Repository = void 0;
const aws_codeartifact_1 = require("@aws-cdk/aws-codeartifact");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const custom_resources_1 = require("@aws-cdk/custom-resources");
/**
 * A CodeArtifact repository.
 */
class Repository extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f;
        super(scope, id);
        _externalConnections.set(this, new Array());
        _repositoryNpmEndpoint.set(this, void 0);
        _publishingRepositoryNpmEndpoint.set(this, void 0);
        _s3BucketArn.set(this, void 0);
        if ((props === null || props === void 0 ? void 0 : props.domainExists) && !props.domainName) {
            throw new Error('domainExists cannot be specified if no domainName is provided');
        }
        if ((props === null || props === void 0 ? void 0 : props.upstreams) && !props.domainExists) {
            throw new Error('upstreams can only be specified if domainExists and domainName are provided');
        }
        const domainName = (_a = props === null || props === void 0 ? void 0 : props.domainName) !== null && _a !== void 0 ? _a : this.node.addr;
        const domain = (props === null || props === void 0 ? void 0 : props.domainExists) ? undefined : new aws_codeartifact_1.CfnDomain(this, 'Domain', { domainName });
        const repositoryName = (_b = props === null || props === void 0 ? void 0 : props.repositoryName) !== null && _b !== void 0 ? _b : this.node.addr;
        /**
         * This repository does not have any upstreams or external connections, so
         * it should be exempt from surprises. Concretely, any apckages published
         * to this repository will go in there, so we can separate "hand-published"
         * from "from upstream or external" packages.
         */
        const publishingUpstream = domain && new aws_codeartifact_1.CfnRepository(this, 'Publishing', {
            description: 'Publishing repository',
            domainName: domain.attrName,
            repositoryName: `${repositoryName}-publish-overlay`,
        });
        const repository = new aws_codeartifact_1.CfnRepository(this, 'Default', {
            description: props === null || props === void 0 ? void 0 : props.description,
            domainName: (_c = domain === null || domain === void 0 ? void 0 : domain.attrName) !== null && _c !== void 0 ? _c : domainName,
            externalConnections: core_1.Lazy.list({
                produce: () => !domain && __classPrivateFieldGet(this, _externalConnections).length > 0
                    ? __classPrivateFieldGet(this, _externalConnections) : undefined,
            }),
            repositoryName,
            // NOTE: Upstream order IS important, as CodeArtifact repositories query
            // their upstreams in the order they are listed in. The publishing
            // upstream is hence last in the list, so we get packages from the
            // "official" sources first.
            upstreams: core_1.Lazy.list({
                produce: () => {
                    var _a;
                    return domain && __classPrivateFieldGet(this, _externalConnections).length > 0
                        ? [
                            ...((_a = props === null || props === void 0 ? void 0 : props.upstreams) !== null && _a !== void 0 ? _a : []),
                            ...__classPrivateFieldGet(this, _externalConnections).map((name) => this.makeUpstreamForId(name, { domain, repositoryName })),
                            ...(publishingUpstream ? [publishingUpstream.attrName] : []),
                        ]
                        : publishingUpstream ? [publishingUpstream === null || publishingUpstream === void 0 ? void 0 : publishingUpstream.attrName] : props === null || props === void 0 ? void 0 : props.upstreams;
                },
            }),
        });
        this.repositoryDomainArn = (_d = domain === null || domain === void 0 ? void 0 : domain.attrArn) !== null && _d !== void 0 ? _d : core_1.Stack.of(this).formatArn({
            service: 'codeartifact',
            resource: 'domain',
            arnFormat: core_1.ArnFormat.SLASH_RESOURCE_NAME,
            resourceName: domainName,
        });
        this.repositoryDomainName = repository.attrDomainName;
        this.repositoryDomainOwner = repository.attrDomainOwner;
        this.repositoryArn = repository.attrArn;
        this.repositoryName = repository.attrName;
        this.publishingRepositoryArn = (_e = publishingUpstream === null || publishingUpstream === void 0 ? void 0 : publishingUpstream.attrArn) !== null && _e !== void 0 ? _e : this.repositoryArn;
        this.publishingRepositoryName = (_f = publishingUpstream === null || publishingUpstream === void 0 ? void 0 : publishingUpstream.attrName) !== null && _f !== void 0 ? _f : repositoryName;
    }
    /**
     * Adds an external connection to this repository.
     *
     * @param id the id of the external connection (i.e: `public:npmjs`).
     */
    addExternalConnection(id) {
        if (!__classPrivateFieldGet(this, _externalConnections).includes(id)) {
            __classPrivateFieldGet(this, _externalConnections).push(id);
        }
    }
    /**
     * The npm repository endpoint to use for interacting with this repository.
     */
    get repositoryNpmEndpoint() {
        if (__classPrivateFieldGet(this, _repositoryNpmEndpoint) == null) {
            const serviceCall = {
                service: 'CodeArtifact',
                action: 'getRepositoryEndpoint',
                parameters: {
                    domain: this.repositoryDomainName,
                    domainOwner: this.repositoryDomainOwner,
                    format: 'npm',
                    repository: this.repositoryName,
                },
                physicalResourceId: custom_resources_1.PhysicalResourceId.fromResponse('repositoryEndpoint'),
            };
            const endpoint = new custom_resources_1.AwsCustomResource(this, 'GetEndpoint', {
                onCreate: serviceCall,
                onUpdate: serviceCall,
                policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({ resources: [this.repositoryArn] }),
                resourceType: 'Custom::CodeArtifactNpmRepositoryEndpoint',
            });
            __classPrivateFieldSet(this, _repositoryNpmEndpoint, endpoint.getResponseField('repositoryEndpoint'));
        }
        return __classPrivateFieldGet(this, _repositoryNpmEndpoint);
    }
    /**
     * The npm repository endpoint to use for interacting with this repository to
     * publish new packages.
     */
    get publishingRepositoryNpmEndpoint() {
        if (__classPrivateFieldGet(this, _publishingRepositoryNpmEndpoint) == null) {
            const serviceCall = {
                service: 'CodeArtifact',
                action: 'getRepositoryEndpoint',
                parameters: {
                    domain: this.repositoryDomainName,
                    domainOwner: this.repositoryDomainOwner,
                    format: 'npm',
                    repository: this.publishingRepositoryName,
                },
                physicalResourceId: custom_resources_1.PhysicalResourceId.fromResponse('repositoryEndpoint'),
            };
            const endpoint = new custom_resources_1.AwsCustomResource(this, 'GetPublishingEndpoint', {
                onCreate: serviceCall,
                onUpdate: serviceCall,
                policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({ resources: [this.publishingRepositoryArn] }),
                resourceType: 'Custom::CodeArtifactNpmRepositoryEndpoint',
            });
            __classPrivateFieldSet(this, _publishingRepositoryNpmEndpoint, endpoint.getResponseField('repositoryEndpoint'));
        }
        return __classPrivateFieldGet(this, _publishingRepositoryNpmEndpoint);
    }
    /**
     * The S3 bucket in which CodeArtifact stores the package data. When using
     * VPC Endpoints for CodeArtifact, an S3 Gateway Endpoint must also be
     * available, which allows reading from this bucket.
     */
    get s3BucketArn() {
        if (__classPrivateFieldGet(this, _s3BucketArn) == null) {
            const domainDescription = new custom_resources_1.AwsCustomResource(this, 'DescribeDomain', {
                onCreate: {
                    service: 'CodeArtifact',
                    action: 'describeDomain',
                    parameters: {
                        domain: this.repositoryDomainName,
                        domainOwner: this.repositoryDomainOwner,
                    },
                    physicalResourceId: custom_resources_1.PhysicalResourceId.fromResponse('domain.s3BucketArn'),
                },
                policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({ resources: [this.repositoryDomainArn] }),
                resourceType: 'Custom::CoreArtifactDomainDescription',
            });
            __classPrivateFieldSet(this, _s3BucketArn, domainDescription.getResponseField('domain.s3BucketArn'));
        }
        return __classPrivateFieldGet(this, _s3BucketArn);
    }
    grantReadFromRepository(grantee) {
        // The Grant API does not allow conditions
        const stsGrantResult = grantee.grantPrincipal.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: ['sts:GetServiceBearerToken'],
            conditions: { StringEquals: { 'sts:AWSServiceName': 'codeartifact.amazonaws.com' } },
            resources: ['*'],
        }));
        if (!stsGrantResult.statementAdded) {
            return aws_iam_1.Grant.drop(grantee, 'CodeArtifact:ReadFromRepository');
        }
        return aws_iam_1.Grant.addToPrincipal({
            grantee,
            actions: [
                'codeartifact:GetAuthorizationToken',
                'codeartifact:GetRepositoryEndpoint',
                'codeartifact:ReadFromRepository',
            ],
            resourceArns: [this.repositoryDomainArn, this.repositoryArn, this.publishingRepositoryArn],
        });
    }
    grantPublishToRepository(grantee, format = 'npm') {
        const readOnlyGrant = this.grantReadFromRepository(grantee);
        if (!readOnlyGrant.success) {
            return readOnlyGrant;
        }
        return aws_iam_1.Grant.addToPrincipal({
            grantee,
            actions: [
                'codeartifact:PublishPackageVersion',
                'codeartifact:PutPackageMetadata',
            ],
            resourceArns: [core_1.Stack.of(this).formatArn({
                    arnFormat: core_1.ArnFormat.SLASH_RESOURCE_NAME,
                    service: 'codeartifact',
                    resource: 'package',
                    resourceName: `${this.repositoryDomainName}/${this.publishingRepositoryName}/${format}/*`,
                })],
        });
    }
    /**
     * Obtains a view of this repository that is intended to be accessed though
     * VPC endpoints.
     *
     * @param apiEndpoint  an `InterfaceVpcEndpoint` to the `codeartifact.api`
     *                     service.
     * @param repoEndpoint an `InterfaceVpcEndpoint` to the
     *                     `codeartifact.repositories` service.
     *
     * @returns a view of this repository that appropriately grants permissions on
     *          the VPC endpoint policies, too.
     */
    throughVpcEndpoint(apiEndpoint, repoEndpoint) {
        return new Proxy(this, {
            get(target, property, _receiver) {
                if (property === 'grantReadFromRepository') {
                    return decoratedGrantReadFromRepository.bind(target);
                }
                return target[property];
            },
            getOwnPropertyDescriptor(target, property) {
                const realDescriptor = Object.getOwnPropertyDescriptor(target, property);
                if (property === 'grantReadFromRepository') {
                    return {
                        ...realDescriptor,
                        value: decoratedGrantReadFromRepository,
                        get: undefined,
                        set: undefined,
                    };
                }
                return realDescriptor;
            },
        });
        function decoratedGrantReadFromRepository(grantee) {
            const mainGrant = this.grantReadFromRepository(grantee);
            if (mainGrant.success) {
                apiEndpoint.addToPolicy(new aws_iam_1.PolicyStatement({
                    effect: aws_iam_1.Effect.ALLOW,
                    actions: ['sts:GetServiceBearerToken'],
                    conditions: { StringEquals: { 'sts:AWSServiceName': 'codeartifact.amazonaws.com' } },
                    resources: ['*'],
                    principals: [grantee.grantPrincipal],
                }));
                apiEndpoint.addToPolicy(new aws_iam_1.PolicyStatement({
                    effect: aws_iam_1.Effect.ALLOW,
                    actions: ['codeartifact:GetAuthorizationToken', 'codeartifact:GetRepositoryEndpoint'],
                    resources: [this.repositoryDomainArn, this.repositoryArn],
                    principals: [grantee.grantPrincipal],
                }));
                repoEndpoint.addToPolicy(new aws_iam_1.PolicyStatement({
                    effect: aws_iam_1.Effect.ALLOW,
                    actions: ['codeartifact:ReadFromRepository'],
                    resources: [this.repositoryArn],
                    principals: [grantee.grantPrincipal],
                }));
            }
            return mainGrant;
        }
    }
    makeUpstreamForId(externalConnection, { domain, repositoryName }) {
        return new aws_codeartifact_1.CfnRepository(this, `Upstream:${externalConnection}`, {
            domainName: domain.attrName,
            description: `Upstream with external connection to ${externalConnection}`,
            externalConnections: [externalConnection],
            repositoryName: `${repositoryName}-${externalConnection.substr(7)}`,
        }).attrName;
    }
}
exports.Repository = Repository;
_externalConnections = new WeakMap(), _repositoryNpmEndpoint = new WeakMap(), _publishingRepositoryNpmEndpoint = new WeakMap(), _s3BucketArn = new WeakMap();
//# sourceMappingURL=data:application/json;base64,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