"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const os = require("os");
const path = require("path");
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const fs = require("fs-extra");
const docgen = require("jsii-docgen");
const markdown_render_1 = require("jsii-docgen/lib/docgen/render/markdown-render");
const caching_1 = require("../../caching");
const aws = require("../shared/aws.lambda-shared");
const code_artifact_lambda_shared_1 = require("../shared/code-artifact.lambda-shared");
const compress_content_lambda_shared_1 = require("../shared/compress-content.lambda-shared");
const constants = require("../shared/constants");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const language_1 = require("../shared/language");
const shell_out_lambda_shared_1 = require("../shared/shell-out.lambda-shared");
const constants_1 = require("./constants");
const util_1 = require("./util");
const ASSEMBLY_KEY_REGEX = new RegExp(`^${constants.STORAGE_KEY_PREFIX}((?:@[^/]+/)?[^/]+)/v([^/]+)${constants.ASSEMBLY_KEY_SUFFIX}$`);
// Capture groups:                                                     ┗━━━━━━━━━1━━━━━━━┛  ┗━━2━━┛
/**
 * This function receives an S3 event, and for each record, proceeds to download
 * the `.jsii` assembly the event refers to, transliterates it to the language,
 * configured in `TARGET_LANGUAGE`, and uploads the resulting `.jsii.<lang>`
 * object to S3.
 *
 * @param event   an S3 event payload
 * @param context a Lambda execution context
 *
 * @returns nothing
 */
function handler(event) {
    console.log(`Event: ${JSON.stringify(event, null, 2)}`);
    // We'll need a writable $HOME directory, or this won't work well, because
    // npm will try to write stuff like the `.npmrc` or package caches in there
    // and that'll bail out on EROFS if that fails.
    return ensureWritableHome(async () => {
        var _a, _b;
        const endpoint = process.env.CODE_ARTIFACT_REPOSITORY_ENDPOINT;
        if (!endpoint) {
            console.log('No CodeArtifact endpoint configured - using npm\'s default registry');
        }
        else {
            console.log(`Using CodeArtifact registry: ${endpoint}`);
            const domain = env_lambda_shared_1.requireEnv('CODE_ARTIFACT_DOMAIN_NAME');
            const domainOwner = process.env.CODE_ARTIFACT_DOMAIN_OWNER;
            const apiEndpoint = process.env.CODE_ARTIFACT_API_ENDPOINT;
            await code_artifact_lambda_shared_1.logInWithCodeArtifact({ endpoint, domain, domainOwner, apiEndpoint });
        }
        // Set up NPM shared cache directory (https://docs.npmjs.com/cli/v7/using-npm/config#cache)
        const npmCacheDir = process.env.NPM_CACHE;
        if (npmCacheDir) {
            // Create it if it does not exist yet...
            await fs.mkdirp(npmCacheDir);
            console.log(`Using shared NPM cache at: ${npmCacheDir}`);
            await shell_out_lambda_shared_1.shellOut('npm', 'config', 'set', `cache=${npmCacheDir}`);
        }
        const created = new Array();
        const deleted = new Array();
        const [, packageName, packageVersion] = (_a = event.assembly.key.match(ASSEMBLY_KEY_REGEX)) !== null && _a !== void 0 ? _a : [];
        if (packageName == null) {
            throw new Error(`Invalid object key: "${event.assembly.key}". It was expected to match ${ASSEMBLY_KEY_REGEX}!`);
        }
        const packageFqn = `${packageName}@${packageVersion}`;
        console.log(`Source Bucket:  ${event.bucket}`);
        console.log(`Source Key:     ${event.assembly.key}`);
        console.log(`Source Version: ${event.assembly.versionId}`);
        console.log(`Fetching assembly: ${event.assembly.key}`);
        const assemblyResponse = await aws.s3().getObject({ Bucket: event.bucket, Key: event.assembly.key }).promise();
        if (!assemblyResponse.Body) {
            throw new Error(`Response body for assembly at key ${event.assembly.key} is empty`);
        }
        const assembly = JSON.parse(assemblyResponse.Body.toString('utf-8'));
        const submodules = Object.keys((_b = assembly.submodules) !== null && _b !== void 0 ? _b : {}).map(s => s.split('.')[1]);
        console.log(`Fetching package: ${event.package.key}`);
        const tarballExists = await aws.s3ObjectExists(event.bucket, event.package.key);
        if (!tarballExists) {
            throw new Error(`Tarball does not exist at key ${event.package.key} in bucket ${event.bucket}.`);
        }
        const readStream = aws.s3().getObject({ Bucket: event.bucket, Key: event.package.key }).createReadStream();
        const tmpdir = fs.mkdtempSync(path.join(os.tmpdir(), 'packages-'));
        const tarball = path.join(tmpdir, 'package.tgz');
        await util_1.writeFile(tarball, readStream);
        const uploads = new Map();
        const deletions = new Map();
        let unprocessable = false;
        function markPackage(e, marker) {
            const key = event.assembly.key.replace(/\/[^/]+$/, marker);
            const upload = uploadFile(event.bucket, key, event.assembly.versionId, Buffer.from(e.message));
            uploads.set(key, upload);
        }
        async function unmarkPackage(marker) {
            const key = event.assembly.key.replace(/\/[^/]+$/, marker);
            const marked = await aws.s3ObjectExists(event.bucket, key);
            if (!marked) {
                return;
            }
            const deletion = deleteFile(event.bucket, key);
            deletions.set(key, deletion);
        }
        console.log(`Generating documentation for ${packageFqn}...`);
        try {
            const docs = await docgen.Documentation.forPackage(tarball, { name: assembly.name });
            // if the package used to not be installable, remove the marker for it.
            await unmarkPackage(constants.UNINSTALLABLE_PACKAGE_SUFFIX);
            for (const language of language_1.DocumentationLanguage.ALL) {
                if (event.languages && !event.languages[language.toString()]) {
                    console.log(`Skipping language ${language} as it was not requested!`);
                    continue;
                }
                const generateDocs = aws_embedded_metrics_1.metricScope((metrics) => async (lang) => {
                    metrics.setDimensions();
                    metrics.setNamespace(constants_1.METRICS_NAMESPACE);
                    async function renderAndDispatch(submodule) {
                        try {
                            console.log(`Rendering documentation in ${lang} for ${packageFqn} (submodule: ${submodule})`);
                            const json = await docs.toJson({
                                submodule,
                                language: docgen.Language.fromString(lang.name),
                            });
                            const jsonPage = Buffer.from(json.render());
                            metrics.putMetric("DocumentSizeBytes" /* DOCUMENT_SIZE */, jsonPage.length, aws_embedded_metrics_1.Unit.Bytes);
                            const { buffer: jsonBody, contentEncoding: jsonContentEncoding } = compress_content_lambda_shared_1.compressContent(jsonPage);
                            metrics.putMetric("CompressedDocumentSizeBytes" /* COMPRESSED_DOCUMENT_SIZE */, jsonBody.length, aws_embedded_metrics_1.Unit.Bytes);
                            const jsonKey = event.assembly.key.replace(/\/[^/]+$/, constants.docsKeySuffix(lang, submodule, 'json'));
                            console.log(`Uploading ${jsonKey}`);
                            const jsonUpload = uploadFile(event.bucket, jsonKey, event.assembly.versionId, jsonBody, jsonContentEncoding);
                            uploads.set(jsonKey, jsonUpload);
                            const markdown = markdown_render_1.MarkdownRenderer.fromSchema(json.content, {
                                anchorFormatter,
                                linkFormatter: linkFormatter(lang),
                            });
                            const page = Buffer.from(markdown.render());
                            metrics.putMetric("DocumentSizeBytes" /* DOCUMENT_SIZE */, page.length, aws_embedded_metrics_1.Unit.Bytes);
                            const { buffer: body, contentEncoding } = compress_content_lambda_shared_1.compressContent(page);
                            metrics.putMetric("CompressedDocumentSizeBytes" /* COMPRESSED_DOCUMENT_SIZE */, body.length, aws_embedded_metrics_1.Unit.Bytes);
                            const key = event.assembly.key.replace(/\/[^/]+$/, constants.docsKeySuffix(lang, submodule, 'md'));
                            console.log(`Uploading ${key}`);
                            const upload = uploadFile(event.bucket, key, event.assembly.versionId, body, contentEncoding);
                            uploads.set(key, upload);
                            // if the package used to have a corrupt assembly, remove the marker for it.
                            await unmarkPackage(constants.corruptAssemblyKeySuffix(language, submodule, 'md'));
                        }
                        catch (e) {
                            if (e instanceof docgen.LanguageNotSupportedError) {
                                markPackage(e, constants.notSupportedKeySuffix(language, submodule, 'json'));
                                markPackage(e, constants.notSupportedKeySuffix(language, submodule, 'md'));
                            }
                            else if (e instanceof docgen.CorruptedAssemblyError) {
                                markPackage(e, constants.corruptAssemblyKeySuffix(language, submodule, 'json'));
                                markPackage(e, constants.corruptAssemblyKeySuffix(language, submodule, 'md'));
                                unprocessable = true;
                            }
                            else {
                                throw e;
                            }
                        }
                    }
                    await renderAndDispatch();
                    for (const submodule of submodules) {
                        await renderAndDispatch(submodule);
                    }
                });
                await generateDocs(language);
            }
        }
        catch (error) {
            if (error instanceof docgen.UnInstallablePackageError) {
                markPackage(error, constants.UNINSTALLABLE_PACKAGE_SUFFIX);
                unprocessable = true;
            }
            else {
                throw error;
            }
        }
        for (const [key, upload] of uploads.entries()) {
            const response = await upload;
            created.push({ bucket: event.bucket, key, versionId: response.VersionId });
            console.log(`Finished uploading ${key} (Version ID: ${response.VersionId})`);
        }
        for (const [key, deletion] of deletions.entries()) {
            const response = await deletion;
            deleted.push({ bucket: event.bucket, key, versionId: response.VersionId });
            console.log(`Finished deleting ${key} (Version ID: ${response.VersionId})`);
        }
        if (unprocessable) {
            // the message here doesn't matter, we only use the error name
            // to divert this message away from the DLQ.
            const error = new Error();
            error.name = constants.UNPROCESSABLE_PACKAGE_ERROR_NAME;
        }
        // output must be compressed to satisfy 262,144 byte limit of SendTaskSuccess command
        const s3OKey = (s3Obj) => s3Obj.key;
        return { created: created.map(s3OKey), deleted: deleted.map(s3OKey) };
    });
}
exports.handler = handler;
async function ensureWritableHome(cb) {
    // Since $HOME is not set, or is not writable, we'll just go make our own...
    const fakeHome = await fs.mkdtemp(path.join(os.tmpdir(), 'fake-home'));
    console.log(`Made temporary $HOME directory: ${fakeHome}`);
    const oldHome = process.env.HOME;
    try {
        process.env.HOME = fakeHome;
        return await cb();
    }
    finally {
        process.env.HOME = oldHome;
        await fs.remove(fakeHome);
        console.log(`Cleaned-up temporary $HOME directory: ${fakeHome}`);
    }
}
function uploadFile(bucket, key, sourceVersionId, body, contentEncoding) {
    const contentType = key.endsWith('.md') ?
        'text/markdown; charset=UTF-8' :
        key.endsWith('.json') ?
            'application/json; charset=UTF-8' :
            'application/octet-stream';
    return aws.s3().putObject({
        Bucket: bucket,
        Key: key,
        Body: body,
        CacheControl: caching_1.CacheStrategy.default().toString(),
        ContentEncoding: contentEncoding,
        ContentType: contentType,
        Metadata: {
            'Origin-Version-Id': sourceVersionId !== null && sourceVersionId !== void 0 ? sourceVersionId : 'N/A',
        },
    }).promise();
}
function deleteFile(bucket, key) {
    return aws.s3().deleteObject({
        Bucket: bucket,
        Key: key,
    }).promise();
}
function anchorFormatter(type) {
    const name = getAssemblyRelativeName(type); // BucketProps.Initializer.parameter.accessControl
    const [base, ...rest] = name.split('.');
    if (rest.length > 0) {
        return sanitize(rest.join('.')); // Initializer.parameter.accessControl
    }
    else {
        return sanitize(base);
    }
}
function linkFormatter(lang) {
    const formatter = (type) => {
        const name = getAssemblyRelativeName(type); // BucketProps.Initializer.parameter.accessControl
        const [baseName, ...rest] = name.split('.');
        const hash = '#' + rest.join('.'); // #Initializer.parameter.accessControl
        const langParam = `?lang=${lang.toString()}`;
        const submoduleParam = type.submodule ? `&submodule=${type.submodule}` : '';
        return `<a href="/packages/${type.packageName}/v/${type.packageVersion}/api/${baseName}${langParam}${submoduleParam}${hash}">${type.displayName}</a>`;
    };
    return formatter;
}
/**
 * Converts a type's id to an assembly-relative version, e.g.:
 * `aws-cdk-lib.aws_s3.Bucket.parameter.accessControl` => `Bucket.parameter.accessControl`
 */
function getAssemblyRelativeName(type) {
    let name = type.id;
    if (!name.startsWith(type.packageName)) {
        throw new Error(`Expected first part of "${type.id}" to start with "${type.packageName}".`);
    }
    name = name.slice(type.packageName.length + 1); // remove "aws-cdk-lib.""
    if (type.submodule) {
        if (!name.startsWith(type.submodule)) {
            throw new Error(`Expected second part of "${type.id}" to start with "${type.submodule}".`);
        }
        name = name.slice(type.submodule.length + 1); // remove "aws_s3."
    }
    return name;
}
;
function sanitize(str) {
    // HTML5 allows any characters in IDs except whitespace
    return str.replace(/ /g, '-');
}
//# sourceMappingURL=data:application/json;base64,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