"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const got_1 = require("got");
const caching_1 = require("../../caching");
const client_lambda_shared_1 = require("../catalog-builder/client.lambda-shared");
const aws = require("../shared/aws.lambda-shared");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const constants_1 = require("./constants");
const npm_downloads_lambda_shared_1 = require("./npm-downloads.lambda-shared");
/**
 * Rebuilds the `stats.json` object in the configured S3 bucket.
 * Validates that the number of packages on a full rebuild should not decrease
 * significantly (due to network errors from e.g. NPM) - can be ignored
 * by passing { ignoreValidation: true }.
 *
 * @param event configuration for the rebuild job.
 * @param context the lambda context in which this execution runs.
 *
 * @returns the information about the updated S3 object.
 */
async function handler(event, context) {
    console.log(JSON.stringify(event, null, 2));
    const STATS_BUCKET_NAME = env_lambda_shared_1.requireEnv('STATS_BUCKET_NAME');
    const STATS_OBJECT_KEY = env_lambda_shared_1.requireEnv('STATS_OBJECT_KEY');
    const catalogClient = await client_lambda_shared_1.CatalogClient.newClient();
    const catalog = catalogClient.packages;
    if (catalog.length === 0) {
        throw new Error('No packages found.');
    }
    const currentDate = new Date().toISOString();
    const stats = { packages: {}, updated: currentDate };
    const npmClient = new npm_downloads_lambda_shared_1.NpmDownloadsClient(got_1.default);
    // remove duplicates from different major versions
    const packageNames = [...new Set(catalog.map(pkg => pkg.name)).values()];
    console.log(`Retrieving download stats for all ${packageNames.length} registered packages: [${packageNames.join(',')}].`);
    const npmDownloads = await npmClient.getDownloads(packageNames, {
        period: npm_downloads_lambda_shared_1.NpmDownloadsPeriod.LAST_WEEK,
        throwErrors: false,
    });
    for (const [pkgName, entry] of npmDownloads.entries()) {
        updateStats(stats, pkgName, entry);
    }
    // Update metrics
    const statsCount = Object.keys(stats.packages).length;
    console.log(`There are now ${statsCount} packages with NPM stats stored.`);
    await aws_embedded_metrics_1.metricScope((metrics) => async () => {
        // Clear out default dimensions as we don't need those. See https://github.com/awslabs/aws-embedded-metrics-node/issues/73
        metrics.setDimensions();
        metrics.setNamespace(constants_1.METRICS_NAMESPACE);
        metrics.putMetric("RegisteredPackagesWithStats" /* REGISTERED_PACKAGES_WITH_STATS */, statsCount, aws_embedded_metrics_1.Unit.Count);
    })();
    // Upload the result to S3 and exit.
    return aws.s3().putObject({
        Bucket: STATS_BUCKET_NAME,
        Key: STATS_OBJECT_KEY,
        Body: JSON.stringify(stats, null, 2),
        ContentType: 'application/json',
        CacheControl: caching_1.CacheStrategy.default().toString(),
        Metadata: {
            'Lambda-Log-Group': context.logGroupName,
            'Lambda-Log-Stream': context.logStreamName,
            'Lambda-Run-Id': context.awsRequestId,
            'Package-Stats-Count': `${statsCount}`,
        },
    }).promise();
}
exports.handler = handler;
function updateStats(stats, pkgName, entry) {
    var _a, _b, _c;
    stats.packages[pkgName] = {
        ...((_a = stats.packages[pkgName]) !== null && _a !== void 0 ? _a : {}),
        downloads: {
            ...((_c = (_b = stats.packages[pkgName]) === null || _b === void 0 ? void 0 : _b.downloads) !== null && _c !== void 0 ? _c : {}),
            npm: entry.downloads,
        },
    };
}
//# sourceMappingURL=data:application/json;base64,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