"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Ingestion = void 0;
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_events_1 = require("@aws-cdk/aws-events");
const aws_events_targets_1 = require("@aws-cdk/aws-events-targets");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_lambda_event_sources_1 = require("@aws-cdk/aws-lambda-event-sources");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const aws_s3_deployment_1 = require("@aws-cdk/aws-s3-deployment");
const aws_sqs_1 = require("@aws-cdk/aws-sqs");
const aws_stepfunctions_1 = require("@aws-cdk/aws-stepfunctions");
const aws_stepfunctions_tasks_1 = require("@aws-cdk/aws-stepfunctions-tasks");
const core_1 = require("@aws-cdk/core");
const deep_link_1 = require("../../deep-link");
const runbook_url_1 = require("../../runbook-url");
const storage_1 = require("../../s3/storage");
const temp_file_1 = require("../../temp-file");
const _lambda_architecture_1 = require("../_lambda-architecture");
const constants_1 = require("../shared/constants");
const constants_2 = require("./constants");
const ingestion_1 = require("./ingestion");
const re_ingest_1 = require("./re-ingest");
/**
 * The ingestion function receives messages from discovery integrations and
 * processes their payloads into the provided S3 Bucket.
 *
 * This function is also an `IGrantable`, so that it can be granted permissions
 * to read from the source S3 buckets.
 */
class Ingestion extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b, _c, _d;
        super(scope, id);
        this.queueRetentionPeriod = core_1.Duration.days(14);
        this.deadLetterQueue = new aws_sqs_1.Queue(this, 'DLQ', {
            encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
            retentionPeriod: this.queueRetentionPeriod,
            visibilityTimeout: core_1.Duration.minutes(15),
        });
        this.queue = new aws_sqs_1.Queue(this, 'Queue', {
            deadLetterQueue: {
                maxReceiveCount: 5,
                queue: this.deadLetterQueue,
            },
            encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
            retentionPeriod: this.queueRetentionPeriod,
            visibilityTimeout: core_1.Duration.minutes(15),
        });
        const configFilename = 'config.json';
        const config = new temp_file_1.TempFile(configFilename, JSON.stringify({
            packageLinks: (_a = props.packageLinks) !== null && _a !== void 0 ? _a : [],
            packageTags: (_b = props.packageTags) !== null && _b !== void 0 ? _b : [],
        }));
        const storageFactory = storage_1.S3StorageFactory.getOrCreate(this);
        const configBucket = storageFactory.newBucket(this, 'ConfigBucket', {
            blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
            enforceSSL: true,
            versioned: true,
        });
        new aws_s3_deployment_1.BucketDeployment(this, 'DeployIngestionConfiguration', {
            sources: [aws_s3_deployment_1.Source.asset(config.dir)],
            destinationBucket: configBucket,
        });
        const environment = {
            AWS_EMF_ENVIRONMENT: 'Local',
            BUCKET_NAME: props.bucket.bucketName,
            CONFIG_BUCKET_NAME: configBucket.bucketName,
            CONFIG_FILE_KEY: configFilename,
            STATE_MACHINE_ARN: props.orchestration.stateMachine.stateMachineArn,
        };
        if (props.codeArtifact) {
            environment.CODE_ARTIFACT_REPOSITORY_ENDPOINT = props.codeArtifact.publishingRepositoryNpmEndpoint;
            environment.CODE_ARTIFACT_DOMAIN_NAME = props.codeArtifact.repositoryDomainName;
            environment.CODE_ARTIFACT_DOMAIN_OWNER = props.codeArtifact.repositoryDomainOwner;
        }
        const handler = new ingestion_1.Ingestion(this, 'Default', {
            description: '[ConstructHub/Ingestion] Ingests new package versions into the Construct Hub',
            environment,
            logRetention: (_c = props.logRetention) !== null && _c !== void 0 ? _c : aws_logs_1.RetentionDays.TEN_YEARS,
            memorySize: 10240,
            timeout: core_1.Duration.minutes(15),
            tracing: aws_lambda_1.Tracing.ACTIVE,
        });
        this.function = handler;
        configBucket.grantRead(handler);
        props.bucket.grantWrite(this.function);
        (_d = props.codeArtifact) === null || _d === void 0 ? void 0 : _d.grantPublishToRepository(handler);
        props.orchestration.stateMachine.grantStartExecution(this.function);
        this.function.addEventSource(new aws_lambda_event_sources_1.SqsEventSource(this.queue, { batchSize: 1 }));
        // This event source is disabled, and can be used to re-process dead-letter-queue messages
        this.function.addEventSource(new aws_lambda_event_sources_1.SqsEventSource(this.deadLetterQueue, { batchSize: 1, enabled: false }));
        // Reprocess workflow
        const reprocessQueue = new aws_sqs_1.Queue(this, 'ReprocessQueue', {
            deadLetterQueue: {
                maxReceiveCount: 5,
                queue: this.deadLetterQueue,
            },
            encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
            retentionPeriod: this.queueRetentionPeriod,
            // Visibility timeout of 15 minutes matches the Lambda maximum execution time.
            visibilityTimeout: core_1.Duration.minutes(15),
        });
        props.bucket.grantRead(this.function, `${constants_1.STORAGE_KEY_PREFIX}*${constants_1.PACKAGE_KEY_SUFFIX}`);
        this.function.addEventSource(new aws_lambda_event_sources_1.SqsEventSource(reprocessQueue, { batchSize: 1 }));
        const reprocessWorkflow = new ReprocessIngestionWorkflow(this, 'ReprocessWorkflow', { bucket: props.bucket, queue: reprocessQueue });
        // Run reprocess workflow on a daily basis
        const updatePeriod = props.reprocessFrequency;
        if (updatePeriod) {
            const rule = new aws_events_1.Rule(this, 'ReprocessCronJob', {
                schedule: aws_events_1.Schedule.rate(updatePeriod),
                description: 'Periodically reprocess all packages',
            });
            rule.addTarget(new aws_events_targets_1.SfnStateMachine(reprocessWorkflow.stateMachine, {
                input: aws_events_1.RuleTargetInput.fromObject({
                    comment: 'Scheduled reprocessing event from cron job.',
                }),
            }));
        }
        this.grantPrincipal = this.function.grantPrincipal;
        props.monitoring.addLowSeverityAlarm('Ingestion Dead-Letter Queue not empty', new aws_cloudwatch_1.MathExpression({
            expression: 'm1 + m2',
            usingMetrics: {
                m1: this.deadLetterQueue.metricApproximateNumberOfMessagesVisible({ period: core_1.Duration.minutes(1) }),
                m2: this.deadLetterQueue.metricApproximateNumberOfMessagesNotVisible({ period: core_1.Duration.minutes(1) }),
            },
        }).createAlarm(this, 'DLQAlarm', {
            alarmName: `${this.node.path}/DLQNotEmpty`,
            alarmDescription: [
                'The dead-letter queue for the Ingestion function is not empty!',
                '',
                `RunBook: ${runbook_url_1.RUNBOOK_URL}`,
                '',
                `Direct link to the queue: ${deep_link_1.sqsQueueUrl(this.deadLetterQueue)}`,
                `Direct link to the function: ${deep_link_1.lambdaFunctionUrl(this.function)}`,
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
            // SQS does not emit metrics if the queue has been empty for a while, which is GOOD.
            treatMissingData: aws_cloudwatch_1.TreatMissingData.NOT_BREACHING,
        }));
        props.monitoring.addHighSeverityAlarm('Ingestion failures', this.function.metricErrors().createAlarm(this, 'FailureAlarm', {
            alarmName: `${this.node.path}/Failure`,
            alarmDescription: [
                'The Ingestion function is failing!',
                '',
                `RunBook: ${runbook_url_1.RUNBOOK_URL}`,
                '',
                `Direct link to the function: ${deep_link_1.lambdaFunctionUrl(this.function)}`,
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 2,
            threshold: 1,
            // Lambda only emits metrics when the function is invoked. No invokation => no errors.
            treatMissingData: aws_cloudwatch_1.TreatMissingData.NOT_BREACHING,
        }));
    }
    metricFoundLicenseFile(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "FoundLicenseFile" /* FOUND_LICENSE_FILE */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    metricIneligibleLicense(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "IneligibleLicense" /* INELIGIBLE_LICENSE */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    metricInvalidAssembly(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "InvalidAssembly" /* INVALID_ASSEMBLY */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    metricInvalidTarball(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "InvalidTarball" /* INVALID_TARBALL */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    /**
     * This metrics is the total count of packages that were rejected due to
     * mismatched identity (name, version, license) between the `package.json`
     * file and te `.jsii` attribute.
     */
    metricMismatchedIdentityRejections(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "MismatchedIdentityRejections" /* MISMATCHED_IDENTITY_REJECTIONS */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
}
exports.Ingestion = Ingestion;
/**
 * A StepFunctions State Machine to reprocess every currently indexed package
 * through the ingestion function. This should not be frequently required, but
 * can come in handy at times.
 *
 * For more information, refer to the runbook at
 * https://github.com/cdklabs/construct-hub/blob/main/docs/operator-runbook.md
 */
class ReprocessIngestionWorkflow extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const lambdaFunction = new re_ingest_1.ReIngest(this, 'Function', {
            architecture: _lambda_architecture_1.gravitonLambdaIfAvailable(this),
            description: '[ConstructHub/Ingestion/ReIngest] The function used to reprocess packages through ingestion',
            environment: { BUCKET_NAME: props.bucket.bucketName, QUEUE_URL: props.queue.queueUrl },
            memorySize: 10240,
            tracing: aws_lambda_1.Tracing.ACTIVE,
            timeout: core_1.Duration.minutes(3),
        });
        props.queue.grantSendMessages(lambdaFunction);
        props.bucket.grantRead(lambdaFunction, `${constants_1.STORAGE_KEY_PREFIX}*${constants_1.METADATA_KEY_SUFFIX}`);
        props.bucket.grantRead(lambdaFunction, `${constants_1.STORAGE_KEY_PREFIX}*${constants_1.PACKAGE_KEY_SUFFIX}`);
        // Need to physical-name the state machine so it can self-invoke.
        const stateMachineName = stateMachineNameFrom(this.node.path);
        const listBucket = new aws_stepfunctions_1.Choice(this, 'Has a ContinuationToken?')
            .when(aws_stepfunctions_1.Condition.isPresent('$.ContinuationToken'), new aws_stepfunctions_tasks_1.CallAwsService(this, 'S3.ListObjectsV2(NextPage)', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                ContinuationToken: aws_stepfunctions_1.JsonPath.stringAt('$.ContinuationToken'),
                Prefix: constants_1.STORAGE_KEY_PREFIX,
            },
            resultPath: '$.response',
        }).addRetry({ errors: ['S3.SdkClientException'] }))
            .otherwise(new aws_stepfunctions_tasks_1.CallAwsService(this, 'S3.ListObjectsV2(FirstPage)', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                Prefix: constants_1.STORAGE_KEY_PREFIX,
            },
            resultPath: '$.response',
        }).addRetry({ errors: ['S3.SdkClientException'] })).afterwards();
        const process = new aws_stepfunctions_1.Map(this, 'Process Result', {
            itemsPath: '$.response.Contents',
            resultPath: aws_stepfunctions_1.JsonPath.DISCARD,
        }).iterator(new aws_stepfunctions_1.Choice(this, 'Is metadata object?')
            .when(aws_stepfunctions_1.Condition.stringMatches('$.Key', `*${constants_1.METADATA_KEY_SUFFIX}`), new aws_stepfunctions_tasks_1.LambdaInvoke(this, 'Send for reprocessing', { lambdaFunction })
            // Ample retries here... We should never fail because of throttling....
            .addRetry({ errors: ['Lambda.TooManyRequestsException'], backoffRate: 1.1, interval: core_1.Duration.minutes(1), maxAttempts: 30 }))
            .otherwise(new aws_stepfunctions_1.Succeed(this, 'Nothing to do')));
        listBucket.next(new aws_stepfunctions_1.Choice(this, 'Is there more?')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), new aws_stepfunctions_1.Wait(this, 'Give room for on-demand work', {
            // Sleep a little before enqueuing the next batch, so that we leave room in the worker
            // pool for handling on-demand work. If we don't do this, 60k items will be queued at
            // once and live updates from NPM will struggle to get in in a reasonable time.
            time: aws_stepfunctions_1.WaitTime.duration(waitTimeBetweenReprocessBatches()),
        }).next(new aws_stepfunctions_tasks_1.StepFunctionsStartExecution(this, 'Continue as new', {
            associateWithParent: true,
            stateMachine: aws_stepfunctions_1.StateMachine.fromStateMachineArn(this, 'ThisStateMachine', core_1.Stack.of(this).formatArn({
                arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME,
                service: 'states',
                resource: 'stateMachine',
                resourceName: stateMachineName,
            })),
            input: aws_stepfunctions_1.TaskInput.fromObject({ ContinuationToken: aws_stepfunctions_1.JsonPath.stringAt('$.response.NextContinuationToken') }),
            integrationPattern: aws_stepfunctions_1.IntegrationPattern.REQUEST_RESPONSE,
            resultPath: aws_stepfunctions_1.JsonPath.DISCARD,
        }).addRetry({ errors: ['StepFunctions.ExecutionLimitExceeded'] }))).afterwards({ includeOtherwise: true })
            .next(process));
        this.stateMachine = new aws_stepfunctions_1.StateMachine(this, 'StateMachine', {
            definition: listBucket,
            stateMachineName,
            timeout: core_1.Duration.hours(1),
        });
        props.bucket.grantRead(this.stateMachine);
        props.queue.grantSendMessages(this.stateMachine);
    }
}
/**
 * This turns a node path into a valid state machine name, to try and improve
 * the StepFunction's AWS console experience while minimizing the risk for
 * collisons.
 */
function stateMachineNameFrom(nodePath) {
    // Poor man's replace all...
    return nodePath.split(/[^a-z0-9+!@.()=_'-]+/i).join('.');
}
/**
 * The time we wait between enqueueing different batches of the reprocessing machine
 */
function waitTimeBetweenReprocessBatches() {
    // Average time per ECS task. We don've have statistics on this, but
    // can be roughly derived from:
    // Every day we process about 60k items with 1000 workers in 4 hours.
    // 4 hours / (60_000 / 1000) ~= 4 minutes
    const avgTimePerTask = core_1.Duration.minutes(4);
    // How many objects are returned by 'listObjectsV2', per call
    const batchSize = 1000;
    // How many workers we have at our disposal
    const workers = 1000;
    // The step functions state machine can't instantaneously start all 1000
    // tasks, they are staggered over time -- so in practice the average load a
    // single batch puts on the ECS cluster at any point in time is a lot lower.
    const sfnStaggerFactor = 0.05;
    // What fraction of capacity [0..1) we want to keep available for on-demand
    // work, while reprocessing.
    const marginFrac = 0.2;
    const seconds = (avgTimePerTask.toSeconds() * sfnStaggerFactor / (1 - marginFrac)) * (batchSize / workers);
    return core_1.Duration.seconds(Math.floor(seconds));
}
//# sourceMappingURL=data:application/json;base64,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