"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DenyList = void 0;
const fs = require("fs");
const os = require("os");
const path = require("path");
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const events = require("@aws-cdk/aws-events");
const targets = require("@aws-cdk/aws-events-targets");
const aws_lambda_event_sources_1 = require("@aws-cdk/aws-lambda-event-sources");
const s3 = require("@aws-cdk/aws-s3");
const s3deploy = require("@aws-cdk/aws-s3-deployment");
const core_1 = require("@aws-cdk/core");
const storage_1 = require("../../s3/storage");
const constants_1 = require("./constants");
const create_map_1 = require("./create-map");
const prune_1 = require("./prune");
/**
 * Manages the construct hub deny list.
 */
class DenyList extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        /**
         * The object key within the bucket with the deny list JSON file.
         */
        this.objectKey = 'deny-list.json';
        const storageFactory = storage_1.S3StorageFactory.getOrCreate(this);
        this.bucket = storageFactory.newBucket(this, 'Bucket', {
            blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
            encryption: s3.BucketEncryption.S3_MANAGED,
            enforceSSL: true,
            removalPolicy: core_1.RemovalPolicy.DESTROY,
            autoDeleteObjects: true,
            versioned: true,
        });
        const directory = this.writeToFile(props.rules, this.objectKey);
        // upload the deny list to the bucket
        const upload = new s3deploy.BucketDeployment(this, 'BucketDeployment', {
            destinationBucket: this.bucket,
            prune: true,
            retainOnDelete: true,
            sources: [s3deploy.Source.asset(directory)],
        });
        this.uploadReady = upload.node.findChild('CustomResource');
        this.prune = new prune_1.Prune(this, 'Prune', {
            packageDataBucket: props.packageDataBucket,
            packageDataKeyPrefix: props.packageDataKeyPrefix,
            monitoring: props.monitoring,
        });
        this.grantRead(this.prune.pruneHandler);
        // trigger prune when the deny list changes
        const pruneOnChange = (_a = props.pruneOnChange) !== null && _a !== void 0 ? _a : true;
        if (pruneOnChange) {
            this.prune.pruneHandler.addEventSource(new aws_lambda_event_sources_1.S3EventSource(this.bucket, {
                events: [s3.EventType.OBJECT_CREATED],
                filters: [{ prefix: this.objectKey, suffix: this.objectKey }],
            }));
            // Add an explicit dep between upload and the bucket notification. We are
            // not using the whole bucket scope to reduce the likelihood of a
            // dependency cycle.
            const notificationsReady = this.bucket.node.findChild('Notifications');
            this.uploadReady.node.addDependency(notificationsReady);
        }
        // trigger prune periodically (every 5 minutes) - just in case
        const prunePeriod = (_b = props.prunePeriod) !== null && _b !== void 0 ? _b : core_1.Duration.minutes(5);
        if (prunePeriod && prunePeriod.toSeconds() > 0) {
            new events.Rule(this, 'PeriodicPrune', {
                schedule: events.Schedule.rate(prunePeriod),
                targets: [new targets.LambdaFunction(this.prune.pruneHandler)],
            });
        }
    }
    /**
     * Grants an AWS Lambda function permissions to read the deny list, and adds
     * the relevant environment variables expected by the `DenyListClient`.
     */
    grantRead(handler) {
        handler.addEnvironment(constants_1.ENV_DENY_LIST_BUCKET_NAME, this.bucket.bucketName);
        handler.addEnvironment(constants_1.ENV_DENY_LIST_OBJECT_KEY, this.objectKey);
        this.bucket.grantRead(handler);
    }
    /**
     * Number of rules in the deny list.
     */
    metricDenyListRules(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "DenyListRuleCount" /* DENY_LIST_RULE_COUNT */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    /**
     * Writes the deny list to a temporary file and returns a path to a directory
     * with the JSON file. The contents of the JSON file is a map where keys are
     * package names (and optionally, version) and the value is the deny list
     * entry. This makes it easier to query by the service.
     *
     * Also performs some validation to make sure there are no duplicate entries.
     *
     * @param list The deny list
     * @returns a path to a temporary directory that can be deployed to S3
     */
    writeToFile(list, fileName) {
        const tmpdir = fs.mkdtempSync(path.join(os.tmpdir(), 'deny-list-'));
        const filePath = path.join(tmpdir, fileName);
        const map = create_map_1.createDenyListMap(list);
        fs.writeFileSync(filePath, JSON.stringify(map, null, 2));
        return tmpdir;
    }
}
exports.DenyList = DenyList;
//# sourceMappingURL=data:application/json;base64,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