# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['esdbclient', 'esdbclient.protos.Grpc']

package_data = \
{'': ['*']}

install_requires = \
['grpcio>=1.51.0,<1.52.0', 'protobuf>=4.21.0,<5.0.0', 'typing_extensions']

setup_kwargs = {
    'name': 'esdbclient',
    'version': '0.10',
    'description': 'Python gRPC Client for EventStoreDB',
    'long_description': '# Python gRPC Client for EventStoreDB\n\nThis package provides a Python gRPC client for\n[EventStoreDB](https://www.eventstore.com/).\n\nThis client is implemented as the Python class `ESDBClient`.\n\nThis client has been developed and tested to work with EventStoreDB LTS\nversions 21.10 and 21.10, without and without SSL/TLS enabled on both\nthese versions, and with Python versions 3.7, 3.8, 3.9, 3.10, and 3.11\nacross all of the above. The test coverage is 100% including branch coverage.\n\nAll the Python code in this package has typing annotations. The static typing\nannotations are checked relatively strictly with mypy. The code is formatted\nwith black and isort, and also checked with flake8. Poetry is used for package\nmanagement during development, and for building and publishing distributions to\n[PyPI](https://pypi.org/project/esdbclient/).\n\nNot all the features of the EventStoreDB API are presented\nby this client in its current form, however many of the most\nuseful aspects are presented in an easy-to-use interface (see below).\n\nProbably the three most useful methods are `append_events()`, `read_stream_events()`\nand `subscribe_all_events()`.\n\n* The `append_events()` method can be used to record atomically all the new\nevents that are generated when executing a command in an application.\n\n* The `read_stream_events()` method can be used to retrieve all the recorded\nevents for an aggregate before executing a command in an application.\n\n* The `subscribe_all_events()` method can be used by a downstream component\nto process recorded events with exactly-once semantics.\n\nFor an example of usage, see the [eventsourcing-eventstoredb](\nhttps://github.com/pyeventsourcing/eventsourcing-eventstoredb) package.\n\n## Table of contents\n\n<!-- TOC -->\n* [Install package](#install-package)\n  * [From PyPI](#from-pypi)\n  * [With Poetry](#with-poetry)\n* [Server container](#server-container)\n  * [Run container](#run-container)\n  * [Stop container](#stop-container)\n* [Client class](#client-class)\n  * [Import class from package](#import-class-from-package)\n  * [Contruct client class](#contruct-client-class)\n* [Streams](#streams)\n  * [Append events](#append-events)\n  * [Append event](#append-event)\n  * [Idempotent append operations](#idempotent-append-operations)\n  * [Read stream events](#read-stream-events)\n  * [Read all events](#read-all-events)\n  * [Get current stream position](#get-current-stream-position)\n  * [Get current commit position](#get-current-commit-position)\n* [Catch-up subscriptions](#catch-up-subscriptions)\n  * [How to implement exactly-once event processing](#how-to-implement-exactly-once-event-processing)\n  * [Subscribe all events](#subscribe-all-events)\n  * [Subscribe stream events](#subscribe-stream-events)\n* [Persistent subscriptions](#persistent-subscriptions)\n  * [Create subscription](#create-subscription)\n  * [Read subscription](#read-subscription)\n  * [Create stream subscription](#create-stream-subscription)\n  * [Read stream subscription](#read-stream-subscription)\n* [Notes](#notes)\n  * [Regular expression filters](#regular-expression-filters)\n  * [New event objects](#new-event-objects)\n  * [Recorded event objects](#recorded-event-objects)\n* [Contributors](#contributors)\n  * [Install Poetry](#install-poetry)\n  * [Setup for PyCharm users](#setup-for-pycharm-users)\n  * [Setup from command line](#setup-from-command-line)\n  * [Project Makefile commands](#project-makefile-commands)\n<!-- TOC -->\n\n## Install package\n\nIt is recommended to install Python packages into a Python virtual environment.\n\n### From PyPI\n\nYou can use pip to install this package directly from\n[the Python Package Index](https://pypi.org/project/esdbclient/).\n\n    $ pip install esdbclient\n\n### With Poetry\n\nYou can use Poetry to add this package to your pyproject.toml and install it.\n\n    $ poetry add esdbclient\n\n## Server container\n\nThe EventStoreDB server can be run locally using the official Docker container image.\n\n### Run container\n\nUse Docker to run EventStoreDB using the official Docker container image on DockerHub.\n\nFor development, you can start a "secure" server locally on port 2113.\n\n    $ docker run -d --name my-eventstoredb -it -p 2113:2113 --env "HOME=/tmp" eventstore/eventstore:22.10.0-buster-slim --dev\n\nAlternatively, you can start an "insecure" server locally on port 2113.\n\n    $ docker run -d --name my-eventstoredb -it -p 2113:2113 eventstore/eventstore:22.10.0-buster-slim --insecure\n\nTo connect to the "insecure" local server using the client in this package, you just need\nto know the local hostname and the port number. To connect to the "secure" local\ndevelopment server, you will also need to know that the username is "admin" and\nthe password is "changeit". You will also need to get the SSL/TLS certificate from\nthe server. You can get the server certificate with the following command.\n\n    $ python -c "import ssl; print(get_server_certificate(addr=(\'localhost\', 2113)))"\n\n\n### Stop container\n\nTo stop and remove the `my-eventstoredb` container created above, use the following Docker commands.\n\n    $ docker stop my-eventstoredb\n\t$ docker rm my-eventstoredb\n\n\n## Client class\n\nThis client is implemented as the Python class `ESDBClient`.\n\n### Import class from package\n\nThe `ESDBClient` class can be imported from the `esdbclient` package.\n\n```python\nfrom esdbclient import ESDBClient\n```\n\n### Contruct client class\n\nThe `ESDBClient` class can be constructed with `host` and `port` arguments.\nThe `host` and `port` arguments indicate the hostname and port number of the\nEventStoreDB server.\n\nIf the EventStoreDB server is "secure", then also use the `server_cert`,\n`username` and `password` arguments.\n\nThe `host` argument is expected to be a Python `str`. The `port` argument is expected\nto be a Python `int`. The `server_cert` is expected to be a Python `str` containing\nthe PEM encoded SSL/TLS server certificate. Both `username` and `password` are expected\nto be a Python `str`.\n\nIn the example below, the constructor argument values are taken from the operating\nsystem environment, because the examples in this document are tested with both\na "secure" and an "insecure" server.\n\n```python\nimport os\n\nclient = ESDBClient(\n    host=os.getenv("ESDB_HOST"),\n    port=int(os.getenv("ESDB_PORT")),\n    server_cert=os.getenv("ESDB_SERVER_CERT"),\n    username=os.getenv("ESDB_USERNAME"),\n    password=os.getenv("ESDB_PASSWORD"),\n)\n```\n\n## Streams\n\nIn EventStoreDB, a "stream" is a sequence of recorded events that all have\nthe same "stream name". Each recorded event has a "stream position" in its stream,\nand a "commit position" in the database. The stream positions of the recorded events\nin a stream is a gapless sequence starting from zero. The commit positions of the\nrecorded events in the database form a sequence that is not gapless.\n\nThe methods `append_events()`, `read_stream_events()` and `read_all_events()` can\nbe used to record and read events in the database.\n\n### Append events\n\nThe `append_events()` method can be used to write a sequence of new events atomically\nto a "stream". Writing new events either creates a stream, or appends events to the end\nof a stream. This method is idempotent (see below).\n\nThis method can be used to record atomically all the new\nevents that are generated when executing a command in an application.\n\nThree arguments are required, `stream_name`, `expected_position`\nand `events`.\n\nThe `stream_name` argument is required, and is expected to be a Python\n`str` object that uniquely identifies the stream in the database.\n\nThe `expected_position` argument is required, is expected to be: `None`\nif events are being written to a new stream, and otherwise an Python `int`\nequal to the position in the stream of the last recorded event in the stream.\n\nThe `events` argument is required, and is expected to be a sequence of new\nevent objects to be appended to the named stream. The `NewEvent` class should\nbe used to construct new event objects (see below).\n\nThis method takes an optional `timeout` argument, which is a Python `float` that sets\na deadline for the completion of the gRPC operation.\n\nStreams are created by writing events. The correct value of the `expected_position`\nargument when writing the first event of a new stream is `None`. Please note, it is\nnot possible to somehow create an "empty" stream in EventStoreDB.\n\nThe stream positions of recorded events in a stream start from zero, and form a gapless\nsequence of integers. The stream position of the first recorded event in a stream is\n`0`. And so when appending the second new event to a stream that has one recorded event,\nthe correct value of the `expected_position` argument is `0`. Similarly, the stream\nposition of the second recorded event in a stream is `1`, and so when appending the\nthird new event to a stream that has two recorded events, the correct value of the\n`expected_position` argument is `1`. And so on... (There is a theoretical maximum\nnumber of recorded events that any stream can have, but I\'m not sure what it is;\nmaybe 9,223,372,036,854,775,807 because it is implemented as a `long` in C#?)\n\nIf there is a mismatch between the given value of the `expected_position` argument\nand the position of the last recorded event in a stream, then an `ExpectedPositionError`\nexception will be raised. This effectively accomplishes optimistic concurrency control.\n\nIf you wish to disable optimistic concurrency control when appending new events, you\ncan set the `expected_position` to a negative integer.\n\nIf you need to discover the current position of the last recorded event in a stream,\nyou can use the `get_stream_position()` method (see below).\n\nPlease note, the append events operation is atomic, so that either all\nor none of the given new events will be recorded. By design, it is only\npossible with EventStoreDB to atomically record new events in one stream.\n\nIn the example below, a new event is appended to a new stream.\n\n```python\nfrom uuid import UUID, uuid4\n\nfrom esdbclient import NewEvent\n\n# Construct new event object.\nevent1 = NewEvent(type=\'OrderCreated\', data=b\'data1\')\n\n# Define stream name.\nstream_name1 = str(uuid4())\n\n# Append list of events to new stream.\ncommit_position1 = client.append_events(\n    stream_name=stream_name1,\n    expected_position=None,\n    events=[event1],\n)\n```\n\nIn the example below, two subsequent events are appended to an existing\nstream.\n\n```python\nevent2 = NewEvent(type=\'OrderUpdated\', data=b\'data2\')\nevent3 = NewEvent(type=\'OrderDeleted\', data=b\'data3\')\n\ncommit_position2 = client.append_events(\n    stream_name=stream_name1,\n    expected_position=0,\n    events=[event2, event3],\n)\n```\n\nIf the operation is successful, this method returns an integer\nrepresenting the overall "commit position" as it was when the operation\nwas completed. Otherwise, an exception will be raised.\n\nA "commit position" is a monotonically increasing integer representing\nthe position of the recorded event in a "total order" of all recorded\nevents in the database across all streams. It is the actual position\nof the event record on disk, and there are usually large differences\nbetween successive commits. In consequence, the sequence of commit\npositions is not gapless. Indeed, there are usually large differences\nbetween the commit positions of successive recorded events.\n\nThe "commit position" returned by `append_events()` is that of the last\nrecorded event in the given batch of new events.\n\nThe "commit position" returned in this way can therefore be used to wait\nfor a downstream component to have processed all the events that were recorded.\n\nFor example, consider a user interface command that results in the recording\nof new events, and a query into an eventually consistent materialized\nview in a downstream component that is updated from these events. If the new\nevents have not yet been processed, the view would be stale. The "commit position"\ncan be used by the user interface to poll the downstream component until it has\nprocessed those new events, after which time the view will not be stale.\n\n\n### Append event\n\nThe `append_event()` method can be used to write a single new event to a stream.\n\nThree arguments are required, `stream_name`, `expected_position` and `event`.\n\nThis method works in the same way as `append_events()`, however `event` is expected\nto be a single `NewEvent`.\n\nThis method takes an optional `timeout` argument, which is a Python `float` that sets\na deadline for the completion of the gRPC operation.\n\nSince the handling of a command in your application may result in one or many\nnew events, and the results of handling a command should be recorded atomically,\nand the writing of new events generated by a command handler is usually a concern\nthat is factored out and used everywhere in a project, it is quite usual in a project\nto only use `append_events()` to record new events. For this reason, an example is\nnot provided here.\n\n\n### Idempotent append operations\n\nSometimes it may happen that a new event is successfully recorded and then somehow\nthe connection to the database gets interrupted before the successful call can return\nsuccessfully to the client. In case of an error when appending an event, it may be\ndesirable to retry appending the same event at the same position. If the event was\nin fact successfully recorded, it is convenient for the retry to return successfully\nwithout raising an error due to optimistic concurrency control (as described above).\n\nThe example below shows the `append_events()` method being called again with\n`event3` and `expected_position=2`. We can see that repeating the call to\n`append_events()` returns successfully.\n\n```python\n# Retry appending event3.\ncommit_position_retry = client.append_events(\n    stream_name=stream_name1,\n    expected_position=0,\n    events=[event2, event3],\n)\n```\n\nWe can see that the same commit position is returned as above.\n\n```python\nassert commit_position_retry == commit_position2\n```\n\nWe can also see the stream has been unchanged despite calling the append_events()\ntwice with the same arguments, by calling `read_stream_events()`. That is, there\nare still only three events in the stream.\n\n```python\nresponse = client.read_stream_events(\n    stream_name=stream_name1\n)\n\nevents = list(response)\nassert len(events) == 3\n```\n\nThis idempotent behaviour is activated because the `NewEvent` class has an `id`\nattribute that, by default, is assigned a new and unique version-4 UUID when an\ninstance of `NewEvent` is constructed. If events with the same `id` are appended\nat the same `expected_position`, the stream will be unchanged, the operation will\ncomplete successfully, and the same commit position will be returned to the caller.\n\n```python\nassert isinstance(event1.id, UUID)\nassert isinstance(event2.id, UUID)\nassert isinstance(event3.id, UUID)\n\nassert event1.id != event2.id\nassert event2.id != event3.id\n\nassert events[0].id == event1.id\nassert events[1].id == event2.id\nassert events[2].id == event3.id\n```\n\nIt is possible to set the `id` constructor argument of `NewEvent` when instantiating\nthe `NewEvent` class, but in the examples above we have been using the default\nbehaviour, which is that the `id` value is generated when the `NewEvent` class is\ninstantiated.\n\n\n### Read stream events\n\nThe `read_stream_events()` method can be used to read the recorded events of a stream.\n\nThis method can be used to retrieve all the recorded events for an aggregate before\nexecuting a command in an application.\n\nThis method has one required argument, `stream_name`, which is the name of\nthe stream from which to read events. By default, the recorded events in the\nstream are returned in the order they were recorded.\n\nThe method `read_stream_events()` also supports four optional arguments,\n`stream_position`, `backwards`, `limit`, and `timeout`.\n\nThe optional `stream_position` argument is an optional integer that can be used to\nindicate the position in the stream from which to start reading. This argument is\n`None` by default, which means the stream will be read either from the start of the\nstream (the default behaviour), or from the end of the stream if `backwards` is\n`True` (see below). When reading a stream from a specific position in the stream, the\nrecorded event at that position WILL be included, both when reading forwards\nfrom that position, and when reading backwards from that position.\n\nThe optional argument `backwards` is a boolean, by default `False`, which means the\nstream will be read forwards by default, so that events are returned in the\norder they were appended, If `backwards` is `True`, the stream will be read\nbackwards, so that events are returned in reverse order.\n\nThe optional argument `limit` is an integer which limits the number of events that will\nbe returned. The default value is `sys.maxint`.\n\nThe optional argument `timeout` is a Python `float` which sets a deadline for the completion of\nthe gRPC operation.\n\nThis method returns a Python iterable object that yields `RecordedEvent` objects.\nThese recorded event objects are instances of the `RecordedEvent` class (see below)\n\nThe example below shows how to read the recorded events of a stream\nforwards from the start of the stream to the end of the stream. The\nname of a stream is given when calling the method. In this example,\nthe iterable response object is converted into a Python `list`, which\ncontains all the recorded event objects that were read from the stream.\n\n```python\nresponse = client.read_stream_events(\n    stream_name=stream_name1\n)\n\nevents = list(response)\n```\n\nNow that we have a list of event objects, we can check we got the\nthree events that were appended to the stream, and that they are\nordered exactly as they were appended.\n\n```python\nassert len(events) == 3\n\nassert events[0].stream_name == stream_name1\nassert events[0].stream_position == 0\nassert events[0].type == event1.type\nassert events[0].data == event1.data\n\nassert events[1].stream_name == stream_name1\nassert events[1].stream_position == 1\nassert events[1].type == event2.type\nassert events[1].data == event2.data\n\nassert events[2].stream_name == stream_name1\nassert events[2].stream_position == 2\nassert events[2].type == event3.type\nassert events[2].data == event3.data\n```\n\nThe example below shows how to read recorded events in a stream forwards from\na specific stream position to the end of the stream.\n\n```python\nevents = list(\n    client.read_stream_events(\n        stream_name=stream_name1,\n        stream_position=1,\n    )\n)\n\nassert len(events) == 2\n\nassert events[0].stream_name == stream_name1\nassert events[0].stream_position == 1\nassert events[0].type == event2.type\nassert events[0].data == event2.data\n\nassert events[1].stream_name == stream_name1\nassert events[1].stream_position == 2\nassert events[1].type == event3.type\nassert events[1].data == event3.data\n```\n\nThe example below shows how to read the recorded events in a stream backwards from\nthe end of the stream to the start of the stream.\n\n```python\nevents = list(\n    client.read_stream_events(\n        stream_name=stream_name1,\n        backwards=True,\n    )\n)\n\nassert len(events) == 3\n\nassert events[0].stream_name == stream_name1\nassert events[0].stream_position == 2\nassert events[0].type == event3.type\nassert events[0].data == event3.data\n\nassert events[1].stream_name == stream_name1\nassert events[1].stream_position == 1\nassert events[1].type == event2.type\nassert events[1].data == event2.data\n```\n\nThe example below shows how to read a limited number (two) of the recorded events\nin a stream forwards from the start of the stream.\n\n```python\nevents = list(\n    client.read_stream_events(\n        stream_name=stream_name1,\n        limit=2,\n    )\n)\n\nassert len(events) == 2\n\nassert events[0].stream_name == stream_name1\nassert events[0].stream_position == 0\nassert events[0].type == event1.type\nassert events[0].data == event1.data\n\nassert events[1].stream_name == stream_name1\nassert events[1].stream_position == 1\nassert events[1].type == event2.type\nassert events[1].data == event2.data\n```\n\nThe example below shows how to read a limited number (one) of the recorded\nevents in a stream backwards from a given stream position.\n\n```python\nevents = list(\n    client.read_stream_events(\n        stream_name=stream_name1,\n        stream_position=2,\n        backwards=True,\n        limit=1,\n    )\n)\n\nassert len(events) == 1\n\nassert events[0].stream_name == stream_name1\nassert events[0].stream_position == 2\nassert events[0].type == event3.type\nassert events[0].data == event3.data\n```\n\n### Read all events\n\nThe method `read_all_events()` can be used to read all recorded events\nin the database in the order they were recorded. An iterable object of\nrecorded events is returned. This iterable object will stop when it has\nyielded the last recorded event.\n\nThis method supports six optional arguments, `commit_position`, `backwards`,\n`filter_exclude`, `filter_include`, `limit`, and `timeout`.\n\nThe optional argument `commit_position` is an optional integer that can be used to\nspecify the commit position from which to start reading. This argument is `None` by\ndefault, meaning that all the events will be read either from the start, or\nfrom the end if `backwards` is `True` (see below). Please note, if specified,\nthe specified position must be an actually existing commit position, because\nany other number will result in a server error (at least in EventStoreDB v21.10).\n\nThe optional argument `backwards` is a boolean which is by default `False` meaning the\nevents will be read forwards by default, so that events are returned in the\norder they were committed, If `backwards` is `True`, the events will be read\nbackwards, so that events are returned in reverse order.\n\nThe optional argument `filter_exclude` is a sequence of regular expressions that\nmatch the type strings of recorded events that should not be included. By default,\nthis argument will match "system events", so that they will not be included.\nThis argument is ignored if `filter_include` is set to a non-empty sequence.\n\nThe optional argument `filter_include` is a sequence of regular expressions\nthat match the type strings of recorded events that should be included. By\ndefault, this argument is an empty tuple. If this argument is set to a\nnon-empty sequence, the `filter_exclude` argument is ignored.\n\nThe optional argument `limit` is an integer which limits the number of events that will\nbe returned. The default value is `sys.maxint`.\n\nThe optional argument `timeout` is a Python `float` which sets a deadline for the completion of\nthe gRPC operation.\n\nThe filtering of events is done on the EventStoreDB server. The\n`limit` argument is applied on the server after filtering. See below for\nmore information about filter regular expressions.\n\nWhen reading forwards from a specific commit position, the event at the specified\nposition WILL be included. However, when reading backwards, the event at the\nspecified position will NOT be included. (This non-inclusive behaviour, of excluding\nthe specified commit position when reading all streams backwards, differs from the\nbehaviour when reading a stream backwards from a specific stream position, I\'m\nnot sure why.)\n\nThe example below shows how to read all events in the database in the\norder they were recorded.\n\n```python\nevents = list(client.read_all_events())\n\nassert len(events) >= 3\n```\n\nThe example below shows how to read all recorded events from a specific commit position.\n\n```python\nevents = list(\n    client.read_all_events(\n        commit_position=commit_position1\n    )\n)\n\nassert len(events) == 3\n\nassert events[0].stream_name == stream_name1\nassert events[0].stream_position == 0\nassert events[0].type == event1.type\nassert events[0].data == event1.data\n\nassert events[1].stream_name == stream_name1\nassert events[1].stream_position == 1\nassert events[1].type == event2.type\nassert events[1].data == event2.data\n\nassert events[2].stream_name == stream_name1\nassert events[2].stream_position == 2\nassert events[2].type == event3.type\nassert events[2].data == event3.data\n```\n\nThe example below shows how to read all recorded events in reverse order.\n\n```python\nevents = list(\n    client.read_all_events(\n        backwards=True\n    )\n)\n\nassert len(events) >= 3\n\nassert events[0].stream_name == stream_name1\nassert events[0].stream_position == 2\nassert events[0].type == event3.type\nassert events[0].data == event3.data\n\nassert events[1].stream_name == stream_name1\nassert events[1].stream_position == 1\nassert events[1].type == event2.type\nassert events[1].data == event2.data\n\nassert events[2].stream_name == stream_name1\nassert events[2].stream_position == 0\nassert events[2].type == event1.type\nassert events[2].data == event1.data\n```\n\nThe example below shows how to read a limited number (one) of the recorded events\nin the database forwards from a specific commit position. Please note, when reading\nall events forwards from a specific commit position, the event at the specified\nposition WILL be included.\n\n\n```python\nevents = list(\n    client.read_all_events(\n        commit_position=commit_position1,\n        limit=1,\n    )\n)\n\nassert len(events) == 1\n\nassert events[0].stream_name == stream_name1\nassert events[0].stream_position == 0\nassert events[0].type == event1.type\nassert events[0].data == event1.data\n\nassert events[0].commit_position == commit_position1\n```\n\nThe example below shows how to read a limited number (one) of the recorded events\nin the database backwards from the end. This gives the last recorded event.\n\n```python\nevents = list(\n    client.read_all_events(\n        backwards=True,\n        limit=1,\n    )\n)\n\nassert len(events) == 1\n\nassert events[0].stream_name == stream_name1\nassert events[0].stream_position == 2\nassert events[0].type == event3.type\nassert events[0].data == event3.data\n```\n\nThe example below shows how to read a limited number (one) of the recorded events\nin the database backwards from a specific commit position. Please note, when reading\nall events backwards from a specific commit position, the event at the specified\nposition WILL NOT be included.\n\n```python\nevents = list(\n    client.read_all_events(\n        commit_position=commit_position2,\n        backwards=True,\n        limit=1,\n    )\n)\n\nassert len(events) == 1\n\nassert events[0].commit_position < commit_position2\n```\n\n\n### Get current stream position\n\nThe `get_stream_position()` method can be used to\nget the "stream position" of the last recorded event in a stream.\n\nThis method has a `stream_name` argument, which is required.\n\nThis method also takes an optional `timeout` argument, that\nis expected to be a Python `float`, which sets a deadline\nfor the completion of the gRPC operation.\n\nThe sequence of positions in a stream is gapless. It is zero-based,\nso that a stream with one recorded event has a current stream\nposition of `0`. The current stream position is `1` when a stream has\ntwo events, and it is `2` when there are events, and so on.\n\nIn the example below, the current stream position is obtained of the\nstream to which events were appended in the examples above.\nBecause the sequence of stream positions is zero-based, and because\nthree events were appended, so the current stream position is `2`.\n\n```python\nstream_position = client.get_stream_position(\n    stream_name=stream_name1\n)\n\nassert stream_position == 2\n```\n\nIf a stream does not exist, the returned stream position value is `None`,\nwhich matches the required expected position when appending the first event\nof a new stream (see above).\n\n```python\nstream_position = client.get_stream_position(\n    stream_name=str(uuid4())\n)\n\nassert stream_position == None\n```\n\nThis method takes an optional argument `timeout` which is a Python `float` that sets\na deadline for the completion of the gRPC operation.\n\n\n### Get current commit position\n\nThe method `get_commit_position()` can be used to get the current\ncommit position of the database.\n\n```python\ncommit_position = client.get_commit_position()\n```\n\nThis method takes an optional argument `timeout` which is a Python `float` that sets\na deadline for the completion of the gRPC operation.\n\nThis method can be useful to measure progress of a downstream component\nthat is processing all recorded events, by comparing the current commit\nposition with the recorded commit position of the last successfully processed\nevent in a downstream component.\n\nThe value of the `commit_position` argument when reading events either by using\nthe `read_all_events()` method or by using a catch-up subscription would usually\nbe determined by the recorded commit position of the last successfully processed\nevent in a downstream component.\n\n\n## Catch-up subscriptions\n\nA "catch-up subscription" can be used to receive already recorded events, but\nit will also return events that are recorded after the subscription was started.\n\nThe method `subscribe_stream_events()` starts a catch-up subscription to receive\nevents from a specific stream. The method `subscribe_all_events()` starts a catch-up\nsubscription to receive all events in the database.\n\nCatch-up subscriptions are simply a streaming gRPC call which is\nkept open by the server, with newly recorded events sent to the client\nas the client iterates over the subscription.\n\nMany catch-up subscriptions can be created, concurrently or successively, and all\nwill receive all the recorded events they have been requested to receive.\n\nReceived recorded events are instances of the `RecordedEvent` class (see below).\nRecorded event objects have a commit position, amonst other attributes.\n\n### How to implement exactly-once event processing\n\nThe commit positions of recorded events that are received and processed by a\ndownstream component are usefully recorded by the downstream component so that\nthe commit position of last processed event can be determined.\n\nThe last recorded commit position can be used to specify the commit position from which\nto subscribe when processing is resumed. Since this commit position will represent the\nposition of the last successfully processed event in a downstream component, so it\nwill be usual to want the next event after this position, because that is the next\nevent that has not yet been processed. For this reason, when subscribing for events\nfrom a specific commit position using a catch-up subscription in EventStoreDB, the\nrecorded event at the specified commit position will NOT be included in the sequence\nof recorded events that are received.\n\nTo accomplish "exactly-once" processing of recorded events in a downstream\ncomponent when using a catch-up subscription, the commit position of a recorded\nevent should be recorded atomically and uniquely along with the result of processing\nrecorded events, for example in the same database as materialised views when\nimplementing eventually-consistent CQRS, or in the same database as a downstream\nanalytics or reporting or archiving application. By recording the commit position\nof recorded events atomically with the new state that results from processing\nrecorded events, "dual writing" in the consumption of recorded events can be\navoided. By also recording the commit position uniquely, the new state cannot be\nrecorded twice, and hence the recorded state of the downstream component will be\nupdated only once for any recorded event. By using the greatest recorded commit\nposition to resume a catch-up subscription, all recorded events will eventually\nbe processed. The combination of the "at-most-once" condition and the "at-least-once"\ncondition gives the "exactly-once" condition.\n\nThe danger with "dual writing" in the consumption of recorded events is that if a\nrecorded event is successfully processed and new state recorded atomically in one\ntransaction with the commit position recorded in a separate transaction, one may\nhappen and not the other. If the new state is recorded but the position is lost,\nand then the processing is stopped and resumed, the recorded event may be processed\ntwice. On the other hand, if the commit position is recorded but the new state is\nlost, the recorded event may effectively not be processed at all. By either\nprocessing an event more than once, or by failing to process an event, the recorded\nstate of the downstream component might be inaccurate, or possibly inconsistent, and\nperhaps catastrophically so. Such consequences may or may not matter in your situation.\nBut sometimes inconsistencies may halt processing until the issue is resolved. You can\navoid "dual writing" in the consumption of events by atomically recording the commit\nposition of a recorded event along with the new state that results from processing that\nevent in the same atomic transaction. By making the recording of the commit positions\nunique, so that transactions will be rolled back when there is a conflict, you will\nprevent the results of any duplicate processing of a recorded event being committed.\n\nRecorded events received from a catch-up subscription cannot be acknowledged back\nto the EventStoreDB server. Acknowledging events, however, is an aspect of "persistent\nsubscriptions" (see below). Hoping to rely on acknowledging events to an upstream\ncomponent is an example of dual writing.\n\n### Subscribe all events\n\nThe`subscribe_all_events()` method can be used to start a "catch-up" subscription\nthat can return all events in the database.\n\nThis method can be used by a downstream component\nto process recorded events with exactly-once semantics.\n\nThis method takes an optional `commit_position` argument, which can be\nused to specify a commit position from which to subscribe for\nrecorded events. The default value is `None`, which means\nthe subscription will operate from the first recorded event\nin the database. If a commit position is given, it must match\nan actually existing commit position in the database. The events\nthat are obtained will not include the event recorded at that commit\nposition.\n\nThis method also takes three other optional arguments, `filter_exclude`,\n`filter_include`, and `timeout`.\n\nThe argument `filter_exclude` is a sequence of regular expressions matching\nthe type strings of recorded events that should be excluded. By default,\nthis argument will match "system events", so that they will not be included.\nThis argument is ignored if `filter_include` is set to a non-empty sequence.\n\nThe argument `filter_include` is a sequence of regular expressions\nmatching the type strings of recorded events that should be included. By\ndefault, this argument is an empty tuple. If this argument is set to a\nnon-empty sequence, the `filter_exclude` argument is ignored.\n\nPlease note, the filtering happens on the EventStoreDB server, and the\n`limit` argument is applied on the server after filtering. See below for\nmore information about filter regular expressions.\n\nThe argument `timeout` is a Python `float` which sets a deadline for the completion of\nthe gRPC operation. This probably isn\'t very useful, but is included for\ncompleteness and consistency with the other methods.\n\nThis method returns a Python iterator that yields recorded events, including events\nthat are recorded after the subscription was created. Iterating over this object will\ntherefore not stop, unless the connection to the database is lost. The connection will\nbe closed when the iterator object is deleted from memory, which will happen when the\niterator object goes out of scope or is explicitly deleted (see below). The connection\nmay also be closed by the server.\n\nThe subscription object can be used directly, but it might be used within a threaded\nloop dedicated to receiving events that can be stopped in a controlled way, with\nrecorded events put on a queue for processing in a different thread. This package\ndoesn\'t provide such a threaded or queuing object class. Just make sure to reconstruct\nthe subscription (and the queue) using the last recorded commit position when resuming\nthe subscription after an error, to be sure all events are processed once.\n\nThe example below shows how to subscribe to receive all recorded\nevents from the start, and then resuming from a specific commit position.\nThree already-recorded events are received, and then three new events are\nrecorded, which are then received via the subscription.\n\n```python\n\n# Append an event to a new stream.\nstream_name2 = str(uuid4())\nevent4 = NewEvent(type=\'OrderCreated\', data=b\'data4\')\nclient.append_events(\n    stream_name=stream_name2,\n    expected_position=None,\n    events=[event4],\n)\n\n# Subscribe from the first recorded event in the database.\nsubscription = client.subscribe_all_events()\nreceived_events = []\n\n# Process events received from the catch-up subscription.\nfor event in subscription:\n    last_commit_position = event.commit_position\n    received_events.append(event)\n    if event.id == event4.id:\n        break\n\nassert received_events[-4].id == event1.id\nassert received_events[-3].id == event2.id\nassert received_events[-2].id == event3.id\nassert received_events[-1].id == event4.id\n\n# Append subsequent events to the stream.\nevent5 = NewEvent(type=\'OrderUpdated\', data=b\'data5\')\nclient.append_events(\n    stream_name=stream_name2,\n    expected_position=0,\n    events=[event5],\n)\n\n# Receive subsequent events from the subscription.\nfor event in subscription:\n    last_commit_position = event.commit_position\n    received_events.append(event)\n    if event.id == event5.id:\n        break\n\n\nassert received_events[-5].id == event1.id\nassert received_events[-4].id == event2.id\nassert received_events[-3].id == event3.id\nassert received_events[-2].id == event4.id\nassert received_events[-1].id == event5.id\n\n\n# Append more events to the stream.\nevent6 = NewEvent(type=\'OrderDeleted\', data=b\'data6\')\nclient.append_events(\n    stream_name=stream_name2,\n    expected_position=1,\n    events=[event6],\n)\n\n\n# Resume subscribing from the last commit position.\nsubscription = client.subscribe_all_events(\n    commit_position=last_commit_position\n)\n\n\n# Catch up by receiving the new event from the subscription.\nfor event in subscription:\n    received_events.append(event)\n    if event.id == event6.id:\n        break\n\nassert received_events[-6].id == event1.id\nassert received_events[-5].id == event2.id\nassert received_events[-4].id == event3.id\nassert received_events[-3].id == event4.id\nassert received_events[-2].id == event5.id\nassert received_events[-1].id == event6.id\n\n\n# Append three more events to a new stream.\nstream_name3 = str(uuid4())\nevent7 = NewEvent(type=\'OrderCreated\', data=b\'data7\')\nevent8 = NewEvent(type=\'OrderUpdated\', data=b\'data8\')\nevent9 = NewEvent(type=\'OrderDeleted\', data=b\'data9\')\n\nclient.append_events(\n    stream_name=stream_name3,\n    expected_position=None,\n    events=[event7, event8, event9],\n)\n\n# Receive the three new events from the resumed subscription.\nfor event in subscription:\n    received_events.append(event)\n    if event.id == event9.id:\n        break\n\nassert received_events[-9].id == event1.id\nassert received_events[-8].id == event2.id\nassert received_events[-7].id == event3.id\nassert received_events[-6].id == event4.id\nassert received_events[-5].id == event5.id\nassert received_events[-4].id == event6.id\nassert received_events[-3].id == event7.id\nassert received_events[-2].id == event8.id\nassert received_events[-1].id == event9.id\n```\n\nThe catch-up subscription gRPC operation is ended as soon as the subscription object\ngoes out of scope or is explicitly deleted from memory.\n\n```python\n# End the subscription.\ndel subscription\n```\n\n### Subscribe stream events\n\nThe`subscribe_stream_events()` method can be used to start a "catch-up" subscription\nthat can return all events in a stream.\n\nThis method takes a `stream_name` argument, which specifies the name of the stream\nfrom which recorded events will be received.\n\nThis method takes an optional `stream_position` argument, which specifies a\nstream position in the stream from which recorded events will be received. The\nevent at the specified stream position will not be included.\n\nThis method takes an optional `timeout` argument, which is a Python `float` that sets\na deadline for the completion of the gRPC operation.\n\nThe example below shows how to start a catch-up subscription to a stream.\n\n```python\n\n# Subscribe to events from stream2, from the start.\nsubscription = client.subscribe_stream_events(stream_name=stream_name2)\n\n# Read from the subscription.\nevents = []\nfor event in subscription:\n    events.append(event)\n    if event.id == event6.id:\n        break\n\n# Check we got events only from stream2.\nassert len(events) == 3\nevents[0].stream_name == stream_name2\nevents[1].stream_name == stream_name2\nevents[2].stream_name == stream_name2\n\n# Append another event to stream1.\nevent10 = NewEvent(type="OrderUndeleted", data=b\'data10\')\nclient.append_events(\n    stream_name=stream_name1,\n    expected_position=2,\n    events=[event10],\n)\n\n# Append another event to stream2.\nevent11 = NewEvent(type="OrderUndeleted", data=b\'data11\')\nclient.append_events(\n    stream_name=stream_name2,\n    expected_position=2,\n    events=[event11]\n)\n\n# Continue reading from the subscription.\nfor event in subscription:\n    events.append(event)\n    if event.id == event11.id:\n        break\n\n# Check we got events only from stream2.\nassert len(events) == 4\nevents[0].stream_name == stream_name2\nevents[1].stream_name == stream_name2\nevents[2].stream_name == stream_name2\nevents[3].stream_name == stream_name2\n```\n\nThe example below shows how to start a catch-up subscription to a stream from a\nspecific stream position.\n\n```python\n\n# Subscribe to events from stream2, from the start.\nsubscription = client.subscribe_stream_events(\n    stream_name=stream_name2,\n    stream_position=1,\n)\n\n# Read event from the subscription.\nevents = []\nfor event in subscription:\n    events.append(event)\n    if event.id == event11.id:\n        break\n\n# Check we got events only after position 1.\nassert len(events) == 2\nevents[0].id == event6.id\nevents[0].stream_position == 2\nevents[0].stream_name == stream_name2\nevents[1].id == event11.id\nevents[1].stream_position == 3\nevents[1].stream_name == stream_name2\n```\n\n## Persistent subscriptions\n\n### Create subscription\n\nThe method `create_subscription()` can be used to create a\n"persistent subscription" to EventStoreDB.\n\nThis method takes a required `group_name` argument, which is the\nname of a "group" of consumers of the subscription.\n\nThis method takes an optional `from_end` argument, which can be\nused to specify that the group of consumers of the subscription should\nonly receive events that were recorded after the subscription was created.\n\nThis method takes an optional `commit_position` argument, which can be\nused to specify a commit position from which the group of consumers of\nthe subscription should receive events. Please note, the recorded event\nat the specified commit position MAY be included in the recorded events\nreceived by the group of consumers.\n\nIf neither `from_end` or `commit_position` are specified, the group of consumers\nof the subscription will receive all recorded events.\n\nThis method also takes option `filter_exclude`, `filter_include`\narguments, which work in the same way as they do in the `subscribe_all_events()`\nmethod.\n\nThis method also takes an optional `timeout` argument, that\nis expected to be a Python `float`, which sets a deadline\nfor the completion of the gRPC operation.\n\nThe method `create_subscription()` does not return a value, because\nrecorded events are obtained by the group of consumers of the subscription\nusing the `read_subscription()` method.\n\n*Please note, in this version of this client the "consumer strategy" is\nset to "DispatchToSingle". Support for choosing other consumer strategies\nsupported by EventStoreDB will in future be supported in this client.*\n\nIn the example below, a persistent subscription is created.\n\n```python\n# Create a persistent subscription.\ngroup_name = f"group-{uuid4()}"\nclient.create_subscription(group_name=group_name)\n```\n\n### Read subscription\n\nThe method `read_subscription()` can be used by a group of consumers to receive\nrecorded events from a persistent subscription created using `create_subscription`.\n\nThis method takes a required `group_name` argument, which is\nthe name of a "group" of consumers of the subscription specified\nwhen `create_subscription()` was called.\n\nThis method also takes an optional `timeout` argument, that\nis expected to be a Python `float`, which sets a deadline\nfor the completion of the gRPC operation.\n\nThis method returns a 2-tuple: a "read request" object and a "read response" object.\n\n```python\nread_req, read_resp = client.read_subscription(group_name=group_name)\n```\n\nThe "read response" object is an iterator that yields recorded events from\nthe specified commit position.\n\nThe "read request" object has an `ack()` method that can be used by a consumer\nin a group to acknowledge to the server that it has received and successfully\nprocessed a recorded event. This will prevent that recorded event being received\nby another consumer in the same group. The `ack()` method takes an `event_id`\nargument, which is the ID of the recorded event that has been received.\n\nThe example below iterates over the "read response" object, and calls `ack()`\non the "read response" object. The for loop breaks when we have received\nthe last event, so that we can continue with the examples below.\n\n```python\nevents = []\nfor event in read_resp:\n    events.append(event)\n\n    # Acknowledge the received event.\n    read_req.ack(event_id=event.id)\n\n    # Break when the last event has been received.\n    if event.id == event11.id:\n        break\n```\n\nThe received events are the events we appended above.\n\n```python\nassert events[-11].id == event1.id\nassert events[-10].id == event2.id\nassert events[-9].id == event3.id\nassert events[-8].id == event4.id\nassert events[-7].id == event5.id\nassert events[-6].id == event6.id\nassert events[-5].id == event7.id\nassert events[-4].id == event8.id\nassert events[-3].id == event9.id\nassert events[-2].id == event10.id\nassert events[-1].id == event11.id\n```\n\nThe "read request" object also has an `nack()` method that can be used by a consumer\nin a group to acknowledge to the server that it has failed successfully to\nprocess a recorded event. This will allow that recorded event to be received\nby this or another consumer in the same group.\n\nIt might be more useful to encapsulate the request and response objects and to iterate\nover the "read response" in a separate thread, to call back to a handler function when\na recorded event is received, and call `ack()` if the handler does not raise an\nexception, and to call `nack()` if an exception is raised. The example below shows how\nthis might be done.\n\n```python\nfrom threading import Thread\n\n\nclass SubscriptionReader:\n    def __init__(self, client, group_name, callback):\n        self.client = client\n        self.group_name = group_name\n        self.callback = callback\n        self.thread = Thread(target=self.read_subscription, daemon=True)\n        self.error = None\n\n    def start(self):\n        self.thread.start()\n\n    def join(self):\n        self.thread.join()\n\n    def read_subscription(self):\n        req, resp = self.client.read_subscription(group_name=self.group_name)\n        for event in resp:\n            try:\n                self.callback(event)\n            except Exception as e:\n                # req.nack(event.id)  # not yet implemented....\n                self.error = e\n                break\n            else:\n                req.ack(event.id)\n\n\n# Create another persistent subscription.\ngroup_name = f"group-{uuid4()}"\nclient.create_subscription(group_name=group_name)\n\nevents = []\n\ndef handle_event(event):\n    events.append(event)\n    if event.id == event11.id:\n        raise Exception()\n\n\nreader = SubscriptionReader(\n    client=client,\n    group_name=group_name,\n    callback=handle_event\n)\n\nreader.start()\nreader.join()\n\nassert events[-1].id == event11.id\n```\n\nPlease note, when processing events in a downstream component, the commit position of\nthe last successfully processed event is usefully recorded by the downstream component\nso that the commit position can be determined by the downstream component from its own\nrecorded when it is restarted. This commit position can be used to specify the commit\nposition from which to subscribe. Since this commit position represents the position of\nthe last successfully processed event in a downstream component, so it will be usual to\nwant to read from the next event after this position, because that is the next event\nthat needs to be processed. However, when subscribing for events using a persistent\nsubscription in EventStoreDB, the event at the specified commit position MAY be returned\nas the first event in the received sequence of recorded events, and so it may\nbe necessary to check the commit position of the received events and to discard\nany  recorded event object that has a commit position equal to the commit position\nspecified in the request.\n\nWhilst there are some advantages of persistent subscriptions, in particular the\nprocessing of recorded events by a group of consumers, by tracking in the\nupstream server the position in the commit sequence of events that have been processed,\nthere is a danger of "dual writing" in the consumption of events. Reliability\nin processing of recorded events by a group of consumers will rely instead on\nidempotent handling of duplicate messages, and resilience to out-of-order delivery.\n\n### Create stream subscription\n\nThe `create_stream_subscription()` method can be used to create a persistent\nsubscription for a stream.\n\nThis method has two required arguments, `group_name` and `stream_name`. The\n`group_name` argument names the group of consumers that will receive events\nfrom this subscription. The `stream_name` argument specifies which stream\nthe subscription will follow. The values of both these arguments are expected\nto be Python `str` objects.\n\nThis method has an optional `stream_position` argument, which specifies a\nstream position from which to subscribe. The recorded event at this stream\nposition will be received when reading the subscription.\n\nThis method has an optional `from_end` argument, which is a Python `bool`.\nBy default, the value of this argument is False. If this argument is set\nto a True value, reading from the subscription will receive only events\nrecorded after the subscription was created. That is, it is not inclusive\nof the current stream position.\n\nThis method also takes an optional `timeout` argument, that\nis expected to be a Python `float`, which sets a deadline\nfor the completion of the gRPC operation.\n\nThis method does not return a value. Events can be received by iterating\nover the value returned by calling `read_stream_subscription()` (see below).\n\nThe example below creates a persistent stream subscription from the start of the stream.\n\n```python\n# Create a persistent stream subscription from start of the stream.\ngroup_name1 = f"group-{uuid4()}"\nclient.create_stream_subscription(\n    group_name=group_name1,\n    stream_name=stream_name1,\n)\n```\n\nThe example below creates a persistent stream subscription from a stream position.\n\n```python\n# Create a persistent stream subscription from a stream position.\ngroup_name2 = f"group-{uuid4()}"\nclient.create_stream_subscription(\n    group_name=group_name2,\n    stream_name=stream_name2,\n    stream_position=2\n)\n```\n\nThe example below creates a persistent stream subscription from the end of the stream.\n\n```python\n# Create a persistent stream subscription from end of the stream.\ngroup_name3 = f"group-{uuid4()}"\nclient.create_stream_subscription(\n    group_name=group_name3,\n    stream_name=stream_name3,\n    from_end=True\n)\n```\n\n### Read stream subscription\n\nThe `read_stream_subscription()` method can be used to create a persistent\nsubscription for a stream.\n\nThis method has two required arguments, `group_name` and `stream_name`, which\nshould match the values of arguments used when calling `create_stream_subscription()`.\n\nThis method also takes an optional `timeout` argument, that\nis expected to be a Python `float`, which sets a deadline\nfor the completion of the gRPC operation.\n\nJust like `read_subscription`, this method returns a 2-tuple: a "read request" object\nand a "read response" object.\n\n```python\nread_req, read_resp = client.read_stream_subscription(\n    group_name=group_name1,\n    stream_name=stream_name1,\n)\n```\n\nThe example below iterates over the "read response" object, and calls `ack()`\non the "read response" object. The for loop breaks when we have received\nthe last event in the stream, so that we can finish the examples in this\ndocumentation.\n\n```python\nevents = []\nfor event in read_resp:\n    events.append(event)\n\n    # Acknowledge the received event.\n    read_req.ack(event_id=event.id)\n\n    # Break when the last event has been received.\n    if event.id == event10.id:\n        break\n```\n\nWe can check we received all the events that were appended to `stream_name1`\nin the examples above.\n\n```python\nassert len(events) == 4\nassert events[0].stream_name == stream_name1\nassert events[0].id == event1.id\nassert events[1].stream_name == stream_name1\nassert events[1].id == event2.id\nassert events[2].stream_name == stream_name1\nassert events[2].id == event3.id\nassert events[3].stream_name == stream_name1\nassert events[3].id == event10.id\n```\n\n\n## Notes\n\n### Regular expression filters\n\nThe filter arguments in `read_all_events()`, `subscribe_all_events()`,\n`create_subscription()` and `commit_position()` are applied to the `type`\nattribute of recorded events.\n\nThe default value of the `filter_exclude` arguments is designed to exclude\nEventStoreDB "system" and "persistence subscription config" events, which\notherwise would be included. System events generated by EventStoreDB all\nhave `type` strings that start with the `$` sign. Persistence subscription\nevents generated when manipulating persistence subscriptions all have `type`\nstrings that start with `PersistentConfig`.\n\nFor example, to match the type of EventStoreDB system events, use the regular\nexpression `r\'\\$.+\'`. Please note, the constant `ESDB_SYSTEM_EVENTS_REGEX` is\nset to `r\'\\$.+\'`. You can import this value\n(`from esdbclient import ESDB_SYSTEM_EVENTS_REGEX`) and use\nit when building longer sequences of regular expressions.\n\nSimilarly, to match the type of EventStoreDB persistence subscription events, use the\nregular expression `r\'PersistentConfig\\d+\'`. The constant `ESDB_PERSISTENT_CONFIG_EVENTS_REGEX`\nis set to `r\'PersistentConfig\\d+\'`. You can also import this value\n(`from esdbclient import ESDB_PERSISTENT_CONFIG_EVENTS_REGEX`) and use it when building\nlonger sequences of regular expressions.\n\nThe constant `DEFAULT_EXCLUDE_FILTER` is a sequence of regular expressions that match\nthe events that EventStoreDB generates internally, events that are extraneous to those\nwhich you append using the `append_events()` method.\n\nFor example, to exclude system events and persistence subscription configuration events,\nand snapshots, you might use the sequence `DEFAULT_EXCLUDE_FILTER + [\'.*Snapshot\']` as\nthe value of the `filter_exclude` argument when calling `read_all_events()`,\n`subscribe_all_events()`, `create_subscription()` or `get_commit_position()`.\n\n### New event objects\n\nThe `NewEvent` class is used when appending events.\n\nThe required argument `type` is a Python `str` object, used to indicate the type of\nthe event that will be recorded.\n\nThe required argument `data` is a Python `bytes` object, used to indicate the data of\nthe event that will be recorded.\n\nThe optional argument `metadata` is a Python `bytes` object, used to indicate any\nmetadata of the event that will be recorded. The default value is an empty `bytes`\nobject.\n\nThe optional argument `content_type` is a Python `str` object, used to indicate the\ntype of the data that will be recorded for this event. The default value is\n`application/json`, which indicates that the `data` was serialised using JSON.\nAn alternative value for this argument is `application/octet-stream`.\n\nThe optional argument `id` is a Python `UUID` object, used to specify the unique ID\nof the event that will be recorded. This value will default to a new version-4 UUID.\n\n```python\nnew_event1 = NewEvent(\n    type=\'OrderCreated\',\n    data=b\'{"name": "Greg"}\',\n)\nassert new_event1.type == \'OrderCreated\'\nassert new_event1.data == b\'{"name": "Greg"}\'\nassert new_event1.metadata == b\'\'\nassert new_event1.content_type == \'application/json\'\nassert isinstance(new_event1.id, UUID)\n\nevent_id = uuid4()\nnew_event2 = NewEvent(\n    type=\'ImageCreated\',\n    data=b\'01010101010101\',\n    metadata=b\'{"a": 1}\',\n    content_type=\'application/octet-stream\',\n    id=event_id,\n)\nassert new_event2.type == \'ImageCreated\'\nassert new_event2.data == b\'01010101010101\'\nassert new_event2.metadata == b\'{"a": 1}\'\nassert new_event2.content_type == \'application/octet-stream\'\nassert new_event2.id == event_id\n```\n\n### Recorded event objects\n\nThe `RecordedEvent` class is used when reading events.\n\nThe attribute `type` is a Python `str` object, used to indicate the type of event\nthat was recorded.\n\nThe attribute `data` is a Python `bytes` object, used to indicate the data of the\nevent that was recorded.\n\nThe attribute `metadata` is a Python `bytes` object, used to indicate the metadata of\nthe event that was recorded.\n\nThe attribute `content_type` is a Python `str` object, used to indicate the type of\ndata that was recorded for this event (usually `application/json` to indicate that\nthis data can be parsed as JSON, but alternatively `application/octet-stream` to\nindicate that it is something else).\n\nThe attribute `id` is a Python `UUID` object, used to indicate the unique ID of the\nevent that was recorded. Please note, when recorded events are returned from a call\nto `read_stream_events()` in EventStoreDB v21.10, the commit position is not actually\nset in the response. This attribute is typed as an optional value (`Optional[UUID]`),\nand in the case of using EventStoreDB v21.10 the value of this attribute will be `None`\nwhen reading recorded events from a stream. Recorded events will however have this\nvalues set when reading recorded events from `read_all_events()` and from both\ncatch-up and persistent subscriptions.\n\nThe attribute `stream_name` is a Python `str` object, used to indicate the name of the\nstream in which the event was recorded.\n\nThe attribute `stream_position` is a Python `int`, used to indicate the position in the\nstream at which the event was recorded.\n\nThe attribute `commit_position` is a Python `int`, used to indicate the commit position\nat which the event was recorded.\n\n```python\nfrom esdbclient.events import RecordedEvent\n\nrecorded_event = RecordedEvent(\n    type=\'OrderCreated\',\n    data=b\'{}\',\n    metadata=b\'\',\n    content_type=\'application/json\',\n    id=uuid4(),\n    stream_name=\'stream1\',\n    stream_position=0,\n    commit_position=512,\n)\n```\n\n## Contributors\n\n### Install Poetry\n\nThe first thing is to check you have Poetry installed.\n\n    $ poetry --version\n\nIf you don\'t, then please [install Poetry](https://python-poetry.org/docs/#installing-with-the-official-installer).\n\n    $ curl -sSL https://install.python-poetry.org | python3 -\n\nIt will help to make sure Poetry\'s bin directory is in your `PATH` environment variable.\n\nBut in any case, make sure you know the path to the `poetry` executable. The Poetry\ninstaller tells you where it has been installed, and how to configure your shell.\n\nPlease refer to the [Poetry docs](https://python-poetry.org/docs/) for guidance on\nusing Poetry.\n\n### Setup for PyCharm users\n\nYou can easily obtain the project files using PyCharm (menu "Git > Clone...").\nPyCharm will then usually prompt you to open the project.\n\nOpen the project in a new window. PyCharm will then usually prompt you to create\na new virtual environment.\n\nCreate a new Poetry virtual environment for the project. If PyCharm doesn\'t already\nknow where your `poetry` executable is, then set the path to your `poetry` executable\nin the "New Poetry Environment" form input field labelled "Poetry executable". In the\n"New Poetry Environment" form, you will also have the opportunity to select which\nPython executable will be used by the virtual environment.\n\nPyCharm will then create a new Poetry virtual environment for your project, using\na particular version of Python, and also install into this virtual environment the\nproject\'s package dependencies according to the project\'s `poetry.lock` file.\n\nYou can add different Poetry environments for different Python versions, and switch\nbetween them using the "Python Interpreter" settings of PyCharm. If you want to use\na version of Python that isn\'t installed, either use your favourite package manager,\nor install Python by downloading an installer for recent versions of Python directly\nfrom the [Python website](https://www.python.org/downloads/).\n\nOnce project dependencies have been installed, you should be able to run tests\nfrom within PyCharm (right-click on the `tests` folder and select the \'Run\' option).\n\nBecause of a conflict between pytest and PyCharm\'s debugger and the coverage tool,\nyou may need to add ``--no-cov`` as an option to the test runner template. Alternatively,\njust use the Python Standard Library\'s ``unittest`` module.\n\nYou should also be able to open a terminal window in PyCharm, and run the project\'s\nMakefile commands from the command line (see below).\n\n### Setup from command line\n\nObtain the project files, using Git or suitable alternative.\n\nIn a terminal application, change your current working directory\nto the root folder of the project files. There should be a Makefile\nin this folder.\n\nUse the Makefile to create a new Poetry virtual environment for the\nproject and install the project\'s package dependencies into it,\nusing the following command.\n\n    $ make install-packages\n\nIt\'s also possible to also install the project in \'editable mode\'.\n\n    $ make install\n\nPlease note, if you create the virtual environment in this way, and then try to\nopen the project in PyCharm and configure the project to use this virtual\nenvironment as an "Existing Poetry Environment", PyCharm sometimes has some\nissues (don\'t know why) which might be problematic. If you encounter such\nissues, you can resolve these issues by deleting the virtual environment\nand creating the Poetry virtual environment using PyCharm (see above).\n\n### Project Makefile commands\n\nYou can start EventStoreDB using the following command.\n\n    $ make start-eventstoredb\n\nYou can run tests using the following command (needs EventStoreDB to be running).\n\n    $ make test\n\nYou can stop EventStoreDB using the following command.\n\n    $ make stop-eventstoredb\n\nYou can check the formatting of the code using the following command.\n\n    $ make lint\n\nYou can reformat the code using the following command.\n\n    $ make fmt\n\nTests belong in `./tests`. Code-under-test belongs in `./esdbclient`.\n\nEdit package dependencies in `pyproject.toml`. Update installed packages (and the\n`poetry.lock` file) using the following command.\n\n    $ make update-packages\n',
    'author': 'John Bywater',
    'author_email': 'john.bywater@appropriatesoftware.net',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/pyeventsourcing/esdbclient',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
