"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cognito_1 = require("@aws-cdk/aws-cognito");
const aws_dynamodb_1 = require("@aws-cdk/aws-dynamodb");
const core_1 = require("@aws-cdk/core");
const path_1 = require("path");
const lib_1 = require("../lib");
const app = new core_1.App();
const stack = new core_1.Stack(app, 'aws-appsync-integ');
const userPool = new aws_cognito_1.UserPool(stack, 'Pool', {
    userPoolName: 'myPool',
});
const api = new lib_1.GraphQLApi(stack, 'Api', {
    name: `demoapi`,
    schemaDefinitionFile: path_1.join(__dirname, 'schema.graphql'),
    authorizationConfig: {
        defaultAuthorization: {
            userPool,
            defaultAction: lib_1.UserPoolDefaultAction.ALLOW,
        },
        additionalAuthorizationModes: [
            {
                apiKeyDesc: 'My API Key',
            },
        ],
    },
});
const customerTable = new aws_dynamodb_1.Table(stack, 'CustomerTable', {
    billingMode: aws_dynamodb_1.BillingMode.PAY_PER_REQUEST,
    partitionKey: {
        name: 'id',
        type: aws_dynamodb_1.AttributeType.STRING,
    },
    removalPolicy: core_1.RemovalPolicy.DESTROY,
});
const orderTable = new aws_dynamodb_1.Table(stack, 'OrderTable', {
    billingMode: aws_dynamodb_1.BillingMode.PAY_PER_REQUEST,
    partitionKey: {
        name: 'customer',
        type: aws_dynamodb_1.AttributeType.STRING,
    },
    sortKey: {
        name: 'order',
        type: aws_dynamodb_1.AttributeType.STRING,
    },
    removalPolicy: core_1.RemovalPolicy.DESTROY,
});
const customerDS = api.addDynamoDbDataSource('Customer', 'The customer data source', customerTable);
const orderDS = api.addDynamoDbDataSource('Order', 'The irder data source', orderTable);
customerDS.createResolver({
    typeName: 'Query',
    fieldName: 'getCustomers',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbScanTable(),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultList(),
});
customerDS.createResolver({
    typeName: 'Query',
    fieldName: 'getCustomer',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbGetItem('id', 'id'),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultItem(),
});
customerDS.createResolver({
    typeName: 'Mutation',
    fieldName: 'addCustomer',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbPutItem(lib_1.PrimaryKey.partition('id').auto(), lib_1.Values.projecting('customer')),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultItem(),
});
customerDS.createResolver({
    typeName: 'Mutation',
    fieldName: 'saveCustomer',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbPutItem(lib_1.PrimaryKey.partition('id').is('id'), lib_1.Values.projecting('customer')),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultItem(),
});
customerDS.createResolver({
    typeName: 'Mutation',
    fieldName: 'saveCustomerWithFirstOrder',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbPutItem(lib_1.PrimaryKey
        .partition('order').auto()
        .sort('customer').is('customer.id'), lib_1.Values
        .projecting('order')
        .attribute('referral').is('referral')),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultItem(),
});
customerDS.createResolver({
    typeName: 'Mutation',
    fieldName: 'removeCustomer',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbDeleteItem('id', 'id'),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultItem(),
});
const ops = [
    { suffix: "Eq", op: lib_1.KeyCondition.eq },
    { suffix: "Lt", op: lib_1.KeyCondition.lt },
    { suffix: "Le", op: lib_1.KeyCondition.le },
    { suffix: "Gt", op: lib_1.KeyCondition.gt },
    { suffix: "Ge", op: lib_1.KeyCondition.ge },
];
for (const { suffix, op } of ops) {
    orderDS.createResolver({
        typeName: 'Query',
        fieldName: 'getCustomerOrders' + suffix,
        requestMappingTemplate: lib_1.MappingTemplate.dynamoDbQuery(op('customer', 'customer')),
        responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultList(),
    });
}
orderDS.createResolver({
    typeName: 'Query',
    fieldName: 'getCustomerOrdersFilter',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbQuery(lib_1.KeyCondition.eq('customer', 'customer').and(lib_1.KeyCondition.beginsWith('order', 'order'))),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultList(),
});
orderDS.createResolver({
    typeName: 'Query',
    fieldName: 'getCustomerOrdersBetween',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbQuery(lib_1.KeyCondition.eq('customer', 'customer').and(lib_1.KeyCondition.between('order', 'order1', 'order2'))),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultList(),
});
app.synth();
//# sourceMappingURL=data:application/json;base64,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