"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.VideoTranscoder = void 0;
const path = require("path");
const ec2 = require("@aws-cdk/aws-ec2");
const efs = require("@aws-cdk/aws-efs");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const aws_lambda_event_sources_1 = require("@aws-cdk/aws-lambda-event-sources");
const s3 = require("@aws-cdk/aws-s3");
const sfn = require("@aws-cdk/aws-stepfunctions");
const tasks = require("@aws-cdk/aws-stepfunctions-tasks");
const cdk = require("@aws-cdk/core");
/**
 * @stability stable
 */
class VideoTranscoder extends cdk.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        var _a, _b, _c, _d, _e, _f;
        super(scope, id);
        this.videoBucket = (_a = props.bucket) !== null && _a !== void 0 ? _a : new s3.Bucket(this, 'VideoInputBucket', {
            removalPolicy: cdk.RemovalPolicy.DESTROY,
        });
        /**
         * if the vpc was not given, then create a new vpc with s3 endpoints
         */
        const vpc = (_b = props.vpc) !== null && _b !== void 0 ? _b : new ec2.Vpc(this, 'vpc', {
            maxAzs: 2,
            natGateways: 1,
            subnetConfiguration: [
                {
                    name: 'PublicSubnet',
                    subnetType: ec2.SubnetType.PUBLIC,
                },
                {
                    name: 'VideoTransSubnet',
                    subnetType: ec2.SubnetType.PRIVATE,
                },
            ],
            gatewayEndpoints: {
                S3: {
                    service: ec2.GatewayVpcEndpointAwsService.S3,
                },
            },
        });
        const segmentTime = (_c = props.segmentTime) !== null && _c !== void 0 ? _c : 5;
        const parallelGroupNo = (_d = props.parallelGroupNo) !== null && _d !== void 0 ? _d : 3;
        const lambdaMemorySize = (_e = props.lambdaMemorySize) !== null && _e !== void 0 ? _e : 3008;
        const lambdaTimeOut = (_f = props.lambdaTimeOut) !== null && _f !== void 0 ? _f : 900;
        /**
         * create the ffmeg layer
         */
        const ffmpegLayer = new lambda.LayerVersion(this, 'FfmpegLayer', {
            code: lambda.AssetCode.fromAsset(path.join(__dirname, '../layers/ffmpeg_layer')),
            compatibleRuntimes: [lambda.Runtime.PYTHON_3_6, lambda.Runtime.PYTHON_3_7, lambda.Runtime.PYTHON_3_8],
        });
        /**
         * create the efs file system for lambda
         */
        const efsFile = new efs.FileSystem(this, 'VideoConvertEFS', {
            vpc,
            removalPolicy: cdk.RemovalPolicy.DESTROY,
        });
        const accessPoint = efsFile.addAccessPoint('AccessPoint', {
            createAcl: {
                ownerGid: '1000',
                ownerUid: '1000',
                permissions: '775',
            },
            path: '/lambda',
            posixUser: {
                gid: '1000',
                uid: '1000',
            },
        });
        /**
         * create the statemachine
         */
        const lambdaRole = new iam.Role(this, 'lambdaRole', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
        });
        // lambdaRole.addManagedPolicy(ManagedPolicy.fromAwsManagedPolicyName("service-role/AWSLambdaBasicExecutionRole"));
        lambdaRole.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaVPCAccessExecutionRole'));
        lambdaRole.addManagedPolicy(iam.ManagedPolicy.fromManagedPolicyArn(this, 'AmazonElasticFileSystemClientReadWriteAccess', 'arn:aws:iam::aws:policy/AmazonElasticFileSystemClientReadWriteAccess'));
        lambdaRole.addManagedPolicy(iam.ManagedPolicy.fromManagedPolicyArn(this, 'AWSStepFunctionsFullAccess', 'arn:aws:iam::aws:policy/AWSStepFunctionsFullAccess'));
        lambdaRole.addManagedPolicy(iam.ManagedPolicy.fromManagedPolicyArn(this, 'AmazonS3FullAccess', 'arn:aws:iam::aws:policy/AmazonS3FullAccess'));
        const lambdaCode = lambda.Code.fromAsset(path.join(__dirname, '../lambda-handler'));
        const splitFunction = new lambda.Function(this, 'splitFunction', {
            functionName: 'splitFunction',
            runtime: lambda.Runtime.PYTHON_3_7,
            handler: 'split.main',
            role: lambdaRole,
            code: lambdaCode,
            vpc,
            layers: [ffmpegLayer],
            filesystem: lambda.FileSystem.fromEfsAccessPoint(accessPoint, '/mnt/lambda'),
            timeout: cdk.Duration.seconds(lambdaTimeOut),
            environment: {
                EFS_PATH: '/mnt/lambda',
                PARALLEL_GROUPS: parallelGroupNo.toString(),
                SEGMENT_TIME: segmentTime.toString(),
            },
        });
        const splitJob = new tasks.LambdaInvoke(this, 'SplitVideo', {
            lambdaFunction: splitFunction,
            outputPath: '$.Payload',
        });
        // const Parallel Video Groups
        const transcodeFunction = new lambda.Function(this, 'transcodeFunction', {
            functionName: 'transcodeFunction',
            runtime: lambda.Runtime.PYTHON_3_7,
            handler: 'transcode.main',
            code: lambdaCode,
            role: lambdaRole,
            vpc,
            allowPublicSubnet: true,
            layers: [ffmpegLayer],
            filesystem: lambda.FileSystem.fromEfsAccessPoint(accessPoint, '/mnt/lambda'),
            timeout: cdk.Duration.seconds(lambdaTimeOut),
            memorySize: lambdaMemorySize,
        });
        const videoSegMapGroup0 = new sfn.Map(this, 'VideoSegmentMapGroup0', {
            inputPath: '$.[0]',
            itemsPath: '$.video_segments',
            parameters: {
                'download_dir.$': '$.download_dir',
                'video_segments.$': '$$.Map.Item.Value',
            },
        });
        videoSegMapGroup0.iterator(new tasks.LambdaInvoke(this, 'TranscodeVideoGroup0', {
            lambdaFunction: transcodeFunction,
            outputPath: '$.Payload',
        }));
        const videoSegMapGroup1 = new sfn.Map(this, 'VideoSegmentMapGroup1', {
            inputPath: '$.[1]',
            itemsPath: '$.video_segments',
            parameters: {
                'download_dir.$': '$.download_dir',
                'video_segments.$': '$$.Map.Item.Value',
            },
        });
        videoSegMapGroup1.iterator(new tasks.LambdaInvoke(this, 'TranscodeVideoGroup1', {
            lambdaFunction: transcodeFunction,
            outputPath: '$.Payload',
        }));
        const videoSegMapGroup2 = new sfn.Map(this, 'VideoSegmentMapGroup2', {
            inputPath: '$.[2]',
            itemsPath: '$.video_segments',
            parameters: {
                'download_dir.$': '$.download_dir',
                'video_segments.$': '$$.Map.Item.Value',
            },
        });
        videoSegMapGroup2.iterator(new tasks.LambdaInvoke(this, 'TranscodeVideoGroup2', {
            lambdaFunction: transcodeFunction,
            outputPath: '$.Payload',
        }));
        const parallelGroup = new sfn.Parallel(this, 'Parallel Video Groups');
        parallelGroup.branch(videoSegMapGroup0);
        parallelGroup.branch(videoSegMapGroup1);
        parallelGroup.branch(videoSegMapGroup2);
        const mergeFunction = new lambda.Function(this, 'mergeFunction', {
            functionName: 'mergeFunction',
            code: lambdaCode,
            handler: 'merge.main',
            runtime: lambda.Runtime.PYTHON_3_7,
            role: lambdaRole,
            vpc,
            layers: [ffmpegLayer],
            filesystem: lambda.FileSystem.fromEfsAccessPoint(accessPoint, '/mnt/lambda'),
            timeout: cdk.Duration.seconds(lambdaTimeOut),
            environment: {
                EFS_PATH: '/mnt/lambda',
                MEDIA_BUCKET: this.videoBucket.bucketName,
                Foo: 'Bar',
            },
            memorySize: lambdaMemorySize,
        });
        const mergeJob = new tasks.LambdaInvoke(this, 'mergeJob', {
            lambdaFunction: mergeFunction,
            outputPath: '$.Payload',
        });
        const chain = sfn.Chain.start(splitJob).next(parallelGroup).next(mergeJob);
        const stateMachine = new sfn.StateMachine(this, 'videoTransMachine', {
            definition: chain,
            stateMachineName: 'VideoTransStateMachine',
        });
        const stateArn = stateMachine.stateMachineArn;
        const triggerFunction = new lambda.Function(this, 'triggerFunction', {
            functionName: 'triggerFunction',
            runtime: lambda.Runtime.PYTHON_3_7,
            handler: 'trigger.main',
            role: lambdaRole,
            code: lambdaCode,
            vpc,
            // allowPublicSubnet: true,
            environment: {
                SFN_ARN: stateArn,
            },
            timeout: cdk.Duration.seconds(lambdaTimeOut),
        });
        triggerFunction.addEventSource(new aws_lambda_event_sources_1.S3EventSource(this.videoBucket, {
            events: [s3.EventType.OBJECT_CREATED],
            filters: [{
                    prefix: 'input/',
                    suffix: '.mp4',
                }],
        }));
    }
}
exports.VideoTranscoder = VideoTranscoder;
//# sourceMappingURL=data:application/json;base64,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