# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['lithic',
 'lithic._utils',
 'lithic.resources',
 'lithic.types',
 'lithic.types.shared',
 'lithic.types.shared_params']

package_data = \
{'': ['*']}

install_requires = \
['anyio>=3.5.0',
 'distro>=1.7.0',
 'httpx>=0.23.0',
 'pydantic>=1.9.0',
 'typing-extensions>=4.1.1']

setup_kwargs = {
    'name': 'lithic',
    'version': '0.7.0',
    'description': 'Client library for the lithic API',
    'long_description': '# Lithic Python API Library\n\n> **Migration Guide**\n>\n> We\'ve made some major improvements to how you pass arguments to methods which will require migrating your existing code.\n>\n> If you want to migrate to the new patterns incrementally you can do so by installing `v0.5.0`. This release contains both\n> the new and old patterns with a backwards compatibility layer.\n>\n> You can find a guide to migrating in [this document](#migration-guide).\n\n[![PyPI version](https://img.shields.io/pypi/v/lithic.svg)](https://pypi.org/project/lithic/)\n\nThe Lithic Python library provides convenient access to the Lithic REST API from any Python 3.7+\napplication. It includes type definitions for all request params and response fields,\nand offers both synchronous and asynchronous clients powered by [httpx](https://github.com/encode/httpx).\n\n## Documentation\n\nThe API documentation can be found [here](https://docs.lithic.com).\n\n## Installation\n\n```sh\npip install lithic\n```\n\n## Usage\n\n```python\nfrom lithic import Lithic\n\nlithic = Lithic(\n    # defaults to os.environ.get("LITHIC_API_KEY")\n    api_key="my api key",\n    # defaults to "production".\n    environment="sandbox",\n)\n\ncard = lithic.cards.create(\n    type="SINGLE_USE",\n)\nprint(card.token)\n```\n\nWhile you can provide an `api_key` keyword argument, we recommend using [python-dotenv](https://pypi.org/project/python-dotenv/)\nand adding `LITHIC_API_KEY="my api key"` to your `.env` file so that your API Key is not stored in source control.\n\n## Async Usage\n\nSimply import `AsyncLithic` instead of `Lithic` and use `await` with each API call:\n\n```python\nfrom lithic import AsyncLithic\n\nlithic = AsyncLithic(\n    # defaults to os.environ.get("LITHIC_API_KEY")\n    api_key="my api key",\n    # defaults to "production".\n    environment="sandbox",\n)\n\n\nasync def main():\n    card = await lithic.cards.create(\n        type="SINGLE_USE",\n    )\n    print(card.token)\n\n\nasyncio.run(main())\n```\n\nFunctionality between the synchronous and asynchronous clients are otherwise identical.\n\n## Using Types\n\nNested request parameters are [TypedDicts](https://docs.python.org/3/library/typing.html#typing.TypedDict), while responses are [Pydantic](https://pydantic-docs.helpmanual.io/) models. This helps provide autocomplete and documentation within your editor.\n\nIf you would like to see type errors in VS Code to help catch bugs earlier, set `python.analysis.typeCheckingMode` to `"basic"`.\n\n## Pagination\n\nList methods in the Lithic API are paginated.\n\nThis library provides auto-paginating iterators with each list response, so you do not have to request successive pages manually:\n\n```python\nimport lithic\n\nlithic = Lithic()\n\nall_cards = []\n# Automatically fetches more pages as needed.\nfor card in lithic.cards.list():\n    # Do something with card here\n    all_cards.append(card)\nprint(all_cards)\n```\n\nOr, asynchronously:\n\n```python\nimport asyncio\nimport lithic\n\nlithic = AsyncLithic()\n\n\nasync def main() -> None:\n    all_cards = []\n    # Iterate through items across all pages, issuing requests as needed.\n    async for card in lithic.cards.list():\n        all_cards.append(card)\n    print(all_cards)\n\n\nasyncio.run(main())\n```\n\nAlternatively, you can use the `.has_next_page()`, `.next_page_info()`, or `.get_next_page()` methods for more granular control working with pages:\n\n```python\nfirst_page = await lithic.cards.list()\nif first_page.has_next_page():\n    print(f"will fetch next page using these details: {first_page.next_page_info()}")\n    next_page = await first_page.get_next_page()\n    print(f"number of items we just fetched: {len(next_page.data)}")\n\n# Remove `await` for non-async usage.\n```\n\nOr just work directly with the returned data:\n\n```python\nfirst_page = await lithic.cards.list()\n\nprint(f"page number: {first_page.page}")  # => "page number: 1"\nfor card in first_page.data:\n    print(card.created)\n\n# Remove `await` for non-async usage.\n```\n\n## Nested params\n\nNested parameters are dictionaries, typed using `TypedDict`, for example:\n\n```py\nfrom lithic import Lithic\n\nlithic = Lithic()\n\nlithic.cards.create(\n    foo={\n        "bar": True\n    },\n)\n```\n\n## Handling errors\n\nWhen the library is unable to connect to the API (e.g., due to network connection problems or a timeout), a subclass of `lithic.APIConnectionError` is raised.\n\nWhen the API returns a non-success status code (i.e., 4xx or 5xx\nresponse), a subclass of `lithic.APIStatusError` will be raised, containing `status_code` and `response` properties.\n\nAll errors inherit from `lithic.APIError`.\n\n```python\nfrom lithic import Lithic\n\nlithic = Lithic()\n\ntry:\n    lithic.cards.create(\n        type="an_incorrect_type",\n    )\nexcept lithic.APIConnectionError as e:\n    print("The server could not be reached")\n    print(e.__cause__)  # an underlying Exception, likely raised within httpx.\nexcept lithic.RateLimitError as e:\n    print("A 429 status code was received; we should back off a bit.")\nexcept lithic.APIStatusError as e:\n    print("Another non-200-range status code was received")\n    print(e.status_code)\n    print(e.response)\n```\n\nError codes are as followed:\n\n| Status Code | Error Type                 |\n| ----------- | -------------------------- |\n| 400         | `BadRequestError`          |\n| 401         | `AuthenticationError`      |\n| 403         | `PermissionDeniedError`    |\n| 404         | `NotFoundError`            |\n| 422         | `UnprocessableEntityError` |\n| 429         | `RateLimitError`           |\n| >=500       | `InternalServerError`      |\n| N/A         | `APIConnectionError`       |\n\n### Retries\n\nCertain errors will be automatically retried 2 times by default, with a short exponential backoff.\nConnection errors (for example, due to a network connectivity problem), 409 Conflict, 429 Rate Limit,\nand >=500 Internal errors will all be retried by default.\n\nYou can use the `max_retries` option to configure or disable this:\n\n```python\nfrom lithic import Lithic\n\n# Configure the default for all requests:\nlithic = Lithic(\n    # default is 2\n    max_retries=0,\n)\n\n# Or, configure per-request:\nlithic.with_options(max_retries=5).cards.list(\n    page_size=10,\n)\n```\n\n### Timeouts\n\nRequests time out after 60 seconds by default. You can configure this with a `timeout` option,\nwhich accepts a float or an [`httpx.Timeout`](https://www.python-httpx.org/advanced/#fine-tuning-the-configuration):\n\n```python\nfrom lithic import Lithic\n\n# Configure the default for all requests:\nlithic = Lithic(\n    # default is 60s\n    timeout=20.0,\n)\n\n# More granular control:\nlithic = Lithic(\n    timeout=httpx.Timeout(60.0, read=5.0, write=10.0, connect=2.0),\n)\n\n# Override per-request:\nlithic.with_options(timeout=5 * 1000).cards.list(\n    page_size=10,\n)\n```\n\nOn timeout, an `APITimeoutError` is thrown.\n\nNote that requests which time out will be [retried twice by default](#retries).\n\n## Advanced: Configuring custom URLs, proxies, and transports\n\nYou can configure the following keyword arguments when instantiating the client:\n\n```python\nimport httpx\nfrom lithic import Lithic\n\nlithic = Lithic(\n    # Use a custom base URL\n    base_url="http://my.test.server.example.com:8083",\n    proxies="http://my.test.proxy.example.com",\n    transport=httpx.HTTPTransport(local_address="0.0.0.0"),\n)\n```\n\nSee the httpx documentation for information about the [`proxies`](https://www.python-httpx.org/advanced/#http-proxying) and [`transport`](https://www.python-httpx.org/advanced/#custom-transports) keyword arguments.\n\n# Migration guide\n\nThis section outlines the features that were deprecated in `v0.5.0`, and subsequently removed in `v0.6.0` and how to migrate your code.\n\n## Breaking changes\n\n### TypedDict → keyword arguments\n\nThe way you pass arguments to methods has been changed from a single `TypedDict` to individual arguments. For example, this snippet:\n\n```python\ncard = await client.cards.create({"type": "VIRTUAL"})\n```\n\nNow becomes:\n\n```python\ncard = await client.cards.create(type="VIRTUAL")\n```\n\n#### Migrating\n\nThe easiest way to make your code compatible with this change is to add `**{`, for example:\n\n```diff\n- card = await client.cards.create({\'type\': \'VIRTUAL\'})\n+ card = await client.cards.create(**{\'type\': \'VIRTUAL\'})\n```\n\nHowever, it is highly recommended to completely switch to explicit keyword arguments:\n\n```diff\n- card = await client.cards.create({\'type\': \'VIRTUAL\'})\n+ card = await client.cards.create(type=\'VIRTUAL\')\n```\n\n### Named path arguments\n\nAll but the last path parameter must now be passed as named arguments instead of positional arguments, for example, for a method that calls the endpoint `/account_holders/{account_holder_token}/documents/{document_token}` you would\'ve been able to call the method like this:\n\n```python\ncard = await client.account_holders.retrieve(\n    "account_holder_token", "my_document_token"\n)\n```\n\nBut now you must call the method like this:\n\n```python\ncard = await client.account_holders.retrieve(\n    "my_document_token", account_holder_token="account_holder_token"\n)\n```\n\nIf you have type checking enabled in your IDE it will tell you which parts of your code need to be updated.\n\n### Request options\n\nYou used to be able to set request options on a per-method basis, now you can only set them on the client. There are two methods that you can use to make this easy, `with_options` and `copy`.\n\nIf you need to make multiple requests with changed options, you can use `.copy()` to get a new client object with those options. This can be useful if you need to set a custom header for multiple requests, for example:\n\n```python\ncopied = client.copy(default_headers={"X-My-Header": "Foo"})\ncard = await copied.cards.create(type="VIRTUAL")\nawait copied.cards.provision(card.token, digital_wallet="GOOGLE_PAY")\n```\n\nIf you just need to override one of the client options for one request, you can use `.with_options()`, for example:\n\n```python\nawait client.with_options(timeout=None).cards.create(type="VIRTUAL")\n```\n\nIt should be noted that the `.with_options()` method is simply an alias to `.copy()`, you can use them interchangeably.\n\nYou can pass nearly every argument that is supported by the Client `__init__` method to the `.copy()` method, except for `proxies` and `transport`.\n\n```python\ncopied = client.copy(\n    api_key="...",\n    environment="sandbox",\n    timeout=httpx.Timeout(read=10),\n    max_retries=5,\n    default_headers={\n        "X-My-Header": "value",\n    },\n    default_query={\n        "my_default_param": "value",\n    },\n)\n```\n\n## New features\n\n### Pass custom headers\n\nIf you need to add additional headers to a request you can easily do so with the `extra_headers` argument:\n\n```python\ncard = await client.cards.create(\n    type="VIRTUAL",\n    extra_headers={\n        "X-Foo": "my header",\n    },\n)\n```\n\n### Pass custom JSON properties\n\nYou can add additional properties to the JSON request body that are not included directly in the method definition through the `extra_body` argument. This can be useful when there are in new properties in the API that are in beta and aren\'t in the SDK yet.\n\n```python\ncard = await client.cards.create(\n    type="VIRTUAL",\n    extra_body={\n        "special_prop": "foo",\n    },\n)\n# sends this to the API:\n# {"type": "VIRTUAL", "special_prop": "foo"}\n```\n\n### Pass custom query parameters\n\nYou can add additional query parameters that aren\'t specified in the method definition through the `extra_query` argument. This can be useful when there are any new/beta query parameters that are not yet in the SDK.\n\n```python\ncard = await client.cards.create(\n    type="VIRTUAL",\n    extra_query={\n        "special_param": "bar",\n    },\n)\n# makes the request to this URL:\n# https://api.lithic.com/v1/cards?special_param=bar\n```\n\n## Status\n\nThis package is in beta. Its internals and interfaces are not stable and subject to change without a major semver bump;\nplease reach out if you rely on any undocumented behavior.\n\nWe are keen for your feedback; please email us at [sdk-feedback@lithic.com](mailto:sdk-feedback@lithic.com) or open an issue with questions,\nbugs, or suggestions.\n\n## Requirements\n\nPython 3.7 or higher.',
    'author': 'Lithic',
    'author_email': 'sdk-feedback@lithic.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/lithic-com/lithic-python',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
