# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['combadge',
 'combadge.core',
 'combadge.support',
 'combadge.support.http',
 'combadge.support.httpx',
 'combadge.support.httpx.backends',
 'combadge.support.rest',
 'combadge.support.soap',
 'combadge.support.zeep',
 'combadge.support.zeep.backends']

package_data = \
{'': ['*']}

install_requires = \
['pydantic>=1.10.4,<2.0.0', 'typing-extensions>=4.4.0,<5.0.0']

extras_require = \
{':python_version < "3.10"': ['get-annotations>=0.1.2,<0.2.0'],
 'httpx': ['httpx>=0.23.3,<0.24.0'],
 'zeep': ['zeep>=4.2.1,<5.0.0']}

setup_kwargs = {
    'name': 'combadge',
    'version': '0.1.0.dev1',
    'description': 'Generic API client based on Pydantic',
    'long_description': '# `combadge`\n\n> 📻 Application to the service, please respond!\n\n[![Checks](https://img.shields.io/github/checks-status/kpn/combadge/main)](https://github.com/kpn/combadge/actions/workflows/check.yaml)\n[![Coverage](https://codecov.io/gh/kpn/combadge/branch/main/graph/badge.svg?token=ZAqYAaTXwE)](https://codecov.io/gh/kpn/combadge)\n![Code style](https://img.shields.io/badge/code%20style-black-000000.svg)\n[![License](https://img.shields.io/github/license/kpn/combadge)](LICENSE)\n\n## Features\n\n- [**Pydantic**](https://docs.pydantic.dev/)-based request and response models\n- Automatically derived exception classes\n- Using [**Protocol**](https://peps.python.org/pep-0544/)s to define service classes\n- Built-in backends:\n  - [HTTPX](https://www.python-httpx.org/), sync and async\n  - [Zeep](https://docs.python-zeep.org/en/master/), sync and async\n- Pluggable backends\n\n## 🚀 Quickstart\n\nℹ️ This `README` is [tested](tests/integration/test_readme.py) and should run «as is».\n\n### 🦋 [HTTPX](https://www.python-httpx.org/) backend\n\n```python\n# test_id=quickstart_httpx\n\nfrom typing import List\n\nfrom httpx import Client\nfrom pydantic import BaseModel, Field\nfrom typing_extensions import Annotated, Protocol\n\nfrom combadge.core.binder import bind\nfrom combadge.support.httpx.backends.sync import HttpxBackend\nfrom combadge.support.rest.marks import QueryParam, method, path\n\n\n# 1️⃣ Declare the response models:\nclass CurrentCondition(BaseModel):\n    humidity: int\n    temperature: Annotated[float, Field(alias="temp_C")]\n\n\nclass Weather(BaseModel):\n    current: Annotated[List[CurrentCondition], Field(alias="current_condition")]\n\n\n# 2️⃣ Declare the protocol:\nclass SupportsWttrIn(Protocol):\n    @method("GET")\n    @path("/{in_}")\n    def get_weather(\n        self,\n        *,\n        in_: str,\n        format_: Annotated[str, QueryParam("format")] = "j1",\n    ) -> Weather:\n        raise NotImplementedError\n\n\n# 3️⃣ Bind the service:\nbackend = HttpxBackend(Client(base_url="https://wttr.in"))\nservice = bind(SupportsWttrIn, backend)\n\n# 🚀 Call the service:\nresponse = service.get_weather(in_="amsterdam")\nassert response.current[0].humidity == 71\nassert response.current[0].temperature == 8.0\n```\n\n### 🧼 [Zeep](https://docs.python-zeep.org/en/master/) backend\n\n```python\n# test_id=quickstart_zeep\n\nfrom typing import Literal, Union\n\nimport zeep\nfrom pydantic import BaseModel, Field\nfrom pytest import raises\nfrom typing_extensions import Annotated\n\nfrom combadge.core.interfaces import SupportsService\nfrom combadge.core.response import FaultyResponse, SuccessfulResponse\nfrom combadge.support.http.marks import Body\nfrom combadge.support.soap.marks import operation_name\nfrom combadge.support.zeep.backends.sync import ZeepBackend\n\n\n# 1️⃣ Declare the request model:\nclass NumberToWordsRequest(BaseModel, allow_population_by_field_name=True):\n    number: Annotated[int, Field(alias="ubiNum")]\n\n\n# 2️⃣ Declare the response model:\nclass NumberToWordsResponse(SuccessfulResponse):\n    __root__: str\n\n\n# 3️⃣ Optionally, declare the error response models:\nclass NumberTooLargeResponse(FaultyResponse):\n    __root__: Literal["number too large"]\n\n\n# 4️⃣ Declare the interface:\nclass SupportsNumberConversion(SupportsService):\n    @operation_name("NumberToWords")\n    def number_to_words(self, request: Body[NumberToWordsRequest]) -> Union[NumberTooLargeResponse, NumberToWordsResponse]:\n        ...\n\n\n# 5️⃣ Bind the service:\nclient = zeep.Client(wsdl="tests/integration/wsdl/NumberConversion.wsdl")\nservice = SupportsNumberConversion.bind(ZeepBackend(client.service))\n\n# 🚀 Call the service:\nresponse = service.number_to_words(NumberToWordsRequest(number=42))\nassert response.unwrap().__root__ == "forty two "\n\n# ☢️ Error classes are automatically derived for error models:\nresponse = service.number_to_words(NumberToWordsRequest(number=-1))\nwith raises(NumberTooLargeResponse.Error):\n    response.raise_for_result()\n```\n',
    'author': 'Pavel Perestoronin',
    'author_email': 'pavel.perestoronin@kpn.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/kpn/combadge',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.8.1,<4.0.0',
}


setup(**setup_kwargs)
